/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                  Authors:  Paul Taylor                                */
/*                  Date   :  Oct 95                                     */
/*-----------------------------------------------------------------------*/
/*                 Event RFC and Tilt labelling                          */
/*                                                                       */
/*=======================================================================*/

#include "EST.h"
#include "EST_rfc.h"
#include "EST_pda.h"

void default_rfc_params(EST_Option &op);


int read_track(EST_Track &tr, const EST_String &in_file, EST_Option &al);

int main(int argc, char *argv[])
{
    EST_Track fz, nfz;
    EST_Stream ev_lab, rfc_lab;
    EST_Option al, op;
    EST_StrList files, pos_list;
    EST_String out_file, pstring;
    
    print_help(argc, argv);

    parse_command_line(argc, argv, "-class:s -e:s -lab:s -o:s -ops -otype:s \
                   -range:f  -pos:s -style:s -sp:s -w", files, al);

    default_rfc_params(op);
    init_lib_ops(al, op);

    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    // if input type is waveform, perform smooth F0 tracking
    if (al.present("-w"))
    {
	EST_Track speech;
	EST_Wave sig;
	EST_Stream sil_lab;

	default_pda_options(op);
	if (read_wave(sig, files.first(), al) == -1)
	    exit(-1);

	if (al.present("-lab")) // use label to show silences
	{
	    sil_lab.load(al.val("-lab"));
	    convert_to_broad_class(sil_lab, al.val("-class") ,op);
	    merge_all_label(sil_lab, "pos");
	    speech = label_to_track(sil_lab, op.fval("pda_frame_shift"));
	}
	else 
	{
	    EST_Stream_Item all;
	    all.set_name("pos");
	    all.set_end(sig.end());
	    sil_lab.append(all);
	    cout << sil_lab;
	    speech = label_to_track(sil_lab, op.fval("pda_frame_shift"));
	}

	fz = pda(sig, op);
	fz = smooth_phrase(fz, speech, op);
    }
    else
    {
//	if (fz.load(files.first()) == misc_read_error)
//	    exit (0);
	if (read_track(fz, files.first(), al) == -1)
	    exit(-1);
    }

    pstring = al.present("-pos") ? al.val("-pos"): EST_String("a b ab pos");
    StringtoStrList(pstring, pos_list);

    if (ev_lab.load(al.val("-e")) != format_ok)
	exit(-1);

    convert_to_broad(ev_lab, pos_list);

    rfc_lab = rfc_label(fz, ev_lab, op);

    if (al.val("-style", 0) == "tilt")
    {
	EST_Stream ev_out;
	ev_out = rfc_to_tilt(rfc_lab);
	write_tilt_fields(ev_out);
	ev_out.save(out_file, al.val("-otype", 0), 2);
    }
    else 
	rfc_lab.save(out_file, al.val("-otype", 0), 2);
    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    // general options
    a_list.override_val("sample_rate", al.val("-f", 0));
    a_list.override_val("rfc_start_range", al.val("-f", 0));
    a_list.override_val("rfc_stop_range", al.val("-f", 0));
    a_list.override_val("rfc_search_range", al.val("-range", 0));
}

void print_help(int argc, char *argv[])
{
    for (int i = 0; i < argc; ++i)
	if (strncmp(argv[i],"-help", 2) == 0)
	{
	    cout << "Usage:\n";
	    cout << "ev_param -e label_file input_file <output file> <options>\n";
	    cout << "use \"-\" to make input and output files stdin/out\n";
	    cout << "-h,-?   options help\n";
	    cout << "-class  <string> name of phoneme set for speech detection\n";
	    cout << "-e      intonation label file\n";
	    cout << "-lab    phoneme label file\n";
	    cout << "-ops    printf command line options\n";
	    cout << "-otype  output file type: esps etc\n";
	    cout << "-style  style of output, eg. \"rfc\", \"tilt\" \n";
	    cout << "-sp     speech class (for pause detection)\n";
	    cout << "-w      Input file is a waveform (default: F0 file)\n";

	    exit (0);
	}
}
