/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Paul Taylor and Alan Black                      */
/*             Date   :  May 1996                                        */
/*-----------------------------------------------------------------------*/
/*            EST_Wave Class source file                                 */
/*                                                                       */
/*=======================================================================*/

#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "EST_cutils.h"
#include "EST_Wave.h"
#include "EST_wave_utils.h"
#include "EST_wave_types.h"
#include "EST_TNamedEnum.h"

#include "waveP.h"

int rateconv(short *in,int isize, short **out, int *osize,
	     int in_samp_freq, int out_samp_freq);

const EST_String DEF_FILE_TYPE = "nist";
const EST_String DEF_SAMPLE_TYPE = "short";

static struct EST_TValuedEnumDefinition<EST_file_format_t, const char *,NO_INFO> ff_names[] = {
{no_format, {"undef"}},
{nist_format, {"nist"}},
{audlab_format, {"audlab"}},
{timit_format, {"timit"}},
{ulaw_format, {"ulaw"}},
{headerless_format, {"headerless"}},
{snd_format, {"snd"}},
{esps_format, {"esps"}},
{raw_format, {"raw"}},
{riff_format, {"riff"}},
{aiff_format, {"aiff"}},
{no_format, {0}}
};


static EST_TValuedEnumDefinition<EST_sample_type_t, const char *,NO_INFO> st_names[] = {
{st_unknown,  {"undef"}},
{st_schar, {"schar"}},
{st_uchar, {"uchar"}},
{st_short, {"short"}},
{st_shorten, {"shorten"}},
{st_int, {"int"}},
{st_float, {"float"}},
{st_double, {"double"}},
{st_mulaw, {"mulaw"}},
{st_adpcm, {"adpcm"}},
{st_alaw, {"alaw"}},
{st_ascii, {"ascii"}},
{st_unknown, {0}}
};

static EST_TNamedEnum<EST_sample_type_t> sample_type_map(st_names);
static EST_TNamedEnum<EST_file_format_t> file_format_map(ff_names);

#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TNamedEnum.cc"
template class EST_TNamedEnum<EST_sample_type_t>; 
template class EST_TNamedEnumI<EST_sample_type_t, NO_INFO>; 
template class EST_TValuedEnum<EST_sample_type_t,const char *>; 
template class EST_TValuedEnumI<EST_sample_type_t,const char *, NO_INFO>; 

template class EST_TNamedEnum<EST_file_format_t>; 
template class EST_TNamedEnumI<EST_file_format_t, NO_INFO>; 
template class EST_TValuedEnum<EST_file_format_t,const char *>; 
template class EST_TValuedEnumI<EST_file_format_t,const char *, NO_INFO>; 
#endif

static int get_word_size(EST_String &sample_type)
{
    EST_sample_type_t st;
    st = sample_type_map.token(sample_type);
    return get_word_size(st);
}

EST_Wave::EST_Wave()
{
    default_vals();
}

EST_Wave::EST_Wave(const EST_Wave &w)
{
    default_vals();
    copy(w);
}

void EST_Wave::default_vals()
{
    // real defaults
    s = 0;
    default_sample_rate = 0;
    default_num_channels = 1;

    // system specific defaults
    bo = EST_NATIVE_BO;
    default_sample_type = DEF_SAMPLE_TYPE;

    p_num_channels = default_num_channels;
    p_sample_rate = default_sample_rate;
    p_sample_type = default_sample_type;
    p_file_type = DEF_FILE_TYPE;
    p_num_samples = 0;
    p_name = "";

    p_word_size = get_word_size(p_sample_type);
}

void EST_Wave::free_wave()
{
    if (p_num_samples != 0)
	delete s;
    s = 0;
}

EST_Wave::~EST_Wave()
{
    free_wave();
}

void EST_Wave::copy_setup(const EST_Wave &w)
{
    default_sample_rate = w.default_sample_rate;
    default_sample_type = w.default_sample_type;
    default_num_channels = w.default_num_channels;

    p_file_type = w.p_file_type;
    p_sample_rate = w.p_sample_rate;
    p_sample_type = w.p_sample_type;
    p_name = w.p_name;
    p_word_size = w.p_word_size;
    bo = w.bo;
}

void EST_Wave::copy(const EST_Wave &w)
{
    free_wave();
    copy_setup(w);

    s = new short[w.p_num_samples*w.p_num_channels];
    memmove(s,w.s,w.p_num_samples*sizeof(short)*w.p_num_channels);

    p_num_channels = w.p_num_channels;
    p_num_samples = w.p_num_samples;
}

float EST_Wave::t(int i) const
{
    return (float)i/(float)p_sample_rate;
}

void EST_Wave::set_num_samples(const int n)
{
    p_num_samples = n;
}

void EST_Wave::set_num_channels(const int n)
{
    p_num_channels = n;
}

void EST_Wave::set_sample_rate(const int n)
{
    p_sample_rate = n;
}

/*float EST_Wave::length()
{
    return (float)num_samples() / (float)sample_rate();
}
*/

float EST_Wave::end()
{
    return (float)num_samples() / (float)sample_rate();
}

void EST_Wave::set_data(short *d,int num_samples,int sr,int nc)
{
    clear();
    s = d;
    p_num_samples = num_samples;
    p_sample_rate = sr;
    p_num_channels = nc;
    bo = (EST_BIG_ENDIAN ? bo_big : bo_little);
}

void EST_Wave::set_data_copy(short *d,int num_samples,int sr,int nc)
{
    // Same as above but copy waveform data
    clear();
    s = new short[num_samples*nc];
    memmove(s,d,nc*num_samples*sizeof(short));
    p_num_samples = num_samples;
    p_sample_rate = sr;
    p_num_channels = nc;
    bo = (EST_BIG_ENDIAN ? bo_big : bo_little);
}

void EST_Wave::set_default_num_channels(const int n)
{
    p_num_channels = n;
}

void EST_Wave::set_default_sample_rate(const int n)
{
    p_sample_rate = n;
}

EST_read_status EST_Wave::load(const EST_String &filename, int offset, int length)
{
    EST_read_status r_val;
    EST_sample_type_t s_type;
    /* try all filestypes in succession */

    if ((r_val = load_wave_nist(filename, &s, &p_num_samples,
				&p_num_channels, &p_word_size, &p_sample_rate,
				&s_type, &bo, offset, length)) == format_ok)
    {
	p_file_type = "nist";
    }
    else if ((r_val = load_wave_sd(filename, &s, &p_num_samples,
			      &p_num_channels, &p_word_size, &p_sample_rate,
			      &s_type, &bo, offset, length)) == format_ok)
    {
	p_file_type = "esps";
    }
    else if ((r_val = load_wave_audlab(filename, &s, &p_num_samples,
				  &p_num_channels, &p_word_size,&p_sample_rate,
				  &s_type, &bo, offset, length)) == format_ok)
    {
	p_file_type = "audlab";
    }
    else if ((r_val = load_wave_snd(filename, &s, &p_num_samples,
			       &p_num_channels, &p_word_size, &p_sample_rate,
			       &s_type, &bo, offset, length)) == format_ok)
    {
	p_file_type = "snd";
    }
    else if ((r_val = load_wave_aiff(filename, &s, &p_num_samples,
				&p_num_channels, &p_word_size, &p_sample_rate,
				&s_type, &bo, offset, length)) == format_ok)
	p_file_type = "aiff";
    else if ((r_val = load_wave_riff(filename, &s, &p_num_samples,
				&p_num_channels, &p_word_size, &p_sample_rate,
				&s_type, &bo, offset, length)) == format_ok)
	p_file_type = "riff";
    else
	return r_val;

    p_name = filename;
    p_sample_type = sample_type_map.name(s_type);
    return format_ok;
}

EST_read_status EST_Wave::load_file(const EST_String &filename, 
		     const EST_String filetype, int sample_rate,
		     const EST_String stype, int bov, int nc, int offset,
		     int length)

{
    // Load a waveform from specified file in specified format
    EST_sample_type_t s_type = sample_type_map.token(stype);
    EST_read_status r_val = wrong_format;
    
    clear();
    
    if (filetype == "undef")
	return load(filename, offset, length);
    else if (filetype == "ulaw")
	r_val = load_wave_ulaw(filename, &s, &p_num_samples,
			       &p_num_channels, &p_word_size, &p_sample_rate,
			       &s_type, &bov, offset, length);
    else if (filetype == "raw")
	r_val = load_wave_raw(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			      &s_type, &bov, offset, length,
			     sample_rate, s_type, bov, nc);
    else if (filetype == "nist")
	r_val = load_wave_nist(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			     &s_type, &bo, offset, length);
    else if (filetype == "esps")
	r_val = load_wave_sd(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			     &s_type, &bo, offset, length);
    else if (filetype == "audlab")
	r_val = load_wave_audlab(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			     &s_type, &bo, offset, length);
    else if (filetype == "snd")
	r_val = load_wave_snd(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			     &s_type, &bo, offset, length);
    else if (filetype == "aiff")
	r_val = load_wave_aiff(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			     &s_type, &bo, offset, length);
    else if (filetype == "riff")
	r_val = load_wave_riff(filename, &s, &p_num_samples,
			     &p_num_channels, &p_word_size, &p_sample_rate,
			     &s_type, &bo, offset, length);
    if (r_val == format_ok)
    {
	p_name = filename;
	p_sample_type = sample_type_map.name(s_type);
	p_file_type = filetype;
    }
    else
    {
	p_sample_type = default_sample_type;
	p_file_type = DEF_FILE_TYPE;
    }
    return r_val;
}    

EST_write_status EST_Wave::save(const EST_String &filename, 
				EST_String ftype)
{
    save_wave_func sf;
    const char *file = filename;
    if ((ftype == "") || (ftype == "undef"))
	ftype = p_file_type;

    EST_file_format_t file_format = file_format_map.token(ftype);
    EST_sample_type_t sample_type = sample_type_map.token(p_sample_type);

    switch (file_format)
    {
    case raw_format:
	return save_wave_raw(file, s, 0, p_num_samples, p_num_channels,
			     p_sample_rate, sample_type, bo);
    case ulaw_format:
	resample(8000);		// must be 8000Hz
	return save_wave_ulaw(file, s, 0, p_num_samples, p_num_channels,
			      p_sample_rate, sample_type, bo);
    default:
	if ((sf = file_format_to_save_func(file_format)) == 0)
	{
	    cerr << "Unknown save file type: " << 
		file_format_to_str(file_format) << "\n";
	    return misc_write_error;
	}
	return (sf)(file, s, 0, p_num_samples, p_num_channels,
		    p_sample_rate, sample_type, bo);
    }
}

EST_write_status EST_Wave::save_file(const EST_String &file,
				     EST_String ftype,
				     EST_String stype,
				     int obo, int offset, int length)
{
    save_wave_func sf;

    if ((ftype == "") || (ftype == "undef"))
	ftype = p_file_type;
    if ((stype == "") || (stype == "undef"))
	stype = p_sample_type;


    EST_file_format_t file_type = file_format_map.token(ftype);
    EST_sample_type_t sample_type = sample_type_map.token(stype);
    
    switch(file_type)
    {
    case no_format:
	cerr << "No output file format for waveform\n";
	return misc_write_error;
    case ulaw_format:
	return save_wave_ulaw(file, s, offset, length,
			      p_num_channels, p_sample_rate,
			      sample_type, obo);
    case raw_format:
	return save_wave_raw(file, s, offset, length,
			     p_num_channels, p_sample_rate,
			     sample_type, obo);
    default:
	if ((sf = file_format_to_save_func(file_type)) == 0)
	{
	    cerr << "Unknown save file type: " << 
		file_format_to_str(file_type) << "\n";
	    return misc_write_error;
	}
	return (sf)(file, s, 0, p_num_samples, p_num_channels,
		    p_sample_rate, sample_type, obo);
    }
}

void EST_Wave::resample(int new_freq)
{
    // Rseample wave to new sample rate
    int new_num_samples;
    short *o;
    
    // check weird frequencies
    if (new_freq != p_sample_rate)
    {
	if (rateconv(s,p_num_samples,&o,&new_num_samples,
		     p_sample_rate,new_freq) == -1)
	{
	    cerr << "rateconv: failed to convert\n";
	    return;
	}
	delete s;
	s = o;
	p_num_samples = new_num_samples;
	p_sample_rate = new_freq;
    }
    
}

void EST_Wave::rescale(float gain,int normalize)
{
    int i,ns;
    float factor = gain;
    
    if (normalize)
    {
	int max = 0;
	for (i = 0; i < p_num_samples; ++i)
	    if (abs(s[i]) > max)
		max = abs(s[i]);
	factor *= 32766.0/(float)max;
    }
    
    for (i = 0; i < p_num_samples; ++i)
    {
	ns = (int) ((float)s[i] * factor);
	if (ns < -32766)
	    s[i] = -32766;
	else if (ns > 32766)
	    s[i] = 32766;
	else
	    s[i] = ns;
    }
}

void EST_Wave::resize(int samples)
{
    // Resize (and zero) the wave
    
    if (p_num_samples < samples)
    {
	short *o = new short[samples];
	memmove(o,s,sizeof(short)*p_num_samples);
	memset(&o[p_num_samples],0,sizeof(short)*(samples-p_num_samples));
	delete s;
	s = o;
    }
    p_num_samples = samples;
}


// REORG - now redundant?? Can use Richard's map thing
EST_file_format_t str_to_file_type(const EST_String &type)
{
    //  Return file_typer from string
    
    if ((type == "nist") || (type == "timit"))
	return nist_format;
    else if ((type == "vox") || (type == "audlab"))
	return audlab_format;
    else if (type == "ulaw")
	return ulaw_format;	/* should be raw */
    else if ((type == "esps") || (type == "sd"))
	return esps_format;
    else if ((type == "snd") || (type == "sun") || (type == "next"))
	return snd_format;
    else if ((type == "wav") || (type == "riff"))
	return riff_format;
    else if (type == "aiff")
	return aiff_format;
    else if (type == "raw")
	return raw_format;
    else
    {
	cerr << "Unknown file format: \"" << type << "\"\n";
	return no_format;
    }
}

EST_Wave &EST_Wave::operator =(const EST_Wave &w)
{
    copy(w);
    return *this;

/* REORG - delet
    s = new short[w.p_num_samples*w.p_num_channels];
    memmove(s,w.s,w.p_num_samples*sizeof(short)*w.p_num_channels);

    bo=w.bo;
    p_num_channels = w.p_num_channels;
    p_num_samples = w.p_num_samples;
    p_word_size = w.p_word_size;
    default_sample_rate = w.default_sample_rate;
    default_num_channels = w.default_num_channels;
    default_sample_type = w.default_sample_type;
    p_file_type = w.p_file_type;
    p_sample_rate = w.p_sample_rate;
    p_sample_type = w.p_sample_type;
    p_name = w.p_name;
    return *this;
*/    

}

EST_Wave &EST_Wave::operator +=(const EST_Wave &w)
{
    short *t_data;
    short *w_data;
    int w_samples, w_num_channels;
    EST_Wave w2;

    if (w.p_num_channels != p_num_channels)
    {
	cerr << "Cannot concatenate waveforms with differing numbers of channels\n";
	return *this;
    }

    if (p_sample_rate != w.sample_rate())
    {
	// I'm not conivnced resampling works with multi channels data
	w2 = w;
	w2.resample(p_sample_rate);
	w_data = w2.s;
	w_samples = w2.num_samples();
	w_num_channels = w2.num_channels();
    }
    else
    {
	w_data = w.s;
	w_samples = w.num_samples();
	w_num_channels = w.num_channels();
    }

    t_data = new short[(p_num_samples*p_num_channels)+
		       (w.p_num_samples*w.p_num_channels)];
    memmove(t_data,s,p_num_samples*p_num_channels*sizeof(short));
    memmove(&t_data[(p_num_samples*p_num_channels)],w_data,
	  (w.p_num_samples*w.p_num_channels)*sizeof(short));
    delete s;
    s = t_data;
    p_num_samples += w.p_num_samples;

    return *this;
}

// add wave s to existing wave in parallel to create multi-channel wave.
EST_Wave &EST_Wave::operator |=(const EST_Wave &iw)
{
    int i,k,newsamples, newchannels;
    short *t_data;
    EST_Wave w=iw;   // this *is* inefficient

    w.resample(p_sample_rate);  // far too difficult otherwise

    // always take number of samples in this as new
    newsamples = p_num_samples;
//    newsamples = (p_num_samples > w.p_num_samples ? 
//		  p_num_samples : w.p_num_samples);
    newchannels = p_num_channels+w.p_num_channels;
    t_data = new short[newsamples*(p_num_channels+w.p_num_channels)];
    memset(t_data,0,
	   sizeof(short)*newsamples*(p_num_channels+w.p_num_channels));

    for (i=0; i < newsamples; i++)
    {
	for (k=0; k < p_num_channels; k++)
	    if (i < p_num_samples)
		t_data[(i*newchannels)+k] = a(i,k+1);
	for (k=0; k < w.p_num_channels; k++)
	    if (i < w.p_num_samples)
		t_data[(i*newchannels)+p_num_channels+k] = w(i,k+1);
    }

    p_num_samples = newsamples;
    p_num_channels = newchannels;
    delete s;
    s = t_data;

    return *this;
}

ostream& operator << (ostream& s, const EST_Wave &sig)
{
    for (int i = 0; i < sig.num_samples(); ++i)
	s << sig(i) << endl;
    
    return s;
}
