/*************************************************************************/
/*                                                                       */
/*                            Richard Tobin                              */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Richard Tobin                           */
/*                     Date   :  July 1996                               */
/*-----------------------------------------------------------------------*/
/*               Matrix inversion                                        */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "EST_cutils.h"

static void swap_rows(int n, double *row1, double *row2);
static void scale_row(int n, double *row, double scale);
static void subtract_scaled_row(int n,double *row1,double *row2,double scale);

/* Invert mat, leaving inverse in inv.  Returns 0 normally, -1 if mat
   is singular.  mat is trashed (actually set to identity if inversion
   succeeds). */

int mat_inv(int n, double **mat, double **inv)
{

    int i, j;

    /* Set inv to identity. */

    for(i=0; i<n; i++)
	for(j=0; j<n; j++)
	    inv[i][j] = 0.0;
    for(i=0; i<n; i++)
	inv[i][i] = 1.0;

    /* We perform manipulations on the matrix to make it into an identity
       matrix, and perform the same manipulations on an identity matrix,
       which magically turns it into the inverse. */

    /* We make the diagonal element on each row be one by scaling the
       row.  Then we make the other rows have zero in that column by
       subtracting multiples of the current.
       I think this is Gauss-Jordan elimination. */

    for(i=0; i<n; i++)
    {
	double best, scale;
	int bestrow = 0;	/* initialize only to keep compiler quiet */

	/* Find row with largest abs value in column i
	   (this reduces numerical instability). */

	best = 0.0;
	for(j=i; j<n; j++)
	    if(fabs(mat[j][i]) > best)
	    {
		best = fabs(mat[j][i]);
		bestrow = j;
	    }

	if(best == 0.0)		/* Singular, I think. */
	    return -1;

	/* Move the best row to row i */

	swap_rows(n, mat[bestrow], mat[i]);
	swap_rows(n, inv[bestrow], inv[i]);

	/* Make the value in column i equal to one by dividing. */
	
	scale = 1.0 / mat[i][i];
	scale_row(n, mat[i], scale);
	scale_row(n, inv[i], scale);

	/* Make every other row have zero in column i by subtracting
	   a multiple of row i. */

	for(j=0; j<n; j++)
	{
	    if(j == i)
		continue;
	    scale = mat[j][i];
	    subtract_scaled_row(n, mat[j], mat[i], scale);
	    subtract_scaled_row(n, inv[j], inv[i], scale);
	}
    }

    return 0;			/* Success! */
}

/* Swap row1 and row2 */

static void swap_rows(int n, double *row1, double *row2)
{
    double *temp;

    temp = walloc(double,n);

    memcpy(temp, row1, n * sizeof(double));
    memcpy(row1, row2, n * sizeof(double));
    memcpy(row2, temp, n * sizeof(double));

    wfree(temp);
}

/* row *= scale */

static void scale_row(int n, double *row, double scale)
{
    int i;

    for(i=0; i<n; i++)
	row[i] *= scale;
}

/* row1 -= (row2 * scale) */

static void subtract_scaled_row(int n,double *row1,double *row2,double scale)
{
    int i;

    for(i=0; i<n; i++)
	row1[i] -= (row2[i] * scale);
}

