open Uri

type request =
  { 
    www_link : Hyper.link;	  (* the link that produced this request *)
    www_url : Url.t;	  (* parsed version *)
    www_fragment : string option; (* because viewer is passed down *)
    mutable www_auth : (string * string) list;
    mutable www_headers : string list;
    mutable www_logging : string -> unit;
    mutable www_error : Error.t

  }

exception Invalid_request of request * string

let make hlink =
  let absuri = Hyper.resolve hlink in {
    www_link = hlink;
    www_url = Lexurl.make absuri.uri_url;(* should not fail ? *)
    www_fragment = absuri.uri_frag;
    www_auth = [];
    www_headers = [];
    www_logging = (fun _ -> ());
    www_error = Error.default
     }
 


(* We need to keep a trace of pending connections, since there is a race
   condition when the user clicks twice rapidly on an anchor. If the second
   click occurs before the document is added to the cache, (e.g. because we
   are waiting for the headers), then the document will be retrieved twice.
   And naturally, for documents that don't enter the cache we always will
   duplicate connexions.
   Retrieve.f is a safe place to add the request to the list of pending
   connexions, because it is synchronous.
   Removing an active connexion must take place when we close the 
   dh.document_fd.
*)
module ConnectionSet =
  Set.Make(struct type t = Url.t let compare = compare end)

let active_connexions = ref ConnectionSet.empty
let is_active_cnx url = ConnectionSet.mem url !active_connexions
and add_active_cnx url = 
  active_connexions := ConnectionSet.add url !active_connexions
and rem_active_cnx url =
  active_connexions := ConnectionSet.remove url !active_connexions

