/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */
package com.kvisco.xsl;

import java.util.Enumeration;
import java.util.Hashtable;

import org.w3c.dom.*;
import com.kvisco.util.List;



/**
 * This class represents an XSLObject in the style tree. It is the most
 * basic of all XSLObjects and contains common functionality
 * across different XSLObjects. I originally had this implemented
 * as an extended W3C DOM XML Element, but due to problems with 
 * extending different implementations, and to make it more
 * cross-DOM accessible I've chosen an this approach. Since this once
 * was a DOM Element, you will notice many of the methods are
 * very DOM-like.
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class XSLObject {
    
    
    private static final String DEFAULT_NAME = "xsl:element";
    
    public static final short    APPLY_IMPORTS         =  0;
    public static final short    APPLY_TEMPLATES       =  1;
    public static final short    ARG                   =  2;
    public static final short    ATTRIBUTE             =  3;
    public static final short    ATTRIBUTE_SET         =  4;
    public static final short    CALL_TEMPLATE         =  5;
    public static final short    CDATA                 =  6;
    public static final short    CHOOSE                =  7;
    public static final short    COMMENT               =  8;
    public static final short    CONTENTS              =  9;
    public static final short    COPY                  = 10;
    public static final short    COPY_OF               = 11;
    public static final short    ELEMENT               = 12;
    public static final short    FOR_EACH              = 13;
    public static final short    FUNCTIONS             = 14;
    public static final short    ID                    = 15;
    public static final short    IF                    = 16;
    public static final short    IMPORT                = 17;
    public static final short    INCLUDE               = 18;
    public static final short    KEY                   = 19;
    public static final short    LITERAL               = 20;
    public static final short    LOCALE                = 21;
    public static final short    MESSAGE               = 22;
    public static final short    NUMBER                = 23;
    public static final short    OTHERWISE             = 24;
    public static final short    OUTPUT                = 25;
    public static final short    PARAM                 = 26;
    public static final short    PARAM_VARIABLE        = 27;
    public static final short    PI                    = 28;
    public static final short    PRESERVE_SPACE        = 29;
    public static final short    SORT                  = 30;
    public static final short    STRIP_SPACE           = 31;
    public static final short    STYLESHEET            = 32;
    public static final short    TEMPLATE              = 33;    
    public static final short    TEXT                  = 34;
    public static final short    USE                   = 35;
    public static final short    VALUE_OF              = 36;
    public static final short    VARIABLE              = 37;
    public static final short    WHEN                  = 38;
    
    // Proprietary XSL elements
    public static final short    ENTITY_REF            = 39;
    public static final short    SCRIPT                = 40;
    
    //-- total number of xsl element types
    private static final short   MAX_TYPE              = 41;
    
    /**
     * The type of this XSLObject
    **/
    private short type = LITERAL; // default
    
    /**
     * AttributeValueTemplate cache for performance
    **/
    private Hashtable avtCache;

    private List children = null;
    
    private Hashtable attributes = null;
    
    private List readOnlyAttrs = null;
    
    private boolean allowActions = true;
    
    private XSLStylesheet parentStylesheet = null;
    
    private XSLObject parent = null;
    
    private String typeName = "xsl:object";
    
    private static Hashtable typeNames = null;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    
    /**
     * Creates an XSLObject using the specified type
     * @param parentStylesheet the owner XSLStylesheet 
     * of the new Element
     * @param type the type of XSLObject that the new instance
     * represents
    **/
    public XSLObject(XSLStylesheet parent, short type) {
        super();
        if (typeNames == null) typeNames = buildNameHash();
        this.type = type;
        this.typeName = getNameFromType(type);
        this.parentStylesheet = parent;
        attributes = new Hashtable();
        children   = new List(0);
        readOnlyAttrs = new List(0);
        // create a new attribute value template Cache
        // for performace.
        avtCache = new Hashtable();
    } //-- XSLObject
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Appends the given XSLObject to this XSLObject's list of
     * actions
     * @param xslObject the XSLObject to add to this XSLObject's
     * list of actions
     * @return true if the given XSLObject has been added to this
     * XSLObject otherwise false 
    **/
    public boolean appendAction(XSLObject xslObject) {
        if (!allowActions) return false;
        if (handleAction(xslObject)) return true;
        
        // normalize text
        if (xslObject.getType() == XSLObject.TEXT) {
            XSLText xslText = (XSLText)xslObject;
            // check last action added
            if (children.size() > 0) {
                XSLObject action = (XSLObject)children.get(children.size()-1);
                if (action.getType() == XSLObject.TEXT) {
                    ((XSLText)action).appendData(xslText.getData());
                    return true;
                }
            }
        }
        children.add(xslObject);
        xslObject.setParentStylesheet(this.parentStylesheet);
        xslObject.setParent(this);
        return true;
    } //-- appendAction
    
    /**
     * Returns the list of actions for this XSLObject
     * @return the list of actions for this XSLObject
    **/
    public List getActions() {
        return (List)children.clone();
    } //-- getActions
    
    /**
     * Returns the value of the attribute whose name is equal to the given name.
     * @return the value of the attribute whose name is equal to the given name 
     * or null if no attribute exist's with such a name.
    **/
    public String getAttribute(String name) {
        if (name != null) return (String)attributes.get(name);
        return null;
    } //-- getAttribute
    
    /**
	 * Returns the value of the specified attribute as an AttributeValueTemplate
	 * @return the value of the specified attribute as an AttributeValueTemplate
	 * @exception XSLException when the Attribute is not a valid 
	 * AttrubueValueTemplate
	**/
    public AttributeValueTemplate getAttributeAsAVT(String name) 
        throws XSLException
    {
        if (name == null) return null;
        
        String attValue = getAttribute(name);
        
        AttributeValueTemplate avt = null;
        
        if ((attValue != null) && (attValue.length() > 0)) {
            // look in cache first
            avt = (AttributeValueTemplate) avtCache.get(attValue);
            if (avt == null) {
                try {
                avt = new AttributeValueTemplate(attValue);
                // add to cache for performace
                // Do we need to clean cache? Yes if we are reusing
                // XSLObjects. No if not. I am currently not 
                // reusing XSLObjects so I am not doing any house 
                // cleaning. This could lead to memory problems if 
                // XSLObjects are reused heavily.
                avtCache.put(attValue, avt);
                }
                catch(InvalidExprException iee) {
                    throw new XSLException
                        (XSLException.INVALID_ATTRIBUTE_VALUE_TEMPLATE,
                            iee.getMessage());
                }
            }
        }
       return avt;
    } //-- getAttributeAsAVT
    
    /**
     * Returns an Enumeration of the names of the attributes of this
     * XSLObject
     * @return an Enumeration of the names of the attributes of this
     * XSLObject
    **/
    public Enumeration getAttributeNames() {
        return attributes.keys();
    } //-- getAttributeNames
    
    /**
     * Returns the nearest ancestor of this XSLObject that is of the given
     * type.
     * @param type the type of ancestor to search for
     * @return the nearest ancestor of this XSLObject that is of the given
     * type.
    **/
    public XSLObject getNearestAncestor(short type) {
        if (parent == null) return null;
        if (parent.getType() == type) return parent;
        else return parent.getNearestAncestor(type);
    } //-- getNearestAncestor
    
	/**
	 * Returns the String value of a DOM Node.
	 * @return the String value of a DOM Node.
	 * @see org.w3c.dom.Node
	**/
    public static String getNodeValue(Node node) {
        if (node == null) return "";
        switch (node.getNodeType()) {
            
            case Node.DOCUMENT_NODE:
                return getNodeValue(((Document)node).getDocumentElement());
            case Node.DOCUMENT_FRAGMENT_NODE:
                StringBuffer sb = new StringBuffer();
                NodeList nl = node.getChildNodes();
                for (int i = 0; i < nl.getLength(); i++) {
                    sb.append(getNodeValue(nl.item(i)));
                }
                return sb.toString();
            // elements
            case Node.ELEMENT_NODE:
                return XSLObject.getText((Element)node);
            // attributes
            case Node.ATTRIBUTE_NODE:
                return ((Attr)node).getValue();
            // Text and Character Data
            case Node.CDATA_SECTION_NODE:
            case Node.TEXT_NODE:
                return ((Text)node).getData();
            // Comments
            case Node.COMMENT_NODE:
                return ((Comment)node).getData();
            // Processing Instructions
            case Node.PROCESSING_INSTRUCTION_NODE:
                return ((ProcessingInstruction)node).getData();
            default:
                break;
        }
        return "";
    } //-- getNodeValue
    
    
    protected XSLObject getParent() {
        return parent;
    } //-- getParent
    
    /**
     * Returns the parent XSLStylesheet of this XSLObject
     * @return the parent XSLStylesheet of this XSLObject
    **/
    public XSLStylesheet getParentStylesheet() {
        return this.parentStylesheet;
    } //-- getParentStylesheet
    
    /**
     * Retrieves the text of an Element
     * @return the text of the given Element
     * @see org.w3c.dom.Element
    **/
    public static String getText(Element element) {
        if (element == null) return null;
        
        NodeList list = element.getChildNodes();
        Node node;
        int size = list.getLength();
        
        StringBuffer sb = new StringBuffer();
        
        for (int i = 0; i < size; i++) {
            node = list.item(i);
            switch(node.getNodeType()) {
                case Node.TEXT_NODE:
                case Node.CDATA_SECTION_NODE:
                    if (size == 1) return ((Text)node).getData();
                    else sb.append(((Text)node).getData());
                    break;
                case Node.ELEMENT_NODE:
                    if (size == 1) return XSLObject.getText((Element)node);
                    else sb.append(XSLObject.getText((Element)node));
                    break;
                default:
                    break;
            }
        }
        return sb.toString();
    } //-- getText
    
    /**
     * Returns the type of XSLObject this Object represents
     * @return the type of XSLObject that this Object represents
    **/
    public final short getType() {
        return type;
    } //-- getType
    
    /**
     * Returns the type of XSL Object that has the given name
     * @param name the name the XSLObject
     * @return the type of XSL Object that has the given name
    **/
    public static short getTypeFromName(String name) {
        Short sh = (Short)typeNames.get(name);
        if (sh == null) return LITERAL;
        else return sh.shortValue();
    } //-- getTypeFromName
    
    /**
     * Returns the name of this XSLObject
     * @return the name of this XSLObject
    **/
    public String getTypeName() {
        return typeName;
    } //-- getTypeName
    
    public void setAllowActions(boolean allow) {
        this.allowActions = allow;
    } //-- allowChildActions
    
    public void setTypeName(String name) {
        this.typeName = name;
    } //-- setTypeName
    
    /**
     * Sets the attribute with the given name to the given value.
     * @param name the name of the attribute to set
     * @param value the value to set the attribute to
     * @throws XSLException if this XSLObject does not allow attributes
     * with the given name, or if the attribute is read only
    **/
    public void setAttribute(String name, String value) 
        throws XSLException
    {
        if ((name != null) && (value != null)) {
            if (readOnlyAttrs.contains(name)) {
                StringBuffer err = new StringBuffer("The attribute '");
                err.append(name);
                err.append("' has been set to read only for this ");
                err.append(getTypeName());
                err.append(" and cannot be changed.");
                throw new XSLException(err.toString());
            }
            else {
                attributes.put(name,value);
            }
        }
    } //-- setAttribute
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    /**
     * Helps keep memory usage to a minimum, by freeing memory
     * from the children list
     * @param deep if true will recursively call this method on
     * all the children as well
    **/
    protected void freeUnusedMemory(boolean deep) {
        children.trimToSize();
        readOnlyAttrs.trimToSize();
        if (deep) {
            for (int i = 0; i < children.size(); i++) {
                XSLObject xslObj = (XSLObject)children.get(i);
                xslObj.freeUnusedMemory(true);
            }
        }
    } //-- freeUnusedMemory
    
    /**
     * Allows subclasses to handle the append operation of the 
     * given XSLObject. Called from #appendAction
     * this method should return true, if there is nothing left to do,
     * otherwise false, if the super class should handle the append
    **/
    protected boolean handleAction(XSLObject xslObject) {
        return false;
    } //--  handleAction
    
    /**
     * Copies the attributes from the given element to
     * this XSLObject. 
    **/
    protected void copyAttributes(Element element) 
        throws XSLException
    {
        NamedNodeMap atts = element.getAttributes();
        // check for null for Oracle Parser
        if (atts == null) return;
        Attr attr;
        for (int i = 0; i < atts.getLength(); i++) {
            attr = (Attr) atts.item(i);
            setAttribute(attr.getName(), attr.getValue());
        }
        
    } //-- copyAttributes
    
    /**
     * Copies the attributes from the given XSLObject to
     * this XSLObject.
    **/
    protected void copyAttributes(XSLObject xslObj) {
        xslObj.copyAttributesInto(this.attributes);
    } //-- copyAttributes
    
    /**
     * Copies the actions of the XSLObject argument and
     * adds them to this XSLObject
     * @param xslObject the XSLObject to copy actions from
    **/
    protected void copyActions(XSLObject xslObject) {
        List list = xslObject.getActions();
        for (int i = 0; i < list.size(); i++) {
            XSLObject action = (XSLObject)list.get(i);
            this.appendAction(action);
        }
    } //-- copyActions
    
    protected void makeAttrReadOnly(String name) {
        readOnlyAttrs.add(name);
    } //-- makeAttrReadOnly
    
    protected void setParent(XSLObject parent) {
        this.parent = parent;
    } //-- setParent
    
    protected void setParentStylesheet(XSLStylesheet parentStylesheet) {
        this.parentStylesheet = parentStylesheet;
    } //-- setParentStylesheet
    
    protected void copyAttributesInto(Hashtable ht) {
        Enumeration enum = attributes.keys();
        while (enum.hasMoreElements()) {
            Object key = enum.nextElement();
            ht.put(key,attributes.get(key));
        }
    } //-- copyAttributesInto
    
      //-------------------/
     //- private methods -/
    //-------------------/
    
    /**
     * Builds a hashtable of all XSL Element names
    **/
    private static Hashtable buildNameHash() {
        Hashtable ht = new Hashtable();
        for (short i = 0; i < MAX_TYPE; i++) {
            ht.put(getNameFromType(i), new Short(i));
        }
        return ht;
    } //-- buildNameHash
    
    /**
     * Returns the Name of XSL Objects that have the given Type
     * @param type the XSL Object type
    **/
    private static String getNameFromType(short type) {
        
        switch (type) {
            case APPLY_IMPORTS:
                return Names.APPLY_IMPORTS;
            case APPLY_TEMPLATES:
                return Names.APPLY_TEMPLATES;
            case ATTRIBUTE:
                return Names.ATTRIBUTE;
            case ATTRIBUTE_SET:
                return Names.ATTRIBUTE_SET;
            case CALL_TEMPLATE:
                return Names.CALL_TEMPLATE;
            case CDATA:
                return Names.CDATA;
            case CHOOSE:
                return Names.CHOOSE;
            case COMMENT:
                return Names.COMMENT;
            case COPY:
                return Names.COPY;
            case COPY_OF:
                return Names.COPY_OF;
            case ELEMENT:
                return Names.ELEMENT;
            case ENTITY_REF:
                return Names.ENTITY_REF;
            case FOR_EACH:
                return Names.FOR_EACH;
            case FUNCTIONS:
                return Names.FUNCTIONS;
            case ID:
                return Names.ID;
            case IF:
                return Names.IF;
            case IMPORT:
                return Names.IMPORT;
            case INCLUDE:
                return Names.INCLUDE;
            case KEY:
                return Names.KEY;
            case LOCALE:
                return Names.LOCALE;
            case MESSAGE:
                return Names.MESSAGE;
            case NUMBER:
                return Names.NUMBER;
            case OTHERWISE:
                return Names.OTHERWISE;
            case OUTPUT:
                return Names.OUTPUT;
            case PARAM:
                return Names.PARAM;
            case PARAM_VARIABLE:
                return Names.PARAM_VARIABLE;
            case PI:
                return Names.PI;
            case PRESERVE_SPACE:
                return Names.PRESERVE_SPACE;
            case SORT:
                return Names.SORT;
            case STRIP_SPACE:
                return Names.STRIP_SPACE;
            case STYLESHEET:
                return Names.STYLESHEET;
            case TEMPLATE:    
                return Names.TEMPLATE;    
            case TEXT:
                return Names.TEXT;
            case USE:
                return Names.USE;
            case VALUE_OF:
                return Names.VALUE_OF;
            case VARIABLE:
                return Names.VARIABLE;
            case WHEN:
                return Names.WHEN;
            case SCRIPT:
                return Names.SCRIPT;
            default:
                return DEFAULT_NAME;
        }
    } //-- getNameFromType
    
    /* */
} //-- XSLObject