/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xsl;

import org.w3c.dom.*;

/**
 * Represents an xsl:sort element
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class XSLSort extends XSLObject {
    
    /**
     * The case-order attribute
    **/
    public static final String CASE_ORDER_ATTR  = "case-order";

    /**
     * The data-type attribute
    **/
    public static final String DATA_TYPE_ATTR   = "data-type";

    /**
     * The lang attribute
    **/
    public static final String LANG_ATTR        = "lang";

    /**
     * The order attribute
    **/
    public static final String ORDER_ATTR       = "order";

    /**
     * The select attribute
    **/
    public static final String SELECT_ATTR      = "select";


    /**
     * The ascending order value
    **/
    public static final String ASCENDING_ORDER  = "ascending";

    /**
     * The descending order value
    **/
    public static final String DESCENDING_ORDER  = "descending";


    /**
     * The number data-type value
    **/
    public static final String NUMBER_TYPE  = "number";

    /**
     * The text data-type value
    **/
    public static final String TEXT_TYPE  = "text";
    

    private SelectExpr select = null;
    private String pattern    = null;
    
      //----------------/
     //- Constructors -/
    //----------------/

    /**
     * Creates a new XSLSort
    **/
    public XSLSort(XSLStylesheet parentStylesheet) {
        super(parentStylesheet, XSLObject.SORT);
        // Set default attributes
        try {
            setAttribute(Names.SELECT_ATTR,".");
        }
        catch(XSLException xslException) {};
    } //-- XSLSort

      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Retrieves the selectExpr for this XSLSort.
    **/
    public SelectExpr getSelectExpr() throws InvalidExprException {
        
        String attValue = getAttribute(Names.SELECT_ATTR);
        if (attValue == null) attValue = "";
        
        // make sure we synchronize select pattern
        if ((!attValue.equals(pattern)) || (select == null)) {
            pattern = attValue;
            // Check for empty string to solve changes from 
            // XML4J 1.1.9 to 1.1.14 as suggested by Domagoj Cosic.
            if ((attValue != null) && (attValue.length() > 0))
                select = ExpressionParser.createSelectExpr(attValue);
        }
        return select;
    } //-- getSelectExpr
    
    public void setAttribute(String name, String value) 
        throws XSLException
    {
        if (name == null) return;
        if (name.equals(Names.SELECT_ATTR)) {
            try {
                setSelectExpr(ExpressionParser.createSelectExpr(value));
            }
            catch(InvalidExprException iee) {
                throw new XSLException("Invalid SelectExpr in xsl:sort - " + iee.getMessage());
            }
        }
        
        super.setAttribute(name,value);
    }
    /**
     * sets the SelectExpr for this XSLSort
    **/
    public void setSelectExpr(SelectExpr selectExpr) {
        
        if (selectExpr == null) {
            try { this.select = ExpressionParser.createSelectExpr("."); }
            catch(InvalidExprException iee) {};
        }
        else select = selectExpr;
    } //-- setSelectExpr

} //-- XSLSort