/*
 * File: RomanNumberFormat.java
 *
 * $Id: RomanNumberFormat.java,v 1.2 1999/01/07 15:25:48 ttaylor Exp $
 */

package org.mitre.tjt.text;

import java.text.FieldPosition;
import java.text.NumberFormat;
import java.text.ParsePosition;

/**
 * This class implements a NumberFormat that generates Roman Numerals that
 * represent the integer value provided.
 *
 * @author Tim Taylor
 * @version 1.0
 */
public class RomanNumberFormat extends NumberFormat {
  // Roman Number Symbols
  /** value of 1 */
  public static final int I = 1;
  /** value of 5 */
  public static final int V = 5;
  /** value of 10 */
  public static final int X = 10;
  /** value of 50 */
  public static final int L = 50;
  /** value of 100 */
  public static final int C = 100;
  /** value of 500 */
  public static final int D = 500;
  /** value of 1000 */
  public static final int M = 1000;

  /** Contains all the valid characters in this set.  Used for parsing. */
  private static final String validChars = "ivxlcdm";
  /**
   * Contains the corresponding value of the character.  Use characters
   * index in <code>validChars</code> to index into this array.
   */
  private static final int[]  charValues = {I, V, X, L, C, D, M};
  
  /** If true, the returned string will use uppercase letters. */
  private boolean upperCaseFormat;

  /**
   * Creates a new RomanNumberSequence object.  The formats generated by this
   * object will use uppercase letters if <code>upperCase</code> is true,
   * lower case otherwise.
   * 
   * @param upperCase True if the format methods should use upper case
   *  letters, (ie: IX).  If false, lowercase letters will be used.
   */
  public RomanNumberFormat(boolean upperCase) {
    upperCaseFormat = upperCase;
  		//{{INIT_CONTROLS
		//}}
}

  /**
   * Formats the provided double as a Roman Number by converting to a long.
   * Loss of precision may result, however, Roman Numbers don't have
   * fractions.
   * 
   * @param number The number to format as a Roman Number.
   * @param result The StringBuffer to write the formated number to.
   * @param fieldPosition The field position in the result.
   * @return The StringBuffer provided in <code>result</code> with the
   *  formated value of <code>number</code> appended.
   */
  public StringBuffer format(double number, StringBuffer result,
                             FieldPosition fieldPosition) {
    return format((long)number, result, fieldPosition);
  }

  /**
   * Formats the provided long as a Roman Number.
   * 
   * @param number The number to format as a Roman Number.
   * @param result The StringBuffer to write the formated number to.
   * @param fieldPosition The field position in the result.
   * @return The StringBuffer provided in <code>result</code> with the
   *  formated value of <code>number</code> appended.
   */
  public StringBuffer format(long number, StringBuffer result,
                             FieldPosition fieldPosition) {
    StringBuffer buf = new StringBuffer();

    while(number > 0) {
      if(number >= M) {
        buf.append('m');
        number -= M;
      } else if(number >= D && number < M - C) {
        buf.append('d');
        number -= D;
      } else if(number >= C) {
        buf.append('c');
        if(number >= D - C) number += C;
        else number -= C;
      } else if(number >= L && number < C - X) {
        buf.append('l');
        number -= L;
      } else if(number >= X) {
        buf.append('x');
        if(number >= L - X) number += X;
        else number -= X;
      } else if(number >= V && number < X - I) {
        buf.append('v');
        number -= V;
      } else {
        buf.append('i');
        if(number >= V - I) number++;
        else number--;
      }
    }

    String formatted = buf.toString();
    if(upperCaseFormat) result.append(formatted.toUpperCase());
    else result.append(formatted);
    return result;
  }

  /**
   * Parses a string containing a number expressed in Roman Numerals and
   * converts it to a <code>Number</code>.  The parsePosition will be set to
   * the character following the last character parsed.
   * @param text The string to parse the number from.
   * @param parsePosition On input, contains the position in text to start
   *  parse at.  On return, contains the position that caused parsing to stop.
   * @return The parsed number.  This will be an instance of java.lang.Long.
   */
  public Number parse(String text, ParsePosition parsePosition) {
    text = text.toLowerCase();
    int len = text.length();
    long result = 0;                    // Holds our parse value
    int pos;

    for(pos = parsePosition.getIndex(); pos < len; pos++) {
      char c = text.charAt(pos);
      int charIndex = validChars.indexOf(c);
      if(charIndex == -1) break;        // Invalid character

      // If the next character has a higher value, this character subtracts,
      // otherwise it adds
      int nextCharIndex =
        (pos + 1 < len) ? validChars.indexOf(text.charAt(pos + 1)) : -1;
      if(nextCharIndex > charIndex) result -= charValues[charIndex];
      else                          result += charValues[charIndex];
    }
    parsePosition.setIndex(pos);
    return new Long(result);
  }

  /**
   * This method is for testing the class.
   * @param args The first argument should be either true or false.  It
   *  determines if the returned string is uppercase (true) or lowercase
   *  (false).  The second argument should be a number to format.
   */
  public static void main(String[] args) throws Exception {
    if(args.length < 2) {
      System.out.println("Usage: java RomanNumberSequence true|false number");
      System.exit(0);
    }

    boolean caps = Boolean.valueOf(args[0]).booleanValue();
    long count = Long.parseLong(args[1]);
    RomanNumberFormat fmt = new RomanNumberFormat(caps);

    // Convert number to string
    System.out.print("Formated number: ");
    String str = fmt.format(count);
    System.out.println(str);

    // Convert string from last step to a number
    System.out.print("Parsed number: ");
    long result = fmt.parse(str).longValue();
    System.out.println(result);

    // Compare the two numbers.  Should be the same
    System.out.println((count == result) ? "Success" : "Failure");
  }
	//{{DECLARE_CONTROLS
	//}}
}
