/*
 * GQmpeg
 * (C)1998, 1999 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License.
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

/*
 * the TODO for this module (using libmikmod)
 *  > move to a forked process with pipes to control playback, to make controls responsive,
 *    remove skipping on playlist/config operations.
 *  > figure out how to correctly calculate a correct mod song length (tricky at best, mod format can jump around)
 *  > implement seek (really requires above, because gqmpeg seeks with seconds, not patterns)
 *  > use the pid of the above fork for cpu %
 *  > add config options for:
 *      > surround option
 *      > output hz
 *      > mono option
 *      > 8bit option
 *      > max voices
 *      > panning option
 *      > fading option
 */

#include "io_mikmod.h"
#include "players.h"
#include "utildlg.h"
#include <mikmod.h>

#define MIKMOD_CONTROL_BINARY "mod_gqmpegctl"

/* ----------------------------------------------------------
   input / output interface to libmikmod
   ----------------------------------------------------------*/

gint		mikmod_buffer_enable = FALSE;
gint		mikmod_buffer = 512;
gint		mikmod_downsample = 0;
gint		mikmod_custom_sample_size = 44100;
gint		mikmod_mono = FALSE;
gint		mikmod_8bit = FALSE;
gint		mikmod_device_enable = FALSE;
gchar		*mikmod_device = NULL;

static SongData *current_sd = NULL;
static gint control_read_id = -1;
static gint child_pid = -1;
static int mikmod_pipe_control[2];
static int mikmod_pipe_status[2];

static void song_finished();
static void stop_playing_file();

/*
 *-----------------------------------------------------------------------------
 * common utils
 *-----------------------------------------------------------------------------
 */

static sig_atomic_t sigpipe_occured = FALSE;

static void sighandler_sigpipe(int sig)
{
	sigpipe_occured = TRUE;
}

static gint pipe_write(int fd, const gchar *text)
{
	struct sigaction new_action, old_action;
	int w;

	if (fd < 0) return -1;

	sigpipe_occured = FALSE;

	new_action.sa_handler = sighandler_sigpipe;
	sigemptyset (&new_action.sa_mask);
	new_action.sa_flags = 0;

	/* setup our signal handler */
	sigaction (SIGPIPE, &new_action, &old_action);

	w = write(fd, text, strlen(text));

	if (sigpipe_occured || w < 0)
		{
		if (sigpipe_occured)
			{
			printf("io_mikmod.c: SIGPIPE writing to pipe\n");
			w = -2;
			}
		else
			{
			printf("io_mikmod.c: ERROR writing to pipe\n");
			}
		}

	/* restore the original signal handler */
	sigaction (SIGPIPE, &old_action, NULL);

	return w;
}

/*
 *-----------------------------------------------------------------------------
 * parent side
 *-----------------------------------------------------------------------------
 */

static gint mikmod_control_send(const gchar *text)
{
	gint w;

	w = pipe_write(mikmod_pipe_control[1], text);

	if (w == -2)
		{
		kill (child_pid, SIGINT);
		waitpid(child_pid, NULL, 0);
		child_pid = -1;
		close(mikmod_pipe_control[1]);
		close(mikmod_pipe_status[0]);
		gdk_input_remove(control_read_id);
		}

	return w;
}

static void mikmod_control_read_cb(gpointer data, int fd, GdkInputCondition condition)
{
	int r;
	gchar buf[2048];
	gchar *text;

	r = read(fd, buf, sizeof(buf));

	if (r == -1)
		{
		gdk_input_remove(control_read_id);
/*		child_running = FALSE;*/
		return;
		}

	if (r < 1) return;

	text = g_strndup(buf, r);

	if (debug_mode) printf("mod_gqmpegctl status text: %s\n", text);

	if (!strncmp(text, "t ", 2))
		{
		sscanf(text, "t %d %d", &seconds, &seconds_remaining);
		}
	else if (!strncmp(text, "end", 3))
		{
		module_playback_end();
		}
	else if (!strncmp(text, "play", 4))
		{
		playback_done_command(EXEC_PLAY, FALSE);
		}
	else if (!strncmp(text, "stop", 4))
		{
		playback_done_command(EXEC_STOP, FALSE);
		seconds = 0;
		seconds_remaining = 0;
		}
	else if (!strncmp(text, "pause", 5))
		{
		playback_done_command(EXEC_PAUSE, FALSE);
		}
	else if (!strncmp(text, "continue", 8))
		{
		playback_done_command(EXEC_CONTINUE, FALSE);
		}
	else if (!strncmp(text, "seek", 4))
		{
		playback_done_command(EXEC_SEEK, FALSE);
		}
	else if (!strncmp(text, "error", 5))
		{
		module_playback_error();
		}
	else
		{
		printf("io_mikmod.c: unknown text from child: %s\n", text);
		}

	g_free(text);
}

static gint mikmod_control_setup()
{
	control_read_id = gdk_input_add (mikmod_pipe_status[0],  GDK_INPUT_READ,
                           mikmod_control_read_cb, NULL);
}

/*
 *-----------------------------------------------------------------------------
 * fork, process setup
 *-----------------------------------------------------------------------------
 */

static gint mikmod_create_process()
{
	pid_t frk_pid;

	if (child_pid != -1) return TRUE;

	/* Create the pipe. */
	if (pipe (mikmod_pipe_control))
		{
		fprintf (stderr, "io_mikmod.c: Pipe failed.\n");
		return FALSE;
		}
	if (pipe (mikmod_pipe_status))
		{
		fprintf (stderr, "io_mikmod.c: Pipe failed.\n");
		close(mikmod_pipe_control[0]);
		close(mikmod_pipe_control[1]);
		return FALSE;
		}

	frk_pid = fork ();
	if (frk_pid == (pid_t) 0)
                {
                /* the child process. */
		char *dt = NULL;

		close(mikmod_pipe_control[1]);
		close(mikmod_pipe_status[0]);

		dup2(mikmod_pipe_control[0], 0);
		dup2(mikmod_pipe_status[1], 1);

		if (debug_mode) dt = "debug";

		execlp (MIKMOD_CONTROL_BINARY, MIKMOD_CONTROL_BINARY, "init", dt, NULL);

		printf("Child exec failed!\n");
		_exit(0);
		}
	else if (frk_pid < (pid_t) 0)
		{
		/* The fork failed. */
		close(mikmod_pipe_control[0]);
		close(mikmod_pipe_control[1]);
		close(mikmod_pipe_status[0]);
		close(mikmod_pipe_status[1]);
		fprintf (stderr, "io_mikmod.c: Fork failed.\n");
		pid = 0;
		child_pid = -1;
		return FALSE;
		}
	else
		{
		/* the parent process. */
		close(mikmod_pipe_control[0]);
		close(mikmod_pipe_status[1]);
		child_pid = (int) frk_pid;
		pid = child_pid;
		mikmod_control_setup();
		}

	return TRUE;
}

static void mikmod_destroy_process()
{
	gint w;

	if (child_pid == -1) return;

	w = mikmod_control_send("exit\n");
	if (w > 0)
		{
		waitpid(child_pid, NULL, 0);
		close(mikmod_pipe_control[1]);
		close(mikmod_pipe_status[0]);
		gdk_input_remove(control_read_id);
		}
	child_pid = -1;
}

static gint shutdown_schedule_id = -1;
static gint mikmod_accessed = FALSE; /* access flag, to abort scheduled shutdown */

static gint mikmod_schedule_shutdown_cb(gpointer data)
{
	if (!mikmod_accessed) mikmod_destroy_process();

	shutdown_schedule_id = -1;

	return FALSE;
}

static void mikmod_schedule_shutdown(gint s)
{
	if (child_pid == -1) return;

	if (shutdown_schedule_id != -1)
		{
		gtk_timeout_remove(shutdown_schedule_id);
		}

	shutdown_schedule_id = gtk_timeout_add(s * 1000, mikmod_schedule_shutdown_cb, NULL);
	mikmod_accessed = FALSE;
}


/*
 *-----------------------------------------------------------------------------
 * real funcs
 *-----------------------------------------------------------------------------
 */

static gint start_playing_file(SongData *sd, gint position)
{
	gchar *buf;
	gint w;

	if (!mikmod_create_process()) return FALSE;

	buf = g_strdup_printf("play \"%s\"\n", sd->path);
	w =  mikmod_control_send(buf);
	g_free(buf);

	if (w < 0)
		{
		mikmod_destroy_process();
		return FALSE;
		}

	mikmod_accessed = TRUE;

	return TRUE;
}

static void stop_playing_file()
{
	gint w;

	if (child_pid == -1) return;

	w =  mikmod_control_send("stop\n");

	if (w < 0)
		{
		mikmod_destroy_process();
		}
	else
		{
		mikmod_schedule_shutdown(20);
		}
}

static gint pause_file()
{
	gint w;

	if (child_pid == -1) return;

	w =  mikmod_control_send("pause\n");
	if (w < 0)
		{
		mikmod_destroy_process();
		return FALSE;
		}

	mikmod_accessed = TRUE;

	return TRUE;
}

static gint continue_file()
{
	gint w;

	if (child_pid == -1) return;

	w =  mikmod_control_send("continue\n");
	if (w < 0)
		{
		mikmod_destroy_process();
		return FALSE;
		}

	mikmod_accessed = TRUE;

	return TRUE;
}

static gint seek_file(gint pos)
{
	gint w;
	gchar *buf;

	if (child_pid == -1) return;

	buf = g_strdup_printf("seek %d\n", pos);
	w =  mikmod_control_send(buf);
	g_free(buf);
	if (w < 0)
		{
		mikmod_destroy_process();
		return FALSE;
		}

	mikmod_accessed = TRUE;

	return TRUE;
}

/*
 *----------------------------------------------------------------------------
 * libmikmod module callback funcs
 *----------------------------------------------------------------------------
 */

static void mikmod_data_init(SongData *sd)
{
	sd->type_description = _("MOD format audio file");
}

static gint mikmod_data_set(SongData *sd, gint generic_info, gint format_info)
{
	if (generic_info)
		{
		sd->generic_info_loaded = TRUE;
		sd->title = Player_LoadTitle(sd->path);
		}
	if (format_info)
		{
		sd->format_info_loaded = TRUE;

		if (TRUE)
			{
			}
		else
			{
			sd->flags |= SONG_FLAG_NOT_FOUND;
			}
		}

	return TRUE;
}

/*
 *----------------------------------------------------------------------------
 * mikmod module play funcs
 *----------------------------------------------------------------------------
 */

static gint mikmod_start(SongData *sd, gint position)
{
	if (debug_mode) printf("play started at %d\n", position);
	current_sd = sd;

	return start_playing_file(sd, position);
}

static gint mikmod_stop(SongData *sd)
{
	if (debug_mode) printf("play stopped\n");
	if (sd != current_sd)
		{
		printf(_("io_mmikmod.c warning: attempt to stop playback of non matching file\n"));
		}
	stop_playing_file();
	current_sd = NULL;
	return TRUE;
}

static gint mikmod_pause(SongData *sd)
{
	if (debug_mode) printf("play paused at %d\n", seconds);
	if (sd != current_sd)
		{
		printf(_("io_mikmod.c warning: attempt to pause playback of non matching file\n"));
		}
	return pause_file();
}

static gint mikmod_continue(SongData *sd)
{
	if (debug_mode) printf("play restarted at %d\n", seconds);
	if (sd != current_sd)
		{
		printf(_("io_mikmod.c warning: attempt to continue playback of non matching file\n"));
		}
	current_sd = sd;
	return continue_file();
}

static gint mikmod_seek(SongData *sd, gint position)
{
	if (debug_mode) printf("play seeking to %d\n", position);
	if (sd != current_sd)
		{
		printf(_("io_mikmod.c warning: attempt to seek in non matching file\n"));
		}

	seconds_remaining += seconds - position;
	seconds = position;

	return seek_file(position);
}

/*
 *----------------------------------------------------------------------------
 * player module interface routines
 *----------------------------------------------------------------------------
 */

void mikmod_init()
{
	IO_ModuleData *imd;
	gchar *mod_desc = "MOD file";
	gint id;

	if (debug_mode) printf("mikmod module initing...\n");

	/* FIXME! should only be done when actually first needed (less start up overhead) */
#if 1
	MikMod_RegisterAllDrivers();
#else
	MikMod_RegisterDriver(&drv_oss);
#endif

	MikMod_RegisterAllLoaders();

	if (debug_mode) 
		{
		gchar *buf = MikMod_InfoDriver();
		printf("%s\n", buf);
		g_free(buf);
		}

	imd = g_new0(IO_ModuleData, 1);

	imd->title = "mikmod";
	imd->description = _("mod player");

	imd->songdata_init_func = mikmod_data_init;
	imd->songdata_info_func = mikmod_data_set;

	imd->start_func = mikmod_start;
	imd->stop_func = mikmod_stop;
	imd->pause_func = mikmod_pause;
	imd->continue_func = mikmod_continue;
	imd->seek_func = mikmod_seek;

	imd->config_load_func = mikmod_config_load;
	imd->config_save_func = mikmod_config_save;
	imd->config_init_func = mikmod_config_init;
	imd->config_apply_func = mikmod_config_apply;
	imd->config_close_func = mikmod_config_close;

	imd->info_func = mod_create_info_window;

	id = player_module_register(imd);

	module_register_file_suffix_type(".mod", mod_desc, id);

	/* this is what happens when everyone expands on the standard in their own way ;) */
	module_register_file_suffix_type(".669", mod_desc, id);
	module_register_file_suffix_type(".amf", mod_desc, id);
	module_register_file_suffix_type(".dsm", mod_desc, id);
	module_register_file_suffix_type(".far", mod_desc, id);
	module_register_file_suffix_type(".gdm", mod_desc, id);
	module_register_file_suffix_type(".it" , mod_desc, id);
	module_register_file_suffix_type(".imf", mod_desc, id);
	module_register_file_suffix_type(".med", mod_desc, id);
	module_register_file_suffix_type(".m15", mod_desc, id);
	module_register_file_suffix_type(".mtm", mod_desc, id);
	module_register_file_suffix_type(".s3m", mod_desc, id);
	module_register_file_suffix_type(".stm", mod_desc, id);
	module_register_file_suffix_type(".stx", mod_desc, id);
	module_register_file_suffix_type(".ult", mod_desc, id);
	module_register_file_suffix_type(".uni", mod_desc, id);
	module_register_file_suffix_type(".xm" , mod_desc, id);
}


