/*\
||| Written By Fredrik Hbinette <hubbe@lysator.liu.se>
||| All rights reserved. No warrenties, use at your own risk.
\*/

#include <X11/X.h>
#include <X11/Xos.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sysexits.h>
#include <signal.h>
#include <stdarg.h>
#include <npapi.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>

#include <sys/socket.h>

/* #define DEBUG */

#define H_LOOP 1
#define H_MANY 2
#define H_STREAM 4
#define H_NOISY 8
#define H_REPEATCOUNT 16
#define H_PRELOAD 32
#define H_DAEMON 64
#define H_EXITS 128
#define H_IGNORE_ERRORS 256

#define BUFSIZE 1024*256
#define PRELOAD 40000

struct data
{
  Window window;
  int pid;

  char *mimetype;
  int repeats;

  int flags;
  char *command;

#ifdef STREAMER
  int fd;
  int peekfd;
  int waitfd;

  int buffering;
  int buffered;
  int offset;
  char *buffer;
#endif
};


#define THIS (*(struct data **) &(instance->pdata))
#define FEND(X,Y) (strchr(X,Y)?strchr(X,Y):(X)+strlen(X))
#define xalloc malloc
#define xfree free

#ifdef DEBUG
static FILE *getout()
{
  static FILE *foo=0;
  if(foo) return foo;
  foo=fopen("/tmp/ndebug","a+");
  fprintf(foo,"------------\n");
  return foo;
}
static int D(char *fmt, ...)
{
  char buffer[9999];
  va_list ap;
  va_start(ap,fmt);
  vsnprintf(buffer,sizeof(buffer),fmt,ap);
  va_end(ap);
  fprintf(getout(),"PID%4d: %s",getpid(),buffer);
  fflush(getout());
}
#else
/* OPTIMIZE ME! */
static int D(char *fmt, ...) { }
#endif

#ifndef MIN
#define MIN(X,Y) ((X)<(Y)?(X):(Y))
#endif

#ifndef MAXINT
#define MAXINT 0x7fffffff
#endif

/* Check weather the first word in 'file' is somewhere to be found in
 * the users PATH. If so, return 1 otherwise 0.
 */
static int inpath(char *file)
{
  char tmp[16384];
  char *path=getenv("PATH");
  char *pos;
  if(!path) return 0;

  D("inpath(%s)\n",file);
  if(file[0]=='/') return 1;
  if(!strncmp(file,"internal:",9)) return 1;

  /* Insert magical shell-script tests here */

  D("Hmm? PATH=%s\n",path);
  pos=path;
  while(1)
  {
    char *next;
    next=FEND(pos,':');
    if(next!=pos)
    {
      char *ptr=tmp;
      struct stat buf;

      memcpy(ptr=tmp,pos,next-pos);
      ptr+=next-pos;
      *(ptr++)='/';
      memcpy(ptr,
	     file,
	     FEND(file,' ')-file);
      ptr+=FEND(file,' ')-file;
      *ptr=0;
      D("stat(%s)\n",tmp);
      if(!stat(tmp, &buf)) return 1;
      D("nope\n");
    }
    if(!*next) return 0;
    pos=next+1;
  }
  D("GAAHHH!\n");
}


/* Shell for fork which handles some magic needed to prevent
 * netscape from freaking out. It also prevents interferrance
 * from signals.
 */
static int my_fork(NPP instance)
{
  int pid;
  sigset_t set,oset;
  D("forking\n");
  sigfillset(& set);
  sigprocmask(SIG_SETMASK,&set,&oset);
  pid=fork();
  if(pid==-1) return pid;
  if(!pid)
  {
    int signum;
    if(!(THIS->flags & H_DAEMON))
      setsid();

    for(signum=0;signum<NSIG;signum++) signal(signum, SIG_DFL);
  }else{
#ifdef STREAMER
    if(THIS->peekfd>=0)
      close(THIS->peekfd);
#endif
    D("Child running\n");
  }
  sigprocmask(SIG_SETMASK,&oset,&set);
  return pid;
}


/* Run chosen commands. */
static void run(NPP instance, const char *file)
{
  while(THIS->repeats>0)
  {
    char *argv[10];
    char buffer[16384];
    char *foo=buffer;
    int loops=1;

    /* If command expects data on stdin but data is actually in
     * a file, open the file and connect it to stdin.
     /*
    if(file && (THIS->flags & H_STREAM))
    {
      int fd=open(file,O_RDONLY);
      dup2(fd,0);
      close(fd);
      D("Stream from file %s\n",file);
    }

    /* Hmm, what does H_REPEATCOUNT do again?? */
    if(THIS->flags & H_REPEATCOUNT)
    {
      loops=THIS->repeats;
    }

    /* This application will play the data forever */
    if(THIS->flags & H_LOOP)
    {
      D("Expecting application to loop.\n");
      loops=0x7fffffff;
    }

    /* setup environment variable $file */
    if(file && !(THIS->flags & H_STREAM))
    {
      if((THIS->flags & H_MANY) && !(THIS->flags & H_REPEATCOUNT))
      {
	int e;
	sprintf(foo,"file=%s",file);
	loops=MIN(THIS->repeats,10);
	for(e=0;e<loops;e++)
	{
	  strcat(foo," ");
	  strcat(foo,file);
	}
      }else{
	sprintf(foo,"file=%s",file);
      }
      putenv(foo);
      foo+=strlen(foo)+1;
    }

    /* setup environment variable $env */
    sprintf(foo,"window=%ld",(long)THIS->window);
    putenv(foo);
    foo+=strlen(foo)+1;

    /* setup environment variable $repeat */
    sprintf(foo,"repeat=%ld",(long)THIS->repeats);
    putenv(foo);
    foo+=strlen(foo)+1;

    /* Create command line */
    argv[0]="/bin/sh";
    argv[1]="-c";
    argv[2]=THIS->command;
    argv[3]=0;


    /* Odd, this seems to be similar to H_REPEATCOUNT */
    if(THIS->flags & H_EXITS) loops=THIS->repeats;

    D("Execing %s (repeats=%d loops=%d)\n",THIS->command,THIS->repeats,loops);
    if(THIS->repeats > loops)
    {
      int pid;
      D("Running %s\n",THIS->command);
      pid=fork();
      if(pid==-1) exit(10);
      
      if(!pid)
      {
	
#ifdef H_NOISY
	if(THIS->flags & H_NOISY)
	{
	  int nl=open("/dev/null", O_RDONLY);
	  D("Redirecting stdout and stderr\n");
	  dup2(nl,1);
	  dup2(nl,2);
	  close(nl);
	}
#endif
	
	execvp(argv[0],argv);
	D("Execvp failed..%d\n",errno);
	exit(EX_UNAVAILABLE);
      }else{
	int status;
	D("waiting for (%d)\n",pid);
	waitpid(pid,&status,0);
	D("wait done\n");
	if(!WIFEXITED(status)) exit(10);
	if(WEXITSTATUS(status) && !(THIS->flags & H_IGNORE_ERRORS))
	  exit(WEXITSTATUS(status));
	D("exited ok!\n");
      }
      if(THIS->repeats < MAXINT)
	THIS->repeats-=loops;
    }else{
#ifdef H_NOISY
      if(THIS->flags & H_NOISY)
      {
	int nl=open("/dev/null", O_RDONLY);
	D("Redirecting stdout and stderr\n");
	dup2(nl,1);
	dup2(nl,2);
	close(nl);
      }
#endif
      D("Execing!!!\n");
      execvp(argv[0],argv);
      D("Execvp failed!!!\n");
      exit(EX_UNAVAILABLE);
    }
  }
  exit(0);
}

struct mimetype
{
  struct mimetype *next;
  char *line;
};

struct command
{
  struct command *next;
  int flags;
  char *cmd;
};

struct handle
{
  struct handle *next;
  struct mimetype *types;
  struct command *commands;
};

static struct handle *first_handle=NULL;

/* read configuration file into memory */
static void read_config(FILE *f)
{
  struct handle **handlep;
  struct command **commandp=0;
  struct mimetype **typep=0;
  char buffer[16384];
  int have_commands=1;
  D("read_config\n");

  handlep=&first_handle;

  while(!feof(f))
  {
    fgets(buffer,sizeof(buffer),f);
    D("::: %s",buffer);
    if(buffer[0]=='#' || !buffer[0] || buffer[0]=='\n') continue;

    if(buffer[strlen(buffer)-1]=='\n')
      buffer[strlen(buffer)-1]=0;

    if(isspace(buffer[0]))
    {
      char *x=buffer+1;
      while(isspace(*x)) x++;
      if(!*x)
      {
	D("Empty line.\n");
	continue;
      }

      D("New command\n");

      *commandp=(struct command *)xalloc(sizeof(struct command));
      if(!*commandp)
      {
	D("xalloc failed\n");
	return;
      }
      (*commandp)->flags=0;
      (*commandp)->cmd=0;
      (*commandp)->next=0;

      /* Command */
      have_commands++;
      while(*x!=':' && *x)
      {
/* 	D("Parsing %s\n",x); */
	switch(*x)
	{
	  case ',':
	  case ' ':
	  case '\t':
	    x++;
	    break;

	  default:
#define GOBBLE(X,Y) if(!strncasecmp(x,X,strlen(X))) { x+=strlen(X); (*commandp)->flags|=Y; break; }
	    GOBBLE("repeat",H_REPEATCOUNT);
	    GOBBLE("loop",H_LOOP);
	    GOBBLE("stream",H_STREAM);
	    GOBBLE("preload",H_PRELOAD);
	    GOBBLE("many",H_MANY);
	    GOBBLE("ignore_errors",H_IGNORE_ERRORS);
	    GOBBLE("exits",H_EXITS);
#ifdef H_NOISY
	    GOBBLE("noisy",H_NOISY);
#endif
	    D("Unknown directive: %s\n",x);

	    /* Unknown directive */
	    fprintf(stderr,"Unknown directive: %s\n",x);
	    if(isalnum(*x))
	    {
	      while(isalnum(*x)) x++;
	    }else{
	      x++;
	    }
	}
      }
      if(*x==':')
      {
	x++;
	while(isspace(*x)) x++;
	(*commandp)->cmd=strdup(x);
      }else{
	D("No colon? (%s)\n",x);
      }
      if(!(*commandp)->cmd)
      {
	xfree(*commandp);
	*commandp=0;
	D("strdup failed %s\n",x);
	return;
      }
      commandp=&((*commandp)->next);
    }else{
      /* Mime type */

      if(have_commands)
      {
  	D("New handle\n");
	if(commandp)
	  D("Commandp=%p\n",*commandp);
	*handlep=(struct handle *)xalloc(sizeof(struct handle));
	if(!*handlep)
	  return;
	
	(*handlep)->commands=0;
	(*handlep)->types=0;
	(*handlep)->next=0;
	commandp=&((*handlep)->commands);
	typep=&((*handlep)->types);
	handlep=&((*handlep)->next);
	have_commands=0;
      }

      D("New mimetype\n");
      *typep=(struct mimetype *)xalloc(sizeof(struct mimetype));
      if(!*typep)
	return;

      (*typep)->next=0;
      (*typep)->line=strdup(buffer);

      if(!(*typep)->line)
      {
	xfree(*typep);
	*typep=0;
	return;
      }
      typep=&((*typep)->next);
    }
  }
}

/* Find configuration file and read it into memory */
static void do_read_config(void)
{
  char fname[8192];
  FILE *f=0;
  if(first_handle) return;
  D("do_read_config\n");
  if(getenv("HOME") && strlen(getenv("HOME"))<8000)
  {
    sprintf(fname,"%s/.netscape/pluggerrc",getenv("HOME"));
    f=fopen(fname,"r");
    D("%s=%p\n",fname,f);
  }

  if(!f)
  {
    if(getenv("MOZILLA_HOME") && strlen(getenv("MOZILLA_HOME"))<8000)
    {
      sprintf(fname,"%s/pluggerrc",getenv("MOZILLA_HOME"));
      f=fopen(fname,"r");
      D("%s=%p\n",fname,f);
    }
  }

  if(!f) f=fopen("/usr/local/netscape/pluggerrc","r");
  if(!f) f=fopen("/etc/pluggerrc","r");
  if(!f) f=fopen("/usr/etc/pluggerrc","r");
  if(!f) f=fopen("/usr/local/etc/pluggerrc","r");
  if(!f) f=fopen("pluggerrc","r");

  if(!f)
  {
    fprintf(stderr,"Plugger: No config file found!\n");
    return;
  }

  read_config(f);
  fclose(f);
  D("do_read_config done\n");
}

/* Construct a MIME Description string for netscape
 * so that netscape shall know when to call us back.
 */
char *NPP_GetMIMEDescription(void)
{
  char *x,*y;
  struct handle *h;
  struct mimetype *m;
  int size_required;

  do_read_config();

  D("Getmimedescription\n");

  size_required=0;
  for(h=first_handle;h;h=h->next)
    for(m=h->types;m;m=m->next)
      size_required+=strlen(m->line)+1;

  D("Size required=%d\n",size_required);
  x=(char *)xalloc(size_required+1);
  if(!x) return 0;

  D("Malloc did not fail\n");

  y=x;

  for(h=first_handle;h;h=h->next)
  {
    D("Foo: %p\n",h->commands);
    for(m=h->types;m;m=m->next)
    {
      char *tmp;
/*       D("appending: %s\n",m->line); */
      memcpy(y,m->line,strlen(m->line));
      y+=strlen(m->line);
      *(y++)=';';
    }
  }
  *(y++)=0;
  D("Getmimedescription done: %s\n",x);
  return x;
}

/* Go through the commands in the config file 
 * and find one that fits our needs.
 */
static int find_command(NPP instance,
			int streaming)
{
  struct handle *h;
  struct mimetype *m;
  struct command *c;

  D("find_command\n");

  do_read_config();

  D("find_command...\n");

  for(h=first_handle;h;h=h->next)
  {
    D("commands for this handle at (%p)\n",h->commands);
    for(m=h->types;m;m=m->next)
    {
      char *tmp1=FEND(m->line,':');
      char tmp2;
      int tmp3;
      D("Checking '%s'\n",m->line);

      while(tmp1>m->line && isspace(tmp1[-1]))
	tmp1--;

      D("tmp1=%s\n",tmp1);

      tmp2=*tmp1;
      *tmp1=0;
      D("Checking '%s' ?= '%s'\n",m->line,THIS->mimetype);
      tmp3=strcasecmp(THIS->mimetype, m->line);
      *tmp1=tmp2;
      if(!tmp3)
      {
	D("Match found!\n");
	break;
      }else{
	D("Not same.\n");
      }
    }

    if(m)
    {
      for(c=h->commands;c;c=c->next)
      {
	D("Checking command: %s\n",c->cmd);
	if((c->flags & H_LOOP) && THIS->repeats != MAXINT) continue;
	if( (!!streaming) != (!!(c->flags & H_STREAM))) continue;
	if(!inpath(c->cmd)) continue;
	D("Match found!\n");
	THIS->command=c->cmd;
	THIS->flags=c->flags;
	return 1;
      }
    }
  }
  D("No match found\n");
  return 0;
}

/* Let netscape know things about plugger */
NPError NPP_GetValue(void *future, NPPVariable variable, void *value)
{
  NPError err = NPERR_NO_ERROR;

  D("Getvalue %d\n",variable);
  
  switch (variable)
  {
  case NPPVpluginNameString:
    D("GET Plugin name\n");
    *((char **)value) = "Plugger "
      VERSION
      ;

  break;

  case NPPVpluginDescriptionString:
    D("GET Plugin description\n");
    *((char **)value) =
      "<img width=40 height=40 border=0 align=left src=http://fredrik.hubbe.net/plugger/logo.gif>"
      "<a href=http://fredrik.hubbe.net/plugger.html>Plugger</a> "
      "version "
      VERSION
      ", written by "
      "<a href=http://fredrik.hubbe.net/>Fredrik Hbinette</a> "
      "<a href=mailto:hubbe@hubbe.net>&lt;hubbe@hubbe.net&gt</a>. "
      "For documentation on how to configure plugger, go to the plugger "
      " <a href=http://fredrik.hubbe.net/plugger.html>homepage</a> "
      "or check the man page. (type <tt>man&nbsp;plugger</tt>)"
      ;
  break;

  default:
    err = NPERR_GENERIC_ERROR;
  }
  return err;
}

/* Initiate another instance of plugger, it is important to know
 * that there might be several instances going at one time.
 */
NPError  NPP_New(NPMIMEType pluginType,
		 NPP instance,
		 uint16 mode,
		 int16 argc,
		 char* argn[],
		 char* argv[],
		 NPSavedData* saved)
{
  int e;
  NPError tmp;

  D("NEW (%s)\n",pluginType);

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  THIS = NPN_MemAlloc(sizeof(struct data));
  if(!THIS) return NPERR_OUT_OF_MEMORY_ERROR;
  memset((char *)THIS, 0, sizeof(*THIS));

  THIS->repeats=MAXINT;
  THIS->pid=-1;
#ifdef STREAMER
  THIS->fd=-1;
  THIS->waitfd=-1;
  THIS->peekfd=-1;
#endif

  if(!pluginType) return NPERR_GENERIC_ERROR;

  for(e=0;e<argc;e++)
  {
    if(!strcasecmp("loop",argn[e]))
    {
      switch(argv[e][0])
      {
      case 't': case 'T': case 'y': case 'Y':
	THIS->repeats=MAXINT;
	break;
      case 'f': case 'F': case 'n': case 'N':
	THIS->repeats=1;
	break;
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
	THIS->repeats=atoi(argv[e]);
      }
    }
  }

  return NPERR_NO_ERROR;
}

/* Free data, kill processes, it is time for this instance to die */
NPError NPP_Destroy(NPP instance, NPSavedData** save)
{
  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  D("Destroy\n");

  if (THIS)
  {
    if(THIS->pid > 0)
      {
	D("killing %d\n",-THIS->pid);
	/* Die Die Die!!! */
	if(!kill(-THIS->pid, SIGTERM))
	{
	  if(!kill(-THIS->pid, SIGTERM))
	  {
	    /* we have failed to kill our children twice,
	     * sleep on it and try again tomorrow..
	     */
	    struct timeval timeout;
	    timeout.tv_sec=0;
	    timeout.tv_usec=200;
	    select(0,0,0,0,&timeout);

	    if(!kill(-THIS->pid, SIGTERM))
	    {
	      /* We have failed to kill our children a third time -
	       * wield nukes and try again.
	       */
	      kill(-THIS->pid, SIGKILL);
	    }
	  }
	}
	THIS->pid=-1;
      }

    D("Freeing memory %p\n",THIS->mimetype);
    
    if(THIS->mimetype)
    {
      NPN_MemFree(THIS->mimetype);
      THIS->mimetype=0;
    }

#ifdef STREAMER
    D("Closing fds\n");
    if(THIS->fd >= 0)
    {
      close(THIS->fd);
      THIS->fd=-1;
    }
    
    if(THIS->peekfd>=0)
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
    
    if(THIS->waitfd >= 0)
    {
      close(THIS->waitfd);
      THIS->waitfd=-1;
    }
    
    if(THIS->buffer)
    {
      NPN_MemFree(THIS->buffer);
      THIS->buffer=0;
    }

    NPN_MemFree(THIS);
    THIS = 0;
#endif
  }
  D("Destroy finished\n");
  
  return NPERR_NO_ERROR;
}

/* Open a new stream,
 * each instance can only handle one stream at a time.
 */
NPError NPP_NewStream(NPP instance,
		      NPMIMEType type,
		      NPStream *stream, 
		      NPBool seekable,
		      uint16 *stype)
{
  int wantstream;
  D("Newstream ... \n");

  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;

  /* This is a stupid special case and should be coded into
   * pluggerc instead...
   */
  if(!strncasecmp("image/",type,6) ||
     !strncasecmp("x-image/",type,6))
    THIS->repeats=1;

  D("Mime type %s\n",type);

  if(THIS->mimetype)
  {
    NPN_MemFree(THIS->mimetype);
    THIS->mimetype=0;
  }
  THIS->mimetype = NPN_MemAlloc(strlen(type)+1);
  if(!THIS->mimetype) return NPERR_OUT_OF_MEMORY_ERROR;
  strcpy(THIS->mimetype, type);

  D("Url is %s (seekable=%d)\n",stream->url,seekable);
#ifdef STREAMER
  /* Hmm, this seems weird... */
  wantstream=!(seekable && !strncasecmp(stream->url,"file:",5));
#else
  wantstream=0;
#endif
  
  if(!find_command(instance, wantstream))
  {
    if(!find_command(instance, !wantstream))
    {
      NPN_Status(instance, "No approperiate application found!");
      return NPERR_GENERIC_ERROR;
    }
  }

#ifdef STREAMER
  if((THIS->flags & H_STREAM) && strncasecmp(stream->url,"file:",5))
  {
    int foo[2],bar[2];

    if(THIS->repeats == 1 || 
      (THIS->flags & H_LOOP) ||
      (THIS->flags & H_DAEMON) ||
      (THIS->flags & H_REPEATCOUNT))
      *stype=NP_NORMAL;
    else
      *stype=NP_ASFILE;

    if(socketpair(AF_UNIX, SOCK_STREAM, 0, foo)<0 ||
       pipe(bar) < 0)
    {
      NPN_Status(instance, "Streamer: Failed to create a pipe!");
      return NPERR_GENERIC_ERROR;
    }

    THIS->pid=my_fork(instance);
    if(THIS->pid==-1)
    {
      NPN_Status(instance, "Streamer: My_Fork failed!");
      return;
    }
    
    if(!THIS->pid)
    {
      D("Streaming child running\n");
      close(bar[0]);

      close(foo[1]);
      dup2(foo[0],0);
      close(foo[0]);

      THIS->repeats=1;
      run(instance, 0);
    }else{
      THIS->buffer=NPN_MemAlloc(BUFSIZE);
      if(!THIS->buffer)
	return NPERR_OUT_OF_MEMORY_ERROR;
      
      if(THIS->repeats < MAXINT)
	THIS->repeats--;
      THIS->fd=foo[1]; /* This is the FD we write data to */
      D("FD to parent = %d\n",THIS->fd);
      fcntl(THIS->fd, F_SETFL, O_NDELAY);

      THIS->waitfd=bar[0];
      close(bar[1]);
      if(THIS->flags & H_PRELOAD)
      {
	THIS->buffering=1;
	THIS->peekfd=foo[0];
      }else{
	close(foo[0]);
      }
    }
    
    D("Ok\n");
  }else
#endif
    *stype = NP_ASFILEONLY;
  
  return NPERR_NO_ERROR;
}


#ifdef STREAMER

/* Check weather there is data to read on 'fd' */
static int data_available(int fd)
{
  fd_set tmp;
  struct timeval timeout;

  do {
    timeout.tv_sec=0;
    timeout.tv_usec=0;
    FD_ZERO(&tmp);
    FD_SET(fd, &tmp);
  } while(select(fd+1, &tmp, 0, 0, &timeout)<0 && errno==EINTR);
  return FD_ISSET(fd, &tmp);
}

/* Try to write data to our children */
static int trywrite(NPP instance)
{
  D("trywrite (%d bytes buffered) fd=%d\n",THIS->buffered,THIS->fd);

  if(THIS->buffering)
  {
    if(THIS->buffered < PRELOAD)
    {
      char b[256];
      sprintf(b,"Buffering ... %2d%%",THIS->buffered * 100 / PRELOAD);
      D("%s\n",b);
      NPN_Status(instance, b);
      return 1;
    }
    THIS->buffering=0;
  }

  if(THIS->peekfd>=0)
  {
    D("Checking waitfd\n");
    if(data_available(THIS->waitfd))
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
  }

  while(THIS->buffered)
  {
    int i;

    do {
      D("trywrite %d bytes (offset = %d)\n",
	MIN(THIS->buffered, BUFSIZE-THIS->offset),
	THIS->offset);
      i=write(THIS->fd,
	      THIS->buffer+THIS->offset,
	      MIN(THIS->buffered, BUFSIZE-THIS->offset) );
      D("Wrote %d bytes (errno = %d)\n",i, errno);
    } while(i<0 && errno==EINTR);

    if(i<0)
    {
      switch(errno)
      {
      case EALREADY:
      case EAGAIN:
	return 1;
      default:
	D("trywrite: Errno = %d\n",errno);
	return 0;
      }
    }

    if(!i) return 1;

    THIS->offset+=i;
    THIS->buffered-=i;
    if(THIS->offset == BUFSIZE) THIS->offset=0;
  }

  D("Checking preload\n");

  /* Hmm, this will not work properly with 
   * programs which buffer more data than we do..
   * With luck it doesn't affect streaming though.
   */
  if((THIS->flags & H_PRELOAD) &&
     THIS->peekfd>=0 &&
     !data_available(THIS->peekfd))
  {
    D("(Re)-starting preload\n");
    THIS->buffering=1;
  }
  D("trywrite-exit: errno = %d\n",errno);
  return 1;
}
#endif

/* */
int32 NPP_WriteReady(NPP instance, NPStream *stream)
{
#ifdef STREAMER
  D("Writeready\n");
  if(!instance) return 0;

  trywrite(instance);

  D("Writeready returns: %d\n", BUFSIZE - THIS->buffered);

  /* Kluge to make netscape behave! */
  /* A small delay is invoked because our children has not
   * ingested all data we have fed them yet. Without this delay
   * netscape would use 100% cpu until all data is ingested.
   */

  if(!BUFSIZE - THIS->buffered)
  {
    struct timeval timeout;
    timeout.tv_sec=0;
    timeout.tv_usec=1000;
    select(0,0,0,0,&timeout);
  }
  return BUFSIZE - THIS->buffered;
#else
  return 0x7fffffff;
#endif
}

int32 NPP_Write(NPP instance,
		NPStream *stream,
		int32 offset,
		int32 len,
		void *buf)
{
#ifdef STREAMER
  int32 origlen=len;
  int i;
  char *buffer=buf;
  D("Write(len=%d, offset=%d)\n",len,offset);
  if (!instance) return 0;
		
  if(!trywrite(instance))
    return -1;

  D("Write: THIS->buffered=%d\n",THIS->buffered);
  if(!THIS->buffered && !THIS->buffering)
  {
    D("Attempting direct write\n");
    do {
      i=write(THIS->fd, buffer, len);
    } while(i<0 && errno==EINTR);

    if(i<0)
    {
      switch(errno)
      {
      case EALREADY:
      case EAGAIN:
	D("Nothing written\n");
	break;

      default:
	D("Errno: %d\n",errno);
	return -1;
      }
    }else{
      D("Wrote %d bytes directly\n",i);
      buffer+=i;
      len-=i;
    }
  }

  while(len>0 && THIS->buffered<BUFSIZE)
  {
    int end=THIS->offset + THIS->buffered;
    end%=BUFSIZE;
    i=MIN(len, BUFSIZE-end);

    i=MIN(i, BUFSIZE-THIS->buffered);

    memcpy(THIS->buffer+end, buffer, i);
    len-=i;
    buffer+=i;
    THIS->buffered+=i;
  }

  D("Write returns %d\n",origlen-len);
  return origlen-len;
#else
  return len;
#endif
}

NPError NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
  D("Destroystream\n");
#ifdef STREAMER
  if(THIS->flags & H_STREAM)
  {
    THIS->buffering=0;

    if(THIS->peekfd>=0)
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
    if(trywrite(instance))
    {
      if(THIS->buffered)
      {
	/* We fork again, somebody has to send the rest of our
	 * stream to our children.
	 */
	int pid=my_fork(instance);
	if(pid==-1) return NPERR_GENERIC_ERROR;
	if(!pid)
	{
	  fcntl(THIS->fd, F_SETFL, 0);
	  while(THIS->buffered && trywrite(instance));
	  D("Buffer-cleanup done\n");
	  _exit(0);
	}
      }
    }
    
    close(THIS->fd);
    THIS->fd=-1;
  }
  D("Destroystream done\n");
#endif
  return NPERR_NO_ERROR;
}

void NPP_StreamAsFile(NPP instance,
		      NPStream *stream,
		      const char* fname)
{
  int pid;
  D("Streamasfile\n");
  if(!fname) return;
  if (instance == NULL) return;

  NPN_Status (instance, "Running helper ...");

  if(!strcmp(THIS->command, "internal:url") && fname)
  {
    int fd;
    char *url=NPN_MemAlloc(stream->end+1);
    D("INTERNAL URL\n");
    fd=open(fname,O_RDONLY);
    if(fd<0)
    {
      NPN_Status(instance,"Plugger: Hey, where did the file go?\n");
    } else {
      if(read(fd, url, stream->end) == stream->end)
      {
	url[stream->end]=0;
	FEND(url,'\n')[0]=0;
	D("URL: %s\n",url);
	NPN_GetURL(instance, url, 0);
      }
      close(fd);
    }
    NPN_MemFree(url);
  }else{
    THIS->pid=my_fork(instance);

    if(THIS->pid==-1) return;
    
    if(!THIS->pid)
    {
#ifdef STREAMER
      if(THIS->flags & H_STREAM)
      {
	char foo[1];
	if(THIS->fd > -1) close(THIS->fd);
	D("Waiting for streaming child to exit.\n");
	while(read(THIS->waitfd, foo, 1) < 0 && errno==EINTR);
	if(THIS->repeats < MAXINT)
	  THIS->repeats--;
      }
#endif

      if(!find_command(instance, 0))
	if(!find_command(instance, 1))
	  exit(EX_UNAVAILABLE);

      run(instance, fname);
    }
  }
}
  
NPError NPP_Initialize(void)
{
  D("init\n");
  return NPERR_NO_ERROR;
}
jref NPP_GetJavaClass()
{
  D("Getjavaclass\n");
  return NULL;
}
void NPP_Shutdown(void)
{
  D("Shutdown\n");
}

NPError NPP_SetWindow(NPP instance, NPWindow* window)
{
  D("SETWINDOW\n");

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  if (!window)
    return NPERR_NO_ERROR;
  
  if (!window->window)
    return (NPERR_NO_ERROR);
  
  if(THIS->window == (Window) window->window)
  {
    /* Window size change.... */
  }else{
    /* New window */
    THIS->window = (Window) window->window;
    D("Received window %x\n",THIS->window);
  }

  return NPERR_NO_ERROR;
}

void NPP_Print(NPP instance, NPPrint* printInfo)
{
  D("PRINT\n");
  if(printInfo == NULL)
    return;
  
  if (instance != NULL)
  {
    if (printInfo->mode == NP_FULL) {
      void* platformPrint =
	printInfo->print.fullPrint.platformPrint;
      NPBool printOne =
	printInfo->print.fullPrint.printOne;
      
      /* Do the default*/
      printInfo->print.fullPrint.pluginPrinted = FALSE;
    }
    else
    {	/* If not fullscreen, we must be embedded */
      NPWindow* printWindow =
	&(printInfo->print.embedPrint.window);
      void* platformPrint =
	printInfo->print.embedPrint.platformPrint;
    }
  }
}


