(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 gelim.ml                                 *)
(****************************************************************************)

open Std;;
open Initial;;
open More_util;;
open Pp;;

open Names;;
open Generic;;
open Term;;
open Printer;;
open Termenv;;
open Reduction;;
open Mach;;
open Trad;;
open Tacmach;;
open Proof_trees;;
open Clenv;;
open Tactics;;
open Wcclausenv;;
open Pattern;;
open Tacticals;;
open Tactics;;
open Elim;;
open Equality;;
open Auto;;

(* I guess that these tactics are used to combine induction with
   inversion. In other words, while the inversion tactics provide a
   special form of case analysis, these tactics provide a special for
   of recursive definition. I have renamed them as "InvElim" instead
   of the former GElim (?).

   Exemple : Consider the following predicate on booleans :

     Inductive P : bool->Prop := 
            c1 : (P true)->(P true)  | 
            c2 : (P true)            | 
            c3 : (P false)->(P false).

   In order to prove ~(P false) we have to proceed by induction. But 
   an application of Elim will leave the unsolvable case corresponding
   to the constructor c2. We hence need to combine inversion and 
   recursion.


   Goal ~(P false).
   Red;Intro H.
   InvElim_clear H;Trivial.
   Qed.

   Eduardo (7/8/97).
   PS : The following comments are from Chet.
*)

(* [make_inv_predicate (ity,args) C]
  
   is given the inductive type, its arguments, both the global
   parameters and its local arguments, and is expected to produce a
   predicate P such that if largs is the "local" part of the
   arguments, then (P largs) will be convertible with a conclusion of
   the form:

   <A1>a1=a1-><A2>a2=a2 ... -> C

   Algorithm: suppose length(largs)=n

   (1) Push the entire arity, [xbar:Abar], carrying along largs and
   the conclusion

   (2) Pair up each ai with its respective Rel version: a1==(Rel n),
   a2==(Rel n-1), etc.

   (3) For each pair, ai,Rel j, if the Ai is dependent - that is, the
   type of [Rel j] is an open term, then we construct the iterated
   tuple, [make_iterated_tuple] does it, and use that for our equation

   Otherwise, we just use <Ai>ai=Rel j

 *)

let named_push_and_liftl n env t l = 
  let rec pushrec = fun p_0 p_1 p_2 ->
    match p_0,p_1,p_2 with
	(0, t, (env,l)) -> (env,t,l)
      | (n, (DOP2(Prod,t,DLAM(na,b))), (env,l)) ->
          pushrec (n-1) b (push_and_lift (Environ.named_hd t na,t) env l)
      | (n, (DOP2(Cast,t,_)), (env,l)) -> pushrec n t (env,l)
      | (_, _, _) -> error "push_and_liftl"
  in
  pushrec n t (env,l)
;;

let mmk = make_module_marker ["#Prelude.obj"];;
let eq_constant = put_pat mmk "eq";;

let make_inv_predicate sigma sign (ity,args) c =
  let nparams = mind_nparams ity in
  let (globargs,largs) = chop_list nparams args in
  let arity = hnf_prod_applist sigma "make_inv_predicate" 
		(mind_arity ity) globargs in
  let len = List.length largs in
  let env = []
  and largs = (List.map insert_lifted largs) in
  let env,_,largs = named_push_and_liftl len env arity largs in

  (* Now the arity is pushed, and we need to construct the pairs
   * ai,Rel(n-i+1)
   *)
  let larg_var_list = 
    map_i (fun i ai -> insert_lifted(DOP2(Implicit,extract_lifted ai,
					  Rel(len-i+1)))) 1 largs
  in
  (* Now, we can recurse down this list, for each ai,(Rel k) whether to
     push <Ai>(Rel k)=ai (when   Ai is closed).

     In any case, we carry along the rest of larg_var_list *)

  let rec build_concl (env,l) =
    match l with
	[] ->
	  let neqns = ((List.length env)-len) in
	  let env,c,_ = prod_and_popl neqns env c [] in
	  let _,c,_ = lam_and_popl (List.length env) env c [] in
	  (c,neqns)
	       
      | t::restlist ->
	 (match extract_lifted t with
          DOP2(Implicit,ai,Rel k) -> 
	  let tk = type_of_rel (Evd.mt_evd()) 
		     (castify_env (Evd.mt_evd()) (ENVIRON(sign,env))) (Rel k) 
	  in
	  let (lhs,eqnty,rhs) =
	    if closed0 tk then
	      (Rel k,tk,ai)
	    else
	      make_iterated_tuple (Evd.mt_evd()) 
		(castify_env (Evd.mt_evd()) (ENVIRON(sign,env)))
                (ai,type_of (Evd.mt_evd()) sign ai)
                (Rel k,tk)
	  in
	  let eqn = applist(get_pat eq_constant,[eqnty;lhs;rhs]) in
	  build_concl(push_and_lift (Anonymous,eqn) env restlist)
          | _ -> assert false)

  in
  
  let (predicate,neqns) = build_concl(env,larg_var_list) in

   (* OK - this predicate should now be usable by res_elimination_then to
      do elimination on the conclusion. *)
               
  (predicate,neqns)
;;

let apply c gl =
    let (wc,kONT) = startWalk gl in
    let clause = mk_clenv_hnf_constr_type_of wc c
    in res_pf kONT clause gl
;;

let res_elimination_then tac indbinding c gl =
  let (wc,kONT) = startWalk gl in

  let t = strong_prodspine (pf_whd_betadeltaiota gl) (pf_type_of gl c) in
  
  let indclause = mk_clenv_from wc (c,t) in
  
  let indclause' = clenv_constrain_with_bindings indbinding indclause in
  
  let newc = clenv_instance_template indclause' in
  
  let (ity,args) = decomp_app (clenv_instance_template_type indclause') in
  
  let (elim_predicate,neqns) =
    make_inv_predicate (project gl) (pf_hyps gl) (ity,args) (pf_concl gl) in

  let nparams = mind_nparams ity in
  
  let largs = snd(chop_list nparams args) in

  let cut_concl = applist(elim_predicate,largs) in

    tclTHENS 
      (cut_intro cut_concl)
      [onLastHyp 
	 (fun id ->
            (tclTHEN 
	       (applyUsing 
		  (applist(VAR (outSOME id),
			   tabulate_list 
			     (fun _ -> DOP0(Meta (newMETA()))) neqns)))
	       default_auto));
       elimination_then_using (tac neqns) (Some elim_predicate) ([],[]) newc
      ]
      gl
;;
  
let res_elim_then tac id = res_elimination_then tac [] (VAR id) ;;

(* The result of the elimination is a bunch of goals like:

           |- (cibar:Cibar)Equands->C

   where the cibar are either dependent or not.  We are fed a
   signature, with "true" for every recursive argument, and false for
   every non-recursive one.  So we need to do the
   sign_branch_len(sign) intros, thinning out all recursive
   assumptions.  This leaves us with exactly length(sign) assumptions.

   We save their names, and then do introductions for all the equands
   (there are some number of them, which is the other argument of the
   tactic)

   This gives us the #neqns equations, whose names we get also, and
   the #length(sign) arguments.

   Suppose that #nodep of these arguments are non-dependent.
   Generalize and thin them.

   This gives us #dep = #length(sign)-#nodep arguments which are
   dependent.

   Now, we want to take each of the equations, and do all possible
   injections to get the left-hand-side to be a variable.  At the same
   time, if we find a lhs/rhs pair which are different, we can
   discriminate them to prove false and finish the branch.

   Then, we thin away the equations, and do the introductions for the
   #nodep arguments which we generalized before.
 *)

let last_n_hyp_clauses n gl =
    tabulate_list (fun i -> nth_clause (-(i+1)) gl) n
;;

let var_occurs_in_pf gl id =
    occur_var id (pf_concl gl) or
    exists_sign (fun _ t -> occur_var id t) (pf_untyped_hyps gl)
;;

let split_dep_and_nodep idl gl =
    (filter (var_occurs_in_pf gl) idl,
     filter (fun x -> not (var_occurs_in_pf gl x)) idl)
;;

(* Called after the case-assumptions have been killed off, and all the
   intros have been done.  Given that the clause in question is an
   equality (if it isn't we fail), we are responsible for projecting
   the equality, using Injection and Discriminate, and applying it to
   the concusion *)

let mmk = make_module_marker ["#Prelude.obj"];;
let eq_pattern = put_pat mmk "(eq ? ? ?)";;

let constructor_heads_match gls m n =
    match (pf_whd_betadeltaiota_stack gls m [],
           pf_whd_betadeltaiota_stack gls n []) with
    ((DOPN(MutConstruct(_,i),_),_),
     (DOPN(MutConstruct(_,j),_),_)) ->
    i=j 
  | _ -> error "ProjectAndApply"
;;

(* invariant: ProjectAndApply is responsible for erasing the clause
   which it is given as input *)
let projectAndApply cls gls =

  match dest_match gls (clause_type cls gls) eq_pattern with
  [t;t1;t2] -> 
  
  if (match strip_outer_cast t1 with VAR _ -> true | _ -> false) or
    (match strip_outer_cast t2 with VAR _ -> true | _ -> false) then
      (tclTHEN 
	 (tclTRY (tclORELSE (hypSubst (outSOME cls) None)
		    (revHypSubst (outSOME cls) None)))
	 (clear_clause cls)) 
      gls

  else
    let deq_trailer neqns =
      tclDO neqns 
	(tclTHEN 
	   (tclTHEN
	      intro
	      (onLastHyp (fun id -> tclTRY(hypSubst (outSOME id) None))))
           (onLastHyp clear_clause))
    in
    (tclTHEN (dEqThen deq_trailer cls) (clear_clause cls)) gls

  | _ -> anomaly "projectAndApply: bad eq_pattern"
;;

let case_trailer neqns ba gl =
  let nargs = List.length ba.assums
  and argids = ba.assums in
  let depids = filter (var_occurs_in_pf gl) argids
  and nodepids = filter (fun x -> not (var_occurs_in_pf gl x)) argids in
  (tclTHEN 
     (tclTHEN 
	(tclTHEN 
	   (tclTHEN 
	      (tclTHEN 
		 (tclTHEN 
		    (tclTHEN 
		       (tclDO neqns intro)
		       (bring_hyps (List.map inSOME nodepids)))
		    (clear_clauses (List.map inSOME nodepids)))
		 (onCL (comp List.rev (nLastHyps neqns)) bring_hyps))
	      (onCL (nLastHyps neqns) clear_clauses))
	   (tclDO neqns
	      (tclTHEN 
		 intro 
		 (onLastHyp (fun cls -> tclTRY (projectAndApply cls))))))
	(tclDO (List.length nodepids) intro))
     (tclMAP (fun id -> tclTRY (clear_clause (Some id))) depids))
  gl
;;

let inv id gls =
    (tclTHEN 
       (res_elim_then 
	  (fun neqns -> introElimAssumsThen (case_trailer neqns)) id)
       (clear_clause (Some id))) 
  gls
;;

let inv_tac = hide_ident_tactic 
                  "InvElimClear" (fun id gls -> inv id gls);;

let half_case_trailer neqns ba gl =
  let nargs = List.length ba.assums
  and argids = ba.assums in
  let depids = filter (var_occurs_in_pf gl) argids
  and nodepids = filter (fun x -> not (var_occurs_in_pf gl x)) argids in
  (tclTHEN 
     (tclTHEN 
	(tclTHEN 
	   (tclTHEN 
	      (tclTHEN 
		 (tclDO neqns intro)
		 (bring_hyps (List.map inSOME nodepids)))
	      (clear_clauses (List.map inSOME nodepids)))
	   (onCL ((comp List.rev ((nLastHyps neqns)))) bring_hyps))
	(onCL (nLastHyps neqns) clear_clauses))
     (tclDO neqns intro))
  gl
;;

let halfInv id gls =
  (res_elim_then 
     (fun neqns -> introElimAssumsThen (half_case_trailer neqns)) id)
  gls
;;

let halfInv_tac = hide_ident_tactic "InvElim" halfInv;;

let introsReplacing ids gls = 
  let rec introrec = function
      [] -> tclIDTAC
    | id::tl -> 
      	(tclTHEN
           (tclORELSE
	      (intro_replacing id)
	      (tclORELSE (intro_erasing id) (intro_using id)))
           (introrec tl))
  in
  introrec ids gls
;;

(* InvIn will bring the specified clauses into the conclusion, and then
 * perform inversion on the named hypothesis.  After, it will intro them
 * back to their places in the hyp-list.
 *)
let invIn id ids gls =
  (tclTHEN
     (tclTHEN (bring_hyps (List.map inSOME ids)) (inv id))
     (introsReplacing ids))
  gls
;;

let invIn_tac =
  let gentac = 
    hide_tactic "InvElimIn"
      (function ((IDENTIFIER id)::hl) ->
	 invIn id (List.map (function (IDENTIFIER id) -> id
			         | _ -> invalid_arg "gentac") hl)
      | _ -> invalid_arg "gentac")
  in
  fun id hl ->
    gentac ((IDENTIFIER id)::(List.map (fun id -> (IDENTIFIER id)) hl))
;;


(* $Id: gelim.ml,v 1.19 1999/08/06 20:49:19 herbelin Exp $ *)
