/*
  main.cc

  Copyright (c) 1996 Roland Wunderling, Malte Zoeckler
  Copyright (c) 1998 Michael Meeks
  Copyright (c) 1998-99 Dragos Acostachioaie

  This file is part of DOC++.

  DOC++ is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License as published by the Free Software Foundation; either
  version 2 of the license, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <assert.h>
#include <fstream.h>
#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#endif
#include <stdio.h>

#include "McString.h"
#include "doc.h"
#include "nametable.h"
#include "tex2gif.h"

#define	GIF_FILE_NAME	"gifs.db"
#define DOC_ENV_VAR     "DOC"

bool	onlyDocs       		= true;
bool	commentExtn    		= false;
bool	HTMLsyntax     		= false;
bool	java           		= false;
bool	trivialGraphs  		= false;
bool	noDefines		= false;
bool	withPrivate    		= false;
bool	fastNotSmall   		= false;
bool	QuantelExtn    		= false;
bool	doTeX          		= false;
bool	upArrows       		= false;
bool	verb			= false;

bool	withTables     		= false;
bool	withBorders    		= false;
char	*ownFooter      	= 0;
bool	showFilenames  		= false;
bool	showFilePath   		= false;
bool	noGifs         		= false;
bool	forceGifs      		= false;
bool	showInherited  		= true;
bool	javaGraphs     		= true;
bool	alwaysPrintDocSection   = true;
bool	showMembersInTOC	= false;
bool	useGeneral     		= true;
bool	shortFilenames 		= false;
bool	sortEntries    		= false;
char	*ownHeader      	= 0;

bool	onlyClassGraph		= false;
char	*texFile        	= 0;
bool	generateIndex		= false;
char	*texOption      	= 0;
char	*texPackages[MAX_TEX_PACKAGES];
int	numTexPackages 		= 0;
char	*texTitle       	= 0;
int	depthTOC		= 1;
bool	noLatex        		= false;
bool	noClassGraph		= false;
char	*texOutputName		= 0;

FILE *out;
Entry *root;
NameTable gifs;

extern void doHTML(char *, Entry *);
extern void mergeCCFunctions(Entry *);

int getNumChildren(Entry *tp)
{
	Entry* tmp=tp;
	int	num=0;

	for (tmp=tp->sub; tmp; tmp=tmp->next)
		num++;

	return num;
}

void checkPackages (Entry* tp)
{
    Entry	*tmp = tp,
		*child,
		*tmp2,
		*tmp3;
    McString 	 newName;


    for (tmp=tp; tmp; tmp=tmp->next)
    {
	if (tmp->sub)
		checkPackages(tmp->sub);

	if (MAKE_DOC(tmp))
		if ((tmp->section==PACKAGE_SEC) && 
		    (getNumChildren(tmp) == 1))
		{
			child = tmp->sub;

			if (child->section==PACKAGE_SEC){
				newName.clear();
				newName += tmp->name;
				newName += "."+(tmp->sub)->name;

				tmp2 = tmp->sub;
				tmp2->parent = tmp->parent;

				for (tmp3=tmp2->sub; tmp3; tmp3=tmp3->next)
					tmp3->parent = tmp;

				tmp->language = tmp2->language;
				tmp->section = tmp2->section;
				tmp->general = tmp2->general;
				tmp->protection = tmp2->protection;
				tmp->ownPage = tmp2->ownPage;
				tmp->sub = tmp2->sub;
				tmp->type = tmp2->type;
				tmp->name = newName;
				tmp->args = tmp2->args;
				tmp->memo = tmp2->memo;
				tmp->doc = tmp2->doc;
				tmp->program = tmp2->program;
				tmp->author = tmp2->author;
				tmp->version = tmp2->version;
				tmp->see = tmp2->see;
				tmp->param = tmp2->param;
				tmp->exception = tmp2->exception;
				tmp->retrn = tmp2->retrn;
				tmp->sublist = tmp2->sublist;
				tmp->pubChilds = tmp2->pubChilds;
				tmp->proChilds = tmp2->proChilds;
				tmp->priChilds = tmp2->priChilds;
				tmp->baseclasses = tmp2->baseclasses;
				tmp->pubBaseclasses = tmp2->pubBaseclasses;
				tmp->proBaseclasses = tmp2->proBaseclasses;
				tmp->priBaseclasses = tmp2->priBaseclasses;
				tmp->otherPubBaseclasses = tmp2->otherPubBaseclasses;
				tmp->otherProBaseclasses = tmp2->otherProBaseclasses;
				tmp->otherPriBaseclasses = tmp2->otherPriBaseclasses;
				tmp->implements = tmp2->implements;
				tmp->extends = tmp2->extends;
				tmp->import = tmp2->import;
				tmp->override = tmp2->override;
			}
		}
	}
}

static void	reNumber( Entry* tp )
{
    Entry* tmp = tp ;
    int	i = 1 ;
    for( tmp = tp->sub ; tmp ; tmp = tmp->next )
    {
	if( MAKE_DOC(tmp) )
	    tmp->subnumber = i++ ;
	reNumber( tmp ) ;
    }
    for( tmp = tp->sub ; tmp ; tmp = tmp->next )
    {
	if( !MAKE_DOC(tmp) )
	    tmp->subnumber = i++ ;
    }
}

void help()
{
  printf("DOC++ %s, a documentation system for C, C++ and Java\n",
    DOCXX_VERSION);
  printf("Usage: doc++ [options] [files]\n\n");
  printf("Options:\n");
  printf("  -A  --all              document all\n");
  printf("  -c  --c-comments       use C++ `//' comments as doc++-comments\n");
  printf("  -h  --help             display this help and exit\n");
  printf("  -H  --html             parse HTML syntax, not only TeX\n");
  printf("  -I  --input FILE       read a list of input files from FILE\n");
  printf("  -J  --java             parse Java instead of C/C++\n");
  printf("  -nd --no-define        ignore `#define' macros\n");
  printf("  -p  --private          document private members\n");
  printf("  -q  --quick            optimise for speed instead of size\n");
  printf("  -Q  --quantel          Quantel extensions, folding & appended comments\n");
  printf("  -t  --tex              generate TeX output instead of HTML\n");
  printf("  -u  --upwards-arrows   upwards arrows in class graphs\n");
  printf("  -v  --verbose          turn verbose mode on\n");
  printf("  -V  --version          output version information and exit\n\n");

  printf("Additional options for HTML output:\n");
  printf("  -a  --tables           use tables instead of description lists\n");
  printf("  -b  --tables-border    use tables with borders\n");
  printf("  -B  --footer FILE      use FILE as footer on HTML pages\n");
  printf("  -d  --dir DIR          use DIR for the output directory\n");
  printf("  -f  --filenames        show filenames in manual pages\n");
  printf("  -F  --filenames-path   show filenames with path in manual pages\n");
  printf("  -g  --no-gifs          do not generate GIFs for equations, etc.\n");
  printf("  -G  --gifs             force generation of GIFs\n");
  printf("  -i  --no-inherited     don't show inherited members\n");
  printf("  -j  --no-java-graphs   suppress Java class graphs\n");
  printf("  -k  --trivial-graphs   keep trivial class graphs\n");
  printf("  -m  --no-members       don't show all members in DOC section\n");
  printf("  -M  --full-toc         show members in TOC\n");
  printf("  -P  --no-general       discard general stuff\n");
  printf("  -r  --short-filenames  use short filenames\n");
  printf("  -S  --sort             sort entries alphabetically\n");
  printf("  -T  --header FILE      use FILE as header on HTML pages\n\n");

  printf("Additional options for TeX output:\n");
  printf("  -ec --class-graph      only generates class graph\n");
  printf("  -ef --env FILE         read TeX environment from FILE\n");
  printf("  -ei --index            generate index\n");
  printf("  -eo --style OPTION     setup TeX style OPTION\n");
  printf("  -ep --package FILE     setup TeX to use package FILE\n");
  printf("  -et --title FILE       use content of FILE as TeX title page\n");
  printf("  -D  --depth DEPTH      set minimum depth (number of levels) in TOC\n");
  printf("  -l  --no-env           do not generate TeX environment\n");
  printf("  -ng --no-class-graph   do not generate class graph\n");
  printf("  -o  --output FILE      set output file name\n");
  printf("  -s  --source           generate source code listing\n\n");
  printf("Mail bug reports and suggestions to <docpp@egroups.com>\n");
}

void missing_arg(char *s)
{
    fprintf(stderr, "Ignoring option `%s': required argument missing\n", s);
}

void missing_arg_e(char c)
{
    fprintf(stderr, "Ignoring option `-e%c': required argument missing\n", c);
}

main(int argc, char **argv)
{
    root = new Entry ;
    McString&	inputFile = root->program ;

    char	Directory[3] = "./" ;
    Directory[1]=PATH_DELIMITER;
    char	*directory  = Directory ;

    int		doListing = 0 ;
    int 	i, c, depth;
    int		gifNum = 0;
    int 	useFileList = 0;
    McString	fileList;

    out  = stdout ;

    // Stick all the arguments in 'args' instead.
    McDArray<char*> args ;
    char *env = getenv(DOC_ENV_VAR) ;
    if (env)
	env = strdup(env) ;
    char *start = env ;
    while (env && *env)
	if (*env == ' ') {
		*env=0 ;
		if (strlen(start)>0)
		    args.append(start) ;
		start = ++env ;
	    }
	else
	    env++ ;
    if (start && strlen(start)>0)	// The last entry
      args.append(start) ;
    for(i = 1; i < argc; i++)	// Append filenames at end !
	args.append(argv[i]) ;

#ifdef HAVE_GETOPT_LONG
    static struct option long_options[] = {

	// general options
	{ "all", no_argument, 0, 'A' },
        { "c-comments", no_argument, 0, 'c' },
        { "help", no_argument, 0, 'h' },
        { "html", no_argument, 0, 'H' },
        { "java", no_argument, 0, 'J' },
        { "trivial-graphs", no_argument, 0, 'k' },
        { "no-defines", no_argument, 0, 'n' },
        { "private", no_argument, 0, 'p' },
        { "quick", no_argument, 0, 'q' },
        { "quantel", no_argument, 0, 'Q' },
        { "tex", no_argument, 0, 't' },
        { "upwards-arrows", no_argument, 0, 'u' },
        { "verbose", no_argument, 0, 'v' },
        { "version", no_argument, 0, 'V' },

        // additional options for HTML output
        { "tables", no_argument, 0, 'a' },
        { "tables-border", no_argument, 0, 'b' },
        { "footer", required_argument, 0, 'B' },
        { "dir", required_argument, 0, 'd' },
        { "filenames", no_argument, 0, 'f' },
        { "filenames-path", no_argument, 0, 'F' },
        { "no-gifs", no_argument, 0, 'g' },
        { "gifs", no_argument, 0, 'G' },
        { "no-inherited", no_argument, 0, 'i' },
        { "no-java-graphs", no_argument, 0, 'j' },
        { "no-members", no_argument, 0, 'm' },
        { "full-toc", no_argument, 0, 'M' },
        { "no-general", no_argument, 0, 'P' },
        { "short-filenames", no_argument, 0, 'r' },
        { "sort", no_argument, 0, 'S' },
        { "header", required_argument, 0, 'T' },

        // additional options for TeX output
        { "class_graph", no_argument, 0, '1' },
        { "env", required_argument, 0, '2' },
        { "index", no_argument, 0, '3' },
        { "style", required_argument, 0, '4' },
        { "package", required_argument, 0, '5' },
        { "title", required_argument, 0, '6' },
        { "depth", required_argument, 0, 'D' },
        { "no-env", no_argument, 0, 'l' },
        { "no-class-graph", no_argument, 0, '7' },
        { "output", required_argument, 0, 'o' },
        { "source", no_argument, 0, 's' },
        { 0, 0, 0, 0 }
    };

    opterr = 0;

    while((c = getopt_long(argc, argv, "+AchHI:JknpqQtuvVabB:d:fFgGijmMPrST:e:D:lo:s",
	long_options, (int *)0)) != EOF)
	switch(c)
	    {
	    case 'A':
		onlyDocs = false;
		withPrivate = true;
		break;
	    case 'c':
		commentExtn = true;
		break;
            case 'h':
		help();
		exit(0);
	    case 'H':
		HTMLsyntax = true;
		break;
	    case 'I':
		if(optarg[0] == '-')
		    {
		    missing_arg("--input");
		    break;
		    }
		useFileList = true;
		fileList = optarg;
		break;
	    case 'J':
		HTMLsyntax = true;
		java = true;
		break;
    	    case 'k':
    		trivialGraphs = true;
		break;
	    case 'n':
		noDefines = true;
		break;
	    case 'p':
    		withPrivate = true;
		break;
	    case 'q':
    		fastNotSmall = true;
		break;
	    case 'Q':
    		QuantelExtn = true;
		break;
    	    case 't':
    		doTeX = true;
		break;
	    case 'u':
    		upArrows = true;
		break;
	    case 'v':
    		verb = true;
		break;
	    case 'V':
    		printf("DOC++ %s\n", DOCXX_VERSION);
		exit(0);
	    case 'a':
    		withTables = true;
		break;
	    case 'b':
    		withTables = true;
		withBorders = true;
		break;
	    case 'B':
		if(optarg[0] == '-')
		    {
		    missing_arg("--footer");
		    break;
		    }
    		ownFooter = optarg;
		break;
	    case 'd':
		if(optarg[0] == '-')
		    {
		    missing_arg("--dir");
		    break;
		    }
    		directory = optarg;
		break;
	    case 'f':
    		showFilenames = true;
		break;
	    case 'F':
    		showFilenames = true;
		showFilePath = true;
		break;
	    case 'g':
    		noGifs = true;
		break;
	    case 'G':
    		forceGifs = true;
		break;
	    case 'i':
    		showInherited = false;
		break;
	    case 'j':
    		javaGraphs = false;
		break;
	    case 'm':
    		alwaysPrintDocSection = false;
		break;
	    case 'M':
    		showMembersInTOC = true;
		break;
            case 'P':
    		useGeneral = false;
		break;
	    case 'r':
    		shortFilenames = true;
		break;
	    case 'S':
    		sortEntries = true;
		break;
	    case 'T':
		if(optarg[0] == '-')
		    {
		    missing_arg("--header");
		    break;
		    }
    		ownHeader = optarg;
		break;
	    case '1':
    		onlyClassGraph = true;
		break;
	    case '2':
		if(optarg[0] == '-')
		    {
		    missing_arg("--env");
		    break;
		    }
    		texFile = optarg;
		break;
	    case '3':
    		generateIndex = true;
		break;
	    case '4':
		if(optarg[0] == '-')
		    {
		    missing_arg("--style");
		    break;
		    }
    		texOption = optarg;
		break;
	    case '5':
		if(optarg[0] == '-')
		    {
		    missing_arg("--package");
		    break;
		    }
		if(numTexPackages < MAX_TEX_PACKAGES - 1)
	          texPackages[numTexPackages++] = optarg;
		break;
	    case '6':
		if(optarg[0] == '-')
		    {
		    missing_arg("--title");
		    break;
		    }
    		texTitle = optarg;
		break;
	    case '7':
    		noClassGraph = true;
		break;
	    case 'D':
		if(optarg[0] == '-')
		    {
		    missing_arg("--depth");
		    break;
		    }
    		if(!sscanf(optarg, "%d", &depth))
	    	    fprintf(stderr, "Ignoring option `--depth': bad argument specified\n");
		else
		    depthTOC = depth;
		break;
	    case 'l':
    		noLatex = true;
		break;
	    case 'o':
		if(optarg[0] == '-')
		    {
		    missing_arg("--output");
		    break;
		    }
		FILE *tmp;
		if((tmp = fopen(optarg, "w")))
		    {
	            texOutputName = optarg;
		    out = tmp;
		    }
		else
		    fprintf(stderr, "Ignoring option `--output': cannot open file `%s'\n",
			optarg);
		break;
	    case 's':
    		doListing = true;
		break;
	    case 'e':
    		switch(optarg[0])
	    	    {
		    case 'c':
        		onlyClassGraph = true;
			break;
		    case 'f':
			if(argv[optind][0] == '-')
	            	    {
			    missing_arg_e('f');
			    break;
			    }
	                break;
    			texFile = argv[optind++];
			break;
		    case 'i':
        		generateIndex = true;
			break;
		    case 'o':
			if(argv[optind][0] == '-')
			    {
			    missing_arg_e('o');
			    break;
			    }
    			texOption = argv[optind++];
			break;
		    case 'p':
			if(argv[optind][0] == '-')
			    {
			    missing_arg_e('p');
			    break;
			    }
			if(numTexPackages < MAX_TEX_PACKAGES - 1)
	    		    texPackages[numTexPackages++] = argv[optind++];
			break;
		    case 't':
			if(argv[optind][0] == '-')
			    {
			    missing_arg_e('t');
			    break;
			    }
        		texTitle = argv[optind++];
			break;
		    default:
			fprintf(stderr, "Unknown option `-e%c'. Try `%s --help'\n", optarg[0],
	    		    argv[0]);
			exit(-1);
		    }
    		break;
	    case '?':
		fprintf(stderr, "Unknown option `-%c'. Try `%s --help'\n", optopt, argv[0]);
    		exit(-1);
            }

    if(optind >= argc && !useFileList)
	{
	fprintf(stderr, "No input files. Try `%s --help'\n", argv[0]);
	exit(-1);
	}

#else
    for(i = 0; i < args.size(); ++i)
	{
	if(args[i][0] != '-')
    	    break;
	switch(args[i][1])
	    {
            case 'A':
		onlyDocs = false;
		withPrivate = true;
		break;
            case 'c':
		commentExtn = true;
		break;
	    case 'h':
		help();
		exit(0);
	    case 'H':
		HTMLsyntax = true;
		break;
	    case 'I':
		if(++i >= args.size())
		    {
		    missing_arg("-I");
		    --i;
		    }
		else
		    {
		    useFileList = true;
		    fileList = args[i];
		    }
		break;
	    case 'J':
		java = true;
		break;
	    case 'k':
		trivialGraphs = true;
		break;
	    case 'n':
    		switch(args[i][2])
		    {
		    case 'd':
			noDefines = true;
			break;
		    case 'g':
			noClassGraph = true;
			break;
		    default:
			fprintf(stderr, "Ignoring unknown option `-n%c'\n", args[i][2]);
			break;
	            }
		break;
	    case 'p':
		withPrivate = true;
		break;
	    case 'q':
		fastNotSmall = true;
		break;
	    case 'Q':
		QuantelExtn = true;
		break;
	    case 't':
		doTeX = true;
		break;
	    case 'u':
		upArrows = true;
		break;
	    case 'v':
		verb = true;
		break;
	    case 'V':
		printf("DOC++ %s\n", DOCXX_VERSION);
		exit(0);
	    case 'a':
		withTables = true;
		break;
	    case 'b':
		withTables = true;
		withBorders = true;
		break;
	    case 'B':
		if(++i >= args.size())
		    {
		    missing_arg("-B");
		    --i;
	    	    }
		else
		    ownFooter = args[i];
		    break;
	    case 'd':
		if(++i >= args.size())
		    {
		    missing_arg("-d");
		    --i;
	            }
		else
		    directory = args[i];
		    break;
	    case 'f':
		showFilenames = true;
		showFilePath = false;
		break;
	    case 'F':
		showFilenames = true;
		showFilePath  = true;
		break;
	    case 'g':
		noGifs = true;
		break;
	    case 'G':
		forceGifs = true;
		break;
	    case 'i':
		showInherited = false;
		break;
	    case 'j':
		javaGraphs = false;
		break;
	    case 'm':
		alwaysPrintDocSection = false;
		break;
	    case 'M':
    		showMembersInTOC = true;
		break;
	    case 'P':
		useGeneral = false;
		break;
	    case 'r':
		shortFilenames = true;
		break;
	    case 'S':
		sortEntries = true;
		break;
	    case 'T':
		if(++i >= args.size())
		    {
		    missing_arg("-T");
		    --i;
		    }
		else
		    ownHeader = args[i];
		break;
	    case 'e':
		switch(args[i][2])
		    {
		    case 'c':
			onlyClassGraph = true;
			break;
		    case 'f':
			if(++i >= args.size())
			    {
			    missing_arg_e('f');
			    --i;
			    }
			else
			    texFile = args[i];
			break;
		    case 'i':
			generateIndex = true;
	    		break;
		    case 'o':
			if(++i >= args.size())
			    {
			    missing_arg_e('o');
			    --i;
			    }
			else
			    texOption = args[i];
			break;
		    case 'p':
			if(++i >= args.size())
			    {
			    missing_arg_e('p');
			    --i;
			    }
			else
			    if(numTexPackages < MAX_TEX_PACKAGES - 1)
				texPackages[numTexPackages++] = args[i];
			break;
		    case 't':
			if(++i >= args.size())
			    {
			    missing_arg_e('t');
			    --i;
			    }
			else
			    texTitle = args[i];
			break;
		    }
		break;
	    case 'D':
    		if(++i >= args.size() || !sscanf(args[i], "%d", &depth))
		    {
 		    fprintf(stderr, "Ignoring option `-D': bad argument specified\n");
 		    --i;
 		    }
		else
		    depthTOC = depth;
 		    break;
	    case 'l':
		noLatex = true;
		break;
	    case 'o':
		if(++i >= args.size())
		    {
		    --i;
		    missing_arg("-o");
		    }
		else
		    {
		    FILE *tmp;
		    if((tmp = fopen(args[i], "w")))
			{
			texOutputName = args[i];
			out = tmp;
			}
		    else
			fprintf(stderr, "Ignoring option `-o': cannot open file `%s'\n", args[i]);
	            }
		break;
	    case 's':
		doListing = true;
		break;
	    default:
		fprintf(stderr, "Unknown option `-%c'. Try `%s -h'\n", args[i][1], argv[0]);
		exit(-1);
	    }
	}

    if(i >= args.size() && !useFileList)
      {
      fprintf(stderr, "No input files. Try `%s -h'\n", argv[0]);
      exit(-1);
      }
#endif

    // Preload GIFs
    if(!noGifs && !forceGifs && !doTeX)
	{
	McString gifDB(directory);
	gifDB += "/";			// this will have problems with msdog
	gifDB += GIF_FILE_NAME;

	ifstream gifFile((const char *)gifDB);
	if(gifFile)
	    {
            gifFile >> gifs;
            gifNum = gifs.num();
    	    }
	}

    // Read input files into buffer
    if(verb)
	printf("Reading files...\n");

    if(useFileList)
	{
	ifstream i_file (fileList);
	if(!i_file)
	    {
            fprintf(stderr, "Error opening file list `%s'\n", (char *)fileList);
            exit(-1);
            }

	char line[1024];
	while(i_file)
    	    {
            i_file.getline(line, 1024);
	    if(i_file)
		readfile(&inputFile, line, 1);
            }
	i_file.close();
	}
    else
#ifdef HAVE_GETOPT_LONG
    for(i = optind; i < argc; i++)
	readfile(&inputFile, argv[i], 1);
#else
    for(; i < args.size(); ++i)
	readfile(&inputFile, args[i], 1);
#endif
    if(verb)
	printf("%d bytes read\n", inputFile.length());

    // Do TeX source code listing if that is what the user want
    if(doTeX && doListing)
	{
	if(verb)
    	    printf("Generating source code listing...\n");
	listing(inputFile);
	if(verb)
	    printf("Done.\n");
	exit(0);
    }

    // Parse buffer
    if(verb)
	printf("Parsing...\n");
    if(java)
	parseJava(root);
    else
	parseCpp(root);

    // Sort entries
    if(verb)
	printf("Sorting entries...\n");
    makeSubLists(root);

    // Create language hash
    if(verb)
	printf("Creating languages hash...\n");
    Entry::setupLangHash();

    // Resolve references
    if(verb)
	printf("Resolving references...\n");
    mergeCCFunctions(root);
    reNumber(root);
    root->makeRefs();
    checkPackages(root);

    // Create user manual
    if(doTeX)
	usermanTeX(inputFile, root);
    else
	{
	usermanHTML(inputFile, root);
	doHTML(directory, root);
	}

    // Create GIFs (if any)
    if(gifNum > 0)
	{
	McString gifDB(directory);
	gifDB += "/";
	gifDB += GIF_FILE_NAME;
	if(verb)
	    printf("Writing GIF database file `%s'...\n", (const char*)gifDB);

	ofstream gifFile((const char *)gifDB);
	gifFile << gifs ;
	makeGifs(directory, GIF_FILE_NAME);
	}

    // That's all
    if(verb)
	printf("Done.\n");

    exit(0);
}
