------------------------------------------------------------------------------
--                                                                          --
--                            GLADE COMPONENTS                              --
--                                                                          --
--                 S Y S T E M . R P C . S T R E A M _ I O                  --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.21 $
--                                                                          --
--         Copyright (C) 1996-1999 Free Software Foundation, Inc.           --
--                                                                          --
-- GARLIC is free software;  you can redistribute it and/or modify it under --
-- terms of the  GNU General Public License  as published by the Free Soft- --
-- ware Foundation;  either version 2,  or (at your option)  any later ver- --
-- sion.  GARLIC is distributed  in the hope that  it will be  useful,  but --
-- WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHANTABI- --
-- LITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public  --
-- License  for more details.  You should have received  a copy of the GNU  --
-- General Public License  distributed with GARLIC;  see file COPYING.  If  --
-- not, write to the Free Software Foundation, 59 Temple Place - Suite 330, --
-- Boston, MA 02111-1307, USA.                                              --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
--               GLADE  is maintained by ACT Europe.                        --
--               (email: glade-report@act-europe.fr)                        --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Exceptions;        use Ada.Exceptions;
with Ada.Streams;           use Ada.Streams;
with System.Garlic;         use System.Garlic;
with System.Garlic.Debug;   use System.Garlic.Debug;
with System.Garlic.Heart;   use System.Garlic.Heart;
with System.Garlic.Streams;
with System.Garlic.Types;
with System.Garlic.Utils;   use System.Garlic.Utils;

package body System.RPC.Stream_IO is

   --  This package needs comments ???

   Private_Debug_Key : constant Debug_Key :=
     Debug_Initialize ("S_RPSTIO", "(s-rpstio): ");
   procedure D
     (Message : in String;
      Key     : in Debug_Key := Private_Debug_Key)
     renames Print_Debug_Info;

   type Partition_Stream_Record is
      record
         Mode      : Stream_Mode;
         Incoming  : aliased Streams.Params_Stream_Type (0);
         Outgoing  : aliased Streams.Params_Stream_Type (0);
         Consumer  : Barrier_Type;
         Available : Mutex_Type;
         Critical  : Mutex_Type;
      end record;
   type Partition_Stream_Access is access Partition_Stream_Record;

   Msgcode : constant Any_Opcode := User_Message;

   Streams : array (Partition_ID'Range) of Partition_Stream_Access;

   Any : Partition_Stream_Access renames Streams (Any_Partition);

   First_Partition : Partition_ID := Partition_ID'Last;
   Last_Partition  : Partition_ID := Partition_ID'First;

   function Fetch
     (Partition : in Partition_ID)
     return Partition_Stream_Access;

   procedure Handle_Request
     (Partition : in Types.Partition_ID;
      Opcode    : in External_Opcode;
      Query     : access Garlic.Streams.Params_Stream_Type;
      Reply     : access Garlic.Streams.Params_Stream_Type;
      Error     : in out Error_Type);

   -----------
   -- Close --
   -----------

   procedure Close
     (Stream : in out Partition_Stream_Type)
   is
      Err : aliased Error_Type;
      Str : Partition_Stream_Access := Fetch (Stream.PID);
   begin
      pragma Debug (D ("Close stream" & Stream.PID'Img));

      if Str.Mode = Out_Mode then
         Send (Types.Partition_ID (Stream.PID),
               Msgcode,
               Str.Outgoing'Access,
               Err);
         if Found (Err) then
            Raise_Exception (Communication_Error'Identity,
                             Content (Err'Access));
         end if;
      end if;

      pragma Debug (D ("Close - Unlock stream" & Stream.PID'Img));
      Leave (Str.Available);
   end Close;

   -----------
   -- Fetch --
   -----------

   function Fetch
     (Partition : in Partition_ID)
     return Partition_Stream_Access is
   begin
      if Streams (Partition) = null then
         pragma Debug (D ("Allocate stream" & Partition'Img));
         Enter (Any.Critical);
         if Streams (Partition) = null then
            Streams (Partition) := new Partition_Stream_Record;
            Create (Streams (Partition).Consumer);
            Create (Streams (Partition).Available);
            Create (Streams (Partition).Critical);
            if First_Partition = Partition_ID'Last
              or else First_Partition > Partition
            then
               First_Partition := Partition;
            end if;
            if Last_Partition = Partition_ID'First
              or else Last_Partition < Partition
            then
               Last_Partition := Partition;
            end if;

         end if;
         Leave (Any.Critical);
      end if;
      return Streams (Partition);
   end Fetch;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
   begin
      Streams (Any_Partition) := new Partition_Stream_Record;
      Create (Streams (Any_Partition).Available);
      Create (Streams (Any_Partition).Critical);
      Register_Handler (Msgcode, Handle_Request'Access);
   end Initialize;

   ----------
   -- Open --
   ----------

   procedure Open
     (Stream    : in out Partition_Stream_Type;
      Partition : in     Partition_ID;
      Mode      : in     Stream_Mode) is
      Str : Partition_Stream_Access;
   begin
      pragma Debug (D ("Open stream" & Partition'Img));

      if Mode = Out_Mode
        and then Partition = Any_Partition
      then
         pragma Debug (D ("Can't write to all partitions"));
         raise Stream_Error;
      end if;

      Str := Fetch (Partition);
      Stream.PID := Partition;

      pragma Debug (D ("Open - Lock stream" & Partition'Img));
      Enter (Str.Available);
      Str.Mode := Mode;

      pragma Debug (D ("Open - Resume stream" & Partition'Img));
   end Open;

   ----------
   -- Read --
   ----------

   procedure Read
     (Stream : in out Partition_Stream_Type;
      Item   : out    Ada.Streams.Stream_Element_Array;
      Last   : out    Ada.Streams.Stream_Element_Offset) is
      FID : Partition_ID;
      LID : Partition_ID;
      Len : Stream_Element_Offset := 0;
      Str : Partition_Stream_Access := Fetch (Stream.PID);
   begin

      if Str.Mode /= In_Mode then
         pragma Debug (D ("Mode should be In_Mode"));
         raise Stream_Error;
      end if;

      pragma Debug (D ("Read new message"));

      while Len = 0 loop
         pragma Debug (D ("Read - Wait for stream" & Stream.PID'Img));
         Wait (Str.Consumer);

         if Stream.PID = Any_Partition then
            FID := First_Partition;
            LID := Last_Partition;
         else
            FID := Stream.PID;
            LID := Stream.PID;
         end if;

         for P in FID .. LID loop
            pragma Debug (D ("Read - Lock stream" & P'Img));
            Enter (Streams (P).Critical);

            pragma Debug (D ("Read from stream" & P'Img));
            System.Garlic.Streams.Read (Streams (P).Incoming, Item, Len);

            pragma Debug (D ("Read - Unlock stream" & P'Img));
            Leave (Streams (P).Critical);

            if Len /= 0 then
               if Streams (P).Incoming.Count /= 0 then
                  pragma Debug (D ("Read - Signal stream" & P'Img));
                  Signal (Streams (P).Consumer);
                  Signal (Any.Consumer);
               end if;
               exit;
            end if;
         end loop;

         exit when Len /= 0;
      end loop;
      Last := Len;
   end Read;

   --------------------
   -- Handle_Request --
   --------------------

   procedure Handle_Request
     (Partition : in Types.Partition_ID;
      Opcode    : in External_Opcode;
      Query     : access Garlic.Streams.Params_Stream_Type;
      Reply     : access Garlic.Streams.Params_Stream_Type;
      Error     : in out Error_Type) is
      SEA : Stream_Element_Array (1 .. Query.Count);
      Len : Stream_Element_Offset;
      Str : Partition_Stream_Access := Fetch (Partition_ID (Partition));
   begin
      pragma Debug (D ("Receive new message"));
      pragma Debug (D ("Receive - Lock stream" & Partition'Img));
      Enter (Str.Critical);

      Garlic.Streams.Read (Query.all, SEA, Len);
      Garlic.Streams.Write (Str.Incoming, SEA);

      pragma Debug (D ("Receive - Unlock stream" & Partition'Img));
      Leave (Str.Critical);

      pragma Debug (D ("Signal to all streams"));
      Signal (Str.Consumer);
      Signal (Any.Consumer);
   end Handle_Request;

   -----------
   -- Write --
   -----------

   procedure Write
     (Stream : in out Partition_Stream_Type;
      Item   : in     Ada.Streams.Stream_Element_Array) is
      Str : Partition_Stream_Access := Fetch (Stream.PID);
   begin
      pragma Debug (D ("Send new message"));

      if Str.Mode /= Out_Mode then
         pragma Debug (D ("Mode should be Out_Mode"));
         raise Stream_Error;
      end if;

      pragma Debug (D ("Write - Lock stream" & Stream.PID'Img));
      Enter (Str.Critical);

      pragma Debug (D ("Write to stream" & Stream.PID'Img));
      Garlic.Streams.Write (Str.Outgoing, Item);

      pragma Debug (D ("Write - Unlock stream" & Stream.PID'Img));
      Leave (Str.Critical);
   end Write;

end System.RPC.Stream_IO;
