/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: iprobe.c,v 6.6 1999/08/27 22:33:48 jsquyres Exp $
 *
 *	Function:	- non-blocking check of incoming messages
 *			- message is not received
 *	Accepts:	- source rank
 *			- message tag
 *			- communicator
 *			- flag (out)
 *			- status (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>
#include <terror.h>


int
MPI_Iprobe(src, tag, comm, flag, stat)

int			src;
int			tag;
MPI_Comm		comm;
int			*flag;
MPI_Status		*stat;

{
	struct _req	request_storage;	/* request structure */
	MPI_Request	req;			/* request */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	int		r;
	double		startt = 0.0;		/* start time */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIIPROBE);
/*
 * Check additional arguments.
 */
	if (flag == 0) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((tag < 0 && tag != MPI_ANY_TAG) || tag > LAM_MAXTAG) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_TAG, 0)));
	}
/*
 * Handle the trivial case.
 */
	if (src == MPI_PROC_NULL) {
		if (stat != MPI_STATUS_IGNORE) {
			stat->MPI_ERROR = MPI_SUCCESS;
			stat->MPI_SOURCE = MPI_PROC_NULL;
			stat->MPI_TAG = MPI_ANY_TAG;
			stat->st_length = 0;
		}
		
		*flag = 1;
		lam_resetfunc_m(BLKMPIIPROBE);
		return(MPI_SUCCESS);
	}

	if ((fl_trace = LAM_TRACE_TOP())) startt = MPI_Wtime();
/*
 * Advance the system to give the probe a better chance of success.
 */
	_mpi_req_blkclr();

	err = _mpi_req_advance();
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	LAM_ZERO_ME(request_storage);
	req = &request_storage;
	err = _mpi_req_build((char *) 0, 0, MPI_BYTE, src, tag, comm,
			LAM_RQIPROBE, &req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	err = _mpi_req_start(req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	r = RPI_SPLIT(_rpi_lamd_iprobe, _rpi_c2c_iprobe, (req));
	if (r < 0) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	err = _mpi_req_end(req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}
/*
 * Copy the status info if synchronized.
 */
	if (r == 1) {
		*flag = 1;
		if (stat != MPI_STATUS_IGNORE) {
			*stat = req->rq_status;
		}
	} else {
		*flag = 0;
	}

	err = _mpi_req_destroy(&req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}
/*
 * Generate a run time trace.
 */
	if (fl_trace) {
		lam_tr_msg(TRTNOIO, startt, LAM_S2US(MPI_Wtime() - startt),
			0, src, tag, comm, 0, 0, 0, 0, 0, LAM_RQIPROBE);
	}

	lam_resetfunc_m(BLKMPIIPROBE);
	return(MPI_SUCCESS);
}
