/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: tgetenvl.c,v 1.0 1999/07/21 19:00:17 jsquyres Exp $
 *
 *	Function:	- get datatype envelope counts
 *	Accepts:	- MPI datatype
 *			- number of integers (out)
 *			- number of addresses (out)
 *			- number of datatypes (out)
 *			- combiner (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


int
MPI_Type_get_envelope(dtype, nints, naddrs, ndtypes, combiner)

MPI_Datatype		dtype;
int			*nints;
int			*naddrs;
int			*ndtypes;
int			*combiner;

{
	lam_initerr_m();
	lam_setfunc_m(BLKMPITGETENV);
/*
 * Check arguments.
 */
	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITGETENV,
					lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (nints == 0 || naddrs == 0 || ndtypes == 0 || combiner == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITGETENV,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (dtype->dt_flags & LAM_DTISDUP) {
		*nints = 0;
		*naddrs = 0;
		*ndtypes = 1;
		*combiner = MPI_COMBINER_DUP;
	}
	else if (dtype->dt_flags & LAM_DTISRESIZE) {
		*nints = 0;
		*naddrs = 2;
		*ndtypes = 1;
		*combiner = MPI_COMBINER_RESIZED;
	}
	else {
		switch (dtype->dt_format) {

		case LAM_DTBASIC:
			*nints = 0;
			*naddrs = 0;
			*ndtypes = 0;
			*combiner = MPI_COMBINER_NAMED;
			break;

		case LAM_DTCONTIG:
			*nints = 1;
			*naddrs = 0;
			*ndtypes = 1;
			*combiner = MPI_COMBINER_CONTIGUOUS;
			break;

		case LAM_DTSTRUCT:
		case LAM_DTSTRUCTCREAT:
			*nints = dtype->dt_count + 1;
			*naddrs = dtype->dt_count;
			*ndtypes = dtype->dt_count;
			if ((dtype->dt_format == LAM_DTSTRUCT)
					&& (dtype->dt_flags & LAM_LANGF77)) {
				*combiner = MPI_COMBINER_STRUCT_INTEGER;
			} else {
				*combiner = MPI_COMBINER_STRUCT;
			}
			break;

		case LAM_DTINDEXED:
			*nints = 2 * dtype->dt_count + 1;
			*naddrs = 0;
			*ndtypes = 1;
			*combiner = MPI_COMBINER_INDEXED;
			break;

		case LAM_DTHINDEXED:
		case LAM_DTHINDEXEDCREAT:
			*nints = dtype->dt_count + 1;
			*naddrs = dtype->dt_count;
			*ndtypes = 1;
			if ((dtype->dt_format == LAM_DTHINDEXED)
					&& (dtype->dt_flags & LAM_LANGF77)) {
				*combiner = MPI_COMBINER_HINDEXED_INTEGER;
			} else {
				*combiner = MPI_COMBINER_HINDEXED;
			}
			break;

		case LAM_DTVECTOR:
			*nints = 3;
			*naddrs = 0;
			*ndtypes = 1;
			*combiner = MPI_COMBINER_VECTOR;
			break;

		case LAM_DTHVECTOR:
		case LAM_DTHVECTORCREAT:
			*nints = 2;
			*naddrs = 1;
			*ndtypes = 1;
			if ((dtype->dt_format == LAM_DTHVECTOR)
					&& (dtype->dt_flags & LAM_LANGF77)) {
				*combiner = MPI_COMBINER_HVECTOR_INTEGER;
			} else {
				*combiner = MPI_COMBINER_HVECTOR;
			}
			break;

		case LAM_DTSUBARRAY:
			*nints = 3 * dtype->dt_uargs[0] + 2;
			*naddrs = 0;
			*ndtypes = 1;
			*combiner = MPI_COMBINER_SUBARRAY;
			break;

		case LAM_DTDARRAY:
			*nints = 4 * dtype->dt_uargs[2] + 4;
			*naddrs = 0;
			*ndtypes = 1;
			*combiner = MPI_COMBINER_DARRAY;
			break;

		case LAM_DTINDEXEDBLK:
		default:
			return(lam_errfunc(MPI_COMM_WORLD, BLKMPITGETENV,
						lam_mkerr(MPI_ERR_TYPE, 0)));
		}
	}

	lam_resetfunc_m(BLKMPITGETENV);
	return(MPI_SUCCESS);
}
