/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMTypeTuple.m,v 1.32 1998/04/06 11:56:42 tiggr Exp $  */

#import "OTMTypeTuple.h"
#import "OTMTuple.h"

@implementation OTMTypeTuple

+(OTMTypeTuple *) typeTupleWithSequence: (id) s
{
  return [[self gcAlloc] initWithSequence: s];
}

+(OTMTypeTuple *) typeTupleWithTuple: (OTMTuple *) t
{
  TLVector *v = [t elements];
  int i, n = [v length];
  TLVector *vt = [CO_TLVector vectorWithCapacity: n];

  for (i = 0; i < n; i++)
    [vt addElement: [[v _elementAtIndex: i] type]];

  return [self typeTupleWithSequence: vt];
}

-(OTMTypeTuple *) actualSelf: (OTMMeta *) either
{
  int i, n = [elements length];
  TLVector *nel = [CO_TLVector vectorWithCapacity: n];
  BOOL different = NO;

  for (i = 0; i < n; i++)
    {
      OTMType *o, *p = [elements _elementAtIndex: i];

      o = [p actualSelf: either];
      if (o != p)
	different = YES;
      [nel addElement: o];
    }
  return different ? [isa typeTupleWithSequence: nel] : self;
}

-(OTMTypeTuple *) actualSelfNonPosing: (OTMMeta *) either
{
  int i, n = [elements length];
  TLVector *nel = [CO_TLVector vectorWithCapacity: n];
  BOOL different = NO;

  for (i = 0; i < n; i++)
    {
      OTMType *o, *p = [elements _elementAtIndex: i];

      o = [p actualSelfNonPosing: either];
      if (o != p)
	different = YES;
      [nel addElement: o];
    }
  return different ? [isa typeTupleWithSequence: nel] : self;
}

-(BOOL) allowedTypeForArgumentRedeclaration: (OTMType *) t
				 inSubclass: (BOOL) subclass_p
{
  id actually = [self tupleSingleElement];

  if (actually != self)
    return [actually allowedTypeForArgumentRedeclaration: t
		     inSubclass: subclass_p];

  if (![t isKindOf: isa])
    return NO;

  {
    TLVector *other = [(OTMTypeTuple *) t elements];
    int i, n = [other length];

    if (n != [elements length])
      return NO;

    for (i = 0; i < n; i++)
      if (![[elements _elementAtIndex: i]
	    allowedTypeForArgumentRedeclaration: [other _elementAtIndex: i]
	    inSubclass: subclass_p])
	return NO;
  }

  return YES;
}

-(void) description: (id <TLMutableStream>) stream
{
  [super description: stream];
  formac (stream, @" %#", elements);
}

-(TLVector *) elements
{
  return elements;
}

-(int) elementCount
{
  return [elements length];
}

-equal: (OTMType *) t
{
  return [t equalTypeTuple: self];
}

-equalTypeTuple: (OTMTypeTuple *) t
{
  TLVector *other = [(OTMTypeTuple *) t elements];
  int i, n = [other length];

  if (n != [elements length])
    return nil;

  for (i = 0; i < n; i++)
    if (![[elements _elementAtIndex: i] equal: [other _elementAtIndex: i]])
      return nil;

  return self;
}

-(int) flatElementCount
{
  int i, t, n = [elements length];

  for (i = t = 0; i < n; i++)
    t += [[elements _elementAtIndex: i] flatElementCount];

  return t;
}

-(id <TLString>) flatFrobnicatedName
{
  int i, n = [elements length];
  TLMutableString *s = nil;

  for (i = 0; i < n; i++)
    s = formac (s, @"%@", [[elements _elementAtIndex: i] flatFrobnicatedName]);

  return formac (s, @"");
}

-(id <TLString>) frobnicatedName
{
  TLMutableString *s = formac (nil, @"(");
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    formac (s, @"%@", [[elements _elementAtIndex: i] frobnicatedName]);

  return formac (s, @")");
}

-(void) gcReference
{
  MARK (elements);
  [super gcReference];
}

-initWithSequence: (id) s
{
  id <TLEnumerator> e = [s enumerator];
  OTMType *t;

  if (![super init])
    return nil;

  elements = [CO_TLVector vector];

  while ((t = [e nextObject]) || [e notEndP])
    if (t)
      [elements addElement: t];
    else
      [elements addElement: the_any_type];

  return self;
}

-(BOOL) isFullyDefinedType
{
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    if (![[elements _elementAtIndex: i] isFullyDefinedType])
      return NO;

  return YES;
}

-(BOOL) isTuple
{
  return YES;
}

-(BOOL) matches: (OTMType *) t
{
  if ([t isKindOf: isa])
    {
      TLVector *other = [(OTMTypeTuple *) t elements];
      int i, n = [other length];

      if (n != [elements length])
	return NO;

      for (i = 0; i < n; i++)
	if (![[elements _elementAtIndex: i]
	      matches: [other _elementAtIndex: i]])
	  return NO;

      return YES;
    }

  /* XXX This is not totally correct, in casu we're a singleton tuple and
     the other is the type of the element.  */
  return NO;
}

-(int) matchesConvertibly: (OTMType *) t
{
  id actually = [self tupleSingleElement];

  if (actually != self)
    return [actually matchesConvertibly: t];

  if ([t isKindOf: isa])
    {
      TLVector *other = [(OTMTypeTuple *) t elements];
      int i, o, offset = 0, n = [other length];

      if (n == [elements length])
	{
	  for (i = offset = 0; i < n; i++)
	    {
	      o = [[elements _elementAtIndex: i]
		   matchesConvertibly: [other _elementAtIndex: i]];
	      if (o < 0)
		return -1;
	      offset += o;
	    }

	  return offset;
	}
    }

  return -1;
}

-(OTMType *) matchesExactly: (OTMType *) t
{
  if ([t isKindOf: isa])
    {
      TLVector *other = [(OTMTypeTuple *) t elements];
      int i, n = [other length];

      if (n != [elements length])
	return nil;

      for (i = 0; i < n; i++)
	{
	  OTMType *et = [elements _elementAtIndex: i];
	  OTMType *t = [et matchesExactly: [other _elementAtIndex: i]];

	  if (!t)
	    return nil;
	  if (t != et)
	    ABORT ();
	}

      return self;
    }

  /* XXX This is not totally correct, in casu we're a singleton tuple and
     the other is the type of the element.  */
  return nil;
}

-(id <TLString>) typeName
{
  return (id) formac (nil, @"%@", self);
}

-(id <TLString>) outputCastName
{
  int i, n = [elements length];
  TLMutableString *s = nil;

  for (i = 0; i < n; i++)
    s = formac (s, i ? @", %@" : @"%@",
		[[elements _elementAtIndex: i] outputCastName]);

  return formac (s, @"");
}

-(id) precompile
{
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    {
      id p = [elements _elementAtIndex: i];
      id o = [p precompile];

      if (o != p)
	[elements _replaceElementAtIndex: i by: o];
    }
  return self;
}

-(OTMType *) tupleFirstFlatElement
{
  return [[elements _elementAtIndex: 0] tupleFirstFlatElement];
}

-(OTMType *) tupleSingleElement
{
  return [elements length] == 1 ? [elements _elementAtIndex: 0] : self;
}

-(OTMType *) typeAt: (int) index in: (int *) indices
{
  return (index < 0
	  ? self
	  : (indices[0] >= [elements length]
	     ? nil
	     : [[elements _elementAtIndex: indices[0]] typeAt: index - 1
						       in: indices + 1]));
}

@end
