(** A table object displaying data in scrollable grid. **)

MODULE VOTable;

(*
    A table object displaying data in scrollable grid.
    Copyright (C) 1998  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       E   := VOEvent,
       F   := VOFrame,
       G   := VOGUIObject,
       O   := VOObject,
       P   := VOPrefs,
       S   := VOScroller,
       T   := VOText,
       TM  := VOTableModel,
       U   := VOUtil,
       V   := VOValue,

       str := Strings;

CONST
  background* = 0;
  selected*   = 1;
  focus*      = 2;

  noSelect*         = 0;
  cellSelect*       = 1;
  singleLineSelect* = 2;

  (* Events *)
  selectionMsg* = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 frame*,
                 headerFrame*,
                 focusFrame*    : LONGINT;
               END;


  (**
    Holds information about a columns.
  **)

  ColumnDesc = RECORD
                 charWidth,
                 width     : LONGINT;
                 title     : G.Object;
                 text      : U.Text;
                 calced    : BOOLEAN;
               END;

  (**
    Object handed to the RedrawCell/RedrawRow methods.
  **)

  DrawInfoDesc* = RECORD
                    x-,y-,
                    width-,
                    height-,
                    wOff-   : LONGINT;
                    text-   : U.Text;
                    font-   : D.Font;
                    flags-  : SET;
                  END;

  Table*     = POINTER TO TableDesc;
  TableDesc* = RECORD (G.GadgetDesc)
                 prefs        : Prefs;
                 frame,
                 headerFrame,
                 focusFrame   : F.Frame;
                 left,top,
                 hTotal,
                 vTotal,
                 hVisible,
                 vVisible     : V.ValueModel;

                 hScroll,
                 vScroll      : S.Scroller;

                 model-       : TM.TableModel;

                 columns      : POINTER TO ARRAY OF ColumnDesc;

                 totalWidth   : LONGINT;
                 headerHeight : LONGINT;

                 iX,iY,
                 iWidth,
                 iHeight      : LONGINT;

                 colHeight    : LONGINT;

                 (* selection stuff *)
                 select       : LONGINT; (* type of selection *)
                 sx,sy        : LONGINT; (* coors of selected cell/row *)
               END;

  SelectionMsg*     = POINTER TO SelectionMsgDesc;
  SelectionMsgDesc* = RECORD (O.MessageDesc)
                        x*,y* : LONGINT;
                      END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
    p.headerFrame:=F.double3DOut;
    p.focusFrame:=F.dottedFocus;
  END Init;

  PROCEDURE (t : Table) Init*;

  BEGIN
    t.Init^;

    t.prefs:=prefs;

    INCL(t.flags,G.canFocus); (* We can show a focus frame *)
    EXCL(t.flags,G.stdFocus); (* we do the displaying of the focus frame ourself *)

    t.SetBackground(D.tableBackgroundColor);

    t.frame:=NIL;
    t.focusFrame:=NIL;

    NEW(t.left);
    t.left.Init;
    t.AttachModel(t.left);

    NEW(t.top);
    t.top.Init;
    t.AttachModel(t.top);

    NEW(t.hTotal);
    t.hTotal.Init;
    NEW(t.vTotal);
    t.vTotal.Init;

    NEW(t.hVisible);
    t.hVisible.Init;
    NEW(t.vVisible);
    t.vVisible.Init;

    t.left.SetLongint(1);
    t.top.SetLongint(1);

    t.hVisible.SetLongint(0);
    t.vVisible.SetLongint(0);

    t.hTotal.SetLongint(0);
    t.vTotal.SetLongint(0);

    t.headerHeight:=0;

    t.model:=NIL;
    t.columns:=NIL;

    t.select:=noSelect;
    t.sx:=-1;
    t.sy:=-1;
  END Init;

  (**
    Set the type of selection. Current supported are:
    cellSelect - select an individual cell
    singleLineSelect - select one row
  **)

  PROCEDURE (t : Table) SetSelectionType*(type : LONGINT);

  BEGIN
    t.select:=type;
    (*
      TODO: If we change the type of the selection we must convert or clear the current
      selection somehoe and then update the display.
    *)
  END SetSelectionType;

  PROCEDURE (t : Table) CalcSize*(display : D.Display);

  VAR
    font  : D.Font;

  BEGIN
    NEW(t.frame);
    t.frame.Init;
    t.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    t.frame.SetInternalFrame(t.prefs.frame);
    t.frame.CalcSize(display);

    NEW(t.headerFrame);
    t.headerFrame.Init;
    t.headerFrame.SetFlags({G.horizontalFlex,G.verticalFlex});
    t.headerFrame.SetInternalFrame(t.prefs.headerFrame);
    t.headerFrame.CalcSize(display);

    NEW(t.focusFrame);
    t.focusFrame.Init;
    t.focusFrame.SetFlags({G.horizontalFlex,G.verticalFlex});
    t.focusFrame.SetInternalFrame(t.prefs.focusFrame);
    t.focusFrame.CalcSize(display);

    t.width:=t.frame.leftBorder+10*display.spaceWidth+t.frame.rightBorder;
    t.height:=t.frame.topBorder+10*display.spaceHeight+t.frame.bottomBorder;

    NEW(t.hScroll);
    t.hScroll.Init;
    t.hScroll.SetFlags({G.horizontalFlex});
    t.hScroll.Set(FALSE);
    t.hScroll.SetModel(t.left,t.hVisible,t.hTotal);
    t.hScroll.CalcSize(display);

    NEW(t.vScroll);
    t.vScroll.Init;
    t.vScroll.SetFlags({G.verticalFlex});
    t.vScroll.Set(TRUE);
    t.vScroll.SetModel(t.top,t.vVisible,t.vTotal);
    t.vScroll.CalcSize(display);

    INC(t.width,t.vScroll.width);
    INC(t.height,t.hScroll.height);

    t.minWidth:=t.width;
    t.minHeight:=t.height;

    font:=display.GetFont(D.normalFont);

    t.colHeight:=font.height+display.spaceHeight DIV 4;

    t.CalcSize^(display);
  END CalcSize;

  PROCEDURE (t : Table) SendLineSelection;

  VAR
    selection : SelectionMsg;

  BEGIN
    NEW(selection);
    selection.y:=t.sy;

    t.Send(selection,selectionMsg);
  END SendLineSelection;

  (**
    Caculate width and height of the columns. Create a textobject for
    column header if necessary.
  **)

  PROCEDURE (t : Table) CalcCol;

  VAR
    x : LONGINT;

  BEGIN
    t.totalWidth:=0;
    t.headerHeight:=0;
    IF t.columns#NIL THEN
      FOR x:=0 TO LEN(t.columns^)-1 DO
        IF (t.columns[x].text#NIL) & (t.columns[x].title=NIL) THEN
          t.columns[x].title:=T.MakeCenterText(t.columns[x].text^);

          t.columns[x].title.CalcSize(t.display);
        END;

        IF (t.columns[x].title#NIL) &
           (t.columns[x].title.oHeight>t.headerHeight) THEN
          t.headerHeight:=t.columns[x].title.oHeight;
        END;

        IF ~t.columns[x].calced THEN
          t.columns[x].width:=t.columns[x].charWidth*t.display.spaceWidth;
          IF t.columns[x].title#NIL THEN
            IF t.columns[x].title.oWidth>t.columns[x].width THEN
              t.columns[x].width:=t.columns[x].title.oWidth;
            END;
          END;
          INC(t.columns[x].width,t.headerFrame.leftBorder+
                                 t.headerFrame.rightBorder+
                                t.display.spaceWidth DIV 2);
          t.columns[x].calced:=TRUE;
        END;
        INC(t.totalWidth,t.columns[x].width);
      END;
      INC(t.headerHeight,t.headerFrame.topBorder+t.headerFrame.bottomBorder);
    END;

    t.iX:=t.x+t.frame.leftBorder;
    t.iY:=t.y+t.frame.topBorder+t.headerHeight;
    t.iWidth:=t.width-t.frame.leftBorder-t.vScroll.oWidth-t.frame.rightBorder;
    t.iHeight:=t.height-t.frame.topBorder-t.hScroll.oHeight-t.frame.bottomBorder-t.headerHeight;
  END CalcCol;

  (**
    DrawCell is responsible of drawing a single cell. You must always render the text given
    info info. You must also render the background, depending of the selete flag set or not,
    if background is set.
  **)

  PROCEDURE (t : Table) DrawCell*(VAR info : DrawInfoDesc);

  VAR
    extent : D.FontExtentDesc;

  BEGIN
    IF info.text#NIL THEN
      info.font.TextExtent(info.text^,str.Length(info.text^),{},extent);
    END;

    IF background IN info.flags THEN
      IF selected IN info.flags THEN
        t.draw.PushForeground(D.fillColor);
      ELSE
        t.draw.PushForeground(t.background);
      END;
      t.draw.FillRectangle(info.x,info.y,info.width,info.height);
      t.draw.PopForeground;
    END;

    IF focus IN info.flags THEN
      t.focusFrame.Resize(info.width,info.height);
      t.focusFrame.Draw(info.x,info.y,t.draw);
    END;

    IF info.text#NIL THEN
      IF extent.width>info.width-2*info.wOff THEN
        (* we must clip the cell *)
         t.draw.InstallClip;
         t.draw.AddRegion(info.x+info.wOff,info.y,info.width-2*info.wOff,info.height);
      END;

      IF selected IN info.flags THEN
        t.draw.PushForeground(D.fillTextColor);
      END;

      t.draw.DrawString(info.x+info.wOff-extent.lbearing,
                        info.y+info.font.ascent+(info.height-info.font.height) DIV 2,
                        info.text^,str.Length(info.text^));

      IF selected IN info.flags THEN
        t.draw.PopForeground;
      END;

      IF extent.width>info.width-2*info.wOff THEN
        t.draw.FreeLastClip;
      END;
    END;
  END DrawCell;

  (**
    DrawRow is responsible for drawing the frame or the background of a complete row.
    Currently you must draw the background if background is set in flags depending
    of selected beeing set or not.
  **)

  PROCEDURE (t : Table) DrawRow*(VAR info : DrawInfoDesc);

  BEGIN
    IF background IN info.flags THEN
      IF selected IN info.flags THEN
        t.draw.PushForeground(D.fillColor);
        t.draw.FillRectangle(info.x,info.y,info.width,info.height);
        t.draw.PopForeground;
      ELSE
        t.draw.PushForeground(t.background);
        t.draw.FillRectangle(info.x,info.y,info.width,info.height);
        t.draw.PopForeground;
      END;
      IF focus IN info.flags THEN
        t.focusFrame.Resize(t.totalWidth,info.height);
        t.focusFrame.Draw(t.iX-t.left.GetLongint()+1,info.y,t.draw);
      END;
    END;
  END DrawRow;

  (**
    Redraw the given row if visible.
  **)

  PROCEDURE (t : Table) RedrawRow(y : LONGINT);

  VAR
    info       : DrawInfoDesc;
    x,column,
    innerWidth : LONGINT;
    text       : U.Text;
    font       : D.Font;

  BEGIN
    IF (y<t.top.GetLongint()) OR (y>t.top.GetLongint()+t.vVisible.GetLongint()) THEN
      RETURN;
    END;

    font:=t.display.GetFont(D.normalFont);
    info.font:=font;
    info.x:=t.iX;
    info.y:=t.iY+y*t.colHeight-t.top.GetLongint()*t.colHeight;
    info.width:=t.iWidth;
    info.height:=t.colHeight;

    (* Evaluate selection *)
    info.flags:={background};
    CASE t.select OF
      singleLineSelect:
        IF t.sy=y THEN
          IF t.HasFocus() THEN
            info.flags:={background,selected,focus};
          ELSE
            info.flags:={background,selected};
          END;
        END;
    ELSE
    END;

    t.draw.InstallClip;
    t.draw.AddRegion(t.iX,t.iY,t.iWidth,t.iHeight);
    t.draw.PushForeground(D.tableTextColor);

    t.DrawRow(info);

    x:=t.iX-t.left.GetLongint()+1;
    FOR column:=1 TO LEN(t.columns^) DO
      IF ~((x>=t.iX+t.iWidth) OR (x+t.columns[column-1].width<t.iX)) THEN
        info.wOff:=G.MaxLong(t.headerFrame.leftBorder,t.headerFrame.rightBorder);
        innerWidth:=t.columns[column-1].width-t.headerFrame.leftBorder-t.headerFrame.rightBorder;
        text:=t.model.GetText(column,y);

        info.x:=x+t.headerFrame.leftBorder;
        info.y:=t.iY+y*t.colHeight-t.top.GetLongint()*t.colHeight;
        info.width:=innerWidth;
        info.height:=t.colHeight;
        info.text:=text;

        (* Evaluate selection *)
        info.flags:={};
        CASE t.select OF
          cellSelect:
            IF (t.sx=column) & (t.sy=y) THEN
              IF t.HasFocus() THEN
                info.flags:={background,selected,focus};
              ELSE
                info.flags:={background,selected};
              END;
            END;
        | singleLineSelect:
          IF t.sy=y THEN
            info.flags:={selected};
          END;
        END;

        t.DrawCell(info);
      END;
      INC(x,t.columns[column-1].width);
    END;

    t.draw.PopForeground;
    t.draw.FreeLastClip;
  END RedrawRow;

  (**
    Redraw the given cell if visible.
  **)

  PROCEDURE (t : Table) RedrawCell(x,y : LONGINT);

  VAR
    info       : DrawInfoDesc;
    xPos,column,
    innerWidth : LONGINT;
    text       : U.Text;
    font       : D.Font;

  BEGIN
    IF (y<t.top.GetLongint()) OR (y>t.top.GetLongint()+t.vVisible.GetLongint()) THEN
      RETURN;
    END;

    xPos:=t.iX-t.left.GetLongint()+1;

    FOR column:=1 TO x-1 DO
      INC(xPos,t.columns[column-1].width);
    END;

    IF ~((xPos>=t.iX+t.iWidth) OR (xPos+t.columns[x-1].width<t.iX)) THEN

      font:=t.display.GetFont(D.normalFont);
      info.font:=font;

      (* Evaluate selection *)
      info.flags:={background};
      CASE t.select OF
        singleLineSelect:
          IF t.sy=y THEN
            IF t.HasFocus() THEN
              info.flags:=info.flags+{background,selected,focus};
            ELSE
              info.flags:=info.flags+{background,selected};
            END;
          END;
      ELSE
      END;

      t.draw.InstallClip;
      t.draw.AddRegion(t.iX,t.iY,t.iWidth,t.iHeight);
      t.draw.PushForeground(D.tableTextColor);

      info.wOff:=G.MaxLong(t.headerFrame.leftBorder,t.headerFrame.rightBorder);
      innerWidth:=t.columns[x-1].width-t.headerFrame.leftBorder-t.headerFrame.rightBorder;
      text:=t.model.GetText(x,y);

      info.x:=xPos+t.headerFrame.leftBorder;
      info.y:=t.iY+y*t.colHeight-t.top.GetLongint()*t.colHeight;
      info.width:=innerWidth;
      info.height:=t.colHeight;
      info.text:=text;

      (* Evaluate selection *)
      CASE t.select OF
        cellSelect:
          IF (t.sx=x) & (t.sy=y) THEN
            IF t.HasFocus() THEN
              info.flags:=info.flags+{background,selected,focus};
            ELSE
              info.flags:=info.flags+{background,selected};
            END;
          END;
      | singleLineSelect:
        IF t.sy=y THEN
          info.flags:={selected};
        END;
      ELSE
      END;

      t.DrawCell(info);

      t.draw.PopForeground;
      t.draw.FreeLastClip;
    END;
  END RedrawCell;

  (**
    Draw the column headers
  **)

  PROCEDURE (t : Table) DrawHeader;

  VAR
    x, column : LONGINT;

  BEGIN
    IF t.model#NIL THEN
      (* Draw header *)
      t.draw.InstallClip;
      t.draw.AddRegion(t.iX,t.iY-t.headerHeight,t.iWidth,t.iHeight+t.headerHeight);

      x:=t.iX-t.left.GetLongint()+1;
      FOR column:=1 TO LEN(t.columns^) DO
        IF t.columns[column-1].title#NIL THEN
          IF ~((x>=t.iX+t.iWidth) OR (x+t.columns[column-1].width<t.iX)) THEN
            t.headerFrame.Resize(t.columns[column-1].width,t.headerHeight);
            t.columns[column-1].title.Resize(t.columns[column-1].width-
                                             t.headerFrame.leftBorder-t.headerFrame.rightBorder,
                                             t.headerHeight-t.headerFrame.topBorder-t.headerFrame.bottomBorder);
            t.columns[column-1].title.SetBackground(D.buttonBackgroundColor);
            t.columns[column-1].title.Draw(x+t.headerFrame.leftBorder,
                                         t.iY-t.headerHeight+t.headerFrame.topBorder,t.draw);
            t.headerFrame.Draw(x,t.iY-t.headerHeight,t.draw);
          END;
        END;
        INC(x,t.columns[column-1].width);
      END;
      IF x<t.iX+t.iWidth THEN
        (* fill space behind last header *)
        t.draw.PushForeground(t.background);
        t.draw.FillRectangle(x,t.iY-t.headerHeight,
                             t.iWidth-(x-t.iX)+1,t.iHeight+t.headerHeight);
        t.draw.PopForeground;
      END;
      t.draw.FreeLastClip;
    END;
  END DrawHeader;

  (**
    Redraw complete table. Redraw header, too, if header=TRUE.
  **)

  PROCEDURE (t : Table) DrawText(header : BOOLEAN);

  VAR
    row,
    column,
    x,
    lastRow,
    innerWidth : LONGINT;
    text       : U.Text;
    font       : D.Font;
    info       : DrawInfoDesc;

  BEGIN
    t.CalcCol;

    (* TODO: Fix setting of visible areas *)
    t.hVisible.SetLongint(t.iWidth);
    t.vVisible.SetLongint(t.iHeight DIV t.colHeight);

    IF t.model#NIL THEN
      t.hTotal.SetLongint(t.totalWidth);
      t.vTotal.SetLongint(t.model.GetRows());
    ELSE
      t.hTotal.SetLongint(0);
      t.vTotal.SetLongint(0);
    END;

    IF t.model#NIL THEN
      IF header THEN
        t.DrawHeader;
      END;

      (* Clip complete content of table *)
      t.draw.InstallClip;
      t.draw.AddRegion(t.iX,t.iY,t.iWidth,t.iHeight);

      font:=t.display.GetFont(D.normalFont);

      info.font:=t.display.GetFont(D.normalFont);

      lastRow:=G.MinLong(t.top.GetLongint()+t.vVisible.GetLongint(),t.model.GetRows());

      t.draw.PushForeground(D.tableTextColor);
      FOR row:=t.top.GetLongint() TO lastRow DO
        (* Clean up row *)

        info.x:=t.iX;
        info.y:=t.iY+row*t.colHeight-t.top.GetLongint()*t.colHeight;
        info.width:=t.iWidth;
        info.height:=t.colHeight;

        (* Evaluate selection *)
        info.flags:={background};
        CASE t.select OF
          singleLineSelect:
            IF t.sy=row THEN
              IF t.HasFocus() THEN
                info.flags:={background,selected,focus};
              ELSE
                info.flags:={background,selected};
              END;
            END;
        ELSE
        END;

        t.DrawRow(info);

        x:=t.iX-t.left.GetLongint()+1;
        FOR column:=1 TO LEN(t.columns^) DO
          IF ~((x>=t.iX+t.iWidth) OR (x+t.columns[column-1].width<t.iX)) THEN
            info.wOff:=G.MaxLong(t.headerFrame.leftBorder,t.headerFrame.rightBorder);
            innerWidth:=t.columns[column-1].width-t.headerFrame.leftBorder-t.headerFrame.rightBorder;
            text:=t.model.GetText(column,row);

            info.x:=x+t.headerFrame.leftBorder;
            info.y:=t.iY+row*t.colHeight-t.top.GetLongint()*t.colHeight;
            info.width:=innerWidth;
            info.height:=t.colHeight;
            info.text:=text;

            (* Evaluate selection *)
            info.flags:={};
            CASE t.select OF
              cellSelect:
                IF (t.sx=column) & (t.sy=row) THEN
                  IF t.HasFocus() THEN
                    info.flags:={background,selected,focus};
                  ELSE
                    info.flags:={background,selected};
                  END;
                END;
            | singleLineSelect:
              IF t.sy=row THEN
                info.flags:={selected};
              END;
            ELSE
            END;

            t.DrawCell(info);
          END;
          INC(x,t.columns[column-1].width);
        END;
      END;
      t.draw.PopForeground;

      (* Clear the area under the last row, if visible *)
      IF (lastRow-t.top.GetLongint()+1)*t.colHeight<t.iHeight THEN
        t.draw.PushForeground(t.background);
        t.draw.FillRectangle(t.iX,
                             t.iY+(lastRow-t.top.GetLongint()+1)*t.colHeight,
                             t.iWidth,
                             t.iHeight-(lastRow-t.top.GetLongint()+1)*t.colHeight);
        t.draw.PopForeground;
      END;

      t.draw.FreeLastClip;
    END;
  END DrawText;

  (**
    Make the given cell visible. Handle an negative number, if the you want
    the method to ignore the column or row.
  **)

  PROCEDURE (t : Table) MakeVisible*(x,y : LONGINT);

  VAR
    width,i : LONGINT;

  BEGIN
    IF t.model#NIL THEN
      IF (y>0) & (y<=t.model.GetRows()) THEN
        IF y<t.top.GetLongint() THEN
          t.top.SetLongint(y);
        ELSIF y>t.top.GetLongint()+t.vVisible.GetLongint()-1 THEN
          t.top.SetLongint(G.MaxLong(1,y-t.vVisible.GetLongint()));
        END;
      END;
      IF (x>0) & (x<=LEN(t.columns^)) THEN
        width:=0;
        FOR i:=0 TO x-2 DO
          INC(width,t.columns[i].width);
        END;

        IF width<t.left.GetLongint() THEN
          t.left.SetLongint(width);
        ELSIF width+t.columns[x-1].width>t.left.GetLongint()+t.hVisible.GetLongint()-1 THEN
          t.left.SetLongint(G.MaxLong(1,width+t.columns[x-1].width-t.hVisible.GetLongint()));
        END;
      END;
    END;
  END MakeVisible;

  (**
    Return the cell under the given coordinates. Returns FALSE if there is no
    cell at the gien position.
  **)

  PROCEDURE (t : Table) GetCell(mx,my : LONGINT; VAR x,y : LONGINT):BOOLEAN;

  VAR
    start : LONGINT;

  BEGIN
    IF (t.model=NIL) OR (t.columns=NIL) THEN
      RETURN FALSE;
    END;

    y:=(my-t.iY) DIV t.colHeight+1;
    IF (y<1) OR (y+t.top.GetLongint()-1>t.vTotal.GetLongint()) THEN
      RETURN FALSE;
    END;

    INC(y,t.top.GetLongint()-1);
    DEC(mx,t.iX-t.left.GetLongint());

    start:=0;
    FOR x:=0 TO LEN(t.columns^)-1 DO
      IF (mx>=start) & (mx<start+t.columns[x].width) THEN
        RETURN TRUE;
      END;
      INC(start,t.columns[x].width);
    END;

    RETURN FALSE;
  END GetCell;

  PROCEDURE (t : Table) GetFocus*(event : E.Event):G.Object;

  VAR
    object : G.Object;
    x,y,oy : LONGINT;

  BEGIN
    IF ~t.visible OR t.disabled OR (t.model=NIL) THEN
      RETURN NIL;
    END;

    object:=t.hScroll.GetFocus(event);
    IF object#NIL THEN
      RETURN object;
    END;

    object:=t.vScroll.GetFocus(event);
    IF object#NIL THEN
      RETURN object;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & t.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        IF t.GetCell(event.x,event.y,x,y) THEN
          CASE t.select OF
            cellSelect:
              IF (t.sy#y) OR (t.sx#x) THEN
                IF y#t.sy THEN
                  oy:=t.sy;
                  t.sy:=y;
                  t.sx:=x;
                  t.RedrawRow(oy);
                  t.RedrawRow(y);
                ELSE
                  t.sx:=x;
                  t.RedrawRow(y);
                END;
              END;
          | singleLineSelect:
              IF y#t.sy THEN
                oy:=t.sy;
                t.sy:=y;
                t.RedrawRow(oy);
                t.RedrawRow(y);
                t.SendLineSelection;
              END;
          ELSE
          END;
        END;
        RETURN t;
      END;
    ELSE
    END;

    RETURN NIL;
  END GetFocus;

  PROCEDURE (t : Table) HandleKeys(event :  E.KeyEvent):BOOLEAN;

  VAR
    keysym  : LONGINT;

  BEGIN
    keysym:=event.GetKey();
    CASE keysym OF
    | E.left:
      CASE t.select OF
        noSelect:
          IF t.left.GetLongint()>1 THEN
            t.left.Dec;
          END;
      | cellSelect:
          IF t.sx>1 THEN
            DEC(t.sx);
            t.RedrawRow(t.sy);
            t.MakeVisible(t.sx,t.sy);
          END;
      | singleLineSelect:
          IF t.left.GetLongint()>1 THEN
            t.left.Dec;
          END;
      ELSE
      END;
    | E.right:
      CASE t.select OF
        noSelect:
          IF t.left.GetLongint()+t.hVisible.GetLongint()-1<t.hTotal.GetLongint() THEN
            t.left.Inc;
          END;
      | cellSelect:
          IF t.sx<LEN(t.columns^) THEN
            INC(t.sx);
            t.RedrawRow(t.sy);
            t.MakeVisible(t.sx,t.sy);
          END;
      | singleLineSelect:
          IF t.left.GetLongint()<t.hTotal.GetLongint()-t.hVisible.GetLongint()+1 THEN
            t.left.Inc;
          END;
      ELSE
      END;
    | E.up:
      CASE t.select OF
        noSelect:
          IF t.top.GetLongint()>1 THEN
            t.top.Dec;
          END;
      | cellSelect,
        singleLineSelect:
          IF t.sy>1 THEN
            DEC(t.sy);
            t.RedrawRow(t.sy+1);
            t.RedrawRow(t.sy);
            t.MakeVisible(t.sx,t.sy);
            t.SendLineSelection;
          END;
     ELSE
      END;
    | E.down:
      CASE t.select OF
        noSelect:
          IF t.top.GetLongint()+t.vVisible.GetLongint()-1<t.model.GetRows() THEN
            t.top.Inc;
          END;
      | cellSelect,
        singleLineSelect:
          IF t.sy<t.model.GetRows() THEN
            INC(t.sy);
            t.RedrawRow(t.sy-1);
            t.RedrawRow(t.sy);
            t.MakeVisible(t.sx,t.sy);
            t.SendLineSelection;
          END;
      ELSE
      END;
    | E.home:
      CASE t.select OF
        noSelect,
        singleLineSelect:
          t.top.SetLongint(1);
      | cellSelect:
          IF t.sx#1 THEN
            t.sx:=1;
            t.RedrawRow(t.sy);
            t.MakeVisible(t.sx,t.sy);
          END;
      ELSE
      END;
    | E.end:
      CASE t.select OF
        noSelect,
        singleLineSelect:
          t.top.SetLongint(G.MaxLong(1,t.vTotal.GetLongint()-t.vVisible.GetLongint()+1));
      | cellSelect:
          IF t.sx#LEN(t.columns^) THEN
            t.sx:=LEN(t.columns^);
            t.RedrawRow(t.sy);
            t.MakeVisible(t.sx,t.sy);
          END;
      ELSE
      END;
    | E.pageUp:
      CASE t.select OF
        noSelect,
        singleLineSelect,
        cellSelect:
          t.top.SetLongint(G.MaxLong(1,t.top.GetLongint()-t.vVisible.GetLongint()+1));
      ELSE
      END;
    | E.pageDown:
      CASE t.select OF
        noSelect,
        singleLineSelect,
        cellSelect:
          t.top.SetLongint(G.MinLong(G.MaxLong(1,t.vTotal.GetLongint()-
                                                 t.vVisible.GetLongint()+1),
                           t.top.GetLongint()+t.vVisible.GetLongint()-1));
      ELSE
      END;
    ELSE
    END;
    RETURN FALSE;
  END HandleKeys;

  PROCEDURE (t : Table) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    WITH
      event : E.MouseEvent DO
        IF (event.button=E.button1) & (event.type=E.mouseUp) THEN
          RETURN TRUE;
        END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (t : Table) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    IF event.type=E.keyDown THEN
      RETURN t.HandleKeys(event);
    ELSE
      RETURN FALSE;
    END;
  END HandleFocusEvent;

  (**
    Completely reinitialize the table.
  **)

  PROCEDURE (t : Table) ReInit;

  VAR
    x    : LONGINT;
    text : U.Text;

  BEGIN
    (* Free old columns *)
    IF t.columns#NIL THEN
      FOR x:=0 TO LEN(t.columns^)-1 DO
        IF t.columns[x].title#NIL THEN
          t.columns[x].title.Free;
          t.columns[x].title:=NIL;
        END;
      END;
    END;

    NEW(t.columns,t.model.GetColumns());
    FOR x:=0 TO LEN(t.columns^)-1 DO
      t.columns[x].charWidth:=t.model.GetColumnWidth(x+1);
      text:=t.model.GetText(x+1,0);
      IF text#NIL THEN
        (* we make a local copy *)
        NEW(t.columns[x].text,str.Length(text^)+1);
        COPY(text^,t.columns[x].text^);
      END;
      t.columns[x].calced:=FALSE;
    END;

    t.left.SetLongint(1);
    t.top.SetLongint(1);

    IF t.visible THEN
      t.Redraw;
    END;
  END ReInit;

  (**
    The the table model for  the table object.
  **)

  PROCEDURE (t : Table) SetModel*(model : TM.TableModel);

  BEGIN
    IF t.model#NIL THEN
      t.UnattachModel(t.model);
    END;

    t.model:=model;

    IF t.model#NIL THEN
      t.AttachModel(model);
      t.ReInit;
    END;
  END SetModel;

  PROCEDURE (t : Table) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    t.Draw^(x,y,draw);

    t.frame.Resize(t.width,t.height);
    t.frame.Draw(t.x,t.y,draw);

    t.hScroll.Resize(t.width-t.frame.leftBorder-t.frame.rightBorder-t.vScroll.width,-1);
    t.vScroll.Resize(-1,t.height-t.frame.topBorder-t.frame.bottomBorder-t.hScroll.height);

    t.hScroll.Draw(t.x+t.frame.leftBorder,
                   t.y+t.height-t.frame.bottomBorder-t.hScroll.oHeight,draw);
    t.vScroll.Draw(t.x+t.width-t.frame.rightBorder-t.vScroll.oWidth,
                   t.y+t.frame.topBorder,draw);

    t.DrawText(TRUE);
  END Draw;

  PROCEDURE (t : Table) Hide*;

  BEGIN
    IF t.visible THEN
      t.hScroll.Hide;
      t.vScroll.Hide;
      t.DrawHide;
      t.Hide^;
    END;
  END Hide;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (t : Table) DrawFocus*;

  BEGIN
    t.DrawText(FALSE);
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (t : Table) HideFocus*;

  BEGIN
    t.DrawText(FALSE);
  END HideFocus;

  PROCEDURE (t : Table) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF model=t.model THEN
      IF msg#NIL THEN
        WITH msg : TM.RefreshCell DO
          t.RedrawCell(msg.x,msg.y);
        END;
      ELSE
        t.ReInit;
      END;
    ELSIF t.visible THEN
      IF model=t.top THEN
        t.DrawText(FALSE);
      ELSE
        t.DrawText(TRUE);
      END;
    END;
  END Resync;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOTable.
