/* pipe.c - for opening a process as a pipe and reading both stderr and stdout together
   Copyright (C) 1997 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */

#include <config.h>
#include "global.h"
#include <stdlib.h>
#include <stdio.h>

#include <errno.h>
#ifdef HAVE_SYS_ERRNO_H
#include <sys/errno.h>
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <signal.h>
#ifdef HAVE_UNISTD_H
#   include <unistd.h>
#endif

#include "my_string.h"
#ifdef HAVE_SYS_SELECT_H
#  include <sys/select.h>
#endif

#include "pool.h"

#include "mad.h"

int data_read_ready (int f);
int data_read_wait (int f);
int data_write_ready (int f);
int data_write_wait (int f);
pid_t triple_pipe_open (int *in, int *out, int *err, int mix, const char *file, char *const argv[]);
char *read_pipe (int fd, int *len);

#undef min
#define min(x,y)     (((x) < (y)) ? (x) : (y))

void set_signal_handlers_to_default (void)
{
    signal (SIGHUP, SIG_DFL);
    signal (SIGQUIT, SIG_DFL);
    signal (SIGINT, SIG_DFL);
    signal (SIGTERM, SIG_DFL);
    signal (SIGABRT, SIG_DFL);
    signal (SIGCHLD, SIG_DFL);
    signal (SIGALRM, SIG_IGN);
}


/*
   This opens a process as a pipe. 'in', 'out' and 'err' are pointers to file handles
   which are filled in by popen. 'in' refers to stdin of the process, to
   which you can write. 'out' and 'err' refer to stdout and stderr of the process
   from which you can read. 'in', 'out' and 'err' can be passed
   as NULL if you want to ignore output or input of those pipes.
   If 'mix' is non-zero, then both stderr and stdout of the process
   can be read from 'out'. If mix is non-zero, then 'err' must be passed as NULL.
   Popen forks and then calls execvp (see execvp(3)) --- which must also take argv[0]
   and args must terminate with a NULL.
   Returns -1 if the fork failed, and -2 if pipe() failed.
   Otherwise returns the pid of the child.
 */
pid_t triple_pipe_open (int *in, int *out, int *err, int mix, const char *file, char *const argv[])
{
    pid_t p;
    int e;
    int f0[2], f1[2], f2[2];

    e = (pipe (f0) | pipe (f1) | pipe (f2));

    if (e) {
	close (f0[0]);
	close (f0[1]);
	close (f1[0]);
	close (f1[1]);
	close (f2[0]);
	close (f2[1]);
	return -2;
    }

    p = fork ();

    if (p == -1) {
	close (f0[0]);
	close (f0[1]);
	close (f1[0]);
	close (f1[1]);
	close (f2[0]);
	close (f2[1]);
	return -1;
    }
    if (p) {
	if (in) {
	    *in = f0[1];
	} else {
	    close (f0[1]);
	}
	if (out) {
	    *out = f1[0];
	} else {
	    close (f1[0]);
	}
	if (err) {
	    *err = f2[0];
	} else {
	    close (f2[0]);
	}
	close (f0[0]);
	close (f1[1]);
	close (f2[1]);
	return p;
    } else {
	int nulldevice_wr, nulldevice_rd;

	nulldevice_wr = open ("/dev/null", O_WRONLY);
	nulldevice_rd = open ("/dev/null", O_RDONLY);

	close (0);
	if (in)
	    dup (f0[0]);
	else
	    dup (nulldevice_rd);
	close (1);
	if (out)
	    dup (f1[1]);
	else
	    dup (nulldevice_wr);
	close (2);
	if (err)
	    dup (f2[1]);
	else {
	    if (mix)
		dup (f1[1]);
	    else
		dup (nulldevice_wr);
	}
	close (f0[0]);
	close (f0[1]);
	close (f1[0]);
	close (f1[1]);
	close (f2[0]);
	close (f2[1]);

	close (nulldevice_rd);
	close (nulldevice_wr);
	set_signal_handlers_to_default ();
	execvp (file, argv);
	exit (1);
    }
    return 0; /* prevents warning */
}

#define CHUNK 8192

/*
   Reads all available data and mallocs space for it plus
   one byte, and sets that byte to zero. If len is non-NULL
   bytes read is placed in len.
 */
char *read_pipe (int fd, int *len)
{
    POOL *p;
    int c, count = 0;
    p = pool_init ();
    for (;;) {
	if (pool_freespace (p) < CHUNK + 1)
	    pool_advance (p, CHUNK + 1);
	do {
	    c = read (fd, pool_current (p), CHUNK);
	} while (c < 0 && errno == EINTR);
	if (c <= 0)
	    break;
	count += c;
	pool_current (p) += c;
    }
    pool_null (p);
    if (len)
	*len = pool_length (p);
    return (char *) pool_break (p);
}

