/* scrollbar.c - scrollbar widget
   Copyright (C) 1997 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */


#include <config.h>
#include <stdio.h>
#include <my_string.h>
#include <stdlib.h>
#include <stdarg.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xatom.h>
#include "lkeysym.h"

#include "stringtools.h"
#include "app_glob.c"
#include "edit.h"
#include "editcmddef.h"

#include "coolwidget.h"
#include "coollocal.h"
#include "mousemark.h"

#include "mad.h"


CWidget *CDrawVerticalScrollbar (const char *identifier, Window parent, int x, int y,
				 int length, int width, int pos, int prop)
{
    CWidget *w;
    if (width == AUTO_WIDTH || width == AUTO_HEIGHT)
	width = 20;
    w = CSetupWidget (identifier, parent, x, y,
	width, length, C_VERTSCROLL_WIDGET, INPUT_MOTION, COLOR_FLAT, 0);
    w->firstline = pos;
    w->numlines = prop;
    set_hint_pos (x + width + WIDGET_SPACING, y + length + WIDGET_SPACING);
    return w;
}

CWidget *CDrawHorizontalScrollbar (const char *identifier, Window parent, int x, int y,
			     int length, int width, int pos, int prop)
{
    CWidget *w;
    if (width == AUTO_WIDTH || width == AUTO_HEIGHT)
	width = 20;
    w = CSetupWidget (identifier, parent, x, y,
		      length, width, C_HORISCROLL_WIDGET,
		      ExposureMask | ButtonPressMask |
	       ButtonReleaseMask | ButtonMotionMask | PointerMotionMask |
		      EnterWindowMask | LeaveWindowMask, COLOR_FLAT, 0);
    w->firstline = pos;
    w->numlines = prop;
    set_hint_pos (x + length + WIDGET_SPACING, y + width + WIDGET_SPACING);
    return w;
}

void CSetScrollbarCallback (const char *scrollbar, const char *wdt,
			    void (*link_to) (CWidget *,
				      CWidget *, XEvent *, CEvent *, int))
{
    CWidget *s, *w;
    s = CIdent (scrollbar);
    w = CIdent (wdt);
    if (!s || !w)
	return;
    s->vert_scrollbar = w;
    s->scroll_bar_link = link_to;
}



static void Crendervertscrollbar (Window win, int x, int y, int w, int h, int pos, int prop, int pos2, int prop2, int flags)
{
    int l = h - 10 * w / 3 - 5;

    render_bevel (win, 0, 0, w - 1, h - 1, 2, 1);
    CSetColor (COLOR_FLAT);
    CRectangle (win, 2, w + 2 * w / 3 + 2, w - 4, (l - 5) * pos / 65535);
    CRectangle (win, 2, w + 2 * w / 3 + 3 + l * (prop + pos) / 65535, w - 4, h - 1 - w - 2 * w / 3 - (w + 2 * w / 3 + 4 + l * (prop + pos) / 65535));

    if (flags & 32) {
	render_bevel (win, 2, 2, w - 3, w + 1, 2 - ((flags & 15) == 1), 2);
	render_bevel (win, 2, w + 2, w - 3, w + 2 * w / 3 + 1, 2 - ((flags & 15) == 2), 2);
	render_bevel (win, 2, h - 2 - w, w - 3, h - 3, 2 - ((flags & 15) == 4), 2);
	render_bevel (win, 2, h - 2 - w - 2 * w / 3, w - 3, h - 3 - w, 2 - ((flags & 15) == 5), 2);
	render_bevel (win, 2, w + 2 * w / 3 + 2 + (l - 5) * pos / 65535, w - 3, w + 2 * w / 3 + 7 + (l - 5) * (prop + pos) / 65535, 2 - ((flags & 15) == 3), 2);
    } else {
	render_bevel (win, 2, 2, w - 3, w + 1, 2, 2 | ((flags & 15) == 1));
	render_bevel (win, 2, w + 2, w - 3, w + 2 * w / 3 + 1, 2, 2 | ((flags & 15) == 2));
	render_bevel (win, 2, h - 2 - w, w - 3, h - 3, 2, 2 | ((flags & 15) == 4));
	render_bevel (win, 2, h - 2 - w - 2 * w / 3, w - 3, h - 3 - w, 2, 2 | ((flags & 15) == 5));
	if ((flags & 15) == 3) {
	    CSetColor (color_widget (5));
	    XDrawRectangle (CDisplay, win, CGC, 4, w + 2 * w / 3 + 4 + (l - 5) * pos2 / 65535, w - 10, 2 + (l - 5) * prop2 / 65535);
	}
	render_bevel (win, 2, w + 2 * w / 3 + 2 + (l - 5) * pos / 65535, w - 3, w + 2 * w / 3 + 7 + (l - 5) * (prop + pos) / 65535, 2, 2 | ((flags & 15) == 3));
    }
}

static void Crenderhoriscrollbar (Window win, int x, int y, int h, int w, int pos, int prop, int flags)
{
    int l = h - 10 * w / 3 - 5, k;
    k = (l - 5) * pos / 65535;

    render_bevel (win, 0, 0, h - 1, w - 1, 2, 1);
    CSetColor (COLOR_FLAT);

    CRectangle (win, w + 2 * w / 3 + 2, 2, (l - 5) * pos / 65535, w - 4);
    CRectangle (win, w + 2 * w / 3 + 3 + l * (prop + pos) / 65535, 2, h - 1 - w - 2 * w / 3 - (w + 2 * w / 3 + 4 + l * (prop + pos) / 65535), w - 4);

    if (flags & 32) {
	render_bevel (win, 2, 2, w + 1, w - 3, 2 - ((flags & 15) == 1), 2);
	render_bevel (win, w + 2, 2, w + 2 * w / 3 + 1, w - 3, 2 - ((flags & 15) == 2), 2);
	render_bevel (win, h - 2 - w, 2, h - 3, w - 3, 2 - ((flags & 15) == 4), 2);
	render_bevel (win, h - 2 - w - 2 * w / 3, 2, h - 3 - w, w - 3, 2 - ((flags & 15) == 5), 2);
	render_bevel (win, w + 2 * w / 3 + 2 + k, 2, w + 2 * w / 3 + 7 + (l - 5) * (prop + pos) / 65535, w - 3, 2 - ((flags & 15) == 3), 2);
    } else {
	render_bevel (win, 2, 2, w + 1, w - 3, 2, 2 | ((flags & 15) == 1));
	render_bevel (win, w + 2, 2, w + 2 * w / 3 + 1, w - 3, 2, 2 | ((flags & 15) == 2));
	render_bevel (win, h - 2 - w, 2, h - 3, w - 3, 2, 2 | ((flags & 15) == 4));
	render_bevel (win, h - 2 - w - 2 * w / 3, 2, h - 3 - w, w - 3, 2, 2 | ((flags & 15) == 5));
	render_bevel (win, w + 2 * w / 3 + 2 + k, 2, w + 2 * w / 3 + 7 + (l - 5) * (prop + pos) / 65535, w - 3, 2, 2 | ((flags & 15) == 3));
    }
}

void render_scrollbar (CWidget * wdt)
{
    int flags = wdt->options;
    if (!wdt)
	return;
    if (wdt->numlines < 0)
	wdt->numlines = 0;
    if (wdt->firstline < 0)
	wdt->firstline = 0;
    if (wdt->firstline > 65535)
	wdt->firstline = 65535;
    if (wdt->firstline + wdt->numlines >= 65535)
	wdt->numlines = 65535 - wdt->firstline;
    if (wdt->kind == C_VERTSCROLL_WIDGET) {
	Crendervertscrollbar (wdt->winid,
			      wdt->x, wdt->y,
			      wdt->width, wdt->height,
			      wdt->firstline, wdt->numlines, wdt->search_start, wdt->search_len, flags);
    } else
	Crenderhoriscrollbar (wdt->winid,
			      wdt->x, wdt->y,
			      wdt->width, wdt->height,
			      wdt->firstline, wdt->numlines, flags);
    if (wdt->scroll_bar_extra_render)
	(*wdt->scroll_bar_extra_render) (wdt);
}

int inbounds (int x, int y, int x1, int y1, int x2, int y2)
{
    if (x >= x1 && x <= x2 && y >= y1 && y <= y2)
	return 1;
    else
	return 0;
}

/*
   Which scrollbar button was pressed: 3 is the middle button ?
 */
int which_scrollbar_button (int bx, int by, CWidget * wdt)
{
    int w, h;
    int pos = wdt->firstline;
    int prop = wdt->numlines;
    int l;

    if (wdt->kind == C_VERTSCROLL_WIDGET) {
	w = wdt->width;
	h = wdt->height;
    } else {
	int t = bx;
	bx = by;
	by = t;
	w = wdt->height;
	h = wdt->width;
    }
    l = h - 10 * w / 3 - 5;

    if (inbounds (bx, by, 2, 2, w - 3, w + 1))
	return 1;
    if (inbounds (bx, by, 2, w + 2, w - 3, w + 2 * w / 3 + 1))
	return 2;
    if (inbounds (bx, by, 2, h - 2 - w, w - 3, h - 3))
	return 4;
    if (inbounds (bx, by, 2, h - 2 - w - 2 * w / 3, w - 3, h - 3 - w))
	return 5;
    if (inbounds (bx, by, 2, w + 2 * w / 3 + 2 + (l - 5) * pos / 65535, w - 3, w + 2 * w / 3 + 7 + (l - 5) * (prop + pos) / 65535))
	return 3;
    return 0;
}

int eh_scrollbar (CWidget * w, XEvent * xevent, CEvent * cwevent)
{
    static int buttonypos, y, whichscrbutton = 0;	/* which of the five scroll bar buttons was pressed */
    int xevent_xbutton_y, length, width;

    if (w->kind == C_VERTSCROLL_WIDGET) {
	xevent_xbutton_y = xevent->xbutton.y;
	length = w->height;
	width = w->width;
    } else {
	xevent_xbutton_y = xevent->xbutton.x;
	length = w->width;
	width = w->height;
    }

    switch (xevent->type) {
    case LeaveNotify:
    case Expose:
	w->options = 0;
	break;
    case ButtonRepeat:
	resolve_button (xevent, cwevent);
	if (cwevent->button == Button1 || cwevent->button == Button2) {
	    int b;
	    b = which_scrollbar_button (cwevent->x, cwevent->y, w);
	    if (b == 3 || !b)
		return 0;
	    y = w->firstline;
	    buttonypos = xevent_xbutton_y;
	    w->options = whichscrbutton = b;
	    cwevent->ident = w->ident;
	    xevent->type = cwevent->type = ButtonPress;
	}
	break;
    case ButtonPress:
	resolve_button (xevent, cwevent);
	if (cwevent->button == Button1 || cwevent->button == Button2) {
	    buttonypos = xevent_xbutton_y;
	    y = w->firstline;
	    w->options = whichscrbutton = which_scrollbar_button (cwevent->x, cwevent->y, w);
	    cwevent->ident = w->ident;
	    w->search_start = w->firstline;
	    w->search_len = w->numlines;
	}
	break;
    case ButtonRelease:
	resolve_button (xevent, cwevent);
	w->options = 32 + whichscrbutton;
	if (whichscrbutton == 3) {
	    y += (double) (xevent_xbutton_y - buttonypos) * (double) 65535.0 / (length - 10 * width / 3 - 10);
	    w->firstline = y;
	    buttonypos = xevent_xbutton_y;
	}
	break;
    case MotionNotify:
	resolve_button (xevent, cwevent);
	if (cwevent->state & (Button1Mask | Button2Mask)) {
	    w->options = whichscrbutton;
	    if (whichscrbutton == 3) {
		y += (double) (xevent_xbutton_y - buttonypos) * (double) 65535.0 / (length - 10 * width / 3 - 10);
		w->firstline = y;
		buttonypos = xevent_xbutton_y;
	    }
	} else
	    w->options = 32 + which_scrollbar_button (xevent->xmotion.x, xevent->xmotion.y, w);
	break;
    default:
	return 0;
    }

    if (w->firstline > 65535)
	w->firstline = 65535;
    if (cwevent->state & (Button1Mask | Button2Mask) || cwevent->type == ButtonPress || cwevent->type == ButtonRelease)
	if (w->scroll_bar_link && w->vert_scrollbar)
	    (*w->scroll_bar_link) (w, w->vert_scrollbar, xevent, cwevent, whichscrbutton);

    if (xevent->type != Expose || !xevent->xexpose.count)
	render_scrollbar (w);

    return 0;
}

