/* vi:set ts=8 sts=0 sw=8:
 * $Id: main.c,v 1.34 2000/03/01 01:46:55 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include "config.h"
#include "prefs.h"
#include "win.h"
#include "doc.h"
#include "msgbar.h"
#include "recent.h"
#include "misc.h"
#include "file.h"
#include "gnpintl.h"

/*** external declarations ***/
extern char *build_user;	/* these are all in appinfo.c */
extern char *build_host;
extern char *build_date;
extern char *gtkver;
#ifdef USE_GNOME
extern char *gnomever;
#endif
extern char *compiler;
extern char *gtk_cflags;
extern char *cflags;
extern char *gtk_ldflags;
extern char *ldflags;
extern char *unamenfo;

extern void     win_set_title(doc_t *d);

#if defined(ENABLE_NLS) && defined(GTK_HAVE_FEATURES_1_1_0)
static char G_GNUC_UNUSED *dummyMsg[] = { N_("Untitled"),
					  N_("Unknown"),
					  N_("Welcome to "),
					  N_("Files left to open: ") };
#endif

/*** local declarations ***/
#define PRINT_VERSION()							\
	do {								\
		printf("%s %s (%s)\n",					\
			APP_NAME, APP_VERSION, APP_DATE);		\
		printf("Compiled by %s@%s on %s\n",			\
			build_user, build_host, build_date);		\
	} while(0)

typedef struct {
#ifdef WANT_SPLASH
	GtkWidget *toplev;	/* top level init status window */
	GtkWidget *fnum;	/* label for num files remaining */
	GtkWidget *fname;	/* label for filename */
	bool_t do_splash;
#endif
	int timeout_id;
	win_t *w;
	char **files;
	int num;
} init_data_t;


/*** local function prototypes ***/
static void app_init(int num, char **files, bool_t);
static int  app_init_open_files(gpointer cbdata);
static void app_init_status_win_create(win_t *w, char **files, int num, bool_t);
#ifdef WANT_SPLASH
static void app_init_status_win_destroy(init_data_t *idp);
static void app_init_status_win_update(init_data_t *idp);
#endif
static void app_parse_args(int *argc, char ***argv, bool_t *, bool_t *);
static void app_print_appinfo(void);
static void app_show_help(char **argv);
static void app_show_info(void);
static void app_show_version(void);


/*** global function definitions ***/

int
main(int argc, char **argv)
{
	bool_t do_fork = FALSE, do_splash = TRUE;

	app_parse_args(&argc, &argv, &do_fork, &do_splash);
	prefs_init_pre();

#ifdef WANT_BACKGROUND
	if (do_fork || IS_DO_FORK()) {
		pid_t pid;

		if ((pid = fork()) == -1)
			perror("could not fork");

		if (pid != 0)
			exit(0);
	}
#endif

	/*
	 * gdk is so braindead in that it catches these "fatal" signals, but
	 * does absolutely nothing with them unless you have compiled it with a
	 * debugging flag (e.g., ./configure --enable-debug=yes).  we'll simply
	 * re-establish the signals to their default behavior.  this should
	 * produce a core dump in the normal case, which is a lot more useful
	 * than what gdk does.
	 *
	 * if you've compiled GDK with debugging or simply don't care, then you
	 * don't need this.
	 */
#ifdef GDK_IS_BRAINDEAD
	signal(SIGHUP, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGQUIT, SIG_DFL);
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);
	signal(SIGPIPE, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
#endif
	gtk_set_locale();
#ifdef USE_GNOME
	gnome_init(APP_NAME, APP_VERSION, argc, argv);
#else
	gtk_init(&argc, &argv);
#endif
#ifdef ENABLE_NLS
	bindtextdomain("gnotepad+", LOCALEDIR);
	textdomain("gnotepad+");
#endif
	prefs_init_post();
	app_init(argc - 1, argv + 1, do_splash);
	gtk_main();

	return 0;
} /* main */


/*** local function definitions ***/
/*
 * PRIVATE: app_init
 *
 * performs all the initial creates of the first window.  creates "Untitled"
 * document if none were specified on the command line.  otherwise, creates a
 * popup showing the number of files remaining to be opened along with the
 * filename to be opened.
 */
static void
app_init(int num, char **files, bool_t do_splash)
{
	win_t *w;

	GNPDBG_GENERIC(("app_init: files to open = %d\n", num));

	/* initialize the window list */
	w = win_new();

	/* add "Untitled" doc if no files specified */
	if (*files == NULL) {
		doc_new(w, NULL, DocText, (DO_LOAD | UPDATE_TITLE));
		gtk_widget_show(w->toplev);

		/*
		 * setting the switch page signal here is kind of hokey, but we
		 * need to do it here because the switchpage handler is where
		 * we determine when to actually load the file from disk.
		 * since we may be opening multiple files, we only want to
		 * enable the switchpage handler **AFTER** we've opened all the
		 * files at the start of the program.  technically, if we're in
		 * this "if" part, we're only opening one file.  however,
		 * there's no clean way of not doing this without passing an
		 * additional parameter to win_new(), which then gets passed to
		 * notebook_init().  we could do it that way, but it's too
		 * messy.
		 *
		 * note that this is done again, at the end of
		 * app_init_open_files().
		 */
		(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
					       GTK_SIGNAL_FUNC(doc_switch_page),
					       w);

		while (gtk_events_pending())
			gtk_main_iteration_do(TRUE);
		msgbar_printf(w, "%s%s %s", WELCOME_MSG, APP_NAME, APP_VERSION, _("."));
#ifdef APP_GNP
#ifdef USE_AUTOSAVE
		win_autosave_init();
#endif	/* USE_AUTOSAVE */
#endif
	} else {
		app_init_status_win_create(w, files, num, do_splash);
	}

} /* app_init */


/*
 * PRIVATE: app_init_open_files
 *
 * callback routine for app_init_status_win_create() which is attached to a
 * timeout during the creation of the status popup window.
 */
static int
app_init_open_files(gpointer cbdata)
{
	init_data_t *idp = (init_data_t *)cbdata;
	win_t *w;

	g_assert(idp != NULL);
	gtk_timeout_remove(idp->timeout_id);
	w = idp->w;
	/*
	 * for each file specified, create a document and add it to the current
	 * window.  since we are inserting into the doc notebook at the head of
	 * the list, we'll do the filenames backwards to preserve the order.
	 */
	while (idp->num > 0) {
#ifdef WANT_SPLASH
		if (idp->do_splash && IS_SHOW_SPLASH())
			app_init_status_win_update(idp);
#endif

#ifdef WANT_DOCUNLOAD
		doc_new(idp->w, idp->files[idp->num - 1], DocText,
			(idp->num - 1 == 0) ? DO_LOAD : 0);
#else
		doc_new(idp->w, idp->files[idp->num - 1], DocText, DO_LOAD);
#endif
#ifdef USE_RECENT
		if (idp->num <= prefs.maxrecent)
			recent_list_add(idp->w, idp->files[idp->num - 1]);
#endif	/* USE_RECENT */
		g_free(idp->files[idp->num - 1]);
		idp->files[idp->num - 1] = NULL;
		idp->num--;
	}

	/*
	 * establish the notebook's switch_page handler.  see app_init() for
	 * more details why it's done here
	 */
	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);

#ifdef WANT_SPLASH
	if (idp->do_splash && IS_SHOW_SPLASH())
		app_init_status_win_destroy(idp);
#endif

	win_set_title((doc_t *)w->curdoc);
	doc_info_label_update(w);
	gtk_widget_show(w->toplev);
	while (gtk_events_pending())
		gtk_main_iteration_do(TRUE);
	msgbar_printf(w, "%s%s %s", WELCOME_MSG, APP_NAME, APP_VERSION, _("."));
#ifdef APP_GNP
#ifdef USE_AUTOSAVE
	win_autosave_init();
#endif	/* USE_AUTOSAVE */
#endif

	return FALSE;
} /* app_init_open_files */


/*
 * PRIVATE: app_init_status_win_create
 *
 * creates all the widgets necessary for the popup window to show the
 * initialization status.  the files are actually opened through a callback
 * routine which is attached to a gtk_timeout near the end of this function.
 */
#define FILES_LEFT	gettext("Files left to open: ")
static void
app_init_status_win_create(win_t *w, char **files, int num, bool_t do_splash)
{
#ifdef WANT_SPLASH
	GtkWidget *vbox;
#endif
	init_data_t *idp;
	int i;
#ifdef DETERMINE_WINDOW_DIMENSIONS
	/* use this to get exact window dimensions */
	int wheight, wwidth, sheight, swidth;
#endif

	idp = g_new(init_data_t, 1);
	idp->w = w;
	idp->num = num;
	idp->files = g_new(char *, num);
	for (i = 0; i < num; i++)
		idp->files[i] = g_strdup(files[i]);

#ifdef WANT_SPLASH
	idp->do_splash = do_splash;
	if (do_splash && IS_SHOW_SPLASH()) {
		idp->toplev = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(idp->toplev),
				     _("gnotepad+ Init..."));
		(void)gtk_signal_connect(GTK_OBJECT(idp->toplev),
					 "delete_event",
					 GTK_SIGNAL_FUNC(gtk_true), NULL);
		gtk_container_border_width(GTK_CONTAINER(idp->toplev), 10);

		vbox = gtk_vbox_new(FALSE, 4);
		gtk_container_add(GTK_CONTAINER(idp->toplev), vbox);

		/* for some reason, the labels below are not centered correctly
		 * when this is compiled with gtk-1.1 */
		idp->fnum = gtk_label_new(FILES_LEFT);
		gtk_box_pack_start(GTK_BOX(vbox), idp->fnum, FALSE, TRUE, 10);

		idp->fname = gtk_label_new(" ");
		gtk_box_pack_start(GTK_BOX(vbox), idp->fname, FALSE, TRUE, 10);
#ifdef DETERMINE_WINDOW_DIMENSIONS
		/* XXX - can we get win dimensions w/o showing the widget?? */
		gtk_widget_show(idp->toplev);
		/* get window width & height, then set it to be larger */
		gdk_window_get_size(idp->toplev->window, &wwidth, &wheight);
		wwidth = 3 * gdk_string_width(idp->toplev->style->font,
					      FILES_LEFT);
		gtk_widget_set_usize(idp->toplev, wwidth, wheight);

		/* get screen width & height, so we can center the window */
		swidth = gdk_screen_width();
		sheight = gdk_screen_height();

		gtk_widget_set_uposition(idp->toplev,
					 (swidth/2) - (wwidth/2),
					 (sheight/2) - (wheight/2));
#else
		gtk_window_position(GTK_WINDOW(idp->toplev),
				    GTK_WIN_POS_CENTER);
#endif
		gtk_widget_show_all(idp->toplev);
	}
#endif	/* WANT_SPLASH */

	/* this will kick-off opening all the files */
	idp->timeout_id = gtk_timeout_add(1,
					(GtkFunction)app_init_open_files, idp);
} /* app_init_status_win_create */


#ifdef WANT_SPLASH
/*
 * PRIVATE: app_init_status_win_destroy
 *
 * cleanup
 */
static void
app_init_status_win_destroy(init_data_t *idp)
{
	g_assert(idp->files[0] == NULL);
	g_free(idp->files);
	if (idp->toplev)
		gtk_widget_destroy(idp->toplev);
	g_free(idp);
} /* app_init_status_win_destroy */
#endif	/* WANT_SPLASH */


#ifdef WANT_SPLASH
/*
 * PRIVATE: app_init_status_win_update
 *
 * update the two labels in the status window.
 */
static void
app_init_status_win_update(init_data_t *idp)
{
	char *buf, *bfname;
	int len;

	bfname = (char *)my_basename(idp->files[idp->num - 1]);
	len = MAX(strlen(bfname) + 1, strlen(FILES_LEFT) + 10);
	buf = g_new(char, len);
	g_snprintf(buf, len, " %s%d ", FILES_LEFT, idp->num);
	gtk_label_set(GTK_LABEL(idp->fnum), buf);
	g_snprintf(buf, len, " %s ", bfname);
	gtk_label_set(GTK_LABEL(idp->fname), buf);
	g_free(buf);
	while (gtk_events_pending())
		gtk_main_iteration_do(TRUE);
	gdk_flush();
} /* app_init_status_win_update */
#endif	/* WANT_SPLASH */


/*
 * PRIVATE: app_parse_args
 *
 * parse for application specific arguments.  return argc and argv with any app
 * specific arguments omitted/removed.
 */
#define	SHIFT_ARGS(zz_argc, zz_argp)					\
	do {								\
		char **aptmp;						\
		(zz_argc)--;						\
		aptmp = zz_argp;					\
		while (*aptmp) {					\
			if (*(aptmp + 1) == NULL) {			\
				*aptmp = NULL;				\
			} else {					\
				*aptmp = *(aptmp + 1);			\
				aptmp++;				\
			}						\
		}							\
	} while (0)

static void
app_parse_args(int *argcp, char ***argvp, bool_t *do_fork, bool_t *do_splash)
{
	char **ap = *argvp;

	while (*ap) {
		GNPDBG_GENERIC(("app_parse_args: *ap = '%s'\n", *ap));
		if (strcmp(*ap, "--help") == 0)
			app_show_help(*argvp);	/* does not return */
		else if (strcmp(*ap, "--version") == 0)
			app_show_version();	/* does not return */
		else if (strcmp(*ap, "--info") == 0)
			app_show_info();	/* does not return */
#ifdef WANT_BACKGROUND
		else if (strcmp(*ap, "--fork") == 0) {
			*do_fork = TRUE;
			SHIFT_ARGS(*argcp, ap);
		}
#endif
#ifdef WANT_SPLASH
		else if (strcmp(*ap, "--nosplash") == 0) {
			*do_splash = FALSE;
			SHIFT_ARGS(*argcp, ap);
		}
#endif
#ifdef APP_DEBUG
		else if (strcmp(*ap, "--debug") == 0) {
			SHIFT_ARGS(*argcp, ap);
			if (*ap) {
				if (*ap[0] == '0')
					sscanf(*ap, "%lx", &dbg_flags);
				else
					sscanf(*ap, "%ld", &dbg_flags);
				SHIFT_ARGS(*argcp, ap);
			}
			GNPDBG_ALWAYS(("dbg_flags = 0x%lx (%ld)\n",
					dbg_flags, dbg_flags));
		}
#endif
		else
			ap++;
	}
} /* app_parse_args */


/*
 * PRIVATE: app_print_appinfo
 *
 * duh
 */
static void
app_print_appinfo(void)
{
	printf("Compile/Link info:\n");
	printf("\tGTK version   = %s\n", gtkver);
#ifdef USE_GNOME
	printf("\tGNOME version = %s\n", gnomever);
#endif
	printf("\tCC            = %s\n", compiler);
	printf("\tGTK_CFLAGS    = %s\n", gtk_cflags);
	printf("\tCFLAGS        = %s\n", cflags);
	printf("\tGTK_LDFLAGS   = %s\n", gtk_ldflags);
	printf("\tLDFLAGS       = %s\n", ldflags);
	printf("\tUNAME         = %s\n", unamenfo);
} /* app_print_appinfo */


/*
 * PRIVATE: app_show_help
 *
 * duh
 */
static void
app_show_help(char **argv)
{
	int i;
	char *usage_msg[] = {
		N_("Options:"),
		N_("  --help       shows this screen"),
		N_("  --version    prints app name and version"),
		N_("  --info       prints compile/link options used"),
		" ",
#ifdef WANT_BACKGROUND
		N_("  --fork       automatically runs gnp+ in the background"),
#endif
#ifdef WANT_SPLASH
		N_("  --nosplash   don't show splash/init screen on startup"),
#endif
		NULL
	};

#ifdef ENABLE_NLS
	gtk_set_locale();
	bindtextdomain("gnotepad+", LOCALEDIR);
	textdomain("gnotepad+");
#endif

	printf(_("Usage: %s [options] [files...]\n"), argv[0]);
	for (i = 0; usage_msg[i]; i++)
		printf("%s\n", gettext(usage_msg[i]));
	exit(0);
} /* app_show_help */


/*
 * PRIVATE: app_show_info
 *
 * duh
 */
static void
app_show_info(void)
{
	PRINT_VERSION();
	app_print_appinfo();
	exit(0);
} /* app_show_help */


/*
 * PRIVATE: app_show_version
 *
 * duh
 */
static void
app_show_version(void)
{
	PRINT_VERSION();
	exit(0);
} /* app_show_version */


/* the end */
