/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appSystem.h>

#   include	<bitmap.h>

#   include	<debugon.h>

#   include	<charnames.h>
#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Save a tag with an argument.					*/
/*									*/
/************************************************************************/
static void docRtfWriteArgTag(	const char *		tag,
				int			arg,
				SimpleOutputStream *	sos )
    {
    char	scratch[20];

    sioOutPutString( tag, sos );

    sprintf( scratch, "%d", arg );
    sioOutPutString( scratch, sos );

    return;
    }

static void docRtfEscapeString(	const unsigned char *	s,
				const unsigned char *	outputMapping,
				int *			pCol,
				int			n,
				SimpleOutputStream *	sos )
    {
    while( n && *s )
	{
	int		c= *s;

	if  ( outputMapping )
	    { c= outputMapping[c];	}

	switch( c )
	    {
	    case '{': case '\\': case '}':
		sioOutPutCharacter( '\\', sos );
		sioOutPutCharacter( c, sos );
		*pCol += 2;
		break;
	    default:
		if  ( c > 127 )
		    {
		    static char	hexdigits[]= "0123456789abcdef";

		    sioOutPutCharacter( '\\', sos );
		    sioOutPutCharacter( '\'', sos );
		    sioOutPutCharacter( hexdigits[ ( c >> 4 ) & 0x0f ], sos );
		    sioOutPutCharacter( hexdigits[ ( c >> 0 ) & 0x0f ], sos );
		    *pCol += 4;
		    }
		else{
		    sioOutPutCharacter( c, sos );
		    *pCol += 1;
		    }
		break;
	    }

	n--; s++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Save border definition.						*/
/*									*/
/************************************************************************/
static void docRtfSaveBorder(	char *				tag,
				const BorderProperties *	bp,
				SimpleOutputStream *		sos )
    {
    if  ( ! bp->bpIsSet )
	{ return;	}

    sioOutPutString( tag, sos );

    switch( bp->bpThickness )
	{
	case 0:
	    break;
	case 1:	sioOutPutString( "\\brdrs", sos );	break;
	case 2:	sioOutPutString( "\\brdrth", sos );	break;
	default:
	    LDEB(bp->bpThickness);
	    break;
	}

    switch( bp->bpStyle )
	{
	case DOCbsSHADOWED:	sioOutPutString( "\\brdrsh", sos ); break;
	case DOCbsDOUBLE:	sioOutPutString( "\\brdrdb", sos ); break;
	case DOCbsDOT:		sioOutPutString( "\\brdrdot", sos ); break;
	case DOCbsDASH:		sioOutPutString( "\\brdrdash", sos ); break;
	case DOCbsHAIR:		sioOutPutString( "\\brdrhair", sos ); break;
	case DOCbsDASHSM:	sioOutPutString( "\\brdrdashsm", sos ); break;
	case DOCbsDASHD:	sioOutPutString( "\\brdrdashd", sos ); break;
	case DOCbsDASHDD:	sioOutPutString( "\\brdrdashdd", sos ); break;
	case DOCbsTRIPLE:	sioOutPutString( "\\brdrtriple", sos ); break;
	case DOCbsTNTHSG:	sioOutPutString( "\\brdrtnthsg", sos ); break;
	case DOCbsTHTNSG:	sioOutPutString( "\\brdrthtnsg", sos ); break;
	case DOCbsTNTHTNSG:	sioOutPutString( "\\brdrtnthtnsg", sos ); break;
	case DOCbsTNTHMG:	sioOutPutString( "\\brdrtnthmg", sos ); break;
	case DOCbsTHTNMG:	sioOutPutString( "\\brdrthtnmg", sos ); break;
	case DOCbsTNTHTNMG:	sioOutPutString( "\\brdrtnthtnmg", sos ); break;
	case DOCbsTNTHLG:	sioOutPutString( "\\brdrtnthlg", sos ); break;
	case DOCbsTHTNLG:	sioOutPutString( "\\brdrthtnlg", sos ); break;
	case DOCbsTNTHTNLG:	sioOutPutString( "\\brdrtnthtnlg", sos ); break;
	case DOCbsWAVY:		sioOutPutString( "\\brdrwavy", sos ); break;
	case DOCbsWAVYDB:	sioOutPutString( "\\brdrwavydb", sos ); break;
	case DOCbsDASHDOTSTR:	sioOutPutString( "\\brdrdashdotstr", sos );
				break;
	case DOCbsEMBOSS:	sioOutPutString( "\\brdremboss", sos ); break;
	case DOCbsENGRAVE:	sioOutPutString( "\\brdrengrave", sos ); break;
	default:
	    LDEB(bp->bpStyle);
	    break;
	}

    if  ( bp->bpWidthTwips != 0 )
	{ docRtfWriteArgTag( "\\brdrw", bp->bpWidthTwips, sos );	}

    if  ( bp->bpColor != 0 )
	{ docRtfWriteArgTag( "\\brdrcf", bp->bpColor, sos );	}

    if  ( bp->bpSpacingTwips != 0 )
	{ docRtfWriteArgTag( "\\brsp", bp->bpSpacingTwips, sos );	}

    sioOutPutString( "\r\n", sos );

    return;
    }

/************************************************************************/
/*									*/
/*  Save possible attribute changes.					*/
/*									*/
/************************************************************************/
static int docRtfSaveAttributeNeg(	SimpleOutputStream *	sos,
					int *			pAttChange,
					int *			pCol,
					TextAttribute		taOld,
					TextAttribute		taNew	)
    {
    if  ( taOld.taSuperSub != DOCfontREGULAR	&&
	  taNew.taSuperSub != taOld.taSuperSub	)
	{
	sioOutPutString( "\\nosupersub", sos );
	*pCol += 11; *pAttChange= 1;
	}

    if  ( ! taNew.taIsUnderlined && taOld.taIsUnderlined )
	{
	sioOutPutString( "\\ul0", sos );
	*pCol += 4; *pAttChange= 1;
	}

    if  ( ! taNew.taFontIsSlanted && taOld.taFontIsSlanted )
	{
	sioOutPutString( "\\i0", sos );
	*pCol += 3; *pAttChange= 1;
	}

    if  ( ! taNew.taFontIsBold && taOld.taFontIsBold )
	{
	sioOutPutString( "\\b0", sos );
	*pCol += 3; *pAttChange= 1;
	}

    return 0;
    }

static int docRtfSaveAttributePos(	SimpleOutputStream *	sos,
					int *			pAttChange,
					int *			pCol,
					TextAttribute		taOld,
					TextAttribute		taNew	)
    {
    if  ( taNew.taFontNumber != taOld.taFontNumber )
	{
	docRtfWriteArgTag( "\\f", taNew.taFontNumber, sos );
	*pCol += 3; *pAttChange= 1;
	}

    if  ( taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints )
	{
	docRtfWriteArgTag( "\\fs", taNew.taFontSizeHalfPoints, sos );
	*pCol += 4; *pAttChange= 1;
	}

    if  ( taNew.taFontIsBold && ! taOld.taFontIsBold )
	{
	sioOutPutString( "\\b", sos );
	*pCol += 2; *pAttChange= 1;
	}

    if  ( taNew.taFontIsSlanted && ! taOld.taFontIsSlanted )
	{
	sioOutPutString( "\\i", sos );
	*pCol += 2; *pAttChange= 1;
	}

    if  ( taNew.taIsUnderlined && ! taOld.taIsUnderlined )
	{
	sioOutPutString( "\\ul", sos );
	*pCol += 3; *pAttChange= 1;
	}

    if  ( taNew.taSuperSub != DOCfontREGULAR		&&
	  taNew.taSuperSub != taOld.taSuperSub	)
	{
	switch( taNew.taSuperSub )
	    {
	    case DOCfontSUPERSCRIPT:
		sioOutPutString( "\\super", sos );
		*pCol += 6; *pAttChange= 1;
		break;
	    case DOCfontSUBSCRIPT:
		sioOutPutString( "\\sub", sos );
		*pCol += 4; *pAttChange= 1;
		break;
	    default:
		LDEB(taNew.taSuperSub); break;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph in RTF format.					*/
/*									*/
/*  Column counting is approximate as it is only for cosmetic reasons.	*/
/*									*/
/************************************************************************/
static void docRtfParaSaveProperties( SimpleOutputStream *	sos,
				int *				pPropertyChange,
				int *				pCol,
				const ParagraphProperties *	newPP,
				const ParagraphProperties *	prevPP )
    {
    int			startFromDefault= 0;
    int			i;

    ParagraphProperties	ppp;

    docInitParagraphProperties( &ppp );
    docCopyParagraphProperties( &ppp, prevPP );

    if  ( newPP->ppTabCount != ppp.ppTabCount )
	{ startFromDefault= 1;	}
    else{
	for ( i= 0; i < newPP->ppTabCount; i++ )
	    {
	    if  ( newPP->ppTabStops[i].tsTwips	!=
		  ppp.ppTabStops[i].tsTwips	)
		{ startFromDefault= 1; break; }
	    }
	}

    if  ( ! docEqualBorder( &(ppp.ppTopBorder),
					    &(newPP->ppTopBorder) )	||
	  ! docEqualBorder( &(ppp.ppBottomBorder),
					    &(newPP->ppBottomBorder) )	)
	{ startFromDefault= 1;	}

    if  ( ppp.ppLineSpacingTwips != newPP->ppLineSpacingTwips		||
	  ppp.ppLineSpacingIsMultiple != newPP->ppLineSpacingIsMultiple )
	{ startFromDefault= 1;	}

    if  ( startFromDefault )
	{
	docCleanParagraphProperties( &ppp );
	docInitParagraphProperties( &ppp );

	/*  1  */
	sioOutPutString( "\\pard", sos );
	*pCol += 5;
	if  ( newPP->ppInTable )
	    { sioOutPutString( "\\intbl", sos ); *pCol += 6;	}

	*pPropertyChange= 1;
	}

    if  ( newPP->ppStartsOnNewPage != ppp.ppStartsOnNewPage )
	{
	if  ( newPP->ppStartsOnNewPage )
	    { sioOutPutString( "\\pagebb",  sos ); *pCol += 7; }
	else{ sioOutPutString( "\\pagebb0", sos ); *pCol += 8; }
	}

    if  ( *pCol >= 72 )
	{ sioOutPutString( "\r\n", sos ); *pCol= 0;	}

    if  ( newPP->ppFirstIndentTwips != ppp.ppFirstIndentTwips )
	{
	docRtfWriteArgTag( "\\fi", newPP->ppFirstIndentTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppLeftIndentTwips != ppp.ppLeftIndentTwips )
	{
	docRtfWriteArgTag( "\\li", newPP->ppLeftIndentTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppRightIndentTwips != ppp.ppRightIndentTwips )
	{
	docRtfWriteArgTag( "\\ri", newPP->ppRightIndentTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppSpaceBeforeTwips != ppp.ppSpaceBeforeTwips )
	{
	docRtfWriteArgTag( "\\sb", newPP->ppSpaceBeforeTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppSpaceAfterTwips != ppp.ppSpaceAfterTwips )
	{
	docRtfWriteArgTag( "\\sa", newPP->ppSpaceAfterTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppLineSpacingTwips != ppp.ppLineSpacingTwips )
	{
	docRtfWriteArgTag( "\\sl", newPP->ppSpaceAfterTwips, sos );
	docRtfWriteArgTag( "\\slmult", newPP->ppLineSpacingIsMultiple, sos );
	*pCol += 14; *pPropertyChange= 1;
	}

    if  ( newPP->ppAlignment != ppp.ppAlignment )
	{
	switch( newPP->ppAlignment )
	    {
	    case DOCiaLEFT:	sioOutPutString( "\\ql", sos ); break;
	    case DOCiaRIGHT:	sioOutPutString( "\\qr", sos ); break;
	    case DOCiaCENTERED:	sioOutPutString( "\\qc", sos ); break;
	    case DOCiaJUSTIFIED:sioOutPutString( "\\qj", sos ); break;
	    default:
		LDEB(newPP->ppAlignment); break;
	    }

	*pCol += 3; *pPropertyChange= 1;
	}

    if  ( startFromDefault )
	{
	TabStop *	ts= newPP->ppTabStops;

	for ( i= 0; i < newPP->ppTabCount; ts++, i++ )
	    {
	    if  ( *pCol >= 65 )
		{ sioOutPutString( "\r\n", sos ); *pCol= 0;	}

	    switch( ts->tsKind )
		{
		case DOCtkLEFT:
		    break;
		case DOCtkRIGHT:
		    sioOutPutString( "\\tqr", sos ); *pCol += 4;
		    break;
		case DOCtkCENTRE:
		    sioOutPutString( "\\tqc", sos ); *pCol += 4;
		    break;
		case DOCtkDECIMAL:
		    sioOutPutString( "\\tqdec", sos ); *pCol += 6;
		    break;
		default:
		    LDEB(ts->tsKind); break;
		}

	    switch( ts->tsLeader )
		{
		case DOCtlNONE:
		    break;
		case DOCtlDOTS:
		    sioOutPutString( "\\tldot", sos ); *pCol += 6;
		    break;
		case DOCtlHYPH:
		    sioOutPutString( "\\tlhyph", sos ); *pCol += 7;
		    break;
		case DOCtlUNDERLINE:
		    sioOutPutString( "\\tlul", sos ); *pCol += 5;
		    break;
		case DOCtlTHICK:
		    sioOutPutString( "\\tlth", sos ); *pCol += 5;
		    break;
		case DOCtlEQUAL:
		    sioOutPutString( "\\tleq", sos ); *pCol += 5;
		    break;
		default:
		    LDEB(ts->tsLeader); break;
		}

	    docRtfWriteArgTag( "\\tx", ts->tsTwips, sos );
	    *pCol += 7;
	    }
	}

    if  ( ! docEqualBorder( &(ppp.ppTopBorder),
					    &(newPP->ppTopBorder) )	)
	{ docRtfSaveBorder( "\\brdrt", &(newPP->ppTopBorder), sos ); }

    if  ( ! docEqualBorder( &(ppp.ppBottomBorder),
					    &(newPP->ppBottomBorder) )	)
	{ docRtfSaveBorder( "\\brdrb", &(newPP->ppBottomBorder), sos ); }

    docCleanParagraphProperties( &ppp );

    return;
    }

int docRtfSaveRuler(	SimpleOutputStream *		sos,
			const ParagraphProperties *	pp )
    {
    int				col= 0;
    int				precedingTags= 1;

    ParagraphProperties		refPP;

    docInitParagraphProperties( &refPP );

    sioOutPutString( "{\\ruler\\pard", sos );

    docRtfParaSaveProperties( sos, &precedingTags, &col, pp, &refPP );

    sioOutPutString( "}", sos );

    docCleanParagraphProperties( &refPP );

    return 0;
    }

static int docRtfSaveObjectData(	const ObjectData *		od,
					SimpleOutputStream *		sos )
    {
    const unsigned char *	s;
    int				i;

    s= od->odBytes;
    for ( i= 0; i < od->odSize; s++, i++ )
	{
	if  ( ! ( i % 78 ) )
	    { sioOutPutString( "\r\n", sos ); }

	sioOutPutCharacter( *s, sos );
	}

    return 0;
    }


static int docRtfSavePictureTags(	InsertedObject *		io,
					SimpleOutputStream *		sos )
    {
    int		xExt= io->ioXExtent;
    int		yExt= io->ioYExtent;

    if  ( xExt == 0 )
	{ xExt= (int)( 1000.0* io->ioTwipsWide )/ ( 20* POINTS_PER_CM ); }
    if  ( yExt == 0 )
	{ yExt= (int)( 1000.0* io->ioTwipsHigh )/ ( 20* POINTS_PER_CM ); }

    docRtfWriteArgTag( "\\picw", xExt, sos );
    docRtfWriteArgTag( "\\pich", yExt, sos );

    if  ( io->ioScaleX != 100 )
	{ docRtfWriteArgTag( "\\picscalex", io->ioScaleX, sos ); }
    if  ( io->ioScaleY != 100 )
	{ docRtfWriteArgTag( "\\picscaley", io->ioScaleY, sos ); }

    docRtfWriteArgTag( "\\picwgoal", io->ioTwipsWide, sos );
    docRtfWriteArgTag( "\\pichgoal", io->ioTwipsHigh, sos );

    if  ( io->ioBliptag == 0 )
	{ io->ioBliptag= appGetTimestamp(); 	}

    if  ( io->ioBliptag != 0 )
	{ docRtfWriteArgTag( "\\bliptag", io->ioBliptag, sos ); }

    return 0;
    }

static int docRtfSaveObject(	SimpleOutputStream *		sos,
				const BufferItem *		bi,
				const TextParticule *		tp )
    {
    InsertedObject *	io= bi->biParaObjects+ tp->tpObjectNumber;


    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    sioOutPutString( "\r\n{\\pict\\wmetafile8", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );

	    return 0;
	case DOCokPICTPNGBLIP:
	    sioOutPutString( "\r\n{\\pict\\pngblip", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokPICTJPEGBLIP:
	    sioOutPutString( "\r\n{\\pict\\jpegblip", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokOLEOBJECT:
	    sioOutPutString( "\r\n{\\object\\objemb", sos );

	    if  ( io->ioObjectClass )
		{
		sioOutPutString( "\r\n{\\*\\objclass ", sos );
		sioOutPutString( (char *)io->ioObjectClass, sos );
		sioOutPutString( "}", sos );
		}

	    if  ( io->ioObjectName )
		{
		sioOutPutString( "\r\n{\\*\\objname ", sos );
		sioOutPutString( (char *)io->ioObjectName, sos );
		sioOutPutString( "}", sos );
		}

	    docRtfWriteArgTag( "\\objw", io->ioTwipsWide, sos );
	    docRtfWriteArgTag( "\\objh", io->ioTwipsHigh, sos );
	    if  ( io->ioScaleX != 100 )
		{ docRtfWriteArgTag( "\\objscalex", io->ioScaleX, sos ); }
	    if  ( io->ioScaleY != 100 )
		{ docRtfWriteArgTag( "\\objscaley", io->ioScaleY, sos ); }

	    sioOutPutString( "\r\n{\\*\\objdata ", sos );
	    docRtfSaveObjectData( &io->ioObjectData, sos );
	    sioOutPutString( "\r\n}", sos );

	    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
		{
		sioOutPutString( "{\\result {\\pict\\wmetafile8", sos );

		docRtfSavePictureTags( io, sos );

		docRtfSaveObjectData( &io->ioResultData, sos );

		sioOutPutString( "\r\n}}", sos );
		}

	    sioOutPutString( "\r\n}", sos );

	    return 0;
	default:
	    LDEB(io->ioKind); return -1;
	}

    /*  Not reached ..
    return 0;
    */
    }

/************************************************************************/
/*									*/
/*  Reserve a number of columns in the output file.			*/
/*									*/
/************************************************************************/
static void docRtfReserveColumns(	SimpleOutputStream *	sos,
					int			cols,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    if  ( col > 0 && col+ cols > 72 )
	{
	sioOutPutString( "\r\n", sos );
	rwc->rwcHasPrecedingTags= 0; col= 0;
	}

    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Switch text attributes.						*/
/*									*/
/************************************************************************/

static void docRtfSaveAttributes(	SimpleOutputStream *	sos,
					int			cols,
					TextAttribute		ta,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    docRtfSaveAttributeNeg( sos, &(rwc->rwcHasPrecedingTags), &col,
						rwc->rwcTextAttribute, ta );

    docRtfReserveColumns( sos, cols, &col, rwc );

    docRtfSaveAttributePos( sos, &(rwc->rwcHasPrecedingTags), &col,
						rwc->rwcTextAttribute, ta );
    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Finish/Begin writing a hyperlink.					*/
/*									*/
/************************************************************************/

static void docRtfFinishFldrslt(	SimpleOutputStream *	sos,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    sioOutPutString( "}}}", sos );
    col += 3;
    rwc->rwcHasPrecedingTags= 0;
    rwc->rwcInFldrslt= 0;
    rwc->rwcTextAttribute= rwc->rwcOutsideLinkAttribute;

    *pCol= col; return;
    }

static void docRtfSaveLinkStart(	SimpleOutputStream *	sos,
					const unsigned char *	bytes,
					int			byteCount,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int			col= *pCol;

    docRtfReserveColumns( sos, 19, &col, rwc );

    sioOutPutString( "{\\field{\\*\\fldinst{", sos );
    col += 19;

    docRtfReserveColumns( sos, byteCount, &col, rwc );

    while( byteCount > 1 && bytes[byteCount- 1] == ' ' )
	{ byteCount--;	}

    docRtfEscapeString( bytes, (const unsigned char *)0, &col, byteCount, sos );

    sioOutPutString( " }}{\\fldrslt{", sos ); col += 13;

    *pCol= col; return;
    }

static int docRtfSaveLinkInstructions(	SimpleOutputStream *	sos,
					int *			pCol,
					RtfWritingContext *	rwc,
					const char *		fileName,
					int			fileSize,
					const char *		markName,
					int			markSize )
    {
    int			col= *pCol;
    int			size;

    size= 11;
    if  ( fileSize > 0 )
	{ size += 1+ fileSize+ 2;	}
    if  ( markSize > 0 )
	{ size += 4+ markSize+ 2;	}

    docRtfReserveColumns( sos, 19, &col, rwc );

    sioOutPutString( "{\\field{\\*\\fldinst{", sos );
    col += 19;

    docRtfReserveColumns( sos, size, &col, rwc );

    sioOutPutString( " HYPERLINK ", sos ); col += 11;

    if  ( fileSize > 0 )
	{
	sioOutPutString( "\"", sos ); col++;
	docRtfEscapeString( (const unsigned char *)fileName,
			    (const unsigned char *)0, &col, fileSize, sos );
	sioOutPutString( "\" ", sos ); col += 2;
	}

    if  ( markSize > 0 )
	{
	docRtfEscapeString( (const unsigned char *)"\\l \"",
			     (const unsigned char *)0, &col, 4, sos );
	docRtfEscapeString( (const unsigned char *)markName,
			     (const unsigned char *)0, &col, markSize, sos );
	sioOutPutString( "\" ", sos ); col += 2;
	}

    sioOutPutString( "}}{\\fldrslt{", sos ); col += 12;
    rwc->rwcHasPrecedingTags= 0;

    *pCol= col; return 0;
    }

static void docRtfStartHyperlink(	SimpleOutputStream *	sos,
					DocumentField *		df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    const char *		fileName= (char *)0;
    int				fileSize;
    const char *		markName= (char *)0;
    int				markSize;

    if  ( rwc->rwcInFldrslt )
	{ docRtfFinishFldrslt( sos, &col, rwc );	}

    rwc->rwcOutsideLinkAttribute= rwc->rwcTextAttribute;

    if  ( ! docGetHyperlink( df, &fileName, &fileSize, &markName, &markSize ) )
	{
	unsigned char *	bytes= df->dfInstructions.odBytes;
	int		byteCount= df->dfInstructions.odSize;

	docRtfSaveLinkStart( sos, bytes, byteCount, &col, rwc );

	rwc->rwcInFldrslt= 1;
	}

    *pCol= col; return;
    }

static void docRtfStartPagefield(	SimpleOutputStream *	sos,
					DocumentField *		df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    if  ( rwc->rwcInFldrslt )
	{ docRtfFinishFldrslt( sos, &col, rwc );	}

    rwc->rwcOutsideLinkAttribute= rwc->rwcTextAttribute;

    docRtfReserveColumns( sos, 19, &col, rwc );

    sioOutPutString( "{\\field{\\*\\fldinst{", sos );
    col += 19;

    sioOutPutString( " PAGE ", sos ); col += 6;

    sioOutPutString( "}}{\\fldrslt{", sos ); col += 12;

    rwc->rwcInFldrslt= 1;
    rwc->rwcHasPrecedingTags= 0;

    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph.							*/
/*									*/
/*  1)  When we are saving a selection, and the selection is inside a	*/
/*	table cell, do not set the \intbl flag.				*/
/*  2)  To make WP 8 happy, always save 'intbl' for the first paragraph	*/
/*	in a table row.							*/
/*									*/
/************************************************************************/

static int docRtfSaveParaItem(	SimpleOutputStream *	sos,
				const BufferItem *	bi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    TextParticule *	tp;
    unsigned char *	s;

    int			col= 0;

    int			upto;
    int			part= 0;
    int			stroff= 0;

    DocumentField *	df;

    if  ( ! bs								||
	  bs->bsBegin.bpBi->biParent != bs->bsEnd.bpBi->biParent	)
	{
	/*  2  */
	if  ( bi->biParaInTable != rwc->rwcParagraphProperties.ppInTable ||
	      ( bi->biParaInTable			&&
		bi->biNumberInParent == 0		&&
		bi->biParent				&&
		bi->biParent->biNumberInParent == 0	)		 )
	    {
	    if  ( bi->biParaInTable )
		{ sioOutPutString( "\\intbl",  sos ); col += 6; }
	    else{ sioOutPutString( "\\intbl0", sos ); col += 7; }
	    }
	}

    docRtfParaSaveProperties( sos, &(rwc->rwcHasPrecedingTags), &col,
		    &(bi->biParaProperties), &(rwc->rwcParagraphProperties) );

    /*  1  */
    docInitTextAttribute( &(rwc->rwcTextAttribute) );
    docInitTextAttribute( &(rwc->rwcOutsideLinkAttribute) );
    rwc->rwcTextAttribute.taFontSizeHalfPoints= 24;
    rwc->rwcOutsideLinkAttribute.taFontSizeHalfPoints= 24;

    /*  2  */
    sioOutPutString( "\\plain", sos ); col += 6;
    rwc->rwcHasPrecedingTags= 1;

    if  ( bi->biParaParticuleCount == 0 )
	{ LDEB(bi->biParaParticuleCount);	}

    if  ( bs && bs->bsBegin.bpBi == bi )
	{
	part= bs->bsBegin.bpParticule;
	stroff= bs->bsBegin.bpStroff;
	}

    tp= bi->biParaParticules+ part;
    s= bi->biParaString+ stroff;

    if  ( bs && bs->bsEnd.bpBi == bi )
	{ upto= bs->bsEnd.bpStroff;	}
    else{ upto= bi->biParaStrlen;	}

    for ( ; part < bi->biParaParticuleCount; part++, tp++ )
	{
	if  ( tp->tpTextAttribute.taFontNumber >= 0 )
	    {
	    docRtfSaveAttributes( sos, tp->tpStrlen,
					    tp->tpTextAttribute, &col, rwc );
	    rwc->rwcTextAttribute= tp->tpTextAttribute;
	    }

	if  ( rwc->rwcSaveAsLink )
	    {
	    if  ( docRtfSaveLinkInstructions( sos, &col, rwc,
		    rwc->rwcSaveAsLinkFileName, rwc->rwcSaveAsLinkFileSize,
		    rwc->rwcSaveAsLinkMarkName, rwc->rwcSaveAsLinkMarkSize ) )
		{ LDEB(1); return -1;	}

	    rwc->rwcOutsideLinkAttribute= rwc->rwcTextAttribute;
	    rwc->rwcInFldrslt= 1;
	    rwc->rwcSaveAsLink= 0;
	    }

	switch( tp->tpKind )
	    {
	    int		n;

	    case DOCkindTAB:
		if  ( stroff >= upto )
		    { break;	}

		sioOutPutString( "\\tab", sos ); s++; stroff++; col += 4;
		rwc->rwcHasPrecedingTags= 1;
		continue;
	    case DOCkindTEXT:
		if  ( stroff >= upto )
		    { break;	}

		if  ( rwc->rwcHasPrecedingTags )
		    {
		    sioOutPutString( " ", sos ); col++;
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( tp->tpStroff+ tp->tpStrlen > upto )
		    { n= upto- stroff;				}
		else{ n= tp->tpStroff+ tp->tpStrlen- stroff;	}

		docRtfEscapeString( s, rwc->rwcOutputMapping, &col, n, sos );
		stroff += n; s += n;

		continue;
	    case DOCkindOBJECT:
		if  ( stroff >= upto )
		    { break;	}

		if  ( docRtfSaveObject( sos, bi, tp ) )
		    { LDEB(tp->tpKind); }
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		rwc->rwcHasPrecedingTags= 0; col= 1;
		continue;

	    case DOCkindFIELDSTART:
		if  ( stroff >= upto )
		    { break;	}

		df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		if  ( df->dfKind == DOCfkHYPERLINK )
		    {
		    docRtfStartHyperlink( sos, df, &col, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( df->dfKind == DOCfkPAGEFIELD )
		    {
		    docRtfStartPagefield( sos, df, &col, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindFIELDEND:
		if  ( rwc->rwcInFldrslt )
		    { docRtfFinishFldrslt( sos, &col, rwc );	}

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindBKMKSTART:
		if  ( stroff >= upto )
		    { break;	}

		if  ( rwc->rwcSaveBookmarks )
		    {
		    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\*\\bkmkstart ", sos ); col += 14;
		    docRtfEscapeString( df->dfInstructions.odBytes,
				    (const unsigned char *)0,
				    &col, df->dfInstructions.odSize, sos );
		    sioOutPutString( "}", sos ); col += 1;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindBKMKEND:
		if  ( rwc->rwcSaveBookmarks )
		    {
		    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\*\\bkmkend ", sos ); col += 12;
		    docRtfEscapeString( df->dfInstructions.odBytes,
				    (const unsigned char *)0,
				    &col, df->dfInstructions.odSize, sos );
		    sioOutPutString( "}", sos ); col += 1;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindXE:
		if  ( stroff >= upto )
		    { break;	}

		if  (  rwc->rwcSaveBookmarks )
		    {
		    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\xe {", sos ); col += 5;
		    docRtfEscapeString( df->dfInstructions.odBytes,
				    (const unsigned char *)0,
				    &col, df->dfInstructions.odSize, sos );
		    sioOutPutString( "}}", sos ); col += 1;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindTC:
		if  ( stroff >= upto )
		    { break;	}

		if  (  rwc->rwcSaveBookmarks )
		    {
		    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\tc {", sos ); col += 5;
		    docRtfEscapeString( df->dfInstructions.odBytes,
				    (const unsigned char *)0,
				    &col, df->dfInstructions.odSize, sos );
		    sioOutPutString( "}}", sos ); col += 1;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		continue;
	    }

	break;
	}

    if  ( rwc->rwcInFldrslt )
	{ docRtfFinishFldrslt( sos, &col, rwc );	}

    if  ( stroff == bi->biParaStrlen )
	{
	if  ( ! bi->biParaInTable					||
	      bi->biNumberInParent < bi->biParent->biGroupChildCount- 1	)
	    { sioOutPutString( "\\par\r\n", sos );	}
	/*
	else{ sioOutPutString( "\\par", sos );		}
	*/
	}

    if  ( docCopyParagraphProperties( &(rwc->rwcParagraphProperties),
						&(bi->biParaProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save Cell properties.						*/
/*									*/
/************************************************************************/

static void docRtfSaveCellProperties(	const CellProperties *	cp,
					int			shiftLeft,
					SimpleOutputStream *	sos )
    {
    int		didAnything= 0;

    if  ( cp->cpFirstInMergedRange )
	{ sioOutPutString( "\\clmgf", sos ); didAnything= 1;	}
    if  ( cp->cpMergedWithPrevious )
	{ sioOutPutString( "\\clmrg", sos ); didAnything= 1;	}

    if  ( didAnything )
	{ sioOutPutString( "\r\n", sos );	}

    docRtfSaveBorder( "\\clbrdrt", &(cp->cpTopBorder), sos );
    docRtfSaveBorder( "\\clbrdrl", &(cp->cpLeftBorder), sos );
    docRtfSaveBorder( "\\clbrdrr", &(cp->cpRightBorder), sos );
    docRtfSaveBorder( "\\clbrdrb", &(cp->cpBottomBorder), sos );

    docRtfWriteArgTag( "\\cellx", cp->cpRightBoundaryTwips- shiftLeft, sos );
    sioOutPutString( "\r\n", sos );
    }

static void docRtfSaveRowProperties(	const RowProperties *	rp,
					int			col0,
					int			col1,
					SimpleOutputStream *	sos )
    {
    const CellProperties *	cp= rp->rpCells;

    int				col;
    int				shiftLeft= 0;

    sioOutPutString( "\\trowd", sos );

    if  ( rp->rpHalfGapWidthTwips != 0 )
	{ docRtfWriteArgTag( "\\trgaph", rp->rpHalfGapWidthTwips, sos ); }

    if  ( rp->rpLeftIndentTwips != 0 )
	{ docRtfWriteArgTag( "\\trleft", rp->rpLeftIndentTwips, sos ); }

    if  ( rp->rpHeightTwips != 0 )
	{ docRtfWriteArgTag( "\\trrh", rp->rpHeightTwips, sos ); }

    if  ( rp->rpIsTableHeader )
	{ sioOutPutString( "\\trhdr", sos ); }

    sioOutPutString( "\r\n", sos );

    docRtfSaveBorder( "\\trbrdrt", &(rp->rpTopBorder), sos );
    docRtfSaveBorder( "\\trbrdrl", &(rp->rpLeftBorder), sos );
    docRtfSaveBorder( "\\trbrdrr", &(rp->rpRightBorder), sos );
    docRtfSaveBorder( "\\trbrdrb", &(rp->rpBottomBorder), sos );

    if  ( rp->rpHasVerticalBorders )
	{ docRtfSaveBorder( "\\trbrdrv", &(rp->rpVerticalBorder), sos ); }
    if  ( rp->rpHasHorizontalBorders )
	{ docRtfSaveBorder( "\\trbrdrh", &(rp->rpHorizontalBorder), sos ); }

    if  ( col0 > 0 )
	{ shiftLeft= cp[col0-1].cpRightBoundaryTwips; }

    for ( col= 0; col < rp->rpCellCount; cp++, col++ )
	{
	if  ( ( col0 < 0 || col >= col0 )	&&
	      ( col1 < 0 || col <= col1 )	)
	    { docRtfSaveCellProperties( cp, shiftLeft, sos );	}
	}

    return;
    }

static int docRtfSaveRowItem(	SimpleOutputStream *	sos,
				const BufferItem *	rowBi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    int			col;

    if  ( ! bs								||
	  bs->bsBegin.bpBi->biParent != bs->bsEnd.bpBi->biParent	)
	{
	if  ( ! docEqualRows( &(rowBi->biRowProperties),
						&(rwc->rwcRowProperties) ) )
	    {
	    int			col0= -1;
	    int			col1= -1;

	    if  ( bs )
		{ col0= bs->bsCol0; col1= bs->bsCol1; 	}

	    docRtfSaveRowProperties( &(rowBi->biRowProperties),
							col0, col1, sos );

	    if  ( docCopyRowProperties( &(rwc->rwcRowProperties),
						&(rowBi->biRowProperties) ) )
		{ LDEB(1);	}
	    }
	}

    for ( col= 0; col < rowBi->biGroupChildCount; col++ )
	{
	int		par;
	BufferItem *	cellBi= rowBi->biGroupChildren[col];

	if  ( bs && docCompareItemPositions( cellBi, bs->bsBegin.bpBi ) < 0 )
	    { continue;	}

	if  ( bs && docCompareItemPositions( cellBi, bs->bsEnd.bpBi ) > 0 )
	    { continue;	}

	if  ( ! bs						||
	      ( ( bs->bsCol0 < 0 || col >= bs->bsCol0 )	&&
		( bs->bsCol1 < 0 || col <= bs->bsCol1 )	)	)
	    {
	    for ( par= 0; par < cellBi->biGroupChildCount; par++ )
		{
		BufferItem *	paraBi= cellBi->biGroupChildren[par];

		if  ( bs && docCompareItemPositions( paraBi,
						    bs->bsBegin.bpBi ) < 0 )
		    { continue;	}

		if  ( bs && docCompareItemPositions( paraBi,
						    bs->bsEnd.bpBi ) > 0 )
		    { continue;	}

		if  ( docRtfSaveParaItem( sos, paraBi, bs, rwc ) )
		    { LLDEB(col,par); return -1;	}
		}

	    if  ( col < rowBi->biGroupChildCount- 1 )
		{ sioOutPutString( "\\cell\r\n", sos );	}
	    else{ sioOutPutString( "\\cell", sos );		}
	    }
	}

    sioOutPutString( "\\row\r\n", sos );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save an inidividual header or footer.				*/
/*									*/
/************************************************************************/

static int docRtfSaveHeaderFooter(	SimpleOutputStream *	sos,
					const BufferItem *	bi,
					const char *		tag,
					RtfWritingContext *	rwc )
    {
    int		i;

    sioOutPutString( tag, sos );

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );

    for ( i= 0; i < bi->biGroupChildCount; i++ )
	{
	if  ( docRtfSaveParaItem( sos, bi->biGroupChildren[i],
						(BufferSelection *)0, rwc ) )
	    { LDEB(i); return -1;	}
	}

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );

    sioOutPutString( "}\r\n", sos );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save the headers and footers of a Section.				*/
/*									*/
/************************************************************************/

static int docRtfSaveHeadersFooters(	SimpleOutputStream *	sos,
					const BufferItem *	bi,
					RtfWritingContext *	rwc )
    {
    if  ( bi->biSectHeader						&&
	  docRtfSaveHeaderFooter( sos, bi->biSectHeader,
					    "{\\header\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectFirstPageHeader					&&
	  docRtfSaveHeaderFooter( sos, bi->biSectFirstPageHeader,
					    "{\\headerf\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectLeftPageHeader					&&
	  docRtfSaveHeaderFooter( sos, bi->biSectLeftPageHeader,
					    "{\\headerl\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectRightPageHeader					&&
	  docRtfSaveHeaderFooter( sos, bi->biSectRightPageHeader,
					    "{\\headerr\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectFooter						&&
	  docRtfSaveHeaderFooter( sos, bi->biSectFooter,
					    "{\\footer\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectFirstPageFooter					&&
	  docRtfSaveHeaderFooter( sos, bi->biSectFirstPageFooter,
					    "{\\footerf\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectLeftPageFooter					&&
	  docRtfSaveHeaderFooter( sos, bi->biSectLeftPageFooter,
					    "{\\footerl\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( bi->biSectRightPageFooter					&&
	  docRtfSaveHeaderFooter( sos, bi->biSectRightPageFooter,
					    "{\\footerr\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a buffer item, hierarchically descending to the leaves of the	*/
/*  document tree.							*/
/*									*/
/************************************************************************/

static int docRtfSaveItem(	SimpleOutputStream *	sos,
				const BufferItem *	bi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    int		i;

    if  ( bs )
	{
	if  ( docCompareItemPositions( bi, bs->bsBegin.bpBi ) < 0	)
	    { return 0;	}

	if  ( docCompareItemPositions( bi, bs->bsEnd.bpBi ) > 0		)
	    { return 0;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( docRtfSaveItem( sos, bi->biGroupChildren[i], bs, rwc ) )
		    { LDEB(i); return -1;	}
		}
	    break;
	case DOClevSECT:
	    sioOutPutString( "\\sectd\r\n", sos );

	    if  ( ! bs						&&
		  docRtfSaveHeadersFooters( sos, bi, rwc )	)
		{ LDEB(1); return -1;	}

	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( docRtfSaveItem( sos, bi->biGroupChildren[i], bs, rwc ) )
		    { LDEB(i); return -1;	}
		}

	    sioOutPutString( "\\sect\r\n", sos );
	    break;
	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{
		docCleanRowProperties( &(rwc->rwcRowProperties) );
		docInitRowProperties( &(rwc->rwcRowProperties) );
		goto rowAsGroup;
		}

	    if  ( docRtfSaveRowItem( sos, bi, bs, rwc ) )
		{ LDEB(1); return -1;	}
	    break;
	case DOClevPARA:
	    if  ( docRtfSaveParaItem( sos, bi, bs, rwc ) )
		{ LDEB(1); return -1;	}
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

static void docRtfSaveInfo(	const char *		tag,
				const unsigned char *	info,
				RtfWritingContext *	rwc,
				SimpleOutputStream *	sos )
    {
    int		col= 0;

    if  ( info )
	{
	sioOutPutString( tag, sos );
	docRtfEscapeString( info, rwc->rwcOutputMapping,
					&col, strlen( (char *)info ), sos );
	sioOutPutString( "}\r\n", sos );
	}

    return;
    }

static int docRtfSaveDocumentProperties(	SimpleOutputStream *	sos,
						RtfWritingContext *	rwc,
						const BufferDocument *	bd )
    {
    const DocumentGeometry *	dg= &(bd->bdGeometry);
    const DocumentProperties *	dp= &(bd->bdProperties);

    int				i;

    int				codePage= bd->bdItem.biDocAnsiCodePage;

    if  ( codePage >= 0 )
	{
	switch( codePage )
	    {
	    case 1252:
		docRtfWriteArgTag( "\\ansicpg", codePage, sos );
		sioOutPutString( "\r\n", sos );
		break;

	    case 1250:
		docRtfWriteArgTag( "\\ansicpg", codePage, sos );
		sioOutPutString( "\r\n", sos );
		memcpy( rwc->rwcOutputMapping, docISO2_to_WIN1250, 256 );
		break;

	    default:
		LDEB(codePage); break;
	    }
	}

    if  ( bd->bdFontList.dflCount > 0 )
	{
	DocumentFont *	df=  bd->bdFontList.dflFonts;

	sioOutPutString( "{\\fonttbl\r\n", sos );

	for ( i= 0; i < bd->bdFontList.dflCount; df++, i++ )
	    {
	    if  ( ! df->dfFamilyStyle && ! df->dfName )
		{ continue;	}

	    docRtfWriteArgTag( "{\\f", df->dfDocFamilyNumber, sos );
	    sioOutPutString( "\\", sos );
	    if  ( df->dfFamilyStyle )
		{ sioOutPutString( df->dfFamilyStyle, sos );	}
	    else{ sioOutPutString( "fnil", sos );		}

	    if  ( df->dfCharset != FONTcharsetDEFAULT )
		{ docRtfWriteArgTag( "\\fcharset", df->dfCharset, sos ); }

	    if  ( df->dfPitch != FONTpitchDEFAULT )
		{ docRtfWriteArgTag( "\\fprq", df->dfPitch, sos ); }

	    if  ( df->dfName )
		{
		sioOutPutString( " ", sos );
		sioOutPutString( df->dfName, sos );
		}
	    sioOutPutString( ";}\r\n", sos );
	    }

	sioOutPutString( "}\r\n", sos );
	}

    if  ( bd->bdColorCount > 0 )
	{
	RGB8Color *			rgb8= bd->bdColors;

	sioOutPutString( "{\\colortbl\r\n", sos );

	for ( i= 0; i < bd->bdColorCount; rgb8++, i++ )
	    {
	    if  ( i != dp->dpDefaultColor )
		{
		docRtfWriteArgTag( "\\red", rgb8->rgb8Red, sos );
		docRtfWriteArgTag( "\\green", rgb8->rgb8Green, sos );
		docRtfWriteArgTag( "\\blue", rgb8->rgb8Blue, sos );
		}
	    sioOutPutString( ";\r\n", sos );
	    }

	sioOutPutString( "}\r\n", sos );
	}

    if  ( docHasDocumentInfo( bd ) )
	{
	sioOutPutString( "{\\info\r\n", sos );

	docRtfSaveInfo( "{\\title ",	bd->bdTitle, rwc, sos );
	docRtfSaveInfo( "{\\author ",	bd->bdAuthor, rwc, sos );
	docRtfSaveInfo( "{\\subject ",	bd->bdSubject, rwc, sos );
	docRtfSaveInfo( "{\\keywords ",	bd->bdKeywords, rwc, sos );
	docRtfSaveInfo( "{\\comment ",	bd->bdComment, rwc, sos );

	sioOutPutString( "}\r\n", sos );
	}

    docRtfWriteArgTag( "\\paperw", dg->dgPaperWideTwips, sos );
    docRtfWriteArgTag( "\\paperh", dg->dgPaperHighTwips, sos );

    docRtfWriteArgTag( "\\margl", dg->dgLeftMarginTwips, sos );
    docRtfWriteArgTag( "\\margr", dg->dgRightMarginTwips, sos );
    docRtfWriteArgTag( "\\margt", dg->dgTopMarginTwips, sos );
    docRtfWriteArgTag( "\\margb", dg->dgBottomMarginTwips, sos );

    if  ( dp->dpTabIntervalTwips > 0 && dp->dpTabIntervalTwips != 720 )
	{ docRtfWriteArgTag( "\\deftab", dp->dpTabIntervalTwips, sos );	}

    return 0;
    }

static void docRtfInitWritingContext(	RtfWritingContext *	rwc )
    {
    int		i;

    docInitTextAttribute( &(rwc->rwcTextAttribute) );
    docInitTextAttribute( &(rwc->rwcOutsideLinkAttribute) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitRowProperties( &(rwc->rwcRowProperties) );

    rwc->rwcInFldrslt= 0;
    rwc->rwcSaveBookmarks= 1;

    rwc->rwcSaveAsLink= 0;
    rwc->rwcSaveAsLinkFileName= (const char *)0;
    rwc->rwcSaveAsLinkFileSize= 0;
    rwc->rwcSaveAsLinkMarkName= (const char *)0;
    rwc->rwcSaveAsLinkMarkSize= 0;

    for ( i= 0; i < 256; i++ )
	{ rwc->rwcOutputMapping[i]= i;	}
    }

static void docRtfCleanWritingContext(	RtfWritingContext *	rwc )
    {
    docCleanParagraphProperties( &rwc->rwcParagraphProperties );
    docCleanRowProperties( &rwc->rwcRowProperties );
    }

int docRtfSaveDocument(		SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const BufferSelection *	bs,
				int			saveBookmarks )
    {
    const BufferItem *		bi= &bd->bdItem;

    RtfWritingContext		rwc;

    docRtfInitWritingContext( &rwc );
    rwc.rwcSaveBookmarks= saveBookmarks;

    sioOutPutString( "{\\rtf0\\ansi\r\n", sos );

    if  ( docRtfSaveDocumentProperties( sos, &rwc, bd ) )
	{ LDEB(1); return -1;	}

    sioOutPutString( "\r\n", sos );

    if  ( docRtfSaveItem( sos, bi, bs, &rwc ) )
	{ LDEB(bi->biLevel); return -1; }

    sioOutPutString( "}\r\n", sos );

    docRtfCleanWritingContext( &rwc );

    return 0;
    }

int docRtfSaveSelectionAsLink(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const BufferSelection *	bs,
				const char *		fileName,
				int			fileSize,
				const char *		markName,
				int			markSize )
    {
    const BufferItem *		bi= &bd->bdItem;

    RtfWritingContext		rwc;

    docRtfInitWritingContext( &rwc );
    rwc.rwcSaveBookmarks= 0;
    rwc.rwcSaveAsLink= 1;
    rwc.rwcSaveAsLinkFileName= fileName;
    rwc.rwcSaveAsLinkFileSize= fileSize;
    rwc.rwcSaveAsLinkMarkName= markName;
    rwc.rwcSaveAsLinkMarkSize= markSize;

    sioOutPutString( "{\\rtf0\\ansi\r\n", sos );

    if  ( docRtfSaveDocumentProperties( sos, &rwc, bd ) )
	{ LDEB(1); return -1;	}

    sioOutPutString( "\r\n", sos );

    if  ( docRtfSaveItem( sos, bi, bs, &rwc ) )
	{ LDEB(bi->biLevel); return -1; }

    sioOutPutString( "}\r\n", sos );

    docRtfCleanWritingContext( &rwc );

    return 0;
    }
