
// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcrbaye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ConcurrentStatement.cc,v 1.4 1999/07/23 21:08:00 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include <iostream.h>
#include <strstream.h>
#include "IIRScram_ConcurrentStatement.hh"
#include "IIR_Declaration.hh"
#include "IIR_ConcurrentStatement.hh"
#include "IIR_Label.hh"
#include "IIR_SignalDeclaration.hh"
#include "symbol_table.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "StandardPackage.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_ComponentInstantiationStatement.hh"

IIRScram_ConcurrentStatement::IIRScram_ConcurrentStatement() { 
  enclosingRegion = NULL;
  enclosing_path = NULL;
}

IIRScram_ConcurrentStatement::~IIRScram_ConcurrentStatement() {
  if(enclosing_path != NULL) {
    delete [] enclosing_path;
  }
}

void 
IIRScram_ConcurrentStatement::_publish_cc_state() {
  _report_undefined_scram_fn("_publish_cc_state()");
}

void 
IIRScram_ConcurrentStatement::_publish_createNetInfo() {
  _report_undefined_scram_fn("_publish_createNetInfo()");
}

void
IIRScram_ConcurrentStatement::_publish_cc_type_info(){
   _report_undefined_scram_fn("_publish_cc_type_info");
}

void
IIRScram_ConcurrentStatement::_publish_cc_extern_type_info(){
   _report_undefined_scram_fn("_publish_cc_extern_type_info");
}

#ifdef PROCESS_GRAPH
void
IIRScram_ConcurrentStatement::_publish_cc_driver_info() {
  _report_undefined_scram_fn("_publish_cc_driver_info");
}
#endif

void 
IIRScram_ConcurrentStatement::_publish_cc_headerfiles_for_cc_generate_statement(){
  char *temp = _current_architecture_name;
  char *old_current_name = _current_publish_name;
  IIR_ComponentInstantiationStatement *compInst = NULL;
  
  IIR_ConcurrentStatement *conc_stmt;
  IIR *stmt = _get_statement_list()->first();

  while( stmt != NULL ) {
    ASSERT( stmt->_is_concurrent_statement() == TRUE );
    conc_stmt = (IIR_ConcurrentStatement *)stmt;

    switch( conc_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
      conc_stmt->_publish_cc_headerfiles_for_cc_default();
      break;

    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
    case IIR_BLOCK_STATEMENT:
      _current_publish_name = "SG";
      conc_stmt->_publish_cc_headerfiles_for_cc_default();
      _current_architecture_name = temp;
      _current_publish_name  = old_current_name;
      
      break;

    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      _cc_out << "#include \"";
      _cc_out << "SCOM";
      ASSERT( conc_stmt->get_instantiated_unit() != NULL );
      ASSERT( conc_stmt->get_instantiated_unit()->_get_declarator() != NULL );
      conc_stmt->get_instantiated_unit()->_get_declarator()->_publish_cc_elaborate();
      _cc_out << "_elab.hh\"\n";

      compInst = (IIR_ComponentInstantiationStatement *) conc_stmt;
      ASSERT ( compInst->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT );
      if (compInst->_get_configuration() != NULL) {
	compInst->_get_configuration()->_publish_cc_headers();
      }
      
      break;
      
    default:
      cerr << "ERROR! IIRScram_ConcurrentGenerateForStatement::"
	   << "_publish_cc_headerfiles_for_cc(): unknown conc_statement "
	   << "type |" << conc_stmt->get_kind_text() << "|" << endl;
    }

    stmt = _get_statement_list()->successor( stmt );
  }
}


void
IIRScram_ConcurrentStatement::_publish_cc_scoping_prefix(IIR *finalScope, IIR *currentScope, ostream& outStream){
  IIR *outerScope = NULL;
  IIR_Declaration* tempDeclaration = NULL;
  IIR_ConcurrentStatement* tempConcStmt = NULL;

  if(currentScope == finalScope){
    return;
  }

  if((finalScope->_is_iir_package_declaration() == TRUE) ||
     (finalScope->get_kind() == IIR_PACKAGE_BODY_DECLARATION)) {
    return;
  }
  
  if ((currentScope->get_kind() == IIR_BLOCK_STATEMENT) || (currentScope->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) || (currentScope->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT)) {
    if (finalScope != currentScope) {
      

      if (currentScope->_is_concurrent_statement() == FALSE ) {
	ASSERT ( currentScope->_is_iir_declaration() == TRUE );
	tempDeclaration = (IIR_Declaration *) this;
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	
	outerScope = tempDeclaration->_get_declarative_region();
      }
      else {
	ASSERT ( currentScope->_get_enclosing_scope() != NULL );
	outerScope = currentScope->_get_enclosing_scope();
      }
      
      ASSERT ( outerScope != NULL );
      
      if (outerScope->_is_iir_declaration() == TRUE ) {
	ASSERT ( outerScope->get_kind() == IIR_ARCHITECTURE_DECLARATION );
	outStream << "enclosingScope->";
      }
      else {
	ASSERT ( outerScope->_is_concurrent_statement() == TRUE );
	tempConcStmt = (IIR_ConcurrentStatement *) outerScope;
	ASSERT ( outerScope->_is_concurrent_statement() == TRUE );
	
	outStream << "enclosingScope->";
	
	tempConcStmt->_publish_cc_scoping_prefix(finalScope, this->_get_enclosing_scope(), outStream);
      }
    }
  }
  else {
    if (currentScope->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      ASSERT (currentScope->_get_enclosing_scope() != NULL);
      currentScope->_get_enclosing_scope()->_publish_cc_scoping_prefix(finalScope, this->_get_enclosing_scope(), outStream);
    }
  }
}

IIR * 
IIRScram_ConcurrentStatement::_transmute() {
  return this;
}

void
IIRScram_ConcurrentStatement::_publish_cc() {
  switch(this->get_kind()) {
  case IIR_COMPONENT_INSTANTIATION_STATEMENT:
    break;
  case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    break;
  default:
    cerr <<  "_publish_cc() not defined for " << this->get_kind_text() << ends;
    break;
  }
}

void
IIRScram_ConcurrentStatement::_publish_cc_binding_name(ostream& outstream) {
  switch(this->get_kind()) {
  case IIR_BLOCK_STATEMENT:
    outstream << "SB";
    break;
  case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
  case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
    outstream << "SG";
    break;
  default:
    cerr << "_publish_cc_binding_name(ostream& outstream) called for node:" 
	 << get_kind_text() << endl;
    break;
  };
  outstream << _current_entity_name << "_"
	    << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path(outstream);
  outstream << "_" << *_get_label();
  // << ends;
}

void 
IIRScram_ConcurrentStatement::_publish_cc_enclosing_stmt_to_architecture_path(ostream& outstream) {
  IIR_Char* ptr = NULL;
  ptr = _get_enclosing_stmt_to_architecture_path();
  if(ptr != NULL) {
    outstream << "_" << ptr;
  }
}

// For example We have the following code:
// architecture a of ent e is
// begin
// Ablk: block
// begin
// Bblk: block
// begin
// Cblk: block
// begin
// end block;
// end block;
// end block;
// end a
// _get_enclosing_stmt_to_architecture_path() of Ablk gives NULL;
// _get_enclosing_stmt_to_architecture_path() of Bblk gives Ablk;
// _get_enclosing_stmt_to_architecture_path() of Cblk gives Ablk_Bblk;
//  IMPORTANT The return value has to be deleted by the caller
IIR_Char*
IIRScram_ConcurrentStatement::_get_enclosing_stmt_to_architecture_path() {
  IIR* enclosing_scope_ptr = _get_enclosing_scope();
  ASSERT(enclosing_scope_ptr != NULL);
  if((enclosing_scope_ptr->get_kind() == IIR_ARCHITECTURE_DECLARATION) || (enclosing_scope_ptr->get_kind() == IIR_ENTITY_DECLARATION)) {
    return NULL;
  }
  if(enclosing_path == NULL) {
    ASSERT(enclosing_scope_ptr->_is_concurrent_statement() == TRUE);
    IIR_ConcurrentStatement* conc_stmt = (IIR_ConcurrentStatement*)enclosing_scope_ptr;
    ostrstream tmp_stream;
    if(conc_stmt->enclosing_path != NULL) {
      tmp_stream << conc_stmt->enclosing_path
		 << "_";
    }
    tmp_stream << *(conc_stmt->_get_label()) << ends;
    enclosing_path = tmp_stream.str();
    return enclosing_path;
  }
  else {
    return enclosing_path;
  }
}

void
IIRScram_ConcurrentStatement::_set_enclosing_scope(IIR* outerRegion)
{
  enclosingRegion = outerRegion;
}

IIR*
IIRScram_ConcurrentStatement::_get_enclosing_scope()
{
  return enclosingRegion;
}

void
IIRScram_ConcurrentStatement::_publish_cc_concurrent_stmt_init(IIR_DeclarationList* decl_list) {
  ASSERT((this->_is_concurrent_generate_statement() == TRUE) ||
	 (this->_is_block_statement() == TRUE));
  IIR_AssociationList* generic_map = this->_get_generic_map_aspect();
  //_cc_out << *(this->_get_label());
  this->_get_label()->_publish_cc_elaborate();
  _cc_out << "_elab_obj = new ";
  this->_publish_cc_binding_name();

  _cc_out << "_elab(this";
  if(generic_map != NULL) {
    if (generic_map->num_elements() > 0) {
      _cc_out << ", \n";
      _current_publish_node = this;
      generic_map->_publish_cc_generic_map_aspect_for_conc_stmts();
      _current_publish_node = NULL;
    }
  }
  _cc_out << ");\n";
}


void
IIRScram_ConcurrentStatement::_publish_cc_instantiate_call() {
  if(this->get_kind() == IIR_PROCESS_STATEMENT) {
    ostrstream objectname;
    if (this->_get_label() != NULL) {
      objectname << *(this->_get_label()) << ends;
    }
    else {
      objectname << "ANON_PROCESS" << this << ends;
    }
    char *objname = objectname.str();
    _cc_out << objname << "_elab_obj = new ";
    _cc_out << objname << "(this);\n";
    _cc_out << "proc_array[";
    _cc_out << objname << "_elab_obj->id";
    _cc_out << "] = (OBJTYPE *) " << objname << "_elab_obj;\n";
    _number_of_processes++;
    delete [] objname;
  }
  else {
    this->_get_label()->_publish_cc_elaborate();
    _cc_out << "_elab_obj->instantiate();\n";
  }
}

void 
IIRScram_ConcurrentStatement::_set_guard_signal( IIR_SignalDeclaration * ){
  _report_undefined_scram_fn("_set_guard_signal( IIR_Declaration * )");
}

void 
IIRScram_ConcurrentStatement::_resolve_guard_signal( symbol_table *sym_tab ){
  // Lookup all signals named "guard"
  IIR_SignalDeclaration *guard_signal = NULL;
  set<IIR_Declaration> *guard_decls;
  guard_decls = new set<IIR_Declaration>(*sym_tab->find_set( "guard" ));
  if( guard_decls == NULL ){
    ostrstream err;
    err << "No signal |guard|, implicit or explicit, has been declared." << ends;
    report_error( this, err );
    goto finish;
  }
  
  guard_decls->reduce_set( &IIR::_is_signal );
  ASSERT( guard_decls->num_elements() == 1 || guard_decls->num_elements() == 0 );

  if( guard_decls->num_elements() == 1 ){
    // Get rid of any that aren't boolean typed.
    guard_signal = (IIR_SignalDeclaration *)guard_decls->get_element();
    if( guard_signal->get_subtype() != StandardPackage::boolean_type ){
      guard_signal = NULL;
    }
  }

  if( guard_signal == NULL ){
    ostrstream err;
    err << "Signal |guard| must be a signal of builtin type |boolean|." << ends;
    report_error( this, err );    
  }
  else{
    _set_guard_signal( guard_signal );
  }
  

 finish:
  delete guard_decls;

}
