#ifndef LIBRARY_MANAGER_HH
#define LIBRARY_MANAGER_HH

// Copyright (c) 1996-1999 The University of Cincinnati.
// All rights reserved. 

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Author: Radharamanan Radhakrishnan          ramanan@ece.uc.edu

//---------------------------------------------------------------------------
//
// $Id: 
//
//---------------------------------------------------------------------------
#include "switch_file.hh"
#include "savant.hh"
#include "IRBasicDataTypes.hh"

class ostream;
class symbol_table;
class symbol_lookup;
class IIR;
class IIR_Declaration;
class IIR_LibraryUnit;
class IIR_ArchitectureDeclaration;
class IIR_TextLiteral;
class IIR_Name;
class IIR_IndexedName;
class IIR_SimpleName;
class IIR_LibraryDeclaration;
class IIRScram_LibraryDeclaration;
class IIR_PackageDeclaration;
class IIR_EntityDeclaration;
class IIR_ConfigurationDeclaration;
class scram;

template <class type> class set;

class library_manager {
  friend class IIRScram_LibraryDeclaration;
  friend class scram;

  // This class implements a library manager whose sole purpose is to
  // create design libraries as specified by the user.

public:
  
  library_manager( char *logical_work_directory_name );
  ~library_manager();
  
  // creates the work directory into which analyzed design units are
  // stored. If user doesn't specify a work directory name through the -lib
  // option the actual name "work" will be used
  int create_work_dir();

  // create the design library so that _publish_vhdl and _publish_cc
  // can dump their output into this directory.
  void create_design_library(char *libName);

  // change directory to libName
  void change_dir(char *libName);

  // As changeDir changes the current directory, this method steps back one
  // directory level.
  int step_back();
  
  // this methods gets a library unit and depending on the type does the
  // appropriate name mangling and returns a handle to a ostream.
  ostream &set_output_file( IIR_LibraryUnit * );
  
  // calls the switch file's flush
  void flush();

  void reset();

  // calls publish vhdl so that analyzed design units can be stored in the
  // work directory
  void parse_into_work_dir( IIR_LibraryUnit * );
  
  // retrieves the work directory's actual name
  char *get_work_dir_name();

  // Get the library that the original parser declared the work library.
  IIR_LibraryDeclaration *get_work_library(){
    return my_work_library;
  }
  
  IIR_PackageDeclaration *get_standard_package(){
    return my_standard_package;
  }


  // the output stream which the library manager uses to create files.
  switch_file _lib_out;
  
  // The following method finds the library referenced, and returns a
  // pointer to it.  (If a library is referenced that the library manager
  // can't find the directory for, it will return NULL, and the caller
  // should complain.)  Note that the library manager is as lazy as
  // possible about finding things - library declaration will initially be
  // empty, and as things within it are accessed, it will grow.  For
  // instance, work._savant_lib might have thousands of things in it.  Even
  // if it does, the first time this library is accessed via
  // "find_library", the declaration will be empty.  If at some point
  // "work.foo.bar" is accessed, the work library declaration will now
  // contain foo which will contain bar.  And, if I then access
  // "work.foo.baz", the there will be one reference to foo, and within
  // foo, both "bar" and "baz".  If I access "work.foo.all", then everthing
  // within "work.foo" will appear in the library.
  IIR_LibraryDeclaration *find_library( IIR_TextLiteral *library_name );
  IIR_LibraryDeclaration *find_library( char *library_name );

  // Primary and secondary units and so forth are magical in that sometimes
  // it's legal to reference them even when they haven't been declared
  // within this particular file.  In these cases, the library manager has
  // to be involved.  If TRUE is passed into these functions, then error
  // messages will be generated for symbols that don't exist, or are not
  // found of the correct type.


  // This method finds the last declared architecture of the name it's
  // called on for the entity passed in.  If complain on error is TRUE.  it
  // will generate an error message when it fails.
  IIR_ArchitectureDeclaration *_lookup_architecture( IIR_Boolean complain_on_error,
						     IIR_EntityDeclaration *,
						     IIR_Name *architecture_name );

  // The name has to be an IIR_Name, or an IIR_TextLiteral, or an assertion
  // will fail.
  IIR_PackageDeclaration *_lookup_package( IIR_Boolean,
					   IIR *package_name,
					   IIR_LibraryDeclaration * = NULL );

  IIR_EntityDeclaration *_lookup_entity( IIR_Boolean complain_on_error,
					 IIR_Name *entity_name,
					 IIR_LibraryDeclaration * = NULL );

  IIR_ConfigurationDeclaration *_lookup_configuration( IIR_Boolean complain_on_error,
						       IIR_Name *configuration_name,
						       IIR_LibraryDeclaration * = NULL );

  void add_declaration( IIR_LibraryUnit * );


  // This method searches for the savant root directory.  It first will try
  // the SAVANTROOT environment variable, and then it will try a list of
  // standard directories that it knows about.  It will return NULL if it
  // can't find a valid match, and the directory name if it can.  Don't
  // delete the string that this returns!
  static char *get_savant_root_directory();

private:
  // Sets the work directory's actual name.  If something like
  // "foo" is passed in, then "foo._savant_lib", or 
  // foo << library_suffix is stored.
  void set_work_dir_name( char *name );

  symbol_lookup *my_symbol_table;

  // Found libraries.
  set<IIR_LibraryDeclaration> *libraries;
  
  static const char *library_suffix;
  static const char *entity_suffix;
  static const char *configuration_suffix;
  static const char *package_suffix;
  static const char *package_body_suffix;
  static const char *secondary_units_suffix;

  // work directory's actual name (logical name is "work").
  IIR_LibraryDeclaration *my_work_library;
  IIR_PackageDeclaration *my_standard_package;
  
  // design library's name
  char *fileName;
  
  void create_directories( IIR_LibraryUnit *lib_unit );

  // All of these methods allocate the string that they return - the memory
  // needs to be deleted or it will leak!
  char *build_arch_file_name( IIR_TextLiteral *entity_name,
			      IIR_TextLiteral *architecture_name,
			      char *library_directory  );

  char *build_package_body_file_name( IIR_TextLiteral *package_body_name,
				      char *library_directory );


  char *build_entity_file_name( IIR_TextLiteral *entity_name,
				char *library_directory );

  char *build_configuration_file_name( IIR_TextLiteral *configuration_name,
				       char *library_directory );

  char *build_package_file_name( IIR_TextLiteral *package_name,
				 char *library_directory );

  char *build_library_directory_name( IIR_TextLiteral *library_name );
  char *build_directory_name_for_secondary_unit( IIR_LibraryUnit * );

  char *find_library_directory( IIR_TextLiteral *logical_library_name );

  
  // This method will look in the hash table first, and then go find a library
  // unit.  (The library declaration can be NULL)
  IIR_LibraryUnit *_lookup_unit( IIR_Boolean complain_on_error,
				 IIR *unit_name,
				 IIR_LibraryDeclaration *,
				 IIR_Kind unit_kind );

  // goes and retrieves a particular primary unit by path.
  IIR_LibraryUnit *get_primary_unit( IIR_LibraryDeclaration *, char *path_to_file );

  // goes and retrieves a particular primary unit given a secondary unit
  IIR_LibraryUnit *get_primary_unit( IIR_LibraryDeclaration *, 
				     IIR_TextLiteral *unit_name, 
				     IIR_Kind unitType,
				     IIR_Boolean complain_on_error = TRUE );

  // Finds unit_name from the specified library.  Only looks for the kind
  // passed in.
  IIR_LibraryUnit *get_primary_unit( IIR_LibraryDeclaration *,
				     IIR_Name *unit_name,
				     IIR_Kind unitType,
				     IIR_Boolean complain_on_error = TRUE );

  // Finds unit_name from the specified library.  If there are multiple
  // possibilies found, it will complain no matter what the "complain_on_error"
  // flag is set to.
  IIR_LibraryUnit *get_primary_unit( IIR_LibraryDeclaration *,
				     IIR_Name *unit_name,
				     IIR_Boolean complain_on_error = TRUE );

  // goes out and retrieves secondary units
  IIR_ArchitectureDeclaration *parse_architecture( IIR_LibraryDeclaration *,
						   IIR_TextLiteral *entity_name, 
						   IIR_TextLiteral *architecture_name );
  
  IIR_ArchitectureDeclaration *parse_architecture( IIR_EntityDeclaration *entity_decl, 
						   IIR_Name *architecture_name );

  // These methods operate either on a simple name, or a selected name.
  // What's passed in will normally be either <unit_name>, or
  // <library_name>.<unit_name>.  In the first case, a library name will
  // never be returned.  If any other type of name is passed in, the result
  // will always be null.
  IIR_TextLiteral *get_library_name( IIR_Name * );
  IIR_TextLiteral *get_unit_name( IIR_Name * );
  
  // This method takes a string with the name of a directory as input and
  // checks to see if this directory is a valid savantroot directory.  It
  // returns true if it is, and false if it's not.
  static bool check_root_dir( char *, bool complain );

  // This method checks for SAVANTROOT to see if it's defined valid.  If
  // you pass it a "true" and the variable isn't defined, or it's defined
  // but invalid, it will report very verbosely what the problem is.
  static char *check_environment_variable( bool );

  // This method looks in the list of standard directories for the savant
  // root dir.  It returns the directory if it finds it, and NULL if it
  // doesn't.
  static char *check_default_directories( bool complain );

  // This is where we store the path to SAVANTROOT.
  static char *savant_root_dir;

  // This method returns a null terminated array of strings containing the
  // directories that we need to search when looking for libraries.
  // Currently directories get listed in here via VHDL_LIBRARY_PATH,
  // SAVANTROOT, and the hardcoded locations in the
  // file_manager::get_library_directories().
  static const char **get_search_dirs();

  // The following methods help build the array of directories.

  // This method parse the environment varible VHDL_LIBRARY_PATH, which
  // should be a colon separated list of directory names.  If a NULL
  // argument is passed in, only the count of entries is returned.  If a
  // non-NULL argument is passed in, the entries from the environment
  // variable are written into the array passed in.
  static unsigned int parse_vhdl_library_path( char ** );

  // Returns the number of entries in the file manager's list.
  static unsigned int get_num_file_manager_directories();
  
  // Returns the next available index for the array of directories.
  static unsigned int get_next_index( const char ** );

};
#endif
