/*
 * Dynamic Encapsulation Gateway Server for Linux.
 * Protocol information sourced from the MFNOS implementation
 * by Barry Siegfried, K2MF
 *
 * Copyright (c) 1999, Terry Dawson, VK2KTJ.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <net/route.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <signal.h>
#include <syslog.h>

#include "dgserver.h"
#include "daemon.h"

static char *
banner="Dynamic Gateway Server";

static char *
copyright="Copyright (c) 1999, Terry Dawson, VK2KTJ";

static char *
usage="\nUsage:\n\tdgserver [-c conffile] [-i device] [-p port] [-t exptimer] [-d]\n\n";

char *conffile=CONFFILE,
	*client=NULL,
	*key;

char *msg;

int msglen=0,
	daemonise=1,
	logging=1,
	verbose=0,
	port=IPPORT_REMOTE,
	exptimer=DEFTIMER;

int opt;

struct sockaddr_in servsa, clientsa;
int clientlen;
struct hostent *hp;
int sockfd;

char request[MAXRQLEN+1];
int rqlen;

char routerq[MAXNETLEN+1];
char *network, *netmask, *gateway;

char *device=DEFDEV;

struct route {
	struct in_addr network;
	struct in_addr netmask;
	struct in_addr gateway;
	char key[MAXKEYLEN];
	int timer;
	struct route *next;
};
struct route *hostedrts=NULL;

int main(int argc, char **argv)
{
	/* for select() */
	int selret;
	fd_set readfds, writefds;
	struct timeval seltimeout;

	struct in_addr networka;
	struct in_addr netmaska;

	/*
	 * Print welcome banner and copyright
	 */
	 fprintf(stderr,"\n%s %s\n%s\n\n",banner,VERSION,copyright);

	/*
	 * Process command line arguments
	 *
	 * command line must contain at least two arguments.
	 */
	if (argc<2) {
		/* FIXME */
		exit(-666);
	}

	while ((opt=getopt(argc, argv, "c:i:p:t:dhlv")) != EOF) {
		switch(opt) {

			case 'c':		/* Configuration file */
				conffile=optarg;
				break;

			case 'i':		/* Encap Interface */
				device=optarg;
				break;

			case 'p':		/* UDP Port */
				port=atoi(optarg);
				break;

			case 't':		/* Expiry timer */
				exptimer=atoi(optarg);
				break;

			case 'd':		/* Daemonise? */
				daemonise=(daemonise ? 0 : 1);
				break;

			case 'l':		/* Log requests/timeouts? */
				logging=(logging ? 0 : 1);
				break;

			case 'v':		/* verbose logging? */
				verbose=(verbose ? 0 : 1);
				break;

			case 'h':
			default:
				fputs(usage,stderr);
				exit(-1);
		}
	}


	/*
	 * Read configuration file
	 */
	if (conffile == NULL) {
		if (parseconf(CONFFILE) < 0)
			exit(-2);
	}
	else {
		if (parseconf(conffile) < 0)
			exit(-2);
	}

	/*
	 * Open the system log
	 */
	openlog("dgserver",LOG_CONS|LOG_PERROR,LOG_DAEMON);

	/*
	 * Open UDP socket
	 */

	if ((sockfd=socket(AF_INET,SOCK_DGRAM,IPPROTO_IP)) < 0) {
		perror("Opening Socket failed");
		exit(-2);
	}

	/*
	 * We'll listen on all of our local network devices
	 */
	bzero((char *) &servsa, sizeof(servsa));
	servsa.sin_family=AF_INET;
	servsa.sin_addr.s_addr=htonl(INADDR_ANY);
	servsa.sin_port=htons(port);

	if (bind(sockfd, (struct sockaddr *) &servsa, sizeof(servsa)) < 0 ) {
		perror("dgserver: attempt to bind to all network devices failed");
		exit(-3);
	}

	/*
	 * Daemonise if necessary
	 */

	if (daemonise) {
		if (! daemon_start(TRUE)) {
			perror("dgserver: attempt to daemonise failed");
			exit(-4);
		}
	}

	/* We'll need this in the recvfrom() */
	clientlen=sizeof(clientsa);

	/* add the socket to the file descriptors our select will listen to */
	FD_ZERO(&writefds);
	FD_ZERO(&readfds);
	FD_SET(sockfd, &readfds);

	/* Set our expiry timeout interval */
	seltimeout.tv_sec = CLOCK_TICK;
	seltimeout.tv_usec = 0;

	/* Loop listening for commands, and calling the route expiry timer */
	while ((selret=select(sockfd+1,&readfds,&writefds,NULL,&seltimeout))!=-1) {

		FD_ZERO(&readfds);
		FD_SET(sockfd, &readfds);

		if (selret==0) {	/* timeout */
			timeoutrt();
			seltimeout.tv_sec = CLOCK_TICK;
		}

		else {
			msglen=recvfrom(sockfd,request,MAXRQLEN,0,&clientsa,&clientlen);

			if (msglen < 0)
				perror("dgserver: recvfrom failed");

			else {

				/* Null terminate the buffer */
				request[msglen]=(char)NULL;

				/* Extract the network route from the request */
				/* FIXME add some validation here */
				strncpy(routerq,request+2,request[1]);
				routerq[(int)request[1]]=(char)NULL;
				network=strtok(routerq,"/");
				netmask=strtok(NULL,"/");
				key=request+request[1]+2;

				switch(request[0]) {
					case ROUTE_ME:
						fprintf(stderr,"main: Add %s with netmask %s and key '%s' from %s\n",
							network,netmask,key,inet_ntoa(clientsa.sin_addr));
						inet_aton(network,&networka);
						cidrtoin(netmask,&netmaska);
						refreshrt(&networka,&netmaska,&clientsa.sin_addr,key);
						break;

					case UNROUTE_ME:
						fprintf(stderr,"main: Remove %s with netmask %s with key '%s' from %s\n",
							network,netmask,key,inet_ntoa(clientsa.sin_addr));
						inet_aton(network,&networka);
						cidrtoin(netmask,&netmaska);
						break;

					default:
						fprintf(stderr,"main: Bad command!");
				}
			}
		}
	}
	exit(0);
}

/*
 * open configuration file, read each line, and add any network routes
 * found.
 *
 * returns 0 for normal operation, and -1 on error.
 */
int
parseconf(char *conffile)
{
	FILE *cf;
	char buff[BUFLEN], *network, *netmask, *key;
	int len;

	struct in_addr networka;
	struct in_addr netmaska;

	network = netmask = key = NULL;

	if ((cf=fopen(conffile,"r")) == NULL) {
		perror("dgserver: unable to open configuration file");
		return(-1);
	}

	while (fgets(buff,BUFLEN,cf) != NULL) {

		/* get the length of the buffer */
		len=strlen(buff);

		/* ignore comment lines */
		if (buff[0]=='#')
			continue;

		/* and empty lines */
		if (len==1)
			continue;

		/* remove any trailing newline */
		if (buff[len-1]=='\n')
			buff[len-1]=(char)NULL;

		/* parse the buffer for our route components */
		network=strtok(buff,"/ \t");
		netmask=strtok(NULL,"/ \t");
		key=strtok(NULL,"/ \t");

		if (logging) {
			if (verbose) {
				fprintf(stderr,"parseconf: found: %s %s %s\n",
					network,netmask,key);
			}
		}
		/* Add the route to the list */
		if (inet_aton(network,&networka)==0) {
			fprintf(stderr,"parseconf: invalid network address: %s",
				network);
		};
		if (cidrtoin(netmask,&netmaska)==0) {
			fprintf(stderr,"parseconf: invalid network mask: %s",
				netmask);
		};
		addhostrt(&networka, &netmaska, key);
	}
	fclose(cf);
	return(0);
}

/* Add a new route to our hosted routes list */
void
addhostrt(struct in_addr *network, struct in_addr *netmask, char *key)
{
	struct route *new;

	/* build new node */
	new=malloc(sizeof(struct route));
	memcpy(&new->network,network,sizeof(network));
	memcpy(&new->netmask,netmask,sizeof(netmask));
	bzero(&new->gateway, sizeof(new->gateway));
	new->timer=0;	/* Routes are added dead, waiting a request */
	strncpy(new->key,key,MAXKEYLEN);

	/* add it to the start of the list */
	new->next=(struct route *)hostedrts;
	hostedrts=new;
}

/* Flush all of the routes from our hosted routes list */
void
flushhostrts(void)
{
	struct route *this, *next;

	this=hostedrts;

	while (this!=(struct route *)NULL) {
		next=this->next;
		free(this->key);
		free(this);
		this=next;
	}
	hostedrts=(struct route *)NULL;
}

/*
 * Convert CIDR style netmasks (/nn) into in_addr form.
 *
 * Returns: non zero on success, 0 on error.
 */
int
cidrtoin(char *netmask, struct in_addr *netmaska)
{
	int bits;
	unsigned int mask,maskl;

	bits=atoi(netmask);
	if (bits>32 || bits<0)
		return(0);

	mask=0;
	while (bits--) {
		mask += (1<<(31-bits));
	maskl=htonl(mask);	
	memcpy(netmaska,&maskl,sizeof(maskl));
	}

	maskl=htonl(mask);	
	memcpy(netmaska,&maskl,sizeof(maskl));

	if (logging) {
		if (verbose) {
			fprintf(stderr,"cidrtoin: converted: %s to %s\n",
				netmask,inet_ntoa(*netmaska));
		}
	}

	return(1);
}


/*
 * Scan the hosted routes list, find the matching route, if it exists and
 * the key matches, then update the timer, if it is unused, or expired then
 * create the kernel route and log the request. If it doesn't exist, then
 * do nothing but log the request.
 *
 * Returns:
 *     0 : Ok route added/updated.
 *    -1 : no match found
 */
int
refreshrt(struct in_addr *network, struct in_addr *netmask,
        struct in_addr *gateway, char *key)
{
	struct route *this;
	struct rtentry newrt;
	struct sockaddr_in* sap;

	this=hostedrts;

	/* find match */
	while (this != (struct route *)NULL) {
		if ((memcmp(&this->network, network, sizeof(network))==0) &&
			(memcmp(&this->netmask, netmask, sizeof(netmask))==0))
			break;	/* match! */
		this=this->next;
	}

	if (this == (struct route *)NULL)	/* no match found */
	{
		if (logging) {
			syslog(LOG_NOTICE,"refreshrt: NOT found: %s %s with key '%s'\n",
				inet_ntoa(*network),inet_ntoa(*netmask),key);
		}
		return(-1);
	}

	/* If the route timer has expired, rebuild the kernel route */
	/*    We assume it has been reaped already                  */
	if (this->timer == 0) {
		/* Add new route to routing table */
		memset((char *) &newrt, 0, sizeof(struct rtentry));

		sap=(struct sockaddr_in *)&newrt.rt_dst;
		sap->sin_addr.s_addr=network->s_addr;
		newrt.rt_dst.sa_family=AF_INET;

		sap=(struct sockaddr_in *)&newrt.rt_genmask;
		sap->sin_addr.s_addr=netmask->s_addr;
		newrt.rt_genmask.sa_family=AF_INET;

		sap=(struct sockaddr_in *)&newrt.rt_gateway;
		sap->sin_addr.s_addr=gateway->s_addr;
		newrt.rt_gateway.sa_family=AF_INET;
		this->gateway.s_addr=gateway->s_addr;

		newrt.rt_dev=device;

		newrt.rt_flags=(RTF_UP|RTF_GATEWAY);

		if (ioctl(sockfd, SIOCADDRT, &newrt)<0) {
			perror("SIOCADDRT");
		};

		/* Fill in gateway and timer fields */
		memcpy(&this->gateway,gateway,sizeof(this->gateway));
		this->timer=exptimer;

		if (logging) {
			syslog(LOG_NOTICE,"refreshrt: added route: %s/%s via %s on %s",
				inet_ntoa(*network),inet_ntoa(*netmask),inet_ntoa(*gateway), device);
		}
	}
	else {
		/* update the timer */
		this->timer=exptimer;
	}
	return(0);
}

/*
 * timeoutrt - scan list, decrement each timer, if timer reaches zero then
 * delete route from routing table.
 */
void
timeoutrt(void)
{
	struct route *this;
	struct rtentry oldrt;
	struct sockaddr_in* sap;

	this=hostedrts;

	/* find match */
	while (this != (struct route *)NULL) {
		if (this->timer) {
			this->timer--;	/* decrement it */

			if (!this->timer) {
				/* the route has expired */
				/* Delete route from routing table */
				memset((char *) &oldrt, 0, sizeof(struct rtentry));

				sap=(struct sockaddr_in *)&oldrt.rt_dst;
				sap->sin_addr.s_addr=this->network.s_addr;
				oldrt.rt_dst.sa_family=AF_INET;

				sap=(struct sockaddr_in *)&oldrt.rt_genmask;
				sap->sin_addr.s_addr=this->netmask.s_addr;
				oldrt.rt_genmask.sa_family=AF_INET;

				sap=(struct sockaddr_in *)&oldrt.rt_gateway;
				sap->sin_addr.s_addr=this->gateway.s_addr;
				oldrt.rt_gateway.sa_family=AF_INET;

				oldrt.rt_dev=device;

				oldrt.rt_flags=(RTF_UP|RTF_GATEWAY);

				if (ioctl(sockfd, SIOCDELRT, &oldrt)<0) {
					perror("SIOCDELRT");
				}
				if (logging) {
					syslog(LOG_NOTICE,"timeoutrt: removed route: %s/%s via %s on %s",
						inet_ntoa(this->network),inet_ntoa(this->netmask),
						inet_ntoa(this->gateway),device);
				}
			}
		}
		if (logging) {
			if (verbose) {
				fprintf(stderr,"timeoutrt: decremented route: %s",inet_ntoa(this->network));
				fprintf(stderr,"/%s via ",inet_ntoa(this->netmask));
				fprintf(stderr,"%s to %d\n",inet_ntoa(this->gateway),this->timer);
			}
		}
		this=this->next;
	}
}

