/* psk31-main.C
 * Soundcard-based implementation of the PSK31 HF keyboard-to-keyboard mode
 * Copyright (C) 1998,1999  Hansi Reiser, DL9RDZ
 * subject to GPL -- see LICENSE for details
 */
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <termios.h>

#include <ncurses.h>

#include "window.h"
#include "psk31.h"

#include "psk31-coder.h"
#include "psk31-transmitter.h"
#include "psk31-receiver.h"


// Naja, das tu ich jetzt einfach mal alles in globale Variablen rein...

static char *version_string="PSK31/Soundblaster for Linux Version V0.61";

// Audio parameters
int full_duplex=0;
#define DMA_BUF_BITS 6
static int audio_fd;
static char *sound_device="/dev/audio";

// 'Window' management
static window *rxwin, *txwin;
window *statwin;
static psk31_transmitter psk31tx;
static psk31_receiver psk31rx;

// Current settings...
static char callsign[40] = "NOCALL";     /* User's callsign */
static int txtrack = 0;         /* Tx frequency follows rx frequency flag */
static float rxfreq = 1150, txfreq = 1150; /* Rx & Tx tone frequencies (Hz) */
static int transmit=0, qpsk=0, lsb=0, dcd=0, passall=0, afc=1, phdelta=0;
static int io_file=0;

static char macro[10][40];

int noiselevel = 0; /* simulated white noise */

void out_status(int force);


/* Use this variable to remember original terminal attributes. */
struct termios saved_attributes;

     
void reset_input_mode (void)
{
	tcsetattr (STDIN_FILENO, TCSANOW, &saved_attributes);
}

void set_input_mode (void)
{
	struct termios tattr;
	
	/* Make sure stdin is a terminal. */
	if (!isatty (STDIN_FILENO)) {
		fprintf (stderr, "Not a terminal.\n");
		exit (EXIT_FAILURE);
	}
	/* Save the terminal attributes so we can restore them later. */
	tcgetattr (STDIN_FILENO, &saved_attributes);
	atexit (reset_input_mode);
	/* Set the funny terminal modes. */
	tcgetattr (STDIN_FILENO, &tattr);
	tattr.c_lflag &= ~(ICANON|ECHO); /* Clear ICANON and ECHO. */
	tattr.c_cc[VMIN] = 1;
	tattr.c_cc[VTIME] = 0;
	tcsetattr (STDIN_FILENO, TCSAFLUSH, &tattr);
}

void change_freq(float offset)
{
	rxfreq+=offset;
	psk31rx.set_freq(rxfreq);
	if( txtrack ) {
		txfreq=rxfreq;
		psk31tx.set_freq(txfreq);
	}
	out_status(0);
}
void set_tx_freq(float freq)
{
	txfreq=freq;
	psk31tx.set_freq(txfreq);
	out_status(0);
}
void set_rx_freq(float freq)
{
	rxfreq=freq;
	psk31rx.set_freq(rxfreq);
	if( txtrack ) set_tx_freq( freq );
	out_status(0);
}
void setmode(int q, int l, int a, int d)
{
	qpsk=q; lsb=l; dcd=d; afc=a;
	psk31rx.set_mode(qpsk,lsb); psk31tx.set_mode(qpsk,lsb);
	psk31rx.set_dcd(dcd); psk31rx.set_afc(afc);
	out_status(0);
}



static void readconfig(void)
{
	int cfgerr=0;
	FILE *fp=(FILE *)0;
	char cfgfile[1024], buf[256], *ptr;
	float f;

	ptr=getenv("HOME");
	if(ptr) {
		strncpy(cfgfile, ptr, 1000);
		strcat(cfgfile, "/.psk31");
		fp = fopen(cfgfile, "r");
	}
	if(!fp && ptr) {
		strncpy(cfgfile, ptr, 1000);
		strcat(cfgfile, "/psk31.ini");
		fp = fopen(cfgfile, "r");
	}
	if(!fp)
		fp = fopen("psk31.ini", "r");
	if(!fp) {
		strncpy(cfgfile, DATADIR, 1000);
		if(cfgfile[strlen(cfgfile)-1]!='/') strcat(cfgfile,"/");
		strcat(cfgfile, "psk31.ini");
		fp = fopen(cfgfile, "r");
	}
	if(!fp) {
		fprintf(stderr, "Configuration file psk31.ini or ~/.psk31 "
			        "not found");
		exit(1);
	}
	while(fgets(buf, 40, fp) != (char *)0) {
		if(sscanf(buf, "CALL=\"%[^\"]", callsign) == 1)
			/**/;
		else if(sscanf(buf, "FREQ=%f", &f) == 1)
			txfreq = rxfreq = f;
		else if(sscanf(buf, "LSB=%d", &lsb) == 1)
			/**/;
		else if(sscanf(buf, "PASSALL=%d", &passall) == 1)
			/**/;
		else if(sscanf(buf, "TXTRACK=%d", &txtrack) == 1)
			/**/;
		else if(sscanf(buf, "MACRO1=\"%[^\"]", macro[0]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO2=\"%[^\"]", macro[1]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO3=\"%[^\"]", macro[2]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO4=\"%[^\"]", macro[3]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO5=\"%[^\"]", macro[4]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO6=\"%[^\"]", macro[5]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO7=\"%[^\"]", macro[6]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO8=\"%[^\"]", macro[7]) == 1 )
			/**/;
		else if(sscanf(buf, "MACRO9=\"%[^\"]", macro[8]) == 1 )
			/**/;
		else {
			fprintf(stderr,"Illegal line in config file%s:\n%s",
				cfgfile, buf);
			cfgerr=1;
		}
	}
	fclose(fp);
	if(cfgerr) exit(1);
}



void usage_exit(char *argv0) {
	fprintf(stderr,
		"%s\n\n"
		"Usage: %s [-d audiodevice] [-q [-l]] [-t] [-x]\n"
		"    -q: Enable QPSK mode\n"
		"    -l: Invert phase shift for LSB mode\n"
		"    -t: Start with transmit mode on (debug only)\n"
		"    -x: TX freq folloes RX freq\n"
		"    -i: io to file (debug only)\n"
		"\n", version_string, argv0);
	exit(1);
}

void out_tuning();


#define STATUSLINE 18
void out_status(int force)
{
	static int oq,ol,odcd,opa,ot,otr,oaf;
	static float orx,otx;
	char buf[128];
	out_tuning();
	if( orx!=rxfreq||otx!=txfreq||oq!=qpsk||ol!=lsb||odcd!=dcd
	    ||opa!=passall||ot!=transmit||otr!=txtrack||oaf!=afc)
		force=1;
	if(!force) return;
	sprintf(buf,"      %4s%4s rx=%6.1f(AFC:%3s) tx=%6.1f(NET:%3s)"
		"  DCD:%3s %7s  %2s",
		qpsk?"QPSK":"BPSK", lsb?"/LSB":"    ", 
		rxfreq, afc?"on ":"off", 
		txfreq, txtrack?"on ":"off", 
		dcd?"on ":"off",
		passall?"PASSALL":"       ", 
		transmit?"TX":"RX");

	orx=rxfreq; otx=txfreq; oq=qpsk; ol=lsb; odcd=dcd;
	ot=transmit; otr=txtrack; opa=passall; oaf=afc;
	window::outputline(STATUSLINE,buf);
}

void out_tuning()
{
	static int oldp=-1;
	int p;
	if(oldp==-1) {
	  window::putsyx(STATUSLINE+0,0,"    ");
	  window::putsyx(STATUSLINE+1,0,"       ");
	  window::putsyx(STATUSLINE+2,0,"       ");
	  window::putsyx(STATUSLINE+3,0,"       ");
	  window::putsyx(STATUSLINE+4,0,"    ");
	}
	// phdelta: 0..255
	// Position:     14 15 0 1 2 ...usw
	int pos[][2]={
	  {0,4}, {0,5}, {0,6}, {1,7}, {1,7}, {2,8}, {3,7}, {3,7},{4,6},{4,5},
	  {4,4}, {4,3}, {4,2}, {3,1}, {3,1}, {2,0}, {1,1}, {1,1},{0,2},{0,3} };
	p=(int)((phdelta/256.0)*20+0.5);   // 0..19
	if(p>19) p=0;
	if(p!=oldp) {
	  window::putsyx(STATUSLINE+pos[p][0], pos[p][1], "O");
	  window::putsyx(STATUSLINE+pos[oldp][0], pos[oldp][1], "");
	  oldp=p;
	}
}

void out_macroinfo()
{
	char buf[128];
	sprintf(buf," F1: %-25s  F2: %-25s\n", macro[0], macro[1]);
	window::putsyx(STATUSLINE+1,10,buf);
	sprintf(buf," F3: %-25s  F4: %-25s\n", macro[2], macro[3]);
	window::putsyx(STATUSLINE+2,10,buf);
	sprintf(buf," F5: %-25s  F6: %-25s\n", macro[4], macro[5]);
	window::putsyx(STATUSLINE+3,10,buf);
	sprintf(buf," F7: %-25s  F7: %-25s\n", macro[6], macro[7]);
	window::putsyx(STATUSLINE+4,10,buf);
	sprintf(buf," F9: %-25s  F10:%-25s\n", macro[8], macro[9]);
	window::putsyx(STATUSLINE+5,10,buf);
}

int main(int argc, char **argv)
{
	int val, res, c, exitfl=0, dd=0;
	fd_set rset, wset, eset;
	struct timeval tm;

	srandom(time((time_t *)0));

	psk31_coder::init_tables();
	/* read config file */
	readconfig();

	/* parse command line arguments */
	while( (c=getopt(argc, argv, "n:iqltxd:?"))!=-1 ) {
		switch(c) {
		case 'n':
			noiselevel=atoi(optarg); break;
		case 'i':
			io_file=1; break;
		case 'x':
			txtrack=1; break;
		case 'q':
			qpsk=1; break;
		case 'l':
			lsb=1; break;
		case 't':
			transmit=1; break;
		case 'd':
			sound_device=strdup(optarg);
			break;
		case '?':
			usage_exit(argv[0]);
			break;
		default:
			fprintf(stderr,"illegal option: %c\n",c);
			usage_exit(argv[0]);
			break;
		}
	}
	/* init FFT */
	/* fftsr_init(); */

	/* init screen */
	window::init_window();
	atexit(window::end_window);
	window::outputline(0,"--------------------------------------------------------------------------------");
	txwin=new window(1, 5, 80, 1);
	window::outputline(6,"--------------------------------------------------------------------------------");
	rxwin=new window(7, 10, 80, 0);
	window::outputline(17,"--------------------------------------------------------------------------------");
	statwin=new window(19, 5, 80, 0);
	out_status(1);
	out_macroinfo();
	
	/* init audio device */
	audio_fd=open(sound_device,O_RDWR|O_NONBLOCK);
	if(audio_fd<0) {
		perror(sound_device); exit(1);
	}
	if(!io_file) {
		val=8000;
		ioctl(audio_fd, SNDCTL_DSP_SPEED, &val);
		val=AFMT_S16_LE;
		ioctl(audio_fd, SNDCTL_DSP_SETFMT, &val);
		val=DMA_BUF_BITS;
		ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &val);
		val=1;
		ioctl(audio_fd, SNDCTL_DSP_NONBLOCK, &val);
		// Check if the device is operating in full duplex mode
		ioctl(audio_fd, SNDCTL_DSP_GETCAPS, &val);
		if(val&DSP_CAP_DUPLEX) full_duplex=1;
	}
	psk31tx.set_audiofd(audio_fd);
	psk31tx.set_mode(qpsk,lsb);
	psk31tx.set_freq(txfreq);

	psk31rx.set_mode(qpsk,lsb);
	psk31rx.set_afc(afc);
	psk31rx.set_freq(rxfreq);

	if(io_file) {
		psk31tx.send_string("TEST:[]{}$%TEST:"
				    "\370\371\372\373\374\375\376\377 "
				    "Test:    ...   test");
		psk31tx.send_char(TX_END);
	}

	/* go into main loop */
	while(!exitfl) {
		fflush(stdout);
		FD_ZERO(&rset); FD_ZERO(&wset); FD_ZERO(&eset);
		FD_SET(STDIN_FILENO, &rset);
		FD_SET(audio_fd, &rset);
		if(transmit) FD_SET(audio_fd, &wset);
		tm.tv_sec=0; tm.tv_usec=50000; /* 50ms */
		res=select(audio_fd+1, &rset, &wset, &eset, &tm);
		if(FD_ISSET(STDIN_FILENO, &rset)) {
			/* handle keyboard input */
			c=getch();
			switch(c) {
			case 0: case 3: 
			case 5:	case 6: case 14:
			case 15: case 17: case 19:
			case 21: case 22: 
			case 25: case 26: case 28: case 29:
			case 30: case 31:
				c=0; /* ignore */
				break;
			case KEY_F(9): case KEY_F(1): case KEY_F(2):
			case KEY_F(3): case KEY_F(4): case KEY_F(5):
			case KEY_F(6): case KEY_F(7): case KEY_F(8):
				psk31tx.send_string( macro[c-KEY_F0-1] );
				txwin->put_string( macro[c-KEY_F0-1] );
				c=0;
				break;
			case 27:
				/* ESC -> Exit */
				/* exitfl=1;*/  c=0; break;
			case 24:
				/* ^X -> command mode */
				c=0; break;
			case 1:
				/* ^A -> afc on/off */
				setmode(qpsk,lsb,!afc,dcd); c=0; break;
			case 11:
				/* ^K -> txtrack on/off */
				txtrack=!txtrack; out_status(0); c=0; break;
			case 2:
				/* ^B -> QPSK <> BPSK */
				setmode(!qpsk,lsb,afc,dcd); c=0; break;
			case 16:
				/* ^P -> PassAll */
				passall=!passall; out_status(0); c=0; break;
			case 4:
				/* ^D -> DCD on/off */
				setmode(qpsk,lsb,afc,!dcd); c=0; break;
			case 23:
				/* ^W -> CWID */
				char buffer[128];
				sprintf(buffer," de %s",callsign);
				psk31tx.send_cw_string(transmit,buffer);
				if(!transmit) {
					transmit=1; out_status(0);
				}
				c=0; break;
			case 12:
				/* ^L -> LSB <> USB */
				setmode(qpsk, !lsb, afc, dcd); c=0; break;
			case KEY_UP:
				change_freq(1); c=0;
				break;
			case KEY_DOWN:
				change_freq(-1); c=0;
				break;
			case KEY_PPAGE:
				change_freq(8); c=0; 
				break;
			case KEY_NPAGE:
				change_freq(-8); c=0; 
				break;
			case 8: case 127: case KEY_BACKSPACE:
				c='\b'; break;    /* Backspace */
			case 7: case 9: case 10: case 13: 
 				break;            /* ok (BEL,TAB,NL,CR) */
			case 18:
				c=TX_END; break;  /* ^R -> switch to RX */
			case 20:
				c=TX_START; transmit=1; out_status(0);

				break;            /* ^T -> switch to TX */
			}
			if(c)
				psk31tx.send_char(c);
			if(c>0&&c<256) 
				txwin->put_char(c);
		}
		if(!transmit || 0&&FD_ISSET(audio_fd, &rset)) {
			/* handle audio input */
			int s; short sample;
			for(;;) {
				res=read(audio_fd, &sample, 2);
				s=(int)((random()-RAND_MAX/2)*2*
					((float)noiselevel/RAND_MAX));
				if(noiselevel) s+=sample; else s=sample;
				if(res==2) {
					res=psk31rx.process_rx_sample(s);
					if(res!=NO_CHAR && (dcd|passall) ) {
						// ignore ctrl chars
						if(res<32 && res!='\n' && 
						   res!='\r'&&res!='\b') 
							res=' ';
						rxwin->put_char(res);
					}
					if( (dd=(dd+1)&63) == 0 ) {
						psk31rx.get_info((int *)0,
								 (int *)0,
								 &rxfreq,
								 (int *)0,
								 (int *)0,
								 &dcd,
								 &phdelta);
						out_status(0);
					}
				}
				else if (res==0 || (res<0&&(errno==EAGAIN||errno==EBUSY))) {
					if(io_file) exit(2);
					break;
				}
				else exit(1);
			} 
		}
		if(transmit) {
			/* generate new audio output */
			for(;;) {
				res=psk31tx.processor();
				if(res==TX_BUSY) break;
				if(res==TX_ERROR) {
					perror("tx error:");
					break;
				}
				if(res==TX_END) { 
					transmit=0; 
					if(io_file) exit(2);
					if(!full_duplex)
						ioctl(audio_fd, SNDCTL_DSP_SYNC);
					break; 
				}
				/* output echo character */
				rxwin->put_char(res&0xFF);
			}
		}
	}
	close(audio_fd);
}



