// ---------------------------------------------------------------------------
// - String.hpp                                                              -
// - standard object library - string class definition                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_STRING_HPP
#define  ALEPH_STRING_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

#ifndef  ALEPH_VECTOR_HPP
#include "Vector.hpp"
#endif

namespace aleph {

  /// The String class is a literal class used to manipulate c-strings.
  /// The basic constructors are provided with a c-string or a regular string.
  /// Internally, the string is a reference counted structure which simplify
  /// the operation during the copy construction or assignment. Note that this
  /// reference count has nothing to do with the object reference count.
  /// Standard methods to retreive the length, a character, or to manipulate
  /// the string are provided.
  /// @author amaury darsch

  class String : public Literal {
  private:
    // the string record is a structure
    struct s_string* p_string;

  public:
    /// Create an empty string
    String (void);

    /// Create a string from a c-string
    /// @param s the c-string to copy
    String (const char* s);

    /// Create a string from a single character
    /// @param c the character to map to a string
    String (const char c);

    /// Copy constructor for this string class
    /// @param that the string to copy
    String (const String& that);

    /// Destroy this string
    ~String (void);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this string
    String toLiteral (void) const;

    /// @return a printable representation of this string
    String toString (void) const;

    /// assign a c-string to this string
    /// @param s the string to assign
    /// @return the current string
    String& operator = (const char* s);

    /// assign a character to this string
    /// @param c the character to assign
    /// @return the current string
    String& operator = (const char c);

    /// assign a string to this string
    /// @param s the string to assign
    /// @return the current string
    String& operator = (const String& s);

    /// get a character at a certain position
    /// @param pos the character position
    /// @return the character at this position
    /// @exception out-of-bound
    char operator [] (const long index) const;

    /// concatenate the string with another one
    /// @param the string to add to the current one
    /// @return a newly created string
    String operator + (const String& s) const;

    /// concatenate the string with a character
    /// @param the character to add to this string
    /// @return a newly created string
    String operator + (const char c) const;

    /// concatenate the string with an integer
    /// @param the integer to add to the string
    /// @return a newly created string
    String operator + (const long value) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator == (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator == (const char* s) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator != (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator != (const char* s) const;

    /// @return the length of this string
    long length (void) const;

    /// @return the length of a c-string
    static long length (const char* s);

    /// @return a character representation of this string or nil.
    char* toChar (void) const;

    /// remove leading blanks and tabs from this string
    /// @return a new clean string
    String stripl (void) const;

    /// remove trailing blanks and tabs from this string
    /// @return a new clean string
    String stripr (void) const;

    /// Remove leading and trailing blanks and tabs from this string
    /// @return a new clean string
    String strip (void) const;

    /// @return a string converter to upper case
    String toUpper (void) const;

    /// @return a string converter to lower case
    String toLower (void) const;

    /// return the right substring starting at index
    /// @param index the start index
    /// @return the extracted substring
    String rsubstr (const long index) const;

    /// return the left substring upto (but not including) the index
    /// @param index the end index
    /// @return the extracted substring
    String lsubstr (const long index) const;

    /// extract a sub string by index
    /// @param lidx the left index
    /// @param ridx the right index
    String substr (const long lidx, const long ridx) const;

    /// @return the hashid for this string
    long hashid (void) const;

    /// fill with string on the left with a character for a given size
    /// @param the fill character
    /// @param the target size
    String lfill (const char c, const long size) const;

    /// fill with string on the rigth with a character for a given size
    /// @param the fill character
    /// @param the target size
    String rfill (const char c, const long size) const;

    /// operate this string with another object
    /// @param type the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// generate a new string
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set an object to this string
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate a string member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this string with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);
  };
}

#endif
