///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  wrapper.cpp
  ------------------------------------------------------------------------
  Implementations of malloc(), free(), etc. in terms of hoard.
  This lets us link in hoard in place of the stock malloc
  (useful to test fragmentation).
  ------------------------------------------------------------------------
  @(#) $Id: wrapper.cpp,v 1.22 2000/02/24 15:27:28 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include <strings.h>
#endif
#include "config.h"

#if USE_PRIVATE_HEAPS
#include "privateheap.h"
#define HEAPTYPE privateHeap
#else
#include "threadheap.h"
#define HEAPTYPE threadHeap
#endif

#include "processheap.h"
#include "wrapper.h"
#include "arch-specific.h"

static wrapper TheWrapper;
char wrapper::buf[sizeof(processHeap)];
processHeap * wrapper::TheRealAllocator;
hoardLockType wrapper::_lock;

void * operator new (size_t size)
{
  return malloc (size);
}

void * operator new[] (size_t size)
{
  return malloc (size);
}

void operator delete (void * ptr)
{
  free (ptr);
}

void operator delete[] (void * ptr)
{
  free (ptr);
}


extern "C" void * malloc (size_t sz)
{
  processHeap * pHeap = TheWrapper.TheAllocator();
  void * addr = pHeap->getHeap(pHeap->getHeapIndex()).malloc (sz);
  return addr;
}

extern "C" void * calloc (size_t nelem, size_t elsize)
{
  processHeap * pHeap = TheWrapper.TheAllocator();
  void * ptr = pHeap->getHeap(pHeap->getHeapIndex()).malloc (nelem * elsize);
  // Zero out the malloc'd block.
  memset (ptr, 0, nelem * elsize);
  return ptr;
}

extern "C" void free (void * ptr)
{
  processHeap * pHeap = TheWrapper.TheAllocator();
#if USE_PRIVATE_HEAPS
  pHeap->getHeap(pHeap->getHeapIndex()).free(ptr);
#else
  pHeap->free (ptr);
#endif
}


extern "C" void * memalign (size_t alignment, size_t size)
{
  processHeap * pHeap = TheWrapper.TheAllocator();
  void * addr = pHeap->getHeap(pHeap->getHeapIndex()).memalign (alignment, size);
  return addr;
}


extern "C" void * valloc (size_t size)
{
  return memalign (hoardGetPageSize(), size);
}


extern "C" void * realloc (void * ptr, size_t sz)
{
  if (ptr == NULL) {
    return malloc (sz);
  }
  if (sz == 0) {
    free (ptr);
    return NULL;
  }

  // Allocate a new block of size sz.

  void * buf = malloc (sz);

  // Find out how large the original object was.

  size_t objSize = HEAPTYPE::objectSize (ptr);

  // Copy the contents of the original object
  // up to the size of the new block.

  size_t minSize = (objSize < sz) ? objSize : sz;
  memcpy (buf, ptr, minSize);

  // Free the old block.

  free (ptr);

  // Return a pointer to the new one.

  return buf;
}


extern "C" void malloc_stats (void)
{
  TheWrapper.TheAllocator()->stats();
}
