/* File "basic.h":
 * Basic types, macros and functions that are used everywhere. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* the Hangul suffix and the Malaga version string ==========================*/

#ifdef HANGUL
#define HANGUL_SUFFIX "-hangul"
#else
#define HANGUL_SUFFIX
#endif

#define MALAGA_VERSION "4.2.5p2"

/* basic types ==============================================================*/

#define GLOBAL extern  /* item that may be used in other modules */
#define LOCAL static   /* item that belongs to this module */
#define FORWARD static /* item that will be defined later in this module */

/* numeric types */
typedef signed char byte_t;           /* signed 8 bits */
typedef unsigned char u_byte_t;       /* unsigned 8 bits */
typedef signed short int short_t;     /* signed 16 bits */
typedef unsigned short int u_short_t; /* unsigned 16 bits */
typedef signed long int long_t;       /* signed 32 bits */
typedef unsigned long int u_long_t;  /* unsigned 32 bits */

/* character types */
typedef char *string_t; /* a EOS-terminated C string */
#define EOS '\0'        /* End-Of-String control character */
#define ORD(char) ((u_byte_t) (char)) /* the ordinal number of a char */

/* boolean type */
#undef bool_t /* <rpc/rpc.h> conflicts with "bool_t" definition */
#undef TRUE
#undef FALSE
typedef enum {FALSE, TRUE} bool_t;

#undef NULL
#define NULL 0 /* null pointer */

#define MIN(a,b) ((a) < (b) ? (a) : (b)) /* minimum of <a> and <b> */
#define MAX(a,b) ((a) > (b) ? (a) : (b)) /* maximum of <a> and <b> */
#define ABS(a) ((a) >= 0 ? (a) : (-a))   /* absolute value of <a> */

/* machine dependencies =====================================================*/

#define BITS_PER_BYTE 8

#define ALIGN(ptr) (((u_long_t) (ptr) + 3UL) & ~3UL) 
/* align to next longword */

#define LONGS(bytes) (((bytes) + 3UL) >> 2) 
/* number of longs needed to store this much bytes */ 

/* error handling ===========================================================*/

#ifdef __GNUC__
extern void error (string_t, ...) __attribute__ ((noreturn));
/* print an error message, like "printf", and return to the user input loop.
 * (exit the program if there is no user input loop).
 * THIS FUNCTION NEVER RETURNS. */
#else
extern void error (string_t, ...);
#endif

/* memory definitions =======================================================*/

#define COPY_MEM(to, from, item_size, items) \
  memcpy (to, from, (item_size)*(items)) 
/* Copy <items> items of size <item_size> each from <from> to <to>.
 * Do not use this macro if <to> and <from> may overlap. */

#define COMP_MEM(ptr1, ptr2, item_size, items) \
  memcmp (ptr1, ptr2, (item_size)*(items)) 
/* the same holds for "COMP_MEM" and "memcmp"... */

#define MOVE_MEM(ptr1, ptr2, item_size, items) \
  memmove (ptr1, ptr2, (item_size)*(items)) 
/* Move <items> items of size <item_size> each from <from> to <to>.
 * Use this macro if <to> and <from> may overlap. */

extern void *new_mem (u_long_t item_size);
/* Allocate a memory block of size <item_size>, clear it and return it.
 * If memory is out, call the function "error". */

extern void *new_vector (u_long_t item_size, u_long_t items);
/* Allocate a vector of <items> items of size <item_size> each, clear it
 * and return it. If memory is out, call the function "error". */

/* string definitions =======================================================*/

extern string_t copy_string (string_t to, string_t from, string_t to_end);
/* Copy <from> to <to>, but do not write behind <to_end>.
 * The string in <to> will be terminated by EOS.
 * If <from> doesn't fit into new string, an error occurs.
 * The pointer that is returned points to the trailing EOS. */

extern string_t copy_string_readable (string_t to, 
				      string_t from, 
				      string_t to_end,
				      string_t from_end);
/* Like "copy_string", but copy a "\" in front of quotes
 * and copy any control chars in octal code: "\000". 
 * If <from_end> != NULL, it marks the end of the string to be copied. */

extern string_t new_string (string_t string);
/* Allocate memory and copy <string> into it.
 * The string can be deleted with "free". */

extern string_t new_string_readable (string_t string, string_t string_end);
/* Like "new_string", but copy a "\" in front of quotes
 * and copy any control chars in octal code: "\000". 
 * If <string_end> != NULL, it marks the end of the string. */

extern string_t new_string_section (string_t string, string_t end);
/* Allocate memory and copy the beginning of <string> into it.
 * <end> points to the first char that is not part of the beginning.
 * The string can be deleted with "free". */

extern string_t concat_strings (string_t first_string, ...);
/* Concatenate a list of strings and return the result string.
 * Must have NULL-terminated list of strings as parameters.
 * The result string can be deleted with "free". */

extern short_t strcmp_no_case (string_t str1, string_t str2);
/* Return (case insensitive) lexical order of <str1> and <str2>:
 * Result is -1 if <str1> < <str2>,
 *            0 if <str1> = <str2>,
 *            1 if <str1> > <str2>. */

extern short_t strncmp_no_case (string_t str1, string_t str2, long_t n);
/* Return (case insensitive) lexical order of <str1> and <str2>,
 * but compare only the first <n> characters.
 * Result is -1 if <str1> < <str2>,
 *            0 if <str1> = <str2>,
 *            1 if <str1> > <str2>. */

extern string_t next_non_space (string_t string);
/* Return <string>, but without leading spaces. */

/* character definitions ====================================================*/

extern char lower_letter[256];
/* This table contains the lower case letter for each letter code,
 * and 0 for each non-letter. */

#define TO_LOWER(c) \
  (lower_letter[ORD(c)] != 0 ? lower_letter[ORD(c)] : (c))
/* Return <c> as a lower letter, or <c> itself if it is not a letter. */

#define IS_LETTER(c) (lower_letter[ORD(c)] != 0)
/* Return whether <c> is a letter */

#define IS_DIGIT(c) (c >= '0' && c <= '9')
/* Return whether <c> is a digit */

#define IS_SPACE(c) ((c) == ' ' || (c) == '\n' || (c) == '\t')
/* Return whether <c> is a whitespace character. */

#define ENCODED_STRING(string) (string)
/* Convert text from external representation to internal representation. */

#define DECODED_STRING(string) (string)
/* Convert text from internal representation to external representation. */

/* debug definitions ========================================================*/

/* The following macros only expand when DEBUG is defined
 * and are intended as debugging aids. */

#ifdef DEBUG

/* If <cond_expr> doesn't hold, print an error message and exit. */ 
#define DB_ASSERT(cond_expr) \
  do \
  { \
    if (!(cond_expr)) \
      { \
        fprintf (stderr, "DEBUG: %s:%d: assertion \"%s\" failed\n", \
	         __FILE__, __LINE__, #cond_expr); \
        exit (1); \
      } \
  } while (FALSE)

#else

#define DB_ASSERT(cond_expr) \
  do {} while (FALSE)

#endif /* DEBUG */

/*---------------------------------------------------------------------------*/
