/* File "breakpoints.c":
 * Administration of breakpoints. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* includes =================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "files.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rules.h"
#include "input.h"
#include "commands.h"

#undef GLOBAL
#define GLOBAL
#include "breakpoints.h"

/* types ====================================================================*/

typedef struct BREAKPOINT_T /* definition of a breakpoint */
{
  struct BREAKPOINT_T *next;

  short_t number;       /* breakpoint number */
  rule_sys_t *rule_sys; /* rule system of breakpoint */
  long_t instr;         /* instruction to break at */
} breakpoint_t;

/* variables ================================================================*/

LOCAL long_t breakpoint_number = 0;     /* number of breakpoints so far */
LOCAL breakpoint_t *breakpoints = NULL; /* list of breakpoints */

LOCAL short_t num_rule_systems;      /* number of loaded rule systems */
LOCAL rule_sys_name_t *rule_systems; /* name for each rule system */

/* functions ================================================================*/

LOCAL long_t instruction_at (rule_sys_t *rule_sys, string_t file, long_t line)
/* Return the index of the first instruction at <rule_sys>, <file>, <line>
 * or -1 if there is no code there. */
{
  src_line_t *src_line;
  
  for (src_line = rule_sys->src_lines; 
       src_line < rule_sys->src_lines + rule_sys->src_lines_size; 
       src_line++)
  {
    if (src_line->file != -1 
	&& strcmp (rule_sys->strings + src_line->file, file) == 0 
	&& src_line->line >= line)
      return (src_line->instr);
  }

  return -1;
}

/*---------------------------------------------------------------------------*/

LOCAL long_t find_rule_by_name (rule_sys_t *rule_sys, string_t rule_name)
/* Find the first line of rule <rule_name> in <rule_sys>.
 * Return the first instruction of this rule. */
{
  long_t i;
  
  /* Look for the rule name. */
  for (i = 0; i < rule_sys->rules_size; i++)
  {
    if (strcmp_no_case (rule_sys->strings + rule_sys->rules[i].name, 
			rule_name) == 0)
      return rule_sys->rules[i].first_instr;
  }
  return -1;
}

/*---------------------------------------------------------------------------*/

LOCAL string_t complete_file_name (rule_sys_t *rule_sys, string_t file_name)
/* Return <file_name> completed if it is in <rule_sys>, else return NULL. */
{
  src_line_t *src_line;
  
  for (src_line = rule_sys->src_lines;
       src_line < rule_sys->src_lines + rule_sys->src_lines_size; 
       src_line++)
  {
    if (src_line->file != -1 
	&& strcmp (name_in_path (rule_sys->strings + src_line->file), 
		   file_name) == 0)
      return rule_sys->strings + src_line->file;
  }
  return NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_breakpoint (string_t arguments, 
			     rule_sys_t **rule_sys, 
			     string_t *file, 
			     long_t *line)
/* Parse a breakpoint specification; <arguments> must be
 * "" (nothing) or
 * "<line_number>" or
 * "[<rule_sys_name>] [<file> <line_number> | <rule_name>]".
 * Set <rule_sys>, <file> and <line> according to it. */
{
  if (pc == -1)
  {
    *rule_sys = NULL;
    *file = NULL;
    *line = -1;
  }
  else
  {
    *rule_sys = executed_rule_sys;
    source_of_instr (executed_rule_sys, pc, line, NULL, file, NULL);
  }

  if (IS_DIGIT (*arguments)) /* Parse a line number. */
    *line = parse_integer (&arguments);
  else if (*arguments != EOS) /* Read file name or rule name. */
  {
    string_t argument;
    long_t i;
    
    argument = parse_word (&arguments);
    
    /* Read rule system name in front of file name or rule name. */
    for (i = 0; i < num_rule_systems; i++)
    {
      if (rule_systems[i].rule_sys == NULL)
	continue;
      
      if (strcmp_no_case (argument, rule_systems[i].name) == 0)
      {
	*rule_sys = rule_systems[i].rule_sys;
	free (argument);
	argument = parse_word (&arguments);
	break;
      }
    }
    
    if (IS_DIGIT (*arguments)) /* A file name followed by a line number. */
    {
      *line = parse_integer (&arguments);

      if (*rule_sys != NULL)
	*file = complete_file_name (*rule_sys, argument);
      else
      {
	for (i = 0; i < num_rule_systems; i++)
	{ 
	  if (rule_systems[i].rule_sys == NULL)
	    continue;
	  
	  *rule_sys = rule_systems[i].rule_sys;
	  *file = complete_file_name (*rule_sys, argument);
	  if (*file != NULL)
	    break;
	}
      }
      if (*file == NULL)
	error ("no source file \"%s\"", argument);
    }
    else
    {
      long_t first_instr;
      
      if (*rule_sys != NULL)
	first_instr = find_rule_by_name (*rule_sys, argument);
      else
      {
	first_instr = -1; /* prevent a "not initialized" warning */
	for (i = 0; i < num_rule_systems; i++)
	{ 
	  if (rule_systems[i].rule_sys == NULL)
	    continue;
	  
	  *rule_sys = rule_systems[i].rule_sys;
	  first_instr = find_rule_by_name (*rule_sys, argument);
	  if (first_instr != -1)
	    break;
	}
      }
      if (first_instr == -1)
	error ("rule \"%s\" is unknown", argument);
      
      /* Find the corresponding source line. */
      source_of_instr (*rule_sys, first_instr, line, NULL, file, NULL);
    }

    free (argument);
  }

  parse_end (arguments);
  
  if (*file == NULL)
    error ("missing file name");

  if (*line == -1)
    error ("missing line number");
}

/*---------------------------------------------------------------------------*/

GLOBAL short_t at_breakpoint (rule_sys_t *rule_sys, long_t instruction)
/* Return breakpoint number if <instruction> in <rule_sys> hits a 
 * breakpoint; return 0 else. */
{
  breakpoint_t *breakpoint;
  
  /* Search the breakpoint list. */
  for (breakpoint = breakpoints; 
       breakpoint != NULL; 
       breakpoint = breakpoint->next)
  {
    if (breakpoint->rule_sys == rule_sys && breakpoint->instr == instruction)
      return breakpoint->number;
  }
  
  return 0;
}

/*---------------------------------------------------------------------------*/

LOCAL void delete_all_breakpoints (void)
/* Run through breakpoint list and free all breakpoints. */
{
  breakpoint_t *breakpoint;
   
  breakpoint = breakpoints;
  while (breakpoint != NULL)
  {
    breakpoint_t *next_breakpoint = breakpoint->next;
    
    free (breakpoint);
    breakpoint = next_breakpoint;
  }
  breakpoints = NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void do_delete (string_t argument)
/* Remove a breakpoint. */
{
  if (IS_LETTER (*argument))
  {
    string_t word = parse_word (&argument);
    
    if (strcmp_no_case (word, "all") != 0)
      error ("\"all\" or breakpoint numbers expected, not \"%s\"", word);
    
    delete_all_breakpoints ();
  }
  else
  {
    while (*argument != EOS)
    {
      long_t breakpoint_number = parse_integer (&argument);
      breakpoint_t **breakpoint_ptr;
      breakpoint_t *breakpoint;
      
      /* Delete breakpoint with <breakpoint_number>. */
      breakpoint_ptr = &breakpoints; 
      while (*breakpoint_ptr != NULL 
	     && (*breakpoint_ptr)->number != breakpoint_number)
	breakpoint_ptr = &(*breakpoint_ptr)->next;
      
      if (*breakpoint_ptr == NULL)
	error ("no breakpoint %d", breakpoint_number);
      
      breakpoint = *breakpoint_ptr;
      *breakpoint_ptr = breakpoint->next;
      free (breakpoint);
    }
  }
  
  parse_end (argument);
}

GLOBAL command_t delete_command =
{
  "delete d", do_delete,
  "Delete breakpoints.\n"
  "Arguments:\n"
  "  <number> ... -- delete specified breakpoints\n"
  "  all -- delete all breakpoints\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_break (string_t argument)
/* Define a breakpoint. */
{
  rule_sys_t *rule_sys;
  string_t file, rule;
  long_t line;
  long_t instr;
  breakpoint_t **breakpoint_ptr;
  breakpoint_t *breakpoint, *new_breakpoint;

  /* Parse breakpoint <argument>. */
  parse_breakpoint (argument, &rule_sys, &file, &line);
  
  /* Find first instruction of this breakpoint */
  instr = instruction_at (rule_sys, file, line);
  if (instr == -1)
    error ("no code at file \"%s\", line %ld", name_in_path (file), line);

  /* Check if other breakpoints exist at this position. */
  for (breakpoint = breakpoints; 
       breakpoint != NULL; 
       breakpoint = breakpoint->next)
  {
    if (breakpoint->rule_sys == rule_sys && breakpoint->instr == instr)
      error ("breakpoint %d already set here", breakpoint->number);
  }

  /* Set source line and source file to the beginning of the statement. */
  source_of_instr (rule_sys, instr, &line, NULL, &file, &rule);

  new_breakpoint = (breakpoint_t *) new_mem (sizeof (breakpoint_t));
  new_breakpoint->number = ++breakpoint_number;
  new_breakpoint->rule_sys = rule_sys;
  new_breakpoint->instr = instr;
  new_breakpoint->next = NULL;

  /* Enter breakpoint at end of list. */
  breakpoint_ptr = &breakpoints;
  while (*breakpoint_ptr != NULL)
    breakpoint_ptr = &(*breakpoint_ptr)->next;
  *breakpoint_ptr = new_breakpoint;
  
  printf ("breakpoint %d in file \"%s\", line %ld, rule \"%s\"\n",
	  new_breakpoint->number, name_in_path (file), line, rule);
}

GLOBAL command_t break_command =
{
  "break b", do_break,
  "Set a breakpoint at the specified position.\n"
  "Arguments:\n"
  "  <rule> -- set a breakpoint at the beginning of <rule>\n"
  "  <file> <line> -- set a breakpoint at <line> in <file>\n"
  "  <line> -- set a breakpoint at <line> in the current rule file\n"
  "  (none) -- set a breakpoint at the current line in the"
  " current rule file\n"
  "The first two forms may begin with a rule system specification.\n"
  "The last two forms can only be used in debug mode or after a rule error.\n"
  "You can't set two breakpoints at the same position.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_list (string_t argument)
/* List breakpoints. */
{
  breakpoint_t *breakpoint;
  
  parse_end (argument);

  if (breakpoints == NULL)
    error ("no breakpoints");
  
  for (breakpoint = breakpoints; 
       breakpoint != NULL; 
       breakpoint = breakpoint->next)
  {
    string_t file, rule;
    long_t line;
    
    source_of_instr (breakpoint->rule_sys, breakpoint->instr, 
		     &line, NULL, &file, &rule);
    printf ("breakpoint %d in file \"%s\", line %ld, rule \"%s\"\n", 
	    breakpoint->number, name_in_path (file), line, rule);
  }
}

GLOBAL command_t list_command =
{
  "list l", do_list,
  "List all breakpoints.\n"
  "Arguments: (none)\n"
};

/*---------------------------------------------------------------------------*/

GLOBAL void init_breakpoints (short_t num_rule_sys, 
			      rule_sys_name_t rule_sys[])
/* Initialise this module. 
 * Pass the number of rule systems in <num_rule_sys> 
 * and their names in <rule_sys>. */
{
  num_rule_systems = num_rule_sys;
  rule_systems = rule_sys;
  breakpoints = NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_breakpoints (void)
/* Terminate this module. */
{
  delete_all_breakpoints ();
}

/*---------------------------------------------------------------------------*/
