/* File "rules.c":
 * All definitions needed for the execution of a rule. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de */ 

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "patterns.h"
#include "files.h"
#include "malaga_files.h"
#include "instr_type.h"
#include "rule_type.h"

#undef GLOBAL
#define GLOBAL
#include "rules.h"

/* variables ================================================================*/

/* the internal state of rule execution */
GLOBAL bool_t executing_rule = FALSE;
GLOBAL long_t pc = -1;
GLOBAL long_t nested_subrules = 0;
GLOBAL long_t executed_rule_number = -1;
GLOBAL rule_sys_t *executed_rule_sys = NULL;
GLOBAL rule_sys_t *debug_rule_sys = NULL;

#define STACK_SIZE 400

LOCAL value_t stack[STACK_SIZE]; /* the rule stack */
LOCAL value_t *top;              /* first stack element that is not used */
LOCAL value_t *base;             /* first stack element used in this rule */
LOCAL value_t *bottom;           /* first stack element used in this branch */

#define BACKUP_STACK_SIZE 50

LOCAL struct 
{ 
  long_t pc;
  long_t nested_subrules;
  value_t *base; 
  value_t *bottom;
} backup_stack[BACKUP_STACK_SIZE];

GLOBAL long_t backup_top = 0; 
/* index of the first free item in <backup_stack> */

/* switch support ===========================================================*/

typedef struct SWITCH_T 
{ 
  struct SWITCH_T *next; 
  symbol_t key; 
  value_t value; 
} switch_t;

switch_t *switches;
switch_t *current_switch;

/*---------------------------------------------------------------------------*/

GLOBAL void set_switch (symbol_t key, value_t value)
/* Set the switch <key> to <value>. */
{
  switch_t **my_switch;

  my_switch = &switches;
  while (*my_switch != NULL && (*my_switch)->key != key)
    my_switch = &(*my_switch)->next;

  if (*my_switch != NULL)
  {
    free ((*my_switch)->value);
    (*my_switch)->value = new_value (value);
  }
  else
  {
    (*my_switch) = new_mem (sizeof (switch_t));
    (*my_switch)->key = key;
    (*my_switch)->value = new_value (value);
    (*my_switch)->next = NULL;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL value_t get_switch (symbol_t key)
/* Return the value of the switch <key>. 
 * Report an error if this switch doesn't exist. */
{
  switch_t *my_switch;

  for (my_switch = switches; my_switch != NULL; my_switch = my_switch->next)
  {
    if (my_switch->key == key)
      return my_switch->value;
  }
  error ("switch \"%s\" is not defined", symbol_name (key));
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t start_switches (void)
/* Must be called before "get_next_switch" is called.
 * Returns TRUE iff there are any switches */
{
  current_switch = switches;
  return (current_switch != NULL);
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_next_switch (symbol_t *key)
/* Return the value of the next switch. Return its key in <*key>.
 * If there is no more switch, return NULL. */
{
  value_t value;

  if (current_switch == NULL)
    return NULL;

  *key = current_switch->key;
  value = current_switch->value;

  current_switch = current_switch->next;
  return value;
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_switches (void)
/* Free all settings. */
{
  switch_t *my_switch;

  my_switch = switches;
  while (my_switch != NULL)
  {
    switch_t *next_switch = my_switch->next;

    free (my_switch->value);
    free (my_switch);
    my_switch = next_switch;
  }
  switches = NULL;
}

/* rule execution ===========================================================*/

LOCAL void check_stack_space (long_t n)
/* Check if there is space enough on the stack to push <n> values. */
{
  if (top + n >= stack + STACK_SIZE)
    error ("too many variables or branches or value too complex");
}

/*---------------------------------------------------------------------------*/

LOCAL value_t standard_function (short_t function, value_t value)
/* Perform function <function> on <value> and return the result. */
{
  switch (function)
  {
  case FUNC_TO_ATOMS:
    return symbol_to_atoms (value_to_symbol (value));
  case FUNC_TO_MULTI:
    return symbol_to_value (find_multi_symbol (value));
  case FUNC_TO_SET:
    return list_to_set (value);
  case FUNC_IS_CAPITAL:
  {
    string_t string;

    string = value_to_string (value);
    if (TO_LOWER (*string) != *string)
      return symbol_to_value (YES_SYMBOL);
    else
      return symbol_to_value (NO_SYMBOL);
  }
  case FUNC_GET_SWITCH:
    return get_switch (value_to_symbol (value));
  case FUNC_GET_LENGTH:
    return double_to_value (get_list_length (value));
  case FUNC_GET_VALUE_TYPE:
    return symbol_to_value (get_value_type (value));
  case FUNC_GET_SYMBOL_NAME:
    return string_to_value (symbol_name (value_to_symbol (value)), NULL);
  case FUNC_TRANSMIT:
    if (transmit_function == NULL)
      error ("no transmit function available");
    return transmit_function (value);
  default:
    error ("internal (unknown standard function)");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void execute_rule (rule_sys_t *rule_sys, long_t rule_number, ...)
/* Execute rule <rule_number> in the rule system <rule_sys>.
 * Add the rule's parameters to the parameter list of "execute_rule". */
{
  static symbol_t nil = NIL_SYMBOL; /* the "nil" symbol */
  bool_t terminate; /* shall we terminate the current rule internal path? */
  long_t i;
  va_list arg;

  /* Initialise the value stack. */
  top = base = bottom = stack;
  backup_top = 0;
  nested_subrules = 0;

  /* See if the rule exists at all. */
  DB_ASSERT (rule_number < rule_sys->rules_size);

  /* Copy for debug purposes and error messages. */
  executed_rule_sys = rule_sys;
  executed_rule_number = rule_number;

  /* Copy the parameters on the stack. */
  check_stack_space (rule_sys->rules[rule_number].num_params);
  va_start (arg, rule_number);
  for (i = 0; i < rule_sys->rules[rule_number].num_params; i++)
  {
    top[0] = va_arg (arg, value_t);
    top++;
  }
  va_end (arg);

  pc = rule_sys->rules[rule_number].first_instr;
  executing_rule = TRUE;

  rule_successful = FALSE;
  terminate = FALSE;
  while (! terminate) 
  {
    instr_t instruction;
    long_t info;

    if (debug_rule_sys == rule_sys)
      debug_rule ();
      
    clear_value_heap_except (top - stack, stack);

    instruction = rule_sys->instrs[pc];
    pc++;
    info = INSTR_INFO (instruction);
    switch (OPCODE (instruction)) 
    {
    case INS_ERROR:
      switch (info)
      {
      case ASSERTION_ERROR:
	error ("assertion failed");
      case NO_RETURN_ERROR:
	error ("missing return statement");
      default:
	error ("%s", rule_sys->strings + info);
      }

    case INS_TERMINATE:
      terminate = TRUE;
      break;

    case INS_NOP:
      break;

    case INS_TERMINATE_IF_NULL:
      DB_ASSERT (top >= base + 1);
      if (top[-1] == NULL)
	terminate = TRUE;
      top--;
      break;

    case INS_ADD_END_STATE:
      DB_ASSERT (top >= base + 1);
      add_end_state (top[-1]);
      top--;
      rule_successful = TRUE;
      break;

    case INS_ADD_STATE:
      DB_ASSERT (top >= base + 1);
      add_running_state (top[-1], info);
      top--;
      rule_successful = TRUE;
      break;

    case INS_ADD_ALLO:
      DB_ASSERT (top >= base + 2);
      add_allo (top[-2], top[-1]);
      top -= 2;
      rule_successful = TRUE;
      break;

    case INS_SUCCEED:
      rule_successful = TRUE;
      break;

    case INS_PUSH_NULL:
    {
      long_t i;

      check_stack_space (info);
      for (i = 0; i < info; i++)
	*top++ = NULL;
      break;
    }

    case INS_PUSH_VAR:
      check_stack_space (1);
      DB_ASSERT (base + info < top);
      *top++ = base[info];
      break;

    case INS_PUSH_CONST:
      check_stack_space (1);
      DB_ASSERT (info < rule_sys->values_size);
      *top++ = rule_sys->values + info;
      break;

    case INS_PUSH_SYMBOL:
      check_stack_space (1);
      top[0] = symbol_to_value (info);
      top++;
      break;

    case INS_PUSH_PATTERN_VAR:
      check_stack_space (1);
      DB_ASSERT (info < PATTERN_VAR_MAX);
      top[0] = string_to_value (pattern_var[info].start, 
                                pattern_var[info].end);
      top++;
      break;

    case INS_POP:
      DB_ASSERT (top >= base + info);
      top -= info;
      break;

    case INS_BUILD_LIST:
      DB_ASSERT (top >= base + info);
      top[-info] = build_list (info, top - info);
      top -= (info-1);
      break;

    case INS_BUILD_RECORD:
      DB_ASSERT (top >= base + 2*info);
      top[-2*info] = build_record (info, top - 2*info);
      top -= (2*info-1);
      break;

    case INS_BUILD_PATH:
      DB_ASSERT (top >= base + info);
      top[-info] = build_path (info, top - info);
      top -= (info-1);
      break;

    case INS_DOT_OPERATION:
      DB_ASSERT (top >= base + 2);
      top[-2] = dot_operation (top[-2], top[-1]);
      if (top[-2] == NULL)
	top[-2] = &nil;
      top--;
      break;
      
    case INS_PLUS_OPERATION:
      DB_ASSERT (top >= base + 2);
      top[-2] = plus_operation (top[-2], top[-1]);
      top--;
      break;
      
    case INS_MINUS_OPERATION:
      DB_ASSERT (top >= base + 2);
      top[-2] = minus_operation (top[-2], top[-1]);
      top--;
      break;

    case INS_ASTERISK_OPERATION:
      DB_ASSERT (top >= base + 2);
      top[-2] = asterisk_operation (top[-2], top[-1]);
      top--;
      break;

    case INS_SLASH_OPERATION:
      DB_ASSERT (top >= base + 2);
      top[-2] = slash_operation (top[-2], top[-1]);
      top--;
      break;

    case INS_UNARY_MINUS_OP:
      DB_ASSERT (top >= base + 1);
      top[-1] = unary_minus_operation (top[-1]);
      break;

    case INS_GET_ATTRIBUTE:
      DB_ASSERT (top >= base + 1);
      top[-1] = get_attribute (top[-1], (symbol_t) info);
      if (top[-1] == NULL)
	top[-1] = &nil;
      break;

    case INS_REMOVE_ATTRIBUTE:
      DB_ASSERT (top >= base + 1);
      top[-1] = remove_attribute (top[-1], (symbol_t) info);
      break;

    case INS_STD_FUNCTION:
      DB_ASSERT (top >= base + 1);
      top[-1] = standard_function (info, top[-1]);
      break;

    case INS_MATCH:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->strings_size);
      if (match_pattern (value_to_string (top[-1]), 
			 rule_sys->strings + info))
	top[-1] = symbol_to_value (YES_SYMBOL);
      else
	top[-1] = symbol_to_value (NO_SYMBOL);
      break;

    case INS_SET_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      base[info] = top[-1];
      top--;
      break;

    case INS_PLUS_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      base[info] = plus_operation (base[info], top[-1]);
      top--;
      break;

    case INS_MINUS_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      base[info] = minus_operation (base[info], top[-1]);
      top--;
      break;

    case INS_ASTERISK_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      base[info] = asterisk_operation (base[info], top[-1]);
      top--;
      break;

    case INS_SLASH_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      base[info] = slash_operation (base[info], top[-1]);
      top--;
      break;

    case INS_SET_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      base[info] = modify_value_part (base[info], top[-2], top[-1],
				      modifier_replace);
      top -= 2;
      break;

    case INS_PLUS_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      base[info] = modify_value_part (base[info], top[-2], top[-1],
				      plus_operation);
      top -= 2;
      break;

    case INS_MINUS_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      base[info] = modify_value_part (base[info], top[-2], top[-1],
				      minus_operation);
      top -= 2;
      break;

    case INS_ASTERISK_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      base[info] = modify_value_part (base[info], top[-2], top[-1],
				      asterisk_operation);
      top -= 2;
      break;

    case INS_SLASH_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      base[info] = modify_value_part (base[info], top[-2], top[-1],
				      slash_operation);
      top -= 2;
      break;

    case INS_PUSH_1ST_ELEMENT:
      check_stack_space (1);
      DB_ASSERT (base + info < top);
      top[0] = first_element (base[info]);
      top++;
      break;

    case INS_ITERATE:
      DB_ASSERT (info >= 1 && base + info < top);
      base[info] = next_element (base[info-1], base[info]);
      break;

    case INS_JUMP:
      DB_ASSERT (info < rule_sys->instrs_size);
      pc = info;
      break;

    case INS_JUMP_IF_EQUAL:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (values_equal (top[-2], top[-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_NOT_EQUAL:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! values_equal (top[-2], top[-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_CONGR:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (values_congruent (top[-2], top[-1]))
	pc = info;
      top -= 2;
      break;
	  
    case INS_JUMP_IF_NOT_CONGR:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! values_congruent (top[-2], top[-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_IN:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_in_value (top[-2], top[-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_NOT_IN:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! value_in_value (top[-2], top[-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_LESS:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_to_double (top[-2]) < value_to_double (top[-1]))
	pc = info;
      top -= 2;
      break;
	  
    case INS_JUMP_IF_NOT_LESS:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! (value_to_double (top[-2]) < value_to_double (top[-1])))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_GREATER:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_to_double (top[-2]) > value_to_double (top[-1]))
	pc = info;
      top -= 2;
      break;
	  
    case INS_JUMP_IF_NOT_GREATER:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! (value_to_double (top[-2]) > value_to_double (top[-1])))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_NULL:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (top[-1] == NULL)
	pc = info;
      top--;
      break;

    case INS_JUMP_IF_NOT_NULL:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (top[-1] != NULL)
	pc = info;
      top--;
      break;

    case INS_JUMP_IF_YES:
    {
      symbol_t symbol;

      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
	  
      symbol = value_to_symbol (top[-1]);
      if (symbol != YES_SYMBOL && symbol != NO_SYMBOL)
	error ("boolean value expected");
      if (symbol == YES_SYMBOL)
	pc = info;
      top--;
      break;
    }

    case INS_JUMP_IF_NO:
    {
      symbol_t symbol;

      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
	  
      symbol = value_to_symbol (top[-1]);
      if (symbol != YES_SYMBOL && symbol != NO_SYMBOL)
	error ("boolean value expected");
      if (symbol == NO_SYMBOL)
	pc = info;
      top--;
      break;
    }
	  
    case INS_JUMP_NOW:
    {
      value_t *old_top = top;

      DB_ASSERT (info < rule_sys->instrs_size);
      if (backup_top >= BACKUP_STACK_SIZE)
	error ("too many branches in a rule");

      check_stack_space (top - bottom);

      backup_stack[backup_top].pc = pc;
      backup_stack[backup_top].nested_subrules = nested_subrules;
      backup_stack[backup_top].base = base;
      backup_stack[backup_top].bottom = bottom;
      while (bottom < old_top)
	*top++ = *bottom++;
      base += top - old_top;
      backup_top++;
      pc = info;
      break;
    }

    case INS_JUMP_LATER:
    {
      value_t *old_top = top;

      DB_ASSERT (info < rule_sys->instrs_size);
      if (backup_top >= BACKUP_STACK_SIZE)
	error ("too many branches in a rule");
 
      check_stack_space (top - bottom);

      backup_stack[backup_top].pc = info;
      backup_stack[backup_top].nested_subrules = nested_subrules;
      backup_stack[backup_top].base = base;
      backup_stack[backup_top].bottom = bottom;
      while (bottom < old_top)
	*top++ = *bottom++;
      base += top - old_top;
      backup_top++;
      break;
    }

    case INS_JUMP_SUBRULE:
      check_stack_space (2);
      top[0] = (value_t) (stack + (base - bottom));
      top[1] = (value_t) (rule_sys->instrs + pc);
      top += 2;
      base = top;
      pc = rule_sys->rules[info].first_instr;
      nested_subrules++;
      break;

    case INS_RETURN:
    {
      value_t *old_base;

      DB_ASSERT (base - bottom >= 2 + info);
      old_base = bottom + (((value_t *) base[-2]) - stack);
      base[-(info + 2)] = top[-1]; /* copy subrule result */
      pc = ((instr_t *) base[-1]) - rule_sys->instrs;
      top = base - (info + 1);
      base = old_base;
      nested_subrules--;
      break;
    }

    default:
      error ("internal (unknown instruction %d)", OPCODE (instruction));
      break;
    }

    if (terminate && backup_top > 0) 
    { /* Load a previously saved rule-internal state and continue. */
      backup_top--;
      top = bottom;
      base = backup_stack[backup_top].base;
      bottom = backup_stack[backup_top].bottom;
      pc = backup_stack[backup_top].pc;
      nested_subrules = backup_stack[backup_top].nested_subrules;
      terminate = FALSE;
    }
  }

  executing_rule = FALSE;
  pc = -1;
  executed_rule_number = -1;
  executed_rule_sys = NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL rule_sys_t *read_rule_sys (string_t file_name)
/* Read rule system from file <file_name>.
 * A symbol file must have already been loaded. */
{
  FILE *stream;
  rule_header_t header;
  rule_sys_t *rule_sys = new_mem (sizeof (rule_sys_t));

  stream = fopen_save (file_name, "rb");

  fread_save (&header, sizeof (header), 1, stream, file_name);

  check_header (&header.common_header, file_name, 
		RULE_FILE, RULE_CODE_VERSION);

  rule_sys->initial_rule_set = header.initial_rule_set;
  rule_sys->initial_cat = header.initial_cat;
  rule_sys->robust_rule = header.robust_rule;
  rule_sys->allo_rule = header.allo_rule;
  rule_sys->pruning_rule = header.pruning_rule;
  rule_sys->filter_rule = header.filter_rule;
  rule_sys->input_rule = header.input_rule;
  rule_sys->rules_size = header.rules_size;
  rule_sys->rules = (rule_t *) fread_block (sizeof (rule_t), 
					    header.rules_size, 
					    stream, file_name);
  rule_sys->rule_sets_size = header.rule_sets_size;
  rule_sys->rule_sets = (long_t *) fread_block (sizeof (long_t),
						header.rule_sets_size, 
						stream, file_name);
  rule_sys->instrs_size = header.instrs_size;
  rule_sys->instrs = (instr_t *) fread_block (sizeof (instr_t), 
					      header.instrs_size, 
					      stream, file_name);
  rule_sys->values_size = header.values_size;
  rule_sys->values = (cell_t *) fread_block (sizeof (cell_t), 
					     header.values_size, 
					     stream, file_name);
  rule_sys->src_lines_size = header.src_lines_size;
  rule_sys->src_lines = (src_line_t *) fread_block (sizeof (src_line_t), 
						    header.src_lines_size, 
						    stream, file_name);
  rule_sys->vars_size = header.vars_size;
  rule_sys->vars = (var_t *) fread_block (sizeof (var_t), header.vars_size, 
					  stream, file_name);
  rule_sys->var_scopes_size = header.var_scopes_size;
  rule_sys->var_scopes = (var_scope_t *) fread_block (sizeof (var_scope_t), 
						      header.var_scopes_size,
						      stream, file_name);
  rule_sys->strings_size = header.strings_size;
  rule_sys->strings = (char *)fread_block (sizeof (char), header.strings_size, 
					   stream, file_name);

  fclose_save (stream, file_name);
  
  return rule_sys;
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_rule_sys (rule_sys_t *rule_sys)
/* Free all memory used by <rule_sys>. */
{ 
  free (rule_sys->rules);
  free (rule_sys->rule_sets);
  free (rule_sys->instrs);
  free (rule_sys->values);
  free (rule_sys->src_lines);
  free (rule_sys->vars);
  free (rule_sys->var_scopes);
  free (rule_sys->strings);
  free (rule_sys);
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t inspect_stack (long_t index)
/* Return S[index] or NULL if it is not defined. */
{
  if (base + index >= top)
    return NULL;
  else
    return base[index];
}

/*---------------------------------------------------------------------------*/

GLOBAL void get_base_and_pc_indexes (long_t index, 
				     long_t *base_index, 
				     long_t *pc_index)
/* If <index> == 0, return the current <pc_index> and the <base_index> of the
 * stack when this rule was called as a subrule. */
{
  if (index == 0)
  {
    *pc_index = pc;
    *base_index = base - bottom;
  }
  else
  {
    *pc_index = ((instr_t *) bottom[index - 1]) - executed_rule_sys->instrs;
    *base_index = ((value_t *) bottom[index - 2]) - stack;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t inspect_stack_pointer (void)
/* Return value of rule stack pointer. */
{
  return top - base;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t first_variable_index (rule_sys_t *rule_sys,
				    long_t instr_index)
/* Return the stack index of the first variable that is visible
 * when pc is at <instr_index> in <rule_sys>. */
{
  rule_t *rule;
  long_t i, first_instr;

  /* Find the rule/subrule we're in. */
  rule = NULL;
  first_instr = -1;
  for (i = 0; i < rule_sys->rules_size; i++)
  {
    rule_t *rule2 = rule_sys->rules + i;

    if (rule2->first_instr <= instr_index 
	&& rule2->first_instr > first_instr)
    {
      rule = rule2;
      first_instr = rule->first_instr;
    }
  }

  DB_ASSERT (rule != NULL);

  if (rule->type == SUBRULE)
    return - (2 + rule->num_params);
  else
    return 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL void source_of_instr (rule_sys_t *rule_sys,
			     long_t instr_index, 
			     long_t *first_line, 
			     long_t *last_line, 
			     string_t *file_name,
			     string_t *rule_name)
/* Set *<first_line>, *<last_line> and *<file_name> to appropriate values
 * for the statement that has generated the instruction at <instr_index>. */
{
  long_t lower, upper, middle;
  src_line_t *src_line;

  if (rule_sys->src_lines_size == 0 
      || rule_sys->src_lines[0].instr > instr_index)
  {
    if (first_line != NULL)
      *first_line = -1;
    if (last_line != NULL)
      *last_line = -1;
    if (file_name != NULL)
      *file_name = NULL;
    if (rule_name != NULL)
      *rule_name = NULL;
    return;
  }

  /* Find the last src_line entry with <instr> <= <instr_index>. */
  lower = 0;
  upper = rule_sys->src_lines_size - 1;
  while (lower < upper) 
  {
    middle = (lower + upper + 1) / 2;
    src_line = rule_sys->src_lines + middle;

    if (src_line->instr <= instr_index)
      lower = middle;
    else
      upper = middle - 1;
  }
  
  src_line = rule_sys->src_lines + lower;

  if (first_line != NULL)
    *first_line = src_line->line;

  if (file_name != NULL)
  {
    if (src_line->file != -1)
      *file_name = rule_sys->strings + src_line->file;
    else
      *file_name = NULL;
  }

  /* Find the last line of the statement. */
  if (last_line != NULL) 
  {
    do
    {
      src_line++;
    } while (src_line < rule_sys->src_lines + rule_sys->src_lines_size
	     && src_line->line == -1);

    if (src_line < rule_sys->src_lines + rule_sys->src_lines_size)
      *last_line = src_line->line - 1;
    else
    *last_line = -1;
  }

  /* Find the rule of the statement */
  if (rule_name != NULL)
  {
    long_t rule_number;
    long_t i;
    long_t first_instr;
    
    rule_number = 0;
    first_instr = -1;
    for (i = 0; i < rule_sys->rules_size; i++)
    {
      rule_t *rule = rule_sys->rules + i;

      if (rule->first_instr <= instr_index && rule->first_instr > first_instr)
      {
	rule_number = i;
	first_instr = rule->first_instr;
      }
    }
    *rule_name = rule_sys->strings + rule_sys->rules[rule_number].name;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL long_t variable_index_local (rule_sys_t *rule_sys, 
				   var_t *var, 
				   long_t instr_index)
/* Return the stack index of variable <var> at <instr_index>.
 * Return -1 if it is not currently defined. */
{
  long_t lower, upper, middle;
  var_scope_t *var_scope;

  /* Find last scope whose <first_instr> is not higher than <instr_index>. */ 
  lower = var->first_scope;
  upper = lower + var->number_of_scopes - 1;

  while (lower < upper) 
  {
    middle = (lower + upper + 1) / 2;
    var_scope = rule_sys->var_scopes + middle;

    if (var_scope->first_instr <= instr_index)
      lower = middle;
    else
      upper = middle - 1;
  }

  /* <lower> is the index of the highest line
   * with an instruction index not more than <instr_index>. */
  if (lower == upper) 
  {
    /* We found a scope. */
    var_scope = rule_sys->var_scopes + lower;
    if (instr_index >= var_scope->first_instr 
	&& instr_index <= var_scope->last_instr)
      /* Variable is defined. */
      return var_scope->stack_index;
  }
  
  return -1;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t variable_at_index (rule_sys_t *rule_sys,
				   long_t stack_index, 
				   long_t instr_index)
/* Return the name of the variable that is defined at <stack_index>
 * when instruction <instr_index> is executed or NULL if there is none. */
{
  long_t i;

  /* There is never a variable at stack index -2 or -1. */
  if (stack_index == -2 || stack_index == -1)
    return NULL;

  /* For each variable name, test if it is the right one. */
  for (i = 0; i < rule_sys->vars_size; i++) 
  {
    var_t *var = rule_sys->vars + i;

    if (stack_index == variable_index_local (rule_sys, var, instr_index))
      return rule_sys->strings + var->name;
  }

  return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t variable_index (rule_sys_t *rule_sys,
			      string_t var_name, 
			      long_t instr_index)
/* Return the stack index of variable <var_name> at <instr_index>. */
{
  long_t lower, upper, middle;
  var_t *var;

  /* Search for the right variable name (binary search). */
  lower = 0;
  upper = rule_sys->vars_size - 1;
  
  while (lower < upper) 
  {
    short_t result;

    middle = (lower + upper) / 2;
    var = rule_sys->vars + middle;
    result = strcmp_no_case (var_name, rule_sys->strings + var->name);

    if (result < 0)
      upper = middle - 1;
    else if (result > 0)
      lower = middle + 1;
    else
      lower = upper = middle;
  }

  /* See if we have found <var_name>. */
  if (lower == upper) 
  {
    var = rule_sys->vars + lower;
    if (strcmp_no_case (var_name, rule_sys->strings + var->name) == 0)
      return variable_index_local (rule_sys, var, instr_index);
  }

  return -1;
}

/*---------------------------------------------------------------------------*/

#define RULE_SET_BUFFER_SIZE 500

GLOBAL string_t rule_set_readable (rule_sys_t *rule_sys, long_t rule_set)
/* Return <rule_set> in <rule_sys> as a readable string.
 * The string is valid only until "rule_set_readable" is called next. */
{
  static char buffer[RULE_SET_BUFFER_SIZE];
  string_t buffer_ptr;
  string_t buffer_end = buffer + RULE_SET_BUFFER_SIZE;

  buffer_ptr = buffer;
  if (rule_set == -1)
    buffer_ptr = copy_string (buffer_ptr, "(end state)", buffer_end);
  else
  { 
    bool_t name_has_been_printed;
    long_t *rule;
      
    buffer_ptr = copy_string (buffer_ptr, "rules ", buffer_end);
    name_has_been_printed = FALSE;
    for (rule = rule_sys->rule_sets + rule_set; *rule != -1; rule++)
    {
      if (name_has_been_printed)
	buffer_ptr = copy_string (buffer_ptr, ", ", buffer_end);
      else
	name_has_been_printed = TRUE;
      buffer_ptr = copy_string (buffer_ptr, 
				rule_sys->strings 
				+ rule_sys->rules[*rule].name,
				buffer_end);
    }
      
    /* Print default rules. */
    for (rule++; *rule != -1; rule++)
    {
      buffer_ptr = copy_string (buffer_ptr, "else ", buffer_end);
      buffer_ptr = copy_string (buffer_ptr,
				rule_sys->strings 
				+ rule_sys->rules[*rule].name,
				buffer_end);
    }
  }

  return buffer;
}

/*---------------------------------------------------------------------------*/
