/* File "rules.h":
 * All definitions needed for the execution of a rule. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de */ 

/* These variables must be set BEFORE "execute_rule" is called. =============*/

GLOBAL void (*add_end_state) (value_t cat);
/* Add a state, consisting of <cat>, as an end state. */

GLOBAL void (*add_running_state) (value_t cat, long_t rule_set);
/* Add a running state, consisting of <cat> and <rule_set>. */

GLOBAL void (*add_allo) (value_t surf, value_t cat);
/* Add an allomorph, consisting of <surf> and <cat>, to the lexicon. */

GLOBAL void (*debug_rule) (void);
/* Called from "execute_rule" before instruction at <pc> is executed. */

GLOBAL value_t (*transmit_function) (value_t argument);
/* Called when "FUNC_TRANSMIT" is executed. */

/*===========================================================================*/

/* These values are used by "execute_rule".
 * but they are global to support debuggers and error messages. */
extern bool_t executing_rule;               /* read only */
extern long_t pc;                           /* read only */
extern long_t nested_subrules;              /* read only */
extern long_t executed_rule_number;         /* read only */
extern rule_sys_t *executed_rule_sys;       /* read only */
extern long_t backup_top;                   /* read only */

extern rule_sys_t *debug_rule_sys; /* rule_sys to debug or NULL */

GLOBAL bool_t rule_successful; /* read only */
/* indicator for execution of result, accept, or allo statement */

extern void execute_rule (rule_sys_t *rule_sys, 
			  long_t rule_number, 
			  ...);
/* Execute rule <rule_number> in the rule system <rule_sys>.
 * Add the rule's parameters to the parameter list of "execute_rule". */

extern rule_sys_t *read_rule_sys (string_t file_name);
/* Read rule system from file <file_name>.
 * A symbol file must have already been loaded. */

extern void free_rule_sys (rule_sys_t *rule_sys);
/* Free all memory used by <rule_sys>. */

/* debug support functions ==================================================*/

extern value_t inspect_stack (long_t index);
/* Return S[index] or NULL if it is not defined. */

extern long_t inspect_stack_pointer (void);
/* Return value of rule stack pointer. */

extern long_t first_variable_index (rule_sys_t *rule_sys,
				    long_t instr_index);
/* Return the stack index of the first variable that is visible
 * when pc is at <instr_index> in <rule_sys>. */

extern void get_base_and_pc_indexes (long_t index, 
				     long_t *base_index, 
				     long_t *pc_index);
/* If <index> == 0, return the current <pc_index> and the <base_index> of the
 * stack when this rule was called as a subrule. */

extern void source_of_instr (rule_sys_t *rule_sys, 
			     long_t instr_index, 
			     long_t *first_line, 
			     long_t *last_line, 
			     string_t *file_name,
			     string_t *rule_name);
/* Set *<first_line>, *<last_line> and *<file_name> to appropriate values
 * for the statement that has generated the instruction at <instr_index>. */

extern long_t rule_of_instr (rule_sys_t *rule_sys,
			     long_t instr);
/* Return the rule index in <rule_sys> where <instr> is located. */

extern long_t variable_index (rule_sys_t *rule_sys, 
			      string_t var_name, 
			      long_t instr_index);
/* Return the stack index of variable <var_name> at <instr_index>. */

extern string_t variable_at_index (rule_sys_t *rule_sys, 
				   long_t stack_index, 
				   long_t instr_index);
/* Return the name of the variable that is defined at <stack_index>
 * when instruction <instr_index> is executed or NULL if there is none. */

extern string_t rule_set_readable (rule_sys_t *rule_sys, long_t rule_set);
/* Return <rule_set> in <rule_sys> as a readable string.
 * The string is valid only until "rule_set_readable" is called next. */

/* switch support ===========================================================*/

extern void set_switch (symbol_t key, value_t value);
/* Set the switch <key> to <value>. */

extern bool_t start_switches (void);
/* Must be called before "get_next_switch" is called.
 * Returns TRUE iff there are any switches */

extern value_t get_next_switch (symbol_t *key);
/* Return the value of the next switch. Return its key in <*key>.
 * If there is no more switch, return NULL. */

extern void free_switches (void);
/* Free all settings. */
