/* File "values.c":
 * Operations to build, modify, and print Malaga values. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include "basic.h"
#include "pools.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

#include "values.h"

/* the internal structure of values =========================================*/

/* The first cell of a value, its type cell, is used to store the value's type
 * together with some type dependent information, which is an unsigned number
 * less than INFO_MAX.
 * Use the macro TYPE to get the type of a value, and INFO to get the type 
 * dependent information. Use TYPE_CELL to create a type-cell. */
#define CELL_BITS BITS_PER_BYTE * sizeof (cell_t)
#define TYPE_BITS 3
#define INFO_BITS (CELL_BITS - TYPE_BITS)
#define TYPE_MAX ((cell_t) 1 << TYPE_BITS)
#define INFO_MAX ((cell_t) 1 << INFO_BITS)
#define TYPE_MASK ((TYPE_MAX - 1) << INFO_BITS)
#define INFO_MASK (INFO_MAX - 1)
#define TYPE(value) ((*(value)) & TYPE_MASK)
#define INFO(value) ((*(value)) & INFO_MASK)
#define TYPE_CELL(type,info) ((type)|(info))

/* There are five different types of values:
 * symbol, string, list, record and number. */

#define SYMBOL_TYPE ((cell_t) 0 << INFO_BITS) 
/* A value of type SYMBOL_TYPE consists only of a type cell. 
 * Its INFO-value is the code for the symbol. */

#define STRING_TYPE ((cell_t) 1 << INFO_BITS) 
/* A value of type STRING_TYPE consists of its type cell,
 * followed by the actual string. Its INFO-value is the
 * number of characters in the string. The actual string
 * is stored in the subsequent cells, two characters
 * paired in a cell, and terminated by one or two
 * NUL-characters, so that the total number of chars is
 * even. The NUL-chars are NOT part of the string. */

#define LIST_TYPE ((cell_t) 2 << INFO_BITS)
/* A value of type LIST_TYPE consists of its type cell and a subsequent cell,
 * which hold the type and the length of the value, followed by 0 or more
 * values of any type, namely the values that form the list. 
 * The length of a list <value>, that is the number of cells needed to store 
 * all the list's values, is (INFO(value[0]) << CELL_BITS) + value[1]. */

#define RECORD_TYPE ((cell_t) 3 << INFO_BITS)
/* A value of type RECORD_TYPE consists of its type cell and a subsequent cell,
 * which hold the type and the length of the value, followed by 0 or more 
 * pairs of values. 
 * In a pair of values, the first value must be a symbol and is considered as 
 * an attribute of that record. The second value is the value of that
 * attribute, and it can be of any type.
 * The length of a record <value>, that is the number of cells
 * needed to store all the record's value pairs, is computed as
 * (INFO(value) << CELL_BITS) + value[1]. */

#define NUMBER_TYPE ((cell_t) 4 << INFO_BITS)
/* A value of type NUMBER_TYPE consists of its type cell,
 * followed by a implementation-dependent number of cells
 * that contain a C "double" value.
 * Its INFO-value is 0. */

/* Use one of the following predicates to test a value
 * against a special type. */
#define IS_SYMBOL(value) (TYPE (value) == SYMBOL_TYPE)
#define IS_STRING(value) (TYPE (value) == STRING_TYPE)
#define IS_RECORD(value) (TYPE (value) == RECORD_TYPE)
#define IS_LIST(value) (TYPE (value) == LIST_TYPE)
#define IS_NUMBER(value) (TYPE (value) == NUMBER_TYPE)

#define NEXT_VALUE(value)    ((value) + length_of_value (value))
/* Return end of <value>.
 * (this may also be the beginning of the next value in a list) */

#define NEXT_ATTRIB(attrib)  ((attrib) + 1 + length_of_value ((attrib) + 1))
/* Return the next attribute (in a record). */

#define CELLS_PER_NUMBER (sizeof (double) / sizeof (cell_t))
/* the number of cells needed to contain a number value
 * sizeof (double) MUST BE A MULTIPLE OF sizeof (cell_t). */

LOCAL cell_t empty_list[] = {TYPE_CELL (LIST_TYPE, 0), 0};
LOCAL cell_t empty_record[] = {TYPE_CELL (RECORD_TYPE, 0), 0};

/* support functions ========================================================*/

GLOBAL long_t length_of_value (value_t value)
/* Return the length of <value> in cells. */
{
  switch (TYPE (value)) 
  {
  case SYMBOL_TYPE:
    return 1;
    
  case STRING_TYPE:
    return 2 + INFO (value) / sizeof (cell_t);
    
  case LIST_TYPE:
  case RECORD_TYPE:
    return 2 + ((long_t) INFO (value) << CELL_BITS) + value[1];

  case NUMBER_TYPE:
    return 1 + CELLS_PER_NUMBER;
    
  default:
    error ("internal (value has bad type)");
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void copy_cells (value_t destination, value_t source, long_t n)
/* Copy <n> cells of value <source> to <destination>. */
{
  value_t source_end = source + n;

  while (source < source_end)
    *destination++ = *source++;
}

/*---------------------------------------------------------------------------*/

LOCAL void copy_value (value_t destination, value_t source)
/* Copy all cells of value <source> to <destination>. */
{
  value_t source_end = NEXT_VALUE (source);

  while (source < source_end)
    *destination++ = *source++;
}

/* value storage support ====================================================*/

GLOBAL value_t new_value (value_t value)
/* Allocate space for <value> and copy it.
 * Values that are allocated are not deleted 
 * when "clear_value_heap" or "clear_value_heap_except" is called.
 * Use "free" to free the value space. */
{
  value_t new_value;

  new_value = (cell_t *) new_vector (sizeof (cell_t), length_of_value (value));
  copy_value (new_value, value);
  return new_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t copy_value_to_pool (pool_t value_pool, 
				   value_t value, 
				   long_t *index)
/* Copy <value> to the pool <value_pool> and store its index in *<index>. */
{
  value_t new_value;

  new_value = (value_t) get_pool_space (value_pool, length_of_value (value), 
					index);
  copy_value (new_value, value);
  return new_value;
}

/* value heap definitions ===================================================*/


LOCAL cell_t *value_heap = NULL; /* the actual heap */
LOCAL cell_t *value_heap_end = NULL; /* pointer to first free cell in heap */
LOCAL long_t value_heap_size = 30000; /* size of the value heap in cells. */

/*---------------------------------------------------------------------------*/

GLOBAL void set_value_heap_size (int size)
/* Set the value heap size to <size> cells. */
{
  if (value_heap != NULL)
    error ("heap size can't be set after heap has been used");

  value_heap_size = size;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t get_value_heap_size (void)
/* Get the current heap size in cells. */
{
  return value_heap_size;
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_value_heap (void)
/* Free the memory that is allocated by the value heap. */
{
  free (value_heap);
  value_heap = NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL void clear_value_heap (void)
/* Clear all values on the heap. */
{
  value_heap_end = value_heap;
}

/*---------------------------------------------------------------------------*/

LOCAL int compare_value_pointers (const void *key1, const void *key2)
/* Return -1/0/1 when the value <value_pointer1> points to is stored on a
 * lower/same/higher address than the value <value_pointer2> points to. */
{
  value_t *value_pointer1;
  value_t *value_pointer2;

  value_pointer1 = * (value_t **) key1;
  value_pointer2 = * (value_t **) key2;

  if (*value_pointer1 < *value_pointer2)
    return -1; 
  else if (*value_pointer1 > *value_pointer2)
    return 1;
  else
    return 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL void clear_value_heap_except (long_t n, value_t values[])
/* If the value heap is almost full, clear all values from the heap,
 * except the <n> values <values>[0..<n>-1]. */ 
{
  long_t i;
  value_t new_value_end;
  value_t old_value_start;
  value_t new_value_start;
  value_t old_value_end;
  value_t **value_pointer;

  /* Clear the heap only if 3/4 of the heap are occupied. */
  if (value_heap == NULL 
      || 4 * (value_heap_end - value_heap) < 3 * value_heap_size)
    return;

  new_value_end = value_heap;

  /* Copy values if there is at least one value to save. */
  if (n > 0) 
  {
    /* Create a table of pointers to the values. */
    value_pointer = (value_t **) new_vector (sizeof (value_t *), n);
    
    for (i = 0; i < n; i++)
      value_pointer[i] = values + i;
    
    /* Sort pointers according to the address of the value they point to. */
    qsort (value_pointer, n, sizeof (value_t *), compare_value_pointers);
    
    /* Find the first index <i> whose value is on the heap. */
    for (i = 0; i < n; i++)
    {
      if (*value_pointer[i] >= value_heap)
	break;
    }
    
    /* Work on all values on the heap. */
    while (i < n && *value_pointer[i] < value_heap_end) 
    {
      /* Copy the value. */
      old_value_start = *value_pointer[i];
      old_value_end = NEXT_VALUE (old_value_start);
      new_value_start = new_value_end;
      new_value_end = new_value_start + (old_value_end - old_value_start);
      copy_value (new_value_start, old_value_start);
      
      /* Adjust the value address and the addresses of all values
       * that are part of that value. */
      while (i < n && *value_pointer[i] < old_value_end) 
      {
	*value_pointer[i] = (new_value_start 
			     + (*value_pointer[i] - old_value_start));
	i++;
      }
    }
    
    free (value_pointer);
  }
  
  value_heap_end = new_value_end;
}

/*---------------------------------------------------------------------------*/

LOCAL value_t space_for_value (long_t size)
/* Get <size> adjacent free cells on the value heap. */
{
  value_t old_value_heap_end;
  
  if (value_heap == NULL)
  {
    value_heap = new_vector (sizeof (cell_t), value_heap_size);
    value_heap_end = value_heap;
  }
  
  if (size + (value_heap_end - value_heap) > value_heap_size)
    error ("value heap is full");
  
  old_value_heap_end = value_heap_end;
  value_heap_end += size;
  return old_value_heap_end;
}

/*---------------------------------------------------------------------------*/

LOCAL value_t space_for_composed_value (short_t type, long_t length)
/* Allocate <length> cells for a composed value of <type>, set its type cell
 * and return the value. */
{
  value_t new_value = space_for_value (length);
  long_t content_size = length - 2;

  if (content_size >= 1L << (INFO_BITS + CELL_BITS))
    error ("value too big");

  new_value[0] = TYPE_CELL (type, content_size >> CELL_BITS);
  new_value[1] = content_size & ((1L << CELL_BITS) - 1);

  return new_value;
}

/* value operations =========================================================*/
 
GLOBAL symbol_t get_value_type (value_t value)
/* Return the type of <value>. Depending of the type, the result value may be
 * SYMBOL_SYMBOL, STRING_SYMBOL, NUMBER_SYMBOL, 
 * LIST_SYMBOL or RECORD_SYMBOL. */
{
  switch (TYPE (value))
  {
  case SYMBOL_TYPE: return SYMBOL_SYMBOL;
  case STRING_TYPE: return STRING_SYMBOL;
  case NUMBER_TYPE: return NUMBER_SYMBOL;
  case LIST_TYPE: return LIST_SYMBOL;
  case RECORD_TYPE: return RECORD_SYMBOL;
  default:
    error ("illegal type");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL symbol_t value_to_symbol (value_t value)
/* Return <value> as a symbol. It is an error if <value> is no symbol. */
{
  if (! IS_SYMBOL (value))
    error ("value is no symbol");

  return *value;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t value_to_string (value_t string)
/* Return the value of <string> as a C style string. */
{
  if (! IS_STRING (string))
    error ("value is no string");

  return (string_t) (string + 1);
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t string_to_value (string_t string_start, string_t string_end)
/* Return the string starting at <string_start> as a Malaga value.
 * If <string_end> != NULL, it marks the end of the string. */
{
  value_t value, value_end;
  long_t length;
  string_t target_ptr, source_ptr;

  if (string_end == NULL)
    string_end = string_start + strlen (string_start);

  length = string_end - string_start;
  if (length > INFO_MAX - 1)
    error ("string too long to be a value");

  value = space_for_value (2 + length / sizeof (cell_t));
  *value = TYPE_CELL (STRING_TYPE, length);
   
  source_ptr = string_start;
  target_ptr = (string_t) (value + 1);
  value_end = NEXT_VALUE (value);

  while (source_ptr < string_end)
    *target_ptr++ = *source_ptr++;

  while (target_ptr < (string_t) value_end)
    *target_ptr++ = '\0';

  return value;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t concat_string_values (value_t string1, value_t string2)
/* Concat <string1> and <string2> and return the result.
 * <string1> and <string2> must be strings. */
{
  long_t new_length;
  string_t new_string, old_string, string_end;
  value_t new_string_value;

  if (! IS_STRING (string1) || ! IS_STRING (string2))
    error ("concatenation operands must be strings");

  new_length = (long_t) INFO (string1) + (long_t) INFO (string2);

  if (new_length > INFO_MAX - 1)
    error ("strings too long for concatenation");
    
  new_string_value = space_for_value (2 + new_length / sizeof (cell_t));
  *new_string_value = TYPE_CELL (STRING_TYPE, new_length);
    
  /* Join the strings. We do it by hand so it's easier to align. */
  new_string = (string_t) (new_string_value + 1);
    
  old_string = (string_t) (string1 + 1);
  while (*old_string != '\0')
    *new_string++ = *old_string++;
  
  old_string = (string_t) (string2 + 1);
  while (*old_string != '\0')
    *new_string++ = *old_string++;
    
  string_end = (string_t) NEXT_VALUE (new_string_value);
  while (new_string < string_end)
    *new_string++ = '\0';
    
  return new_string_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t build_record (long_t n, value_t values[])
/* Return a record of <n> attributes. The <i>-th attribute-value pair is
 * <values>[2<i>-2] : <values>[2<i>-1]" for <i>=1,..,<n>. */
{
  value_t new_record, v;
  long_t i, j, new_record_length;
  
  /* Check that all attributes are different. */
  for (i = 0; i < n; i++) 
  {
    if (! IS_SYMBOL (values[2*i]))
      error ("attribute must be symbol");
    
    for (j = 0; j < i; j++) 
    {
      if (*values[2*i] == *values[2*j])
	error ("attribute twice in record");
    }
  }

  new_record_length = 2;
  for (i = 0; i < n; i++)
    new_record_length += 1 + length_of_value (values[2*i+1]);

  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
  v = new_record + 2;
  for (i = 0; i < n; i++) 
  {
    *v++ = *values[2*i];
    copy_value (v, values[2*i+1]);
    v = NEXT_VALUE (v);
  }
  
  return new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t join_records (value_t record1, value_t record2)
/* Return a record of all attributes of <record1> and <record2>, and their
 * associated values. If an attribute has different values in <record1> and 
 * <record2>, the value in <record2> will be taken). 
 * <record1> and <record2> must be records. */
{
  value_t v, v1, v2, new_record, record1_end, record2_end;
  long_t new_record_length;

  if (! IS_RECORD (record1) || ! IS_RECORD (record2))
    error ("join operands must be records");

  record1_end = NEXT_VALUE (record1);
  record2_end = NEXT_VALUE (record2);
    
  /* Calculate the space needed. This is the length of the
   * first record plus the length of the second record minus the
   * sum of the length of all attribute-value-pairs in <record1> whose
   * attributes are also in <record2>. */
  new_record_length
    = length_of_value (record1) + length_of_value (record2) - 2;
  for (v1 = record1 + 2; v1 < record1_end; v1 = NEXT_ATTRIB (v1)) 
  {
    for (v2 = record2 + 2; v2 < record2_end; v2 = NEXT_ATTRIB (v2)) 
    {
      if (*v1 == *v2) 
	/* We've discovered two identical attributes */
      { 
	new_record_length -= (1 + length_of_value (v1 + 1));
	break;
      }
    }
  }
  
  /* Allocate a new record value. */
  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
  
  /* Copy the attributes of the first record. If an attribut
   * belongs to both <value1> and <value2>, don't copy its value. */
  v = new_record + 2;
  for (v1 = record1 + 2; v1 < record1_end; v1 = NEXT_ATTRIB (v1)) 
  {
    /* Go through <record2> until we reach end or find same attribute. */
    for (v2 = record2 + 2; v2 < record2_end; v2 = NEXT_ATTRIB (v2)) 
    {
      if (*v1 == *v2)
	break;
    }
    
    if (v2 >= record2_end) 
      /* If did not find the attribute in <record2>,
       * copy the attribute value of <record1>. */
    {
      *v = *v1;
      copy_value (v + 1, v1 + 1);
      v = NEXT_ATTRIB (v);
    }
  }
  
  /* Append the attributes of the second record. */
  copy_cells (v, record2 + 2, length_of_value (record2) - 2);
  
  return new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_attribute (value_t record, symbol_t attribute)
/* Return the value of <attribute> in the record <record> 
 * or NULL if it doesn't exist. */
{
  value_t record_end = NEXT_VALUE (record);
  value_t v;
 
  /* No error when getting an attribute from "nil". */
  if (*record == NIL_SYMBOL)
    return NULL;

  if (! IS_RECORD (record)) 
    error ("can get an attribute value of a record only");
  
  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v)) 
  {
    if (*v == attribute)
      return v + 1;
  }
  
  return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t select_attribute (value_t record, symbol_t attribute)
/* Return the record that contains <attribute> and its value in <record>.
 * <record> must be a record. */
{
  value_t record_end, v, new_record;
  long_t new_record_length;

  if (! IS_RECORD (record))
    error ("can select attributes from record only");

  record_end = NEXT_VALUE (record);

  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v))
  {
    if (*v == attribute)
    {
      new_record_length = 3 + length_of_value (v + 1);
      new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
      new_record[2] = attribute;
      copy_value (new_record + 3, v + 1);
      return new_record;
    }
  }
  return empty_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t select_attributes (value_t record, value_t list)
/* Return the record with all attributes of <record> that are in <list>.
 * <record> must be a record and <list> must be a list of symbols. */
{
  value_t record_end, list_end, v, v1, v2, new_record;
  long_t new_record_length;

  if (! IS_RECORD (record))
    error ("can select attributes from record only");

  if (! IS_LIST (list))
    error ("attribute selection list must be a list");

  record_end = NEXT_VALUE (record);
  list_end = NEXT_VALUE (list);

  /* Check that <value2> is a list with symbols only. */
  for (v2 = list + 2; v2 < list_end; v2 = NEXT_VALUE (v2))
  {
    if (! IS_SYMBOL (v2))
      error ("attribute selection list must contain symbols only");
  }
      
  /* Calculate size of new value */
  new_record_length = 2;
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1))
  {
    for (v2 = list + 2; v2 < list_end; v2++)
    {
      if (*v1 == *v2)
      {
	new_record_length += 1 + length_of_value (v1 + 1);
	break;
      }
    }
  }

  /* We don't create a new record if no attributes are deleted. */
  if (new_record_length == length_of_value (record))
    return record;
      
  /* Allocate and copy new value. */
  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
  v = new_record + 2;
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1))
  {
    for (v2 = list + 2; v2 < list_end; v2++)
    {
      if (*v1 == *v2)
      {
	*v++ = *v1;
	copy_value (v, v1 + 1);
	v = NEXT_VALUE (v);
	break;
      }
    }
  }
  return new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t remove_attribute (value_t record, symbol_t attribute)
/* Remove <attribute> from <record> and return the result.
 * <record> must be a record. */
{
  value_t new_record, record_end, v, v1;
  long_t new_record_length;

  if (! IS_RECORD (record))
    error ("can remove attributes from record only");

  record_end = NEXT_VALUE (record);

  /* Find the attribute that is to be deleted. */
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1)) 
  {
    if (*v1 == attribute) 
    {
      /* Compute its length and get space for the new record. */
      new_record_length = (length_of_value (record)
			   - (length_of_value (v1 + 1) + 1));
      new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
	  
      /* Copy the record. */
      v = new_record + 2;
      for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1)) 
      {
	if (*v1 != attribute) 
	{
	  *v = *v1;
	  copy_value (v + 1, v1 + 1);
	  v = NEXT_ATTRIB (v);
	}
      }
	  
      return new_record;
    }
  }
      
  /* We didn't find the attribute, so return the original record. */
  return record;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t remove_attributes (value_t record, value_t list)
/* Remove all attributes in <list> from <record> and return the result.
 * <list> must be a list of symbols; <record> must be a record. */
{
  value_t v, v1, v2, list_end, record_end, new_record;
  long_t new_record_length;

  if (! IS_RECORD (record))
    error ("can remove attributes from record only");

  if (! IS_LIST (list))
    error ("attribute list must be a list");
  
  list_end = NEXT_VALUE (list);
  record_end = NEXT_VALUE (record);

  /* Check if the list consists of symbols only. */
  for (v2 = list + 2; v2 < list_end; v2++)
  {
    if (! IS_SYMBOL (v2))
      error ("attribute list must contain symbols only");
  }
      
  /* Compute the length of the new record. */
  new_record_length = 2;
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1))
  {
    for (v2 = list + 2; v2 < list_end; v2++)
    {
      if (*v1 == *v2)
	break;
    }
    if (v2 == list_end)
      new_record_length += 1 + length_of_value (v1 + 1);
  }
      
  /* We don't create a new record if no attributes will be deleted. */
  if (new_record_length == length_of_value (record))
    return record;
      
  /* Allocate the new record and copy it. */
  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
  v = new_record + 2;
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1)) 
  {
    for (v2 = list + 2; v2 < list_end; v2++)
    {
      if (*v1 == *v2)
	break;
    }
	
    if (v2 == list_end)
    {
      *v = *v1;
      copy_value (v + 1, v1 + 1);
      v = NEXT_ATTRIB (v);
    }
  }
	  
  return new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t replace_attribute (value_t record, 
				  symbol_t attribute, value_t value)
/* Replace the attribute <attribute> of <record> by <value>.
 * <record> must be a record which contains <attribute>. */
{
  value_t record_end, new_record, v, nv;
  long_t new_record_length;

  if (! IS_RECORD (record))
    error ("value must be a record");

  record_end = NEXT_VALUE (record);

  /* Find the attribute to replace. */
  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v))
  {
    if (*v == attribute)
    {
      new_record_length = (length_of_value (record) +
			   length_of_value (value) - length_of_value (v + 1));
      new_record = space_for_composed_value (RECORD_TYPE, new_record_length);

      /* Copy left part. */
      nv = new_record + 2;
      copy_cells (nv, record + 2, v - (record + 2));

      /* Copy changed attribute. */
      nv += v - (record + 2);
      *nv = attribute;
      copy_value (nv + 1, value);

      /* Copy right part. */
      nv = NEXT_ATTRIB (nv);
      copy_cells (nv, NEXT_ATTRIB (v), record_end - NEXT_ATTRIB (v));

      return new_record;
    }
  }
  error ("missing attribute to replace");
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t build_list (long_t n, value_t elements[])
/* Return a list of the <n> given elements in <elements>. */
{
  value_t new_list, v;
  long_t i, new_list_length;
 
  new_list_length = 2;
  for (i = 0; i < n; i++)
    new_list_length += length_of_value (elements[i]);

  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (i = 0; i < n; i++) 
  {
    copy_value (v, elements[i]);
    v = NEXT_VALUE (v);
  }
  
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t concat_lists (value_t list1, value_t list2)
/* Concat <list1> and <list2> and return the result.
 * <list1> and <list2> must be lists. */
{
  long_t list1_length, list2_length, new_list_length;
  value_t new_list;

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("concatenation operands must be lists");

  list1_length = length_of_value (list1);
  list2_length = length_of_value (list2);
  new_list_length = list1_length + list2_length - 2;
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
    
  /* Copy all elements of the first and the second list. */
  copy_cells (new_list + 2, list1 + 2, list1_length - 2);
  copy_cells (new_list + list1_length, list2 + 2, list2_length - 2);

  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_list_difference (value_t list1, value_t list2)
/* Compute the list difference of <list1> and <list2> and return it.
 * An element that appears <m> times in <list1> and <n> times in <list2> 
 * will appear <m-n> times in the result.
 * <list1> and <list2> must be lists. */
{
  value_t list1_end, list2_end, new_list, v, v1, v2;
  long_t new_list_length, appearances;

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("list difference operands must be lists");

  list1_end = NEXT_VALUE (list1);
  list2_end = NEXT_VALUE (list2);

  /* Calculate the size of the new value. */
  new_list_length = 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    /* Check whether <v1> will be included in the list.
     * It will be included if the ordinal number of its appearence is 
     * higher than the number of appearences in <list2>. */
    
    /* Count appearences in <list1> up to (including) <v1>. */
    appearances = 1;
    for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances++;
    }
    
    /* Subtract appearences in <value2>. */
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances--;
    }
    
    if (appearances > 0)
      new_list_length += length_of_value (v1);
  }
  
  /* We don't create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list1))
    return list1;
  
  /* Allocate and copy new value. */
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    /* Check whether <v1> will be included in the list.
     * It will be included if the ordinal number of its appearence is 
     * higher than the number of appearences in <value2>. */
    
    /* Count appearences in <value1> up to (including) <v1>. */
    appearances = 1;
    for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances++;
    }
    
    /* Subtract appearences in <value2>. */
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances--;
    }
    
    if (appearances > 0)
    {
      copy_value (v, v1);
      v = NEXT_VALUE (v);
    }
  }
  
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_set_difference (value_t list1, value_t list2)
/* Compute the set difference of <list1> and <list2> and return it.
 * Each element of <list1> will be in the result list if it is not 
 * an element of <list2>.
 * <list1> and <list2> must be lists. */
{
  value_t list1_end, list2_end, new_list, v, v1, v2;
  long_t new_list_length;

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("set difference operands must be lists");

  list1_end = NEXT_VALUE (list1);
  list2_end = NEXT_VALUE (list2);

  /* Compute the length of the new list. */
  new_list_length = 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	break;
    }
    
    if (v2 == list2_end)
      new_list_length += length_of_value (v1);
  }
  
  /* No need to create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list1))
    return list1;
  
  /* Allocate the new list and copy it. */
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1)) 
  {
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	break;
    }
    
    if (v2 == list2_end)
    {
      copy_value (v, v1);
      v = NEXT_VALUE (v);
    }
  }
  
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t intersect_lists (value_t list1, value_t list2)
/* Return the list intersection of <list1> and <list2>.
 * An element that occurs <m> times in <list1> and <n> times in <list2>
 * will occur min(<m>, <n>) times in the result. */
{
  value_t new_list, list1_end, list2_end, v1, v2, v;
  long_t new_list_length, appearances;

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("operands for intersection must be lists");

  list1_end = NEXT_VALUE (list1);
  list2_end = NEXT_VALUE (list2);

  /* Calculate the size of the new list. */
  new_list_length = 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    /* Check whether <v1> will be included in the list.
     * It will be included if the ordinal number of its appearence is 
     * not higher than the number of appearences in <list2>. */
      
    /* Count appearences in <list1> up to (including) <v1>. */
    appearances = 1;
    for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances++;
    }
      
    /* Subtract appearences in <list2>. */
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances--;
    }
    
    if (appearances <= 0)
      new_list_length += length_of_value (v1);
  }
  
  /* We don't create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list1))
    return list1;
    
  /* Allocate and copy new value. */
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    /* Check whether <v1> will be included in the list.
     * It will be included if the ordinal number of its appearence is 
     * not higher than the number of appearences in <value2>. */
      
    /* Count appearences in <value1> up to (including) <v1>. */
    appearances = 1;
    for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances++;
    }
      
    /* Subtract appearences in <value2>. */
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances--;
    }
      
    if (appearances <= 0)
    {
      copy_value (v, v1);
      v = NEXT_VALUE (v);
    }
  }
    
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t get_list_length (value_t list)
/* Return the number of elements in <list>. 
 * <list> must be a list. */
{
  long_t elements;
  value_t list_end = NEXT_VALUE (list);
  value_t v;

  if (! IS_LIST (list))
    error ("can get length of a list only");

  elements = 0;
  for (v = list + 2; v < list_end; v = NEXT_VALUE (v))
    elements++;

  return elements;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_element (value_t list, long_t n)
/* Return the <n>-th element of the list <list>,
 * or NULL, if that element doesn't exist.
 * If <n> is positive, elements will be counted from the left border.
 * If it's negative, elements will be counted from the right border. */
{
  value_t list_end = NEXT_VALUE (list);
  value_t v;
  
  /* No error when getting an element from "nil". */
  if (*list == NIL_SYMBOL)
    return NULL;

  if (! IS_LIST (list)) 
    error ("can get an element of a list only");

  if (n < 0)
    n = get_list_length (list) + n + 1;

  if (n <= 0)
    return NULL;

  for (v = list + 2; v < list_end; v = NEXT_VALUE (v)) 
  { 
    n--;
    if (n == 0)
      return v;
  }

  return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t remove_element (value_t list, long_t n)
/* Return the list <list> without element at index <n>.
 * If <n> is positive, the elements will be counted from the left border;
 * if <n> is negative, they will be counted from the right border.
 * If the list contains less than abs(<n>) elements, do nothing. */
{
  value_t list_end, new_list, element, v;
  long_t new_list_length;

  if (! IS_LIST (list))
    error ("can remove an element in a list only");

  list_end = NEXT_VALUE (list);

  /* Find the first/last value in the list that will/won't be copied. */
  element = get_element (list, n);
  if (element == NULL)
    return list;

  new_list_length = length_of_value (list) - length_of_value (element);
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);

  v = new_list + 2;
  copy_cells (v, list + 2, element - (list + 2));
  v += element - (list + 2);
  copy_cells (v, NEXT_VALUE (element), list_end - NEXT_VALUE (element));
 
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t replace_element (value_t list, long_t n, value_t value)
/* Replace the <n>-th element of <list> by <value>.
 * If <n> is negative, count from the right end.
 * <list> must be a list which contains at least <n> Elements. */
{
  value_t new_list, element, nv;
  long_t new_list_length;

  element = get_element (list, n);
  if (element == NULL)
    error ("missing element to replace");

  new_list_length = (length_of_value (list) +
		     length_of_value (value) - length_of_value (element));
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);

  /* Copy left part */
  nv = new_list + 2;
  copy_cells (nv, list + 2, element - (list + 2));

  /* Copy changed element. */
  nv += element - (list + 2);
  copy_value (nv, value);
  
  /* Copy right part. */
  nv = NEXT_VALUE (nv);
  copy_cells (nv, NEXT_VALUE (element), 
	      NEXT_VALUE (list) - NEXT_VALUE (element));
  
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t list_to_set (value_t list)
/* Return the list <list>, but with multiple occurrences of elements deleted.
 * That means, convert the multi-set <list> to a set. */
{
  value_t v1, v2, v, new_list, list_end;
  long_t new_list_length;

  if (! IS_LIST (list))
    error ("can only convert a list to a set");
  
  list_end = NEXT_VALUE (list);

  /* Compute the length of the new list. */
  new_list_length = 2;
  for (v1 = list + 2; v1 < list_end; v1 = NEXT_VALUE (v1))
  {
    /* Check if <v1> already occurred in the list. */
    for (v2 = list + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	break;
    }
    
    if (v2 == v1)
      new_list_length += length_of_value (v1);
  }
  
  /* No need to create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list))
    return list;
	
  /* Allocate the new list and copy it. */
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (v1 = list + 2; v1 < list_end; v1 = NEXT_VALUE (v1))
  {
    /* Check if <v1> already occurred in the list. */
    for (v2 = list + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	break;
    }
    
    if (v2 == v1)
    {
      copy_value (v, v1);
      v = NEXT_VALUE (v);
    }
  }
  
  return new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL double value_to_double (value_t value)
/* Return the value of <value> which must be a number value. */
{
  double number;
  value_t number_ptr;
  short_t i;

  if (! IS_NUMBER (value))
    error ("value is no number");

  number_ptr = (value_t) &number;
  for (i = 0; i < (short_t) CELLS_PER_NUMBER; i++)
    number_ptr[i] = value[i+1];

  return number;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t value_to_long (value_t value)
/* Return the value of <value> which must be an integral number value. */
{
  double number = value_to_double (value);
  long_t result;

  result = (long_t) number;
  if (result != number)
    error ("number too big or not integral");

  return result;
}

/*---------------------------------------------------------------------------*/
GLOBAL value_t double_to_value (double number)
/* Convert <number> to a Malaga value. */
{
  short_t i;
  value_t value, number_ptr;

  number_ptr = (value_t) &number;
  value = space_for_value (1 + CELLS_PER_NUMBER);
  *value = TYPE_CELL (NUMBER_TYPE, 0);
  for (i = 0; i < (short_t) CELLS_PER_NUMBER; i++)
    value[i+1] = number_ptr[i];

  return value;
}

/* type dependent Malaga operations =========================================*/

GLOBAL value_t dot_operation (value_t value1, value_t value2)
/* Perform "."-operation, which depends on the type of the values. */
{
  switch (TYPE (value2))
  {
  case SYMBOL_TYPE:
    return get_attribute (value1, value_to_symbol (value2));
  case NUMBER_TYPE:
    return get_element (value1, value_to_long (value2));
  case LIST_TYPE:
    return get_value_part (value1, value2);
  default:
    error ("in \"<v1> . <v2>\", <v2> must be symbol, number or list");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t plus_operation (value_t value1, value_t value2)
/* Perform "+"-operation, which depends on the type of the values. */
{
  switch (TYPE (value1))
  {
  case STRING_TYPE:
    return concat_string_values (value1, value2);
  case LIST_TYPE:
    return concat_lists (value1, value2);
  case RECORD_TYPE:
    return join_records (value1, value2);
  case NUMBER_TYPE:
    return double_to_value (value_to_double (value1) 
			    + value_to_double (value2));
  default:
    error ("\"+\"-operands must be strings, lists, records, or numbers");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t minus_operation (value_t value1, value_t value2)
/* Perform "-"-operation, which depends on the type of the values. */
{
  switch (TYPE (value1))
  {
  case LIST_TYPE:
    switch (TYPE (value2))
    {
    case NUMBER_TYPE:
      return remove_element (value1, value_to_long (value2)); 
    case LIST_TYPE:
      return get_list_difference (value1, value2);
    default:
      error ("in \"<list> - <value>\", <value> must be number or list");
    }

  case RECORD_TYPE:
    switch (TYPE (value2))
    {
    case SYMBOL_TYPE:
      return remove_attribute (value1, value_to_symbol (value2));
    case LIST_TYPE:
      return remove_attributes (value1, value2);
    default: 
      error ("in \"<record> - <value>\", <value> must be symbol or list");
    }

  case NUMBER_TYPE:
    return double_to_value (value_to_double (value1) 
			    - value_to_double (value2));
  default:
    error ("in \"<v1> - <v2>\", <v1> must be list, record, or number");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t asterisk_operation (value_t value1, value_t value2)
/* Perform "*"-operation, which depends on the type of the values. */
{
  switch (TYPE (value1))
  {
  case LIST_TYPE:
    return intersect_lists (value1, value2);
  case RECORD_TYPE:
    switch (TYPE (value2))
    {
    case SYMBOL_TYPE:
      return select_attribute (value1, value_to_symbol (value2));
    case LIST_TYPE:
      return select_attributes (value1, value2);
    default:
      error ("in \"<record> * <value>\", <value> must be symbol or list");
    }
  case NUMBER_TYPE:
    return double_to_value (value_to_double (value1) 
			    * value_to_double (value2));
  default:
    error ("in \"<v1> *<v2>\", <v1> must be list, record, or number");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t slash_operation (value_t value1, value_t value2)
/* Perform "/"-operation, which depends on the type of the values. */
{
  switch (TYPE (value1))
  {
  case LIST_TYPE:
    return get_set_difference (value1, value2);

  case NUMBER_TYPE:
  {
    double divisor;
    
    divisor = value_to_double (value2);
    if (divisor == 0.0)
      error ("division by 0.0");

    return double_to_value (value_to_double (value1) / divisor);
  }

  default:
    error ("\"/\"-operands must be lists or numbers");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t unary_minus_operation (value_t value)
/* Perform unary "-"-operation, which depends on the type of the values. */
{
  if (! IS_NUMBER (value))
    error ("unary-\"-\" operand must be number");

  return double_to_value (- value_to_double (value));
}

/* attribute path functions =================================================*/

GLOBAL value_t build_path (long_t n, value_t elements[])
/* Return a path which contains of  <n> given elements in <elements>. 
 * Any element must be a number, a symbol or a list of numbers and symbols. 
 * If an element is a list, the elements of this list are inserted into the
 * path. */
{
  value_t new_list, v, element_end;
  long_t i, new_list_length;
  
  new_list_length = 2;
  for (i = 0; i < n; i++)
  {
    switch (TYPE (elements[i]))
    {
    case LIST_TYPE:
      element_end = NEXT_VALUE (elements[i]);
      for (v = elements[i] + 2; v < element_end; v = NEXT_VALUE (v))
      {
	if (! IS_SYMBOL (v) && ! IS_NUMBER (v))
	  error ("sublist in path may contain symbols and numbers only");
      }
      new_list_length += length_of_value (elements[i]) - 2;
      break;
      
    case SYMBOL_TYPE:
    case NUMBER_TYPE:
      new_list_length += length_of_value (elements[i]);
      break;
      
    default:
      error ("value path may contain symbols, numbers and lists only");
    }
  }
    
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (i = 0; i < n; i++) 
  {
    if (IS_LIST (elements[i]))
    {
      copy_cells (v, elements[i] + 2, length_of_value (elements[i]) - 2);
      v += length_of_value (elements[i]) - 2;
    }
    else
    {
      copy_value (v, elements[i]);
      v = NEXT_VALUE (v);
    }
  }
  
  return new_list;
}

/*---------------------------------------------------------------------------*/

LOCAL value_t modify_value_part_local (value_t value, 
				       cell_t path[], long_t path_size,
				       value_t new_value,
				       value_t (*modifier) (value_t old_part, 
							    value_t new_value))
/* See "modify_value_part" below for description. This function doesn't take a 
 * <path> value; instead, it takes an array <path> of symbols and numbers
 * and the length of this array in <path_size>. */
{
  if (path_size == 0)
    return modifier (value, new_value);
  else
  {
    value_t subvalue, new_subvalue;
    
    /* Find attribute in <value>. */
    if (IS_SYMBOL (path))
      subvalue = get_attribute (value, *path);
    else if (IS_NUMBER (path))
      subvalue = get_element (value, value_to_long (path));
    else
      error ("path must consist of symbols and numbers");

    if (subvalue == NULL)
      error ("can't find attribute path in value");
    
    /* Go down recursively */
    new_subvalue = modify_value_part_local (subvalue, NEXT_VALUE (path),
					    path_size - length_of_value (path),
					    new_value, modifier);
    if (new_subvalue == subvalue)
      return value;
    else if (IS_SYMBOL (path))
      return replace_attribute (value, *path, new_subvalue);
    else /* IS_NUMBER (path) */
      return replace_element (value, value_to_long (path), new_subvalue);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t modify_value_part (value_t value, value_t path, 
                                  value_t new_value,
				  value_t (*modifier) (value_t old_part, 
						       value_t new_value))
/* Return <value>, but modify the part that is described by <path>.
 * <path> must be a list of symbols and numbers <e1, e2, .. , en>.
 * They will be used as nested attributes and indexes, so the part of <value>
 * that is actually modified is <value>.<e1>.<e2>..<en>. If this part does not
 * exist, an error will be reported. Else the function <modifier> will be 
 * called with this part and <new_value> as arguments.
 * The value returned by <modifier> will be entered in <value> in place of
 * <old_part>. */
{
  if (! IS_LIST (path))
    error ("path must be a list");

  return modify_value_part_local (value, path + 2, length_of_value (path) - 2, 
				  new_value, modifier);
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t modifier_replace (value_t old_part, value_t new_value)
/* Modifier for "modify_value_part" that replaces <old_part> by <new_value>. */
{
  return new_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_value_part (value_t value, value_t path)
/* Return the value part of <value> that is specified by the path <path>. 
 * If that value part doesn't exist, return NULL. */
{
  value_t component;
  value_t path_end = NEXT_VALUE (path);
  
  if (! IS_LIST (path))
    error ("path must be a list");

  for (component = path + 2; component < path_end; 
       component = NEXT_VALUE (component))
  {
    if (IS_SYMBOL (component))
      value = get_attribute (value, *component);
    else if (IS_NUMBER (component))
      value = get_element (value, value_to_long (component));
    else
      error ("path must contain symbols and numbers only");

    if (value == NULL)
      return NULL;
  }
  
  return value;
}

/* functions for list/record iteration ======================================*/

GLOBAL value_t first_element (value_t value)
/* If <value> is a list, return the first element.
 * If <value> is a record, return the first attribute.
 * Return NULL if such an element doesn't exist. */
{
  long_t number;

  if (*value == NIL_SYMBOL)
    return NULL;

  switch (TYPE (value))
  {
  case RECORD_TYPE:
  case LIST_TYPE:

    /* Return NULL if list or record is empty. */
    if (value + 2 >= NEXT_VALUE (value))
      return NULL;
    else
      return value + 2;

  case NUMBER_TYPE:
    number = value_to_long (value);
    if (number > 0)
      return double_to_value (1.0);
    else if (number < 0)
      return double_to_value (-1.0);
    else
      return NULL;

  default:
    error ("can iterate on lists, records and numbers only");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t next_element (value_t value, value_t element)
/* Return successor of <element> in <value> or NULL, if no successor exists.
 * <value> must be a list or a record and <element> must be an element or
 * attribute that has been extracted of <value> either by 
 * "first_element" or by "next_element". */
{
  long_t number, max_number;
  value_t value_end;

  value_end = NEXT_VALUE (value);

  switch (TYPE (value))
  {
  case RECORD_TYPE:
  case LIST_TYPE:

    if (element == NULL)
      return NULL;

    if (IS_LIST (value))
      element = NEXT_VALUE (element);
    else
      element = NEXT_ATTRIB (element);

    if (element < value_end)
      return element;
    else
      return NULL;

  case NUMBER_TYPE:
    max_number = value_to_long (value);
    number = value_to_long (element);

    if (max_number > 0 && number < max_number)
      return double_to_value (number + 1);
    else if (max_number < 0 && number > max_number)
      return double_to_value (number - 1);
    else
      return NULL;

  default:
    error ("can iterate on lists, records and numbers only");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t value_from_element_onward (value_t value, value_t element)
/* Return the list/record of all elements/attributes of <value> from <element>
 * onward. If <element> is NULL, return an empty list/record.
 * If <value> is a list, <element> must be NULL or an element of it.
 * If <value> is a record, <element> must be NULL or an attribute of it. */
{
  value_t new_value;
  value_t value_end = NEXT_VALUE (value);
  long_t new_value_length;

  if (! IS_LIST (value) && ! IS_RECORD (value))
    error ("can create subvalue of a list or a record only.");

  /* Return the empty list/record if <element> == NULL */
  if (element == NULL)
  {
    if (IS_LIST (value))
      return empty_list;
    else
      return empty_record;
  }
  
  new_value_length = value_end - element + 2;
  new_value = space_for_composed_value (TYPE (value), new_value_length);
  copy_cells (new_value + 2, element, new_value_length - 2);

  return new_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t value_without_element (value_t value, value_t element)
/* Return the value <value> without its element <element>.
 * If <value> is a list, <element> must be one of its elements.
 * If <value> is a record, <element> must be one of its attributes.
 * If <element> is NULL, the original list/record is returned. */
{
  value_t new_value, v, nv, value_end;
  long_t new_value_length;

  value_end = NEXT_VALUE (value);
  if (! IS_LIST (value) && ! IS_RECORD (value))
    error ("can delete an element in a list/record only");

  if (element == NULL)
    return value;

  DB_ASSERT (element > value && element < value_end);

  /* Calculate length of new value. */
  if (IS_LIST (value))
    new_value_length = length_of_value (value) - length_of_value (element);
  else /* if (IS_RECORD (value)) */
    new_value_length = (length_of_value (value) 
			- (1 + length_of_value (element + 1)));
  
  new_value = space_for_composed_value (TYPE (value), new_value_length);
  
  /* Copy all elements but <element> to the new list. */
  nv = new_value + 2;
  for (v = value + 2; v < value_end; v = NEXT_VALUE (v)) 
  {
    if (v != element) 
    {
      copy_value (nv, v);
      nv = NEXT_VALUE (nv);
    }
    else if (IS_RECORD (value))
      /* In a record, also omit the value of the attribute. */
      v = NEXT_VALUE (v);
  }
 
  return new_value;
}

/* functions for handling symbols ===========================================*/

LOCAL int compare_symbols (const void *symbol1, const void *symbol2)
/* Return -1 if *<symbol1> <  *<symbol2>,
 *         0 if *<symbol1> == *<symbol2>,
 *         1 if *<symbol1> >  *<symbol2>. */
{
  const symbol_t sym1 = *((symbol_t *) symbol1);
  const symbol_t sym2 = *((symbol_t *) symbol2);

  if (sym1 < sym2) 
    return -1;
  else if (sym1 > sym2)
    return 1;
  else
    return 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t canonise_atom_list (value_t atom_list)
/* Convert the list of <atom_list> to canonical format. */
{
  value_t atom, symbols;
  value_t atom_list_end = NEXT_VALUE (atom_list);

  /* Check <atom_list> for right format. */
  if (! IS_LIST (atom_list))
    error ("atom list is no list");
  for (atom = atom_list + 2; atom < atom_list_end; atom++)
  {
    if (! IS_SYMBOL (atom))
      error ("atom list may only contain symbols");
  }

  /* Copy <atom list> to heap. */
  symbols = space_for_value (atom_list_end - atom_list);
  copy_value (symbols, atom_list);

  /* Sort the symbols with ascending symbol codes. */
  qsort (symbols + 2, length_of_value (symbols) - 2, sizeof (symbol_t), 
	 compare_symbols);

  return symbols;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t symbol_to_atoms (symbol_t symbol)
/* Return the list of atoms for <symbol>. */
{
  value_t atoms;
  
  /* Look for the symbol in the symbol table. */
  atoms = values_atoms_of_symbol (symbol);
  if (atoms == NULL) 
  {
    atoms = space_for_composed_value (LIST_TYPE, 3);
    atoms[2] = symbol;
  }

  return atoms;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t symbols_to_list (long_t n, symbol_t symbol_list[])
/* Convert <symbol_list>, consisting of <n> symbols, to a value. */
{
  value_t new_value;

  new_value = space_for_composed_value (LIST_TYPE, n + 2);
  copy_cells (new_value + 2, symbol_list, n);
  return new_value;
}

/* functions to compare values ==============================================*/

GLOBAL bool_t values_equal (value_t value1, value_t value2)
/* Return a truth value indicating whether <value1> and <value2> are equal.
 * <value1> an <value2> must be of same type.
 * Refer to documentation to see what "equal" in Malaga really means. */
{
  if (TYPE (value1) != TYPE (value2))
  {
    if (*value1 != NIL_SYMBOL && *value2 != NIL_SYMBOL)
      error ("can compare values of same type only");
    
    return FALSE;
  }

  switch (TYPE (value1)) 
  {
  case SYMBOL_TYPE:
    return (*value1 == *value2);
    
  case STRING_TYPE:
    return (strcmp_no_case ((string_t) (value1 + 1), 
			    (string_t) (value2 + 1)) == 0);

  case LIST_TYPE:
  {
    value_t value1_end = NEXT_VALUE (value1);
    value_t value2_end = NEXT_VALUE (value2);
    
    /* Look for each value pair if they are equal. */ 
    value1 += 2;
    value2 += 2;
    while (value1 < value1_end && value2 < value2_end) 
    {
      if (! values_equal (value1, value2))
	return FALSE;
      
      value1 = NEXT_VALUE (value1);
      value2 = NEXT_VALUE (value2);
    }

    if (value1 == value1_end && value2 == value2_end)
      return TRUE;
    else
      return FALSE;
  }
  
  case RECORD_TYPE:
  {
    value_t value1_end = NEXT_VALUE (value1);
    value_t value2_end = NEXT_VALUE (value2);
    value_t v1, v2;
    
    /* Do the records have the same length? */
    if (length_of_value (value1) != length_of_value (value2))
      return FALSE;
    
    /* Check whether for every attribute in <value1>, there is one
     * in <value2> and that their values are equal. */
    for (v1 = value1 + 2; v1 < value1_end; v1 = NEXT_ATTRIB (v1))
    {
      /* Look for the same attribute in <value2>. */
      for (v2 = value2 + 2; v2 < value2_end; v2 = NEXT_ATTRIB (v2)) 
      {
	if (*v1 == *v2)
	  break;
      }
      
      if (v2 == value2_end || ! values_equal (v1 + 1, v2 + 1)) 
	/* We looked 'till end of value2 and didn't find attribute.
	 * Or they don't have the same values. */
	return FALSE;
    }
    
    return TRUE;
  }
  
  case NUMBER_TYPE:
    return (value_to_double (value1) == value_to_double (value2));
    
  default:
    error ("internal (value has bad type)");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t values_congruent (value_t value1, value_t value2)
/* Return a truth value indicating whether <value1> and <value2> have
 * at least one element in common.
 * <value1> and <value2> must both be symbols or lists. */
{
  value_t value1_end, value2_end, v1, v2;

  if (TYPE (value1) != TYPE (value2))
    error ("for congruency test, values must be of same type");

  if (IS_SYMBOL (value1))
  {
    value1 = symbol_to_atoms (value_to_symbol (value1));
    value2 = symbol_to_atoms (value_to_symbol (value2));
  }
  else if (! IS_LIST (value1))
    error ("for congruency test, values must be lists or symbols");

  value1_end = NEXT_VALUE (value1);
  value2_end = NEXT_VALUE (value2);
    
  /* Look for a common element. */
  for (v1 = value1 + 2; v1 < value1_end; v1 = NEXT_VALUE (v1)) 
  {
    for (v2 = value2 + 2; v2 < value2_end; v2 = NEXT_VALUE (v2)) 
    {
      if (values_equal (v1, v2))
	return TRUE;
    }
  }
  
  /* No common symbol found. */
  return FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t value_in_value (value_t value1, value_t value2)
/* Return bool value saying if <value1> is element or attribute of <value2>.
 * <value2> must be a list or a record.
 * If <value2> is a record, then <value1> must be a symbol. */
{
  value_t value2_end = NEXT_VALUE (value2);

  if (IS_LIST (value2)) 
  {
    for (value2 += 2; value2 < value2_end; value2 = NEXT_VALUE (value2)) 
    {
      if (values_equal (value1, value2))
	return TRUE;
    }
  }
  else if (IS_RECORD (value2)) 
  {
    if (! IS_SYMBOL (value1))
      error ("only attributes can be found in a record using \"in\"");
    
    for (value2 += 2; value2 < value2_end; value2 = NEXT_ATTRIB (value2)) 
    {
      if (*value1 == *value2)
	return TRUE;
    }
  }
  else 
    error ("can use \"in\" with records and lists only");

  return FALSE;
}

/* functions to print values ================================================*/

GLOBAL attribute_order_t attribute_order = ALPHABETIC_ORDER;

LOCAL string_t attribute_as_text (string_t output, 
				  value_t attribute, 
				  string_t output_end,
				  bool_t full_value)
/* Print <attribute> and its value (behind attribute). */
{
  bool_t attribute_hidden;
  long_t i;

  /* Is <attribute> a hidden attribute? */
  attribute_hidden = FALSE;
  if (! full_value)
  {
    for (i = 0; i < num_hidden_attributes; i++) 
    {
      if (*attribute == hidden_attributes[i])
      {
	attribute_hidden = TRUE;
	break;
      }
    }
  }

  if (attribute_hidden) 
  {
    output = copy_string (output, "(", output_end);
    output = value_as_text (output, attribute, output_end, full_value);
    output = copy_string (output, ")", output_end);
  } 
  else 
  {
    output = value_as_text (output, attribute, output_end, full_value);
    output = copy_string (output, ": ", output_end);
    output = value_as_text (output, attribute + 1, output_end, full_value);
  }

  return output;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t value_as_text (string_t output, 
			       value_t value, 
			       string_t output_end,
			       bool_t full_value)
/* Convert <value> to a format readable for humans and store it in <output>
 * which extends to <output_end> (this is a pointer to the first byte after
 * <output>. The pointer returned points to the EOS of the built string.
 * If <full_value> == TRUE, show all attributes, even those that are hidden. */
{
  value_t value_end;

  /* Return an empty string. */
  if (value == NULL)
    return copy_string (output, "", output_end);

  value_end = NEXT_VALUE (value);

  switch (TYPE (value)) 
  {
  case SYMBOL_TYPE:
    output = copy_string (output, values_symbol_name (*value), output_end);
    break;

  case STRING_TYPE:
    output = copy_string (output, "\"", output_end);
    output = copy_string_readable (output, 
				   DECODED_STRING ((string_t) (value + 1)), 
				   output_end, NULL);
    output = copy_string (output, "\"", output_end);
    break;

  case LIST_TYPE: 
    output = copy_string (output, "<", output_end);
    value += 2;
    
    /* Print elements. */
    while (value < value_end) 
    {
      output = value_as_text (output, value, output_end, full_value);
      value = NEXT_VALUE (value);
      if (value < value_end)
	output = copy_string (output, ", ", output_end);
    }
    
    output = copy_string (output, ">", output_end);
    break; 
    
  case RECORD_TYPE: 
    output = copy_string (output, "[", output_end);

    switch (attribute_order)
    {
      
    case INTERNAL_ORDER:
    {
      value_t attribute;
      
      for (attribute = value + 2; 
	   attribute < value_end;
	   attribute = NEXT_ATTRIB (attribute))
      {
	if (attribute > value + 2)
	  output = copy_string (output, ", ", output_end);
	output = attribute_as_text (output, attribute, output_end, full_value);
      } 
      break;
    }

    case ALPHABETIC_ORDER:
    {
      string_t last_name; /* last attribute name that has been printed */
      
      last_name = NULL;
      while (TRUE) 
      {
	value_t attrib, next_attrib;
	string_t name, next_name; /* names of the above attributes */

	/* Find the next attribute to be printed. */
	next_attrib = NULL;
	next_name = NULL;
	for (attrib = value + 2; 
	     attrib < value_end; 
	     attrib = NEXT_ATTRIB (attrib)) 
	{
	  name = values_symbol_name (*attrib);
	  if ((last_name == NULL || strcmp_no_case (name, last_name) > 0)
	      && (next_name == NULL || strcmp_no_case (name, next_name) < 0))
	  {
	    next_attrib = attrib;
	    next_name = name;
	  }
	}
	
	if (next_attrib == NULL)
	  break;
	
	if (last_name != NULL)
	  output = copy_string (output, ", ", output_end);
	
	output = attribute_as_text (output, next_attrib, output_end, 
				    full_value);	
	last_name = next_name; 
      }
      break;
    }

    case DEFINITION_ORDER:
    {
      symbol_t last_symbol; /* last attribute symbol that has been printed */
      
      last_symbol = INFO_MAX;
      while (TRUE) 
      {
	value_t attrib, next_attrib;
	
	/* Find the next attribute to be printed. */
	next_attrib = NULL;
	for (attrib = value + 2; 
	     attrib < value_end; 
	     attrib = NEXT_ATTRIB (attrib)) 
	{
	  if ((last_symbol == INFO_MAX || *attrib > last_symbol)
	      && (next_attrib == NULL || *attrib < *next_attrib))
	    next_attrib = attrib;
	}
	
	if (next_attrib == NULL)
	  break;
	
	if (last_symbol != INFO_MAX)
	  output = copy_string (output, ", ", output_end);
	
	output = attribute_as_text (output, next_attrib, output_end, 
				    full_value);
	last_symbol = *next_attrib;
      }
      break;
    }
    }

    output = copy_string (output, "]", output_end);
    break;
    
  case NUMBER_TYPE:
  {
    char string_buffer[25];
    
    sprintf (string_buffer, "%.11G", value_to_double (value));
    output = copy_string (output, string_buffer, output_end);
    break;
  }

  default:
    error ("internal (value has bad type)");
  }
  
  return output;
}

/*---------------------------------------------------------------------------*/

#define PRINT_BUFFER_SIZE 150000 /* maximum size of a printed Malaga value */

GLOBAL short_t fprint_value (FILE *stream, value_t value)
/* Print <value> in a format readable for humans on <stream>. */
{
  char print_buffer[PRINT_BUFFER_SIZE];

  value_as_text (print_buffer, value, print_buffer + PRINT_BUFFER_SIZE, FALSE);
  return fprintf (stream, "%s", print_buffer);
}

/*---------------------------------------------------------------------------*/

GLOBAL short_t print_value (value_t value)
/* Print <value> in a format readable for humans on stdout. */
{
  return fprint_value (stdout, value);
}

/*---------------------------------------------------------------------------*/
