# File "path.tcl":
# Part of "display.tcl", displays a single analysis path.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#------------------------------------------------------------------------------

# Draw the selected path or path piece with categories displayed as
# matrices.

proc path_show {start_node node} {

  global state font_size font char_set

  set char_height $font_size(path)
  set start_x 15
  set canvas .path.frame.canvas
  
  if {$start_node == -1 || $node == $start_node} {
    # initial node -- initialize local variables
    set current_y 15
    set max_x $start_x
    set path_counter 0
    set first_node $node
    while {$first_node != 0} {
      set first_node $state($first_node,ancestor)
      incr path_counter
    }
  } else {
    # not initial node -- print previous nodes recursively
    
    set x_y_counter [path_show $start_node $state($node,ancestor)]
    set max_x [lindex $x_y_counter 0]
    set current_y [lindex $x_y_counter 1]
    set path_counter [expr [lindex $x_y_counter 2] + 1]
    
    # a little vertical space between rule applications
    set current_y [expr $current_y + $char_height / 2]
  }
  
  if {$node != $start_node} {
    # Print next surface and category.
    set surface $state($node,next_surf)
    set value $state($node,next_cat)
    if {$value != ""} {
      
      # segment index
      set counter_id [.path.frame.canvas create matrix \
		      $start_x $current_y "! \"$path_counter: \"" \
		      -font $font(path) \
		      -tags path_tag]
      set counter_y [widget_bottom $canvas $counter_id]
      set counter_x [widget_right $canvas $counter_id]
      
      # next surface
      set string_id [.path.frame.canvas create matrix \
		     $counter_x $current_y "\"$surface\"" \
		     -tags path_tag \
		     -char_set $char_set \
		     -font $font(path)]
      set max_x [maximum $max_x [widget_right $canvas $string_id]]
      
      # next category
      set matrix_id [.path.frame.canvas create matrix \
		     $counter_x [widget_bottom $canvas $string_id] $value \
		     -tags path_tag \
		     -char_set $char_set \
		     -font $font(path)]
      set max_x [maximum $max_x [widget_right $canvas $matrix_id]]
      set current_y [widget_bottom $canvas $matrix_id]
      
      # center counter vertically
      $canvas move $counter_id 0 [expr ($current_y - $counter_y) / 2]
    }
  }
  
  if {$start_node != -1} {
    
    # Only print rule name if this is not the start node.
    if {$node != $start_node} {
      set rule_name $state($node,rule)
    } else {set rule_name ""}
    
    if {$rule_name != ""} {
      
      # Insert a small vertical space
      set current_y [expr $current_y + $char_height / 2]
      
      # rule name (without arrow yet)
      set rule_id [.path.frame.canvas create matrix \
		   $start_x $current_y "! \"$rule_name \"" \
		   -tags path_tag \
		   -char_set $char_set \
		   -font $font(path)]
      set rule_name_x [widget_right $canvas $rule_id]
      set rule_name_y [widget_bottom $canvas $rule_id]
      set current_x [expr $rule_name_x + $char_height/2]
      
    } else {set current_x $start_x}
    
    # result surface
    set surface $state($node,result_surf)
    set value $state($node,result_cat)
    if {$value != ""} {
      set string_id [.path.frame.canvas create matrix \
		     $current_x $current_y "\"$surface\"" \
		     -tags path_tag \
		     -char_set $char_set \
		     -font $font(path)]
      set max_x [maximum $max_x [widget_right $canvas $string_id]]
      
      # result category
      set matrix_id [.path.frame.canvas create matrix \
		     $current_x [widget_bottom $canvas $string_id] $value \
		     -tags path_tag \
		     -char_set $char_set \
		     -font $font(path)]
      set max_x [maximum $max_x [widget_right $canvas $matrix_id]]
      set current_y [widget_bottom $canvas $matrix_id]
      
      # rule set
      set string_id [.path.frame.canvas create matrix \
		     $current_x $current_y "! \"$state($node,rule_set)\"" \
		     -tags path_tag \
		     -char_set $char_set \
		     -font $font(path)]
      set max_x [maximum $max_x [widget_right $canvas $string_id]]
      set current_y [widget_bottom $canvas $string_id]
    }
    
    if {$rule_name != ""} {
      
      # center rule name vertically
      $canvas move $rule_id 0 [expr ($current_y - $rule_name_y) / 2]
      set rule_name_y [expr $rule_name_y + ($current_y - $rule_name_y) / 2]
      
      # draw rule arrow
      $canvas create line \
      $start_x $rule_name_y $rule_name_x $rule_name_y \
      -width 2 \
      -arrow last \
      -fill black \
      -tags path_tag
    }
  }

  return [list $max_x $current_y $path_counter]

}

#------------------------------------------------------------------------------

proc path_close_window {} {

  global selected_start_node path_geometry
  
  if [winfo exists .path] {
    if [info exists selected_start_node] {
      tree_mark_path false
      unset selected_start_node
    }
    set path_geometry [winfo geometry .path]
    destroy .path
  }
}

#------------------------------------------------------------------------------

proc path_open_window {} {

  global path_geometry
  
  if [winfo exists .path] {return}
  
  if {! [info exists path_geometry]} {set path_geometry 600x600}
  
  toplevel .path
  wm minsize .path 100 100
  wm geometry .path $path_geometry
  wm protocol .path WM_DELETE_WINDOW path_close_window
  wm iconname .path "Path"
  wm focusmodel .path active

  frame .path.menu -relief raised -borderwidth 1
  pack .path.menu -side top -fill x
  create_window_menu path
  create_font_menu path
  create_result_menu path
  
  create_scroll_frame path
  focus .path.frame.canvas
}

#------------------------------------------------------------------------------

proc path_redraw {mode} {

  global selected_start_node selected_end_node
  
  path_open_window
  
  if {$selected_start_node == $selected_end_node} {
    wm title .path "Malaga Analysis State \#$selected_start_node"
  } elseif {$selected_start_node == -1} {
    wm title .path "Malaga Analysis Category"
  } else {wm title .path "Malaga Analysis Path"}

  .tree.frame.canvas configure -cursor watch
  .path.frame.canvas configure -cursor watch

  .path.frame.canvas delete path_tag

  set x_y [path_show $selected_start_node $selected_end_node]

  .path.frame.canvas configure \
  -scrollregion "0 0 [expr [lindex $x_y 0] + 15] [expr [lindex $x_y 1] + 15]"

  .path.frame.canvas configure -cursor ""
  .tree.frame.canvas configure -cursor ""
}

#------------------------------------------------------------------------------

proc path_display {start_node end_node} {

  global selected_start_node selected_end_node

  tree_mark_path false
  
  # set selected path
  set selected_start_node $start_node
  set selected_end_node $end_node

  # Highlight new path in tree
  tree_mark_path true
  
  path_redraw new_mode

  if {[wm state .path] == "normal"} {raise .path}
}

#------------------------------------------------------------------------------

proc path_display_first_result {} {
  
  global final_nodes

  set length [llength $final_nodes]
  if {[llength $final_nodes] > 0} {
    set first_node [lindex $final_nodes 0]
    path_display $first_node $first_node
  }
}

#------------------------------------------------------------------------------

proc path_display_last_result {} {

  global final_nodes
  
  set length [llength $final_nodes]
  if {$length > 0} {
    set last_node [lindex $final_nodes [expr $length - 1]]
    path_display $last_node $last_node
  }
}

#------------------------------------------------------------------------------

# Display the next result in the tree, if there is any.
# A result must already be selected.

proc path_display_next_result {} {

  global final_nodes selected_start_node

  set length [llength $final_nodes]
  if [info exists selected_start_node] {
    set index [lsearch $final_nodes $selected_start_node]
    if {$index != -1 && $index < $length - 1} {
      set last_node [lindex $final_nodes [expr $index + 1]]
      path_display $last_node $last_node
    }
  }
}

#------------------------------------------------------------------------------

# Display the previous result in the tree, if there is any.
# A result must already be selected.

proc path_display_previous_result {} {
  
  global final_nodes selected_start_node

  if [info exists selected_start_node] {
    set index [lsearch $final_nodes $selected_start_node]
    if {$index != -1 && $index > 0} {
      set last_node [lindex $final_nodes [expr $index - 1]]
      path_display $last_node $last_node
    }
  }
}

#------------------------------------------------------------------------------
