# File "variables.tcl":
# Main module to display Malaga variables.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de

# Global variables:
# $variable_names -- the list of variable names
# $variable_value($name) -- the value of variable $name
# $displayed($name) -- is 1 if variable $name should be displayed, else 0

#------------------------------------------------------------------------------

# Enter all local variables in the variables menu.

proc variables_fill_menu {w} {
  
  global variable_names displayed menu_font
  
  catch {$w delete 0 last}
  foreach name $variable_names {
    $w add checkbutton \
    -label "$name" \
    -variable displayed($name) \
    -font $menu_font \
    -command {variables_display}
  }
}

#------------------------------------------------------------------------------

# Create a menu with all malaga variables currently defined.

proc variables_create_list_menu {} {

  global menu_font displayed variable_names
    
  # Variable button:
  menubutton .variables.menu.variable \
  -text " Variables " -font $menu_font -menu .variables.menu.variable.m
  menu .variables.menu.variable.m
  
  .variables.menu.variable.m add cascade \
  -label " Show selected variables " \
  -font $menu_font \
  -accelerator => \
  -menu .variables.menu.variable.m.vars
    
  menu .variables.menu.variable.m.vars \
  -postcommand [list variables_fill_menu .variables.menu.variable.m.vars]
    
  # Draw all variables:
  .variables.menu.variable.m add command \
  -label " Show all variables " \
  -font $menu_font \
  -accelerator "Control+a " \
  -command {
    foreach name $variable_names {set displayed($name) 1}
    variables_display
  }
  bind .variables <Control-a> {
    foreach name $variable_names {set displayed($name) 1}
    variables_display
  }
  bind .variables <Control-n> {
    foreach name $variable_names {set displayed($name) 0}
    variables_display
  }
    
  # Delete all variables:
  .variables.menu.variable.m add command \
  -label " Show no variables " \
  -font $menu_font \
  -accelerator "Control+n " \
  -command {
    foreach name $variable_names {set displayed($name) 0}
    variables_display
  }
    
  pack .variables.menu.variable -side left
}

#------------------------------------------------------------------------------

proc variables_display {} {
    
  global font font_size variable_names displayed char_set variable_value
  
  .variables.frame.canvas delete matrix_tag
  
  set canvas_border 10
  set char_height $font_size(variables)
  set current_y $canvas_border
  set start_x $canvas_border
  set max_x $start_x
  
  # Run through the variables list:
  set first_variable 1
  foreach name $variable_names {
    
    if {$displayed($name)} {
      
      # Space between variables
      if {! $first_variable} {incr current_y $char_height}
      
      # Display variable name
      set name_id [.variables.frame.canvas create matrix \
		   $start_x $current_y \
		   "! \"$name = \"" \
		   -font $font(variables) \
		   -char_set $char_set \
		   -tags matrix_tag]
      
      # Display variable content
      set matrix_id [.variables.frame.canvas create matrix \
		     [widget_right .variables.frame.canvas $name_id] \
		     $current_y \
		     $variable_value($name) \
		     -font $font(variables) \
		     -char_set $char_set \
		     -tags matrix_tag]
      
      set max_x [maximum $max_x [widget_right \
				 .variables.frame.canvas $matrix_id]]
      set current_y [widget_bottom .variables.frame.canvas $matrix_id]
      
      # Center variable name vertically
      .variables.frame.canvas move $name_id \
      0 [expr ($current_y - [widget_bottom \
			     .variables.frame.canvas $name_id]) / 2]
    
      set first_variable 0
    }
  }
  
  # Reconfigure scroll bars:
  .variables.frame.canvas configure \
  -scrollregion \
  "0 0 [expr $max_x + $canvas_border] [expr $current_y + $canvas_border]"
}

#------------------------------------------------------------------------------

proc variables_open_window {} {
  
  global variables_geometry
  
  if [winfo exists .variables] {return}
  
  if {! [info exists variables_geometry]} {set variables_geometry 600x400}
  
  # Define start window.
  toplevel .variables
  wm title .variables "Malaga Variables"
  wm iconname .variables "Variables"
  wm minsize .variables 100 100
  wm geometry .variables $variables_geometry
  wm focusmodel .variables active
  
  # Define the different menus.
  frame .variables.menu -relief raised -borderwidth 1
  pack .variables.menu -side top -fill x
  create_window_menu variables
  create_font_menu variables
  variables_create_list_menu
  tk_menuBar .variables.menu \
  .variables.menu.window .variables.menu.fonts .variables.menu.variable
  
  create_scroll_frame variables
  focus .variables.frame.canvas
}

#------------------------------------------------------------------------------

proc variables_close_window {} {
  
  global variables_geometry
  
  if [winfo exists .variables] {
    set variables_geometry [winfo geometry .variables]
    destroy .variables
  }
}

#------------------------------------------------------------------------------

proc variables_redraw {mode} {
    
  variables_open_window
  
  if {$mode == "reload"} {variables_read}
  variables_display
  if {$mode == "reload" && [wm state .variables] == "normal"} {
    raise .variables
  }
}

#------------------------------------------------------------------------------

proc variables_read {} {
  
  global variable_names variable_value displayed
  
  set variable_names {}
  
  # Read all variable definitions and set the display state to 1 if not set
  while {[set line [read_line stdin]] != "end" } {
    set var_name [lindex $line 0]
    lappend variable_names $var_name
    set variable_value($var_name) [lindex $line 1]
    if {! [info exists displayed($var_name)]} {set displayed($var_name) 1}
  }
}

#------------------------------------------------------------------------------

proc variables_init {} {

  global font font_size

  set font_size(variables) 14
  set font(variables) $font($font_size(variables))
}

#------------------------------------------------------------------------------
