/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * restrict3rpl_def.c
 */

#include <model_headers.h>

/*
	x' = vx
	y' = vy
	vx' = x + 2*vy - (1-mu) * r1x / r1cube - mu * r2x / r2cube
	vy' = y - 2*vx - (1-mu) * r1y / r1cube - mu * r2y / r2cube

   where
        r1x = x + mu
	r1y = y
	r2x = x - (1-mu)
	r2y = y
	r1sqr = r1x*r1x+r1y*r1y
	r2sqr = r2x*r2x+r2y*r2y
	r1cube = sqrt(r1sqr) * r1sqr
	r2cube = sqrt(r2sqr) * r2sqr


	Translation table:

		x[0] <--> x
		x[1] <--> y
		x[2] <--> vx
		x[3] <--> vy
		p[0] <--> mu
*/

/* function used to define dynamical system */
int 
restrict3rpl_ds_func( double* f, double* x, double* p )
{
	double r1x,r2x,r1sqr,r2sqr,r1cube,r2cube;

	r1x = x[0] + p[0];
	r2x = x[0] - (1.0-p[0]);
	r1sqr = r1x*r1x + x[1]*x[1];
	r2sqr = r2x*r2x + x[1]*x[1];
	r1cube = r1sqr * sqrt(r1sqr);
	r2cube = r2sqr * sqrt(r2sqr);

	f[0] = x[2];
	f[1] = x[3];
	f[2] = x[0] + 2.0*x[3] - (1-p[0]) * r1x / r1cube - p[0] * r2x / r2cube;
	f[3] = x[1] - 2.0*x[2] - (1-p[0]) * x[1] / r1cube - p[0] * x[1] / r2cube;

	return 0;
}

/* function used to define aux functions */
int restrict3rpl_aux_func( double* f, double* x, double* p )
{
  double c,s;

  c = cos(x[4]);
  s = sin(x[4]);

  f[0] = c*x[0]-s*x[1];
  f[1] = s*x[0]+c*x[1];
  return 0;
}

/* function used to define jacobian. NOT AUTOMATICALLY GENERATED.
	input explicit jacobian in the  form
	m[i][j] = d f_i / d x_j; (starting with 0)
*/
/*
int 
restrict3rpl_jac( double** m, double* x, double* p )
{

  return 0;
}
*/
/* function for geomview output */
int 
restrict3rpl_gv( double* f, double* x, double* p )
{
  int i;
  double c,s,xr,yr;

  c = cos(x[4]);
  s = sin(x[4]);
  xr = c*x[0]-s*x[1];
  yr = s*x[0]+c*x[1];

  for (i=0; i<48; i++) f[i] = 0.0;
  
  f[0] = f[5] = f[10] = 1.0;
  /* limit how puch the first body can shrink so that we can still see it */
  if (p[0] <= 0.8) f[15] = 1.0/(1.0-p[0]);
  else f[31] = 5.0;
  f[12] = -c*p[0]*f[15];
  f[13] = -s*p[0]*f[15];

  f[16] = f[21] = f[26] = 1.0;
  /* limit how much the second body can shrink so that we can still see it */
  if (p[0] >= 0.2) f[31] = 1.0/p[0];
  else f[31] = 5.0;
  f[28] = (1.0-p[0])*c*f[31];
  f[29] = (1.0-p[0])*s*f[31];

  f[32] = f[37] = f[42] = f[47] = 1.0;
  f[44] = xr;
  f[45] = yr;

  return 0;  
}

/* function used to define default data */
int 
restrict3rpl_init(void)
{
	int n_varb=4;
	static char *variable_names[]={"x","y","vx","vy"};
	static double variables[]={0.9,0,-0.13,-1.0};
	static double variable_min[]={-2,-2,-2,-2};
	static double variable_max[]={2,2,2,2};

	static char *indep_varb_name="time";
	double indep_varb_min=0;
	double indep_varb_max=10000;

	int n_param=1;
	static char *parameter_names[]={"mu"};
	static double parameters[]={0.3};
	static double parameter_min[]={0.0};
	static double parameter_max[]={1.0};

	int n_funct=2;
	static char *funct_names[]={"xrot","yrot"};
	static double funct_min[]={-10.0,-10.0};
	static double funct_max[]={10.0,10.0};

	int manifold_type=EUCLIDEAN;
	static int periodic_varb[]={FALSE,FALSE,FALSE,FALSE};
	static double period_start[]={0,0,0,0};
	static double period_end[]={1,0,0,0};

	int mapping_toggle=FALSE;
	int inverse_toggle=FALSE;

	int (*def_name)()=restrict3rpl_ds_func;
	int (*jac_name)()=NULL;
	int (*aux_func_name)()=restrict3rpl_aux_func;
	int (*inv_name)()=NULL;
	int (*dfdt_name)()=NULL;
	int (*dfdparam_name)()=NULL;
#define GV
  int (*gv_funct)()=restrict3rpl_gv;
  int (*gv_custom_funct)()=NULL;    /* nontransformation geomview commands */
  int gv_n = 3;
  static char *gv_filename = "restrict3.oogl";

  c_filename = __FILE__;  /* display this file for help with this system */

#include <ds_define.c>

  return 0;
}

