/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#if !lexer_h
#define lexer_h 1

/* Returns nonzero if character CH may be the first character in an
   identifier. */
#define is_id1(CH)							\
	(isalpha ((unsigned char) (CH)) || (CH) == '@' || (CH) == '#'	\
	 || (CH) == '$')

/* Returns nonzero if character CH may be a character in an
   identifier other than the first. */
#define is_idn(CH)					\
	(is_id1 (CH) || (CH) == '.' || (CH) == '_')

/* Token types. */
enum
  {
    /* Basic tokens. */
    ID = 256, NUM, STRING, STOP,

    /* Keywords.  The order is important here. */
    AND, OR, NOT,
    EQ, GE, GT, LE, LT, NE,	/* Do not separate the relational operators. */
    ALL, BY, TO, WITH, EXP,	/* Do not separate the ** operator. */

    /* Tagged quotes. */
    TAGGED_QUOTE
  };

extern int token;
extern double tokval;
extern unsigned char *tokstr;
extern int tokstr_len;
extern int tokstr_size;
extern unsigned char *toklongstr;
extern long int tokint;

/* Main lexer routines. */
int dump_token (void);
int yylex (void);

#if PRODUCTION
#define get_token() (token = yylex ())
#else
#define get_token() (dump_token ())
#endif

int match_id (char *);
int match_tok (int);
int match_int (int);

int syntax_error (const char *, ...);

#if __GNUC__ && !__STRICT_ANSI__
/* Macro version of function in lexer.c. */
#define id_match(KW, TOK)					\
	({							\
	  const char *_kw = (KW);				\
	  const char *_tok = (TOK);				\
	  size_t _klen = strlen (_kw);				\
	  size_t _tlen = strlen (_tok);				\
	  (_klen == _tlen && !memcmp (_kw, _tok, _klen))	\
	    || (_tlen >= 3 && !strncmp (_kw, _tok, _tlen));	\
	})
#endif
    
/* If the current token is the identifier S (specified without
   quotes), skips the token and returns 1; otherwise, returns 0. */
#define match_id(S)							\
	(token == ID && id_match (#S, tokstr) ? (get_token (), 1) : 0)

/* If TOK is the current token, skips it and returns 1; otherwise,
   returns 0. */
#define match_tok(TOK)				\
	(token == (TOK) ? (get_token (), 1) : 0)

/* If the current token is an integer having value N, skips it and
   returns 1; otherwise, returns 0. */
#define match_int(N)						\
	(token == NUM && tokint == (N) ? (get_token (), 1) : 0)
     
/* If the current token is the identifier S (specified without
   quotes), skips the token; otherwise, reports an error and returns
   from the current function with return value 0. */
#define force_match_id(S)					\
	if (token != ID || !id_match (#S, tokstr))		\
	  return syntax_error ("expecting `" #S "'");		\
	else							\
	  get_token ()
	
/* If the current token is T, skips the token; otherwise, reports an
   error and returns from the current function with return value 0. */
#define force_match(T)					\
	if (token != T)					\
	  return syntax_error ("expecting " #T);	\
	else						\
	  get_token ()
	
/* If the current token is not a string, reports an error and returns
   from the current function with return value 0. */
#define force_string()					\
	do						\
	  {						\
	    if (token != STRING)			\
              return syntax_error ("expecting string");	\
	  }						\
	while (0)

/* If the current token is not an integer, reports an error and
   returns from the current function with return value 0. */
#define force_int()						\
	do							\
	  {							\
	    if (token != NUM || tokint == NOT_LONG)		\
              return syntax_error ("expecting integer");	\
	  }							\
	while (0)
	
/* If the current token is not a number, reports an error and returns
   from the current function with return value 0. */
#define force_num()					\
	do						\
	  {						\
	    if (token != NUM)				\
	      return syntax_error ("expecting number");	\
	  }						\
	while (0)
	
/* If the current token is not an identifier, reports an error and
   returns from the current function with return value 0. */
#define force_id() 						\
	do							\
	  {							\
	    if (token != ID)					\
	      return syntax_error ("expecting identifier");	\
	  }							\
	while (0)

int (id_match) (const char *keyword_string, const char *token_string);

#endif /* !lexer_h */
