/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998,1999 Bert Vermeulen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <gtk/gtk.h>
#include <glib.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>

#include "common.h"
#include "configfile.h"
#include "browse.h"
#include "mainwin.h"
#include "query.h"
#include "util.h"
#include "detail.h"
#include "errorchain.h"
#include "ldif.h"
#include "debug.h"

#include "warning.xpm"

extern GtkWidget *statusbar, *mainwin;
extern struct gq_config config;

GNode *root = NULL;
GtkWidget *pane1_scrwin = NULL;
GtkWidget *pane2_scrwin = NULL;
GtkWidget *browsemode_vbox = NULL;

/* these keep track of the edit box in the right pane */
GString *cur_dn = NULL;
struct ldapserver *cur_dn_server = NULL;
int cur_show_values = 0;


GtkWidget *create_mainwin_browsemode(void)
{
     GtkWidget *hbox1;
     GtkWidget *mainpane;
     GtkWidget *pane2_vbox;
     GtkWidget *treeroot;

     browsemode_vbox = gtk_vbox_new(FALSE, 0);
     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), hbox1, FALSE, FALSE, 3);

     mainpane = gtk_hpaned_new();
     gtk_container_border_width(GTK_CONTAINER(mainpane), 2);
     gtk_widget_show(mainpane);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), mainpane, TRUE, TRUE, 0);

     root = g_node_new(new_node_entry("root node"));

     treeroot = gtk_tree_new();
     gtk_signal_connect(GTK_OBJECT(treeroot), "select_child",
			GTK_SIGNAL_FUNC(server_item_selected), NULL);
     gtk_widget_show(treeroot);

     add_all_servers();
     gnode_to_tree(root, treeroot);

     pane1_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane1_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane1_scrwin);
     gtk_widget_set_usize(pane1_scrwin, 300, -1);
     gtk_paned_add1(GTK_PANED(mainpane), pane1_scrwin);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane1_scrwin),
					   treeroot);

     pane2_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane2_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane2_scrwin);
     pane2_vbox = gtk_vbox_new(FALSE, 5);
     gtk_widget_show(pane2_vbox);
     gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
					   pane2_vbox);
     gtk_paned_add2(GTK_PANED(mainpane), pane2_scrwin);

     gtk_widget_show(browsemode_vbox);
     gtk_object_set_data(GTK_OBJECT(browsemode_vbox), "focus", pane2_scrwin);

     cur_dn = g_string_sized_new(128);
     g_string_assign(cur_dn, "some dummy string");

     return(browsemode_vbox);
}


/*
 * add all configured servers to the root node
 */
void add_all_servers(void)
{
     GNode *new_node;
     struct ldapserver *server;
     struct node_entry *entry;
     int server_cnt;
     char message[128];

     server_cnt = 0;
     server = config.ldapservers;
     while(server) {
	  entry = new_node_entry(server->name);
	  entry->status = IS_SERVER;
	  new_node = g_node_new(entry);
	  g_node_append(root, new_node);
	  server = server->next;
	  server_cnt++;
     }

     make_message(message, server_cnt, "server", "servers", "found");
     statusbar_msg(message);

}


/*
 * build gtk_tree from GNode
 */
GtkWidget *gnode_to_tree(GNode *node, GtkWidget *tree)
{

     g_node_children_foreach(node, G_TRAVERSE_ALL,
			     (GNodeForeachFunc) gnode_to_tree_callback,
			     (gpointer) tree);

     return(tree);
}


gboolean gnode_to_tree_callback(GNode *node, gpointer tree)
{
     GtkWidget *new_item, *new_subtree;
     struct node_entry *entry;

     entry = (struct node_entry *) node->data;

     new_item = gtk_tree_item_new_with_label(dn_by_node(node));
     gtk_signal_connect(GTK_OBJECT(new_item), "button_press_event",
			GTK_SIGNAL_FUNC(button_press_on_tree_item), NULL);
     gtk_object_set_data(GTK_OBJECT(new_item), "node", (gpointer) node);
     gtk_widget_show(new_item);
     gtk_tree_append(GTK_TREE(tree), new_item);

     if(node->children) {
	  new_subtree = gtk_tree_new();
	  gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	  gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
			     GTK_SIGNAL_FUNC(edit_right_pane_values), NULL);
	  gnode_to_tree(node, new_subtree);
     }
     else {
	  switch(entry->status) {
	  case STATUS_UNKNOWN:
	       /* insert dummy subtree, connect expand to onelevel search */
	       new_subtree = gtk_tree_new();
	       gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	       gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
				  GTK_SIGNAL_FUNC(edit_right_pane_values), NULL);
	       gtk_signal_connect(GTK_OBJECT(new_item), "expand",
				  GTK_SIGNAL_FUNC(attach_children_to_node),
				  NULL);
	       break;

	  case IS_SERVER:
	       /* insert dummy subtree, connect expand to suffix scanner */
	       new_subtree = gtk_tree_new();
	       gtk_tree_item_set_subtree(GTK_TREE_ITEM(new_item), new_subtree);
	       gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
				  GTK_SIGNAL_FUNC(edit_right_pane_values), NULL);
	       gtk_signal_connect(GTK_OBJECT(new_item), "expand",
				  GTK_SIGNAL_FUNC(attach_server_suffixes),
				  NULL);
	       break;

	  default:
	       break;
	  }
     }

     return(FALSE);
}


/*
 * remove server from visual and gnode trees
 */
void remove_server_node_tree(struct ldapserver *server)
{

     /* this will be called when a server was deleted with
	the preferences dialog */




}


/*
 * an item with a dummy subtree was selected or is about to be expanded. Find out
 * the child nodes for this DN, or (if there are none) delete the subtree
 */
void attach_children_to_node(GtkWidget *item)
{
     GNode *node;
     struct node_entry *entry;

     /* this function only gets called the first time an item's subtree
	is expanded -- should know whether it has children or not when done */
     gtk_signal_disconnect_by_func(GTK_OBJECT(item),
				   GTK_SIGNAL_FUNC(attach_children_to_node),
				   NULL);

     /* connect next expand sig */
     gtk_signal_connect(GTK_OBJECT(item), "expand",
			GTK_SIGNAL_FUNC(attach_children_to_tree), NULL);

     node = (GNode *) gtk_object_get_data(GTK_OBJECT(item), "node");

     set_busycursor();

     descend_onelevel(node);

     set_normalcursor();

     entry = (struct node_entry *) node->data;
     if(entry->status == HAS_CHILDREN) {
	  /* connect next expand sig */
	  gtk_signal_connect(GTK_OBJECT(item), "expand",
			     GTK_SIGNAL_FUNC(attach_children_to_tree), NULL);
     }
     else
	  gtk_tree_item_remove_subtree(GTK_TREE_ITEM(item));

}


/*
 * a subtree that's known to have children in it is about to be expanded.
 * attach_children_to_node() has already run, so just need to attach them to
 * the tree here
 */
void attach_children_to_tree(GtkWidget *item)
{
     GNode *node;
     GtkWidget *subtree;
     struct node_entry *entry;

     gtk_signal_disconnect_by_func(GTK_OBJECT(item),
				   GTK_SIGNAL_FUNC(attach_children_to_tree),
				   NULL);

     node = (GNode *) gtk_object_get_data(GTK_OBJECT(item), "node");

     entry = (struct node_entry *) node->data;
     if(entry->status == HAS_CHILDREN) {
	  subtree = GTK_TREE_ITEM_SUBTREE(GTK_TREE_ITEM(item));
	  gnode_to_tree(node, subtree);
     }

}


/*
 * a server with a dummy subtree is about to be expanded. Run the suffix
 * discovery stuff and attach any suffixes found. If none found, delete
 * the subtree. Which makes the server totally useless :-(
 */
void attach_server_suffixes(GtkWidget *item)
{
     GNode *server_node, *new_node;
     GSList *suffixes, *slist;
     GtkWidget *subtree;
     struct ldapserver *server;

     /* this function only gets called the first time a server is expanded */
     gtk_signal_disconnect_by_func(GTK_OBJECT(item),
				   GTK_SIGNAL_FUNC(attach_server_suffixes),
				   NULL);

     server_node = (GNode *) gtk_object_get_data(GTK_OBJECT(item), "node");
     server = server_by_node(server_node);

     suffixes = get_suffix(server);	 
     if( (slist = suffixes) ) {
	  /* add suffixes to server node */
	  while(slist) {
	       new_node = g_node_new(new_node_entry(slist->data));
	       g_node_append(server_node, new_node);
	       g_free(slist->data);
	       slist = slist->next;
	  }
	  g_slist_free(suffixes);

	  subtree = GTK_TREE_ITEM_SUBTREE(GTK_TREE_ITEM(item));
	  gnode_to_tree(server_node, subtree);
     }
     else
	  /* no suffixes found -- delete subtree */
	  gtk_tree_item_remove_subtree(GTK_TREE_ITEM(item));

}


/*
 * do a one level search of node's DN, and append any resulting entries
 * to the node
 */
void descend_onelevel(GNode *node)
{
     GNode *new_node;
     LDAP *ld;
     LDAPMessage *res, *e;
     struct ldapserver *server;
     struct node_entry *entry;
     int msg, rc, num_children;
     char *dn, message[MAX_DN_LEN + 21];
     char *dummy[] = { "dummy", NULL };

     entry = (struct node_entry *) node->data;
     server = server_by_node(node);
     if(!server)
	  return;

     if( (ld = open_connection(server)) == NULL)
	  return;

     sprintf(message, "onelevel search on %s", dn_by_node(node));
     statusbar_msg(message);

     if(config.sort_browse)
	  msg = ldap_search_s(ld, dn_by_node(node), LDAP_SCOPE_ONELEVEL,
			      "objectclass=*", dummy, 1, &res);
     else
	  msg = ldap_search(ld, dn_by_node(node), LDAP_SCOPE_ONELEVEL,
			    "objectclass=*", dummy, 1);

     if(msg == -1) {
	  statusbar_msg(ldap_err2string(msg));
	  return;
     }

     num_children = 0;

     if(config.sort_browse) {
	  if( (ldap_sort_entries(ld, &res, NULL, strcasecmp)) == LDAP_SUCCESS)
	       for(e = ldap_first_entry(ld, res); e != NULL; e=ldap_next_entry(ld, e)) {
		    dn = ldap_get_dn(ld, e);
		    new_node = g_node_new(new_node_entry(dn));
		    g_node_append(node, new_node);
		    free(dn);
		    num_children++;
	       }
     }
     else {
	  while( (rc = ldap_result(ld, msg, 0, NULL, &res)) == LDAP_RES_SEARCH_ENTRY)
	       for(e = ldap_first_entry(ld, res); e != NULL; e=ldap_next_entry(ld, e)) {
		    dn = ldap_get_dn(ld, e);
		    new_node = g_node_new(new_node_entry(dn));
		    g_node_append(node, new_node);
		    free(dn);
		    num_children++;
	       }
     }

     ldap_msgfree(res);

     make_message(message, num_children, "entry", "entries", "found");
     statusbar_msg(message);

     if(num_children)
	  entry->status = HAS_CHILDREN;
     else
	  entry->status = HAS_NO_CHILDREN;

}


/*
 * Button pressed on a tree item. Button 3 gets intercepted and puts up
 * a popup menu, all other buttons get passed along to the default handler
 */
gboolean button_press_on_tree_item(GtkWidget *tree_item, GdkEventButton *event)
{
     GNode *node;
     GtkWidget *root_menu, *menu, *menu_item;
     struct node_entry *entry;

     if(event->type == GDK_BUTTON_PRESS && event->button == 3) {
	  if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	       return(TRUE);

	  if( (entry = (struct node_entry *) node->data) == NULL)
	       return(TRUE);

	  root_menu = gtk_menu_item_new_with_label("Root");
	  gtk_widget_show(root_menu);
	  menu = gtk_menu_new();
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);

	  menu_item = gtk_tearoff_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);

	  /* Refresh */
	  menu_item = gtk_menu_item_new_with_label("Refresh");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);
	  if(entry->status == IS_SERVER)
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  GTK_SIGNAL_FUNC(refresh_server),
				  (gpointer) tree_item);
	  else
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  (GtkSignalFunc) refresh_subtree,
				  (gpointer) tree_item);

	  /* Use as template */
	  menu_item = gtk_menu_item_new_with_label("Use as template");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) edit_right_pane_novalues,
			     (gpointer) tree_item);
	  if(entry->status == IS_SERVER)
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);

	  /* Export to LDIF */
	  menu_item = gtk_menu_item_new_with_label("Export to LDIF");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) dump_subtree,
			     (gpointer) tree_item);
	  gtk_widget_show(menu_item);

	  menu_item = gtk_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);

	  menu_item = gtk_menu_item_new_with_label("Delete");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) delete_browse_entry,
			     (gpointer) tree_item);
	  if(entry->status == IS_SERVER)
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);


	  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			 event->button, event->time);

	  gtk_signal_emit_stop_by_name(GTK_OBJECT(tree_item), "button_press_event");
     }

     return(TRUE);
}


void refresh_server(GtkWidget *widget, GtkWidget *tree_item)
{
     GNode *node;
     GList *items;
     GtkTree *tree;
     GtkWidget *new_subtree;
     struct ldapserver *server;
     int was_selected, was_expanded;

     if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	  return;

     if( (tree = (GtkTree *) GTK_TREE_ITEM(tree_item)->subtree) == NULL) {
	  new_subtree = gtk_tree_new();
	  gtk_tree_item_set_subtree(GTK_TREE_ITEM(tree_item), new_subtree);
	  tree = (GtkTree *) new_subtree;
	  gtk_signal_connect(GTK_OBJECT(tree_item), "expand",
			     GTK_SIGNAL_FUNC(attach_server_suffixes), NULL);
	  attach_server_suffixes(tree_item);

	  if(tree->children)
	       /* this server magically grew suffixes since last time we
		  checked. Might as well show it... */
	       gtk_signal_emit_by_name(GTK_OBJECT(tree_item), "expand");
     }

     if( (items = tree->children) == NULL)
	  return;

     was_selected = (GTK_WIDGET_STATE(tree_item) == GTK_STATE_SELECTED);
     was_expanded = GTK_TREE_ITEM(tree_item)->expanded;

     /* clear cached connection, if any */
     if( (server = server_by_node(node)) )
	  close_connection(server, TRUE);

     g_node_children_foreach(node, G_TRAVERSE_ALL,
			     (GNodeForeachFunc) node_entry_destroy, NULL);

     gtk_tree_remove_items(tree, items);

     new_subtree = gtk_tree_new();
     gtk_tree_item_set_subtree(GTK_TREE_ITEM(tree_item), new_subtree);
     gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
			GTK_SIGNAL_FUNC(edit_right_pane_values), NULL);
     gtk_signal_connect(GTK_OBJECT(tree_item), "expand",
			GTK_SIGNAL_FUNC(attach_server_suffixes),
			NULL);

     /* was_selected? */

     if(was_expanded)
	  gtk_signal_emit_by_name(GTK_OBJECT(tree_item), "expand");

}


void refresh_subtree(GtkWidget *widget, GtkWidget *tree_item)
{
     GNode *node;
     GList *items;
     GtkTree *tree;
     GtkWidget *new_subtree;
     struct node_entry *entry;
     int was_selected, was_expanded;

     if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     was_selected = (GTK_WIDGET_STATE(tree_item) == GTK_STATE_SELECTED);
     was_expanded = GTK_TREE_ITEM(tree_item)->expanded;

     if( (GTK_TREE_ITEM(tree_item)->subtree) == NULL) {
	  /* item has no subtree at all (i.e. dummy subtree was removed)
	     so just refresh in right pane */
	  refresh_right_pane(NULL);
	  return;
     }

     tree = GTK_TREE(GTK_TREE_ITEM(tree_item)->subtree);

     if( (items = tree->children) == NULL)
	  return;

     g_node_children_foreach(node, G_TRAVERSE_ALL,
			     (GNodeForeachFunc) node_entry_destroy, NULL);

     gtk_tree_remove_items(tree, items);

     new_subtree = gtk_tree_new();
     gtk_tree_item_set_subtree(GTK_TREE_ITEM(tree_item), new_subtree);
     gtk_signal_connect(GTK_OBJECT(new_subtree), "select_child",
			GTK_SIGNAL_FUNC(edit_right_pane_values), NULL);

     gtk_signal_connect(GTK_OBJECT(tree_item), "expand",
			GTK_SIGNAL_FUNC(attach_children_to_node),
			NULL);

     if(was_selected)
	  refresh_right_pane(NULL);

     if(was_expanded)
	  gtk_signal_emit_by_name(GTK_OBJECT(tree_item), "expand");

}


void delete_browse_entry(GtkWidget *widget, GtkWidget *tree_item)
{
     GNode *node;
     struct ldapserver *server;
     struct node_entry *entry;
     int do_delete;

     if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     if( (server = server_by_node(node)) == NULL)
	  return;

     do_delete = 0;
     if( (GTK_TREE_ITEM(tree_item)->subtree) == NULL) {
	  /* item is a leaf node */
	  do_delete = 1;
     }
     else {
	  /* maybe delete everything in the subtree as well?
	     should do another LDAP_SCOPE_SUBTREE search after
	     each batch of deletes, in case the server is limiting
	     the number of entries returned per search. This could
	     get hairy...

	     For now, just pop up a dialog box with a warning
	  */

	  /* make sure the subtree isn't a dummy */
	  if(GTK_TREE(GTK_TREE_ITEM(tree_item)->subtree)->children == NULL) {
	       /* it's a dummy subtree -- check server for children */
	       if(is_leaf_entry(server, entry->dn))
		    do_delete = 1;
	  }

     }

     if(do_delete)
	  delete_entry(server, entry->dn);
     else
	  delete_browse_warning_popup(tree_item);

}


void delete_browse_warning_popup(GtkWidget *tree_item)
{
     GtkWidget *window, *vbox1, *vbox2, *vbox3, *label;
     GtkWidget *hbox1, *ok_button;
     GtkWidget *pixmap;
     GdkPixmap *warning;
     GdkBitmap *warning_mask;

     window = gtk_dialog_new();
     gtk_container_border_width(GTK_CONTAINER(window), 12);
     gtk_window_set_title(GTK_WINDOW(window), "Warning");
     gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_widget_show(vbox1);

     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 10);
     warning = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(mainwin)->window,
					    &warning_mask,
					    &mainwin->style->white,
					    warning_xpm);
     pixmap = gtk_pixmap_new(warning, warning_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox1), pixmap, TRUE, TRUE, 10);

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 10);

     label = gtk_label_new("This entry has a subtree, and cannot be deleted.");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(vbox2), label, TRUE, TRUE, 0);
     label = gtk_label_new("Delete all entries under it first.");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(vbox2), label, TRUE, TRUE, 0);

     /* OK button */
     vbox3 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox3);

     ok_button = gtk_button_new_with_label("   OK   ");
     gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) window);
     gtk_box_pack_start(GTK_BOX(vbox3), ok_button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(ok_button);
     gtk_widget_show(ok_button);

     gtk_signal_connect_object(GTK_OBJECT(window), "destroy",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) window);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) window);

     gtk_widget_show(window);

}


/*
 * a server was selected in the tree widget.
 *
 * FIXME: put up the server's editable info here? Easy to do, but
 *        does it belong here?
 */
void server_item_selected(GtkWidget *treeroot, GtkWidget *tree_item)
{
     GNode *node;
     GtkWidget *pane2_vbox, *label;
     char *server_name;

     if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	  return;

     server_name = dn_by_node(node);

     if(strncmp(cur_dn->str, server_name, cur_dn->len)) {
	  g_string_assign(cur_dn, server_name);

	  pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
	  gtk_widget_destroy(pane2_vbox);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
	  gtk_widget_show(pane2_vbox);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
						pane2_vbox);

	  label = gtk_label_new(server_name);
	  gtk_widget_show(label);
	  gtk_box_pack_start(GTK_BOX(pane2_vbox), label, FALSE, FALSE, 0);

     }

}


void edit_right_pane_values(GtkWidget *treeroot, GtkWidget *tree_item)
{

     edit_right_pane(tree_item, TRUE);

}


void edit_right_pane_novalues(GtkWidget *widget, GtkWidget *tree_item)
{

     edit_right_pane(tree_item, FALSE);

}


/*
 * display an inputbox for the selected item in the right pane
 */
void edit_right_pane(GtkWidget *tree_item, int show_values)
{
     GNode *node;
     GtkWidget *pane2_vbox;
     struct ldapserver *server;
     struct node_entry *entry;
     char *dn;

     if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	  return;

     if( (server = server_by_node(node)) == NULL)
	  return;

     dn = dn_by_node(node);

     if(strncmp(cur_dn->str, dn, cur_dn->len)
	|| server != cur_dn_server
	|| show_values != cur_show_values) {
	  cur_dn_server = server;
	  g_string_assign(cur_dn, dn);
	  cur_show_values = show_values;

	  pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
	  gtk_widget_destroy(pane2_vbox);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
	  gtk_widget_show(pane2_vbox);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
						pane2_vbox);

	  if(show_values)
	       gtk_object_set_data(GTK_OBJECT(pane2_vbox), "dn", dn);

	  set_busycursor();

	  edit_entry_table(pane2_vbox, server, dn, show_values);

	  /* while we're at it: if this DN still has a dummy tree, scan it */
	  entry = (struct node_entry *) node->data;
	  if(entry->status == STATUS_UNKNOWN)
	       attach_children_to_node(tree_item);

	  set_normalcursor();

     }

}


/*
 * refresh entry in right pane (reload from server)
 */
void refresh_right_pane(GtkWidget *dummy)
{
     GNode *node;
     GtkWidget *pane2_vbox;
     struct ldapserver *server;

     /* if the parent tree was refreshed while closed, the current dn
	isn't actually in the node tree. Just ignore requests for a
	detail refresh for now */
     if( (node = node_by_cur_dn()) == NULL)
	  return;

     pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
     gtk_widget_destroy(pane2_vbox);

     pane2_vbox = gtk_vbox_new(FALSE, 2);
     gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
     gtk_widget_show(pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
					   pane2_vbox);

     if( (server = server_by_node(node)) == NULL)
	  return;

     gtk_object_set_data(GTK_OBJECT(pane2_vbox), "dn", dn_by_node(node));

     set_busycursor();

     edit_entry_table(pane2_vbox, server, dn_by_node(node), 1);

     set_normalcursor();

}


/*
 * clear right pane
 */
void clear_right_pane(void)
{
     GtkWidget *pane2_vbox;

     pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
     gtk_widget_destroy(pane2_vbox);

     pane2_vbox = gtk_vbox_new(FALSE, 2);
     gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
     gtk_widget_show(pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
					   pane2_vbox);

}


/*
 * figure out the GNode by cur_dn and cur_dn_server
 */
GNode *node_by_cur_dn(void)
{
     GNode *node, *server_node;

     server_node = node = NULL;

     /* find server in root */
     g_node_children_foreach(root, G_TRAVERSE_ALL,
			     (GNodeForeachFunc) set_server_callback, &server_node);

     if(server_node)
	  /* find dn under server */
	  g_node_traverse(server_node, G_IN_ORDER, G_TRAVERSE_ALL, -1,
			  (GNodeTraverseFunc) set_node_callback, &node);

     return(node);
}


void set_server_callback(GNode *node, gpointer *server)
{
     struct node_entry *entry;

     entry = node->data;

     if(server_by_name(dn_by_node(node)) == cur_dn_server)
	  *server = node;

}


gboolean set_node_callback(GNode *node, gpointer *data)
{
     struct node_entry *entry;

     entry = node->data;

     if(!strncmp(cur_dn->str, entry->dn, cur_dn->len)) {
	  *data = node;
	  return(TRUE);
     }

     return(FALSE);
}


/*
 * move up the gnode tree until root + 1 level, which is the server entry
 */
struct ldapserver *server_by_node(GNode *node)
{
     GNode *prevnode;

     prevnode = node;
     while(!G_NODE_IS_ROOT(node)) {
	  prevnode = node;
	  node = node->parent;
     }

     return(server_by_name(dn_by_node(prevnode)));
}


/*
 * returns this node's DN
 */
char *dn_by_node(GNode *node)
{
     struct node_entry *entry;

     if(node) {
	  entry = node->data;
	  if(entry && entry->dn)
	       return(entry->dn);
     }

     return("");
}


/*
 * allocate and initialize the node_entry struct which
 * lives in the node's data field
 */
struct node_entry *new_node_entry(char *dn)
{
     struct node_entry *new_entry;

     new_entry = MALLOC(sizeof(struct node_entry), "struct node_entry");
     new_entry->status = 0;
     new_entry->dn = g_strdup(dn);

     return(new_entry);
}


/*
 * get all suffixes a server considers itself authorative for.
 */
GSList *get_suffix(struct ldapserver *server)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     BerElement *ptr;
     GSList *suffixes;
     int msg, i, j, s, num_suffixes;
     char *attr, **vals, message[128], suffix[MAX_SUFFIX_LEN];
     char *ldapv3_config[] = {
	  "namingcontexts",
	  NULL
     };

     num_suffixes = 0;

     set_busycursor();

     if( (ld = open_connection(server)) == NULL) {
	  set_normalcursor();
	  return(NULL);
     }

     suffixes = NULL;

     /* try LDAP V3 style config */
     statusbar_msg("base search on NULL DN");
     msg = ldap_search_s(ld, "", LDAP_SCOPE_BASE, "(objectclass=*)",
			 ldapv3_config, 0, &res);
     if(msg == LDAP_SUCCESS) {
	  e = ldap_first_entry(ld, res);
	  if(e) {
	       for(attr = ldap_first_attribute(ld, e, &ptr); attr;
		   attr = ldap_next_attribute(ld, e, ptr)) {

		    if(!strcasecmp(attr, "namingcontexts")) {
			 vals = ldap_get_values(ld, e, attr);
			 for(i = 0; vals[i]; i++) {
			      suffixes = g_slist_append(suffixes, g_strdup(vals[i]));
			      num_suffixes++;
			 }
		    }
	       }
	  }
     }

     if(num_suffixes == 0) {

	  /* try Umich style config */
	  statusbar_msg("base search on cn=config");
	  msg = ldap_search_s(ld, "cn=config", LDAP_SCOPE_BASE,
			      "(objectclass=*)", NULL, 0, &res);

	  if(msg == LDAP_SUCCESS) {
	       e = ldap_first_entry(ld, res);
	       if(e) {
		    for(attr = ldap_first_attribute(ld, e, &ptr); attr;
			attr = ldap_next_attribute(ld, e, ptr)) {
			 if(!strcasecmp(attr, "database")) {
			      vals = ldap_get_values(ld, e, attr);
			      i = s = 0;
			      /* skip first value (backend name) */
			      while(vals[0][i] != ':' && vals[0][i])
				   i++;

			      while(vals[0][i]) {
				   while( (vals[0][i] == ':' || vals[0][i] == ' ')
					  && vals[0][i])
					i++;
				   j = 0;
				   while(vals[0][i] && vals[0][i] != ':')
					suffix[j++] = vals[0][i++];

				   /* delete trailing spaces */
				   while(suffix[j - 1] == ' ')
					j--;
				   suffix[j] = 0;

				   /* four11 and whowhere used to return nothing, recently
				      started to return empty suffixes on 'cn=config'.
				      Can't wait to see what they'll come up with next */
				   if(strlen(suffix)) {
					suffixes = g_slist_append(suffixes, g_strdup(suffix));
					num_suffixes++;
				   }
			      }
			 }
		    }
	       }
	  }

     }

     set_normalcursor();
     close_connection(server, FALSE);

     make_message(message, num_suffixes, "suffix", "suffixes", "found");
     statusbar_msg(message);

     return(suffixes);
}


void cleanup_browse_mode(void)
{

     free_browse_all();

     if(cur_dn)
	  g_string_free(cur_dn, TRUE);

}


/*
 * free() all data in the GNode tree, and destroy the tree
 */
void free_browse_all(void)
{

     if(root) {
	  /* free all node_entry structs in gnode tree */
	  g_node_traverse(root, G_LEVEL_ORDER, G_TRAVERSE_ALL, -1,
			  (GNodeTraverseFunc) node_entry_free_callback, NULL);

	  g_node_destroy(root);
	  root = NULL;
     }

}


/*
 * this does the real work for free_browse_all()
 */
gboolean node_entry_free_callback(GNode *node, gpointer dummy)
{
     struct node_entry *entry;

     if(node && node->data) {
	  entry = node->data;
	  if(entry->dn)
	       free(entry->dn);
	  FREE(entry, "struct node_entry");
	  node->data = NULL;
     }

     return(FALSE);
}


/*
 * free() all data in the node, and destroy the node for good measure
 */
void node_entry_destroy(GNode *node, gpointer dummy)
{

     node_entry_free_callback(node, dummy);
     g_node_destroy(node);

}


void dump_subtree(GtkWidget *widget, GtkWidget *tree_item)
{
     LDAPMessage *res, *e;
     LDAP *ld;
     GNode *node;
     GString *out, *gmessage;
     GSList *bases;
     GtkWidget *filesel;
     struct ldapserver *server;
     struct node_entry *entry;
     int msg, num_entries;
     char message[128];

     if( (node = (GNode *) gtk_object_get_data(GTK_OBJECT(tree_item), "node")) == NULL)
	  return;

     if( (server = server_by_node(node)) == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     bases = NULL;

     out = g_string_sized_new(2048);

     if(entry->status == IS_SERVER) {

	  if(node->children == NULL && GTK_TREE_ITEM(tree_item)->subtree)
	  /* an unexpanded server tree hasn't been scanned for suffixes yet,
	     do that now */
	       attach_server_suffixes(tree_item);

	  g_node_children_foreach(node, G_TRAVERSE_ALL,
				  (GNodeForeachFunc) add_dn_to_gslist, &bases);
     }
     else
	  bases = g_slist_append(bases, strdup(entry->dn));

     if(g_slist_length(bases) == 0)
	  return;

     /* AFAIK, the UMich LDIF format doesn't take comments or a version string */
     if(config.ldifformat != LDIF_UMICH)
	  prepend_ldif_header(out, server, bases);

     set_busycursor();

     if( (ld = open_connection(server)) == NULL) {
	  /* bases and suffixes/dn leak here... */
	  set_normalcursor();
	  return;
     }

     num_entries = 0;
     gmessage = g_string_sized_new(256);
     while(bases) {
	  g_string_sprintf(gmessage, "subtree search on %s", (char *) bases->data);
	  statusbar_msg(gmessage->str);
	  msg = ldap_search_s(ld, (char *) bases->data, LDAP_SCOPE_SUBTREE,
			      "(objectclass=*)", NULL, 0, &res);
	  if(msg != -1)
	       for(e = ldap_first_entry(ld, res); e; e = ldap_next_entry(ld, e)) {
		    ldif_entry_out(out, ld, e);
		    num_entries++;
	       }

	  free(bases->data);
	  bases = bases->next;
     }

     close_connection(server, FALSE);
     make_message(message, num_entries, "entry", "entries", "found");
     statusbar_msg(message);

     set_normalcursor();

     filesel = gtk_file_selection_new("Save LDIF");
     gtk_object_set_data(GTK_OBJECT(filesel), "out", out);
     gtk_object_set_data(GTK_OBJECT(filesel), "num_entries", GINT_TO_POINTER(num_entries));
     gtk_object_set_data(GTK_OBJECT(filesel), "server", server);
     gtk_signal_connect(GTK_OBJECT(filesel), "destroy",
			(GtkSignalFunc) dump_subtree_filesel_destroy,
			filesel);
     gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",
			(GtkSignalFunc) dump_subtree_ok_callback,
			filesel);
     gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",
			       (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(filesel));
     gtk_signal_connect_object(GTK_OBJECT(filesel), "key_press_event",
			       (GtkSignalFunc) close_on_esc, (gpointer) filesel);
     gtk_widget_show(filesel);

}


void dump_subtree_ok_callback(GtkWidget *button, GtkWidget *filesel)
{
     FILE *outfile;
     GString *out, *bigmessage;
     struct ldapserver *server;
     int written, num_entries;
     char *filename, message[128];

     out = (GString *) gtk_object_get_data(GTK_OBJECT(filesel), "out");
     server = (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(filesel), "server");
     num_entries = GPOINTER_TO_INT((GString *) gtk_object_get_data(GTK_OBJECT(filesel),
								   "num_entries"));
     filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));

     if( (outfile = fopen(filename, "w")) == NULL)
	  error_popup("Save failed", strerror(errno));
     else {
	  written = fwrite(out->str, 1, out->len, outfile);
	  if(written != out->len) {
	       sprintf(message, "%d of %d bytes written", written, out->len);
	       error_popup("Save failed", message);
	  }
	  else {
	       bigmessage = g_string_sized_new(128);
	       make_message(message, num_entries, "entry", "entries", "exported");
	       g_string_sprintf(bigmessage, "%s to %s", message, filename);
	       statusbar_msg(bigmessage->str);
	       g_string_free(bigmessage, TRUE);
	  }
	  fclose(outfile);
     }


     gtk_widget_destroy(filesel);

}


void dump_subtree_filesel_destroy(GtkWidget *button, GtkWidget *filesel)
{
     GString *out;

     out = (GString *) gtk_object_get_data(GTK_OBJECT(filesel), "out");
     g_string_free(out, TRUE);

     gtk_widget_destroy(filesel);

}


void add_dn_to_gslist(GNode *node, gpointer *bases)
{
     struct node_entry *entry;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     *bases = g_slist_append(*bases, strdup(entry->dn));

}


