/**************************************************************************** 
** File: ip.c
**
** Author: Mike Borella
**
** Comments: Dump IP header information
**
*****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "config.h"
#include "ip.h"
#include "esp.h"
#include "ah.h"
#include "icmp.h"
#include "tcp.h"
#include "udp.h"

extern struct arg_t *my_args;

/*----------------------------------------------------------------------------
**
** in_cksum.c
**
** Do IP checksum
**
**----------------------------------------------------------------------------
*/

u_int16_t in_cksum(u_int16_t *addr, int len) 
{ 
  int nleft = len; 
  u_int16_t *w = addr; 
  u_int32_t sum = 0; 
  u_int16_t answer = 0; 
     
  /* 
   * Our algorithm is simple, using a 32 bit accumulator (sum), we add 
   * sequential 16 bit words to it, and at the end, fold back all the 
   * carry bits from the top 16 bits into the lower 16 bits. 
   */ 
  while (nleft > 1)  
    { 
      sum += *w++; 
      nleft -= 2; 
    } 
     
  /* 
   * mop up an odd byte, if necessary 
   */ 
   
  if (nleft == 1) 
    { 
      *(u_int8_t *)(&answer) = *(u_int8_t *)w ; 
      sum += answer; 
    } 
     
    /* 
     * add back carry outs from top 16 bits to low 16 bits 
     */ 
   
  sum = (sum >> 16) + (sum & 0xffff); /* add hi 16 to low 16 */ 
  sum += (sum >> 16);         /* add carry */ 
  answer = ~sum;              /* truncate to 16 bits */ 
  return(answer); 
} 

/*----------------------------------------------------------------------------
**
** dump_ip()
**
** Parse IP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_ip(const u_char *bp, int length)
{
  IPHdr *ip, ip2;
  u_int hlen, len, off;
  u_char *cp = NULL;
  u_int frag_off;
  u_char tos;
  u_int16_t csum;
  u_int16_t my_csum;

  /*
   * Dump header announcement
   */

  printf("-----------------------------------------------------------------\n");
  printf("                        IP Header\n");
  printf("-----------------------------------------------------------------\n");

  /*
   * View the packet as an IP header
   */

  ip = (IPHdr *) bp;

  /*
   * Check for truncated header and truncated packet
   */
  
  if (length < sizeof(IPHdr))
    {
      printf("Truncated header, length = %d bytes\n", length);
      return;
    }

  len = ntohs(ip->ip_len);
  if (length < len)
    {
      printf("Truncated packet: length field = %d, actual length = %d\n", 
	     len, length);
      return;
    }

  /*
   * Dump header fields
   */
  
  csum = ntohs(ip->ip_csum);
  hlen = ip->ip_hl*4;

  if (!my_args->n)
    {
      /* Don't print anything except src and dst addresses in minmal mode */
      if (!my_args->m)
	{
	  printf("Version:                %d\n", ip->ip_v);
	  printf("Header length:          %d\n", hlen);
	  
	  tos = ip->ip_tos;
	  printf("Type of service:        %d", tos);
	  printf(" (precedence=%d, D=%d, T=%d, R=%d, U=%d)\n",
		 tos >> 5, (tos & 0x10) >> 4, (tos & 0x08) >> 3,
		 (tos & 0x04) >> 2, tos & 0x03);
	  
	  printf("Total length:           %d\n", ntohs(ip->ip_len));
	  printf("Identification #:       %d\n", ntohs(ip->ip_id));
	  
	  frag_off = ntohs(ip->ip_off);
	  printf("Fragmentation offset:   %d", (frag_off & 0x1fff) * 8);
	  frag_off &= 0xe000;
	  printf(" (U=%d, DF=%d, MF=%d)\n", (frag_off & 0x8000) >> 15, 
		 (frag_off & 0x4000) >> 14, (frag_off & 0x2000) >> 13);
	  
	  printf("Time to live:           %d\n", ip->ip_ttl);
	  printf("Protocol:               %d\n", ip->ip_p);
	  
	  printf("Header checksum:        %d ", csum);
	  memcpy((void *) &ip2, (void *) ip, sizeof(IPHdr));
	  ip2.ip_csum = 0;
	  my_csum = ntohs(in_cksum((u_int16_t *) &ip2, sizeof(IPHdr)));
	  if (my_csum != csum) printf("(error: should be %d)", my_csum);
	  printf("\n");
	}
      
      printf("Source address          %s\n", inet_ntoa(ip->ip_src));
      printf("Destination address     %s\n", inet_ntoa(ip->ip_dst));
    }


  /*
   * If this is fragment zero, hand it to the next higher
   * level protocol.
   */

  len -= hlen;
  off = ntohs(ip->ip_off);
  if ((off & 0x1fff) == 0) 
    {
      cp = (u_char *) ip + hlen;
      switch (ip->ip_p) 
	{
	case TCP_NEXT_HEADER:
	  dump_tcp(cp, len);
	  break;
       
	case UDP_NEXT_HEADER:
	  dump_udp(cp, len);
	  break;
	  
	case ICMP_NEXT_HEADER:
	  dump_icmp(cp);
	  break; 
	
	case IP_NEXT_HEADER:
	  dump_ip(cp, len);
	  break; 
	  
	case ESP_NEXT_HEADER:
	  dump_esp(cp, len);
	  break;

	case AH_NEXT_HEADER:
	  dump_ah(cp, len);
	  break;

	default:
	  break;
	}
    }
}
