#include "links.h"

struct list_head downloads = {&downloads, &downloads};

struct list_head sessions = {&sessions, &sessions};

int session_id = 1;

unsigned char *get_err_msg(int state)
{
	if (state <= S_OK || state >= S_WAIT) {
		int i;
		for (i = 0; msg_dsc[i].msg; i++)
			if (msg_dsc[i].n == state) return stracpy(get_text(msg_dsc[i].msg));
		return stracpy(get_text("Unknown error"));
	}
	return stracpy(get_text(strerror(-state)));
}

void add_xnum_to_str(unsigned char **s, int *l, int n)
{
	unsigned char suff = 0;
	int d = -1;
	if (n >= 1000000) suff = 'M', d = (n / 100000) % 10, n /= 1000000;
	else if (n >= 1000) suff = 'k', d = (n / 100) % 10, n /= 1000;
	add_num_to_str(s, l, n);
	if (n < 10 && d != -1) add_chr_to_str(s, l, '.'), add_num_to_str(s, l, d);
	add_chr_to_str(s, l, ' ');
	if (suff) add_chr_to_str(s, l, suff);
	add_chr_to_str(s, l, 'B');
}

void add_time_to_str(unsigned char **s, int *l, ttime t)
{
	unsigned char q[64];
	t /= 1000;
	t &= 0xffffffff;
	if (t >= 86400) sprintf(q, "%dd ", (int)(t / 86400)), add_to_str(s, l, q), t %= 86400;
	if (t >= 3600) sprintf(q, "%2d:%02d", (int)(t / 3600), (int)(t / 60 % 60)), add_to_str(s, l, q);
	else sprintf(q, "%d", (int)(t / 60)), add_to_str(s, l, q);
	sprintf(q, ":%02d", (int)(t % 60)), add_to_str(s, l, q);
}

unsigned char *get_stat_msg(struct status *stat)
{
	if (stat->state == S_TRANS && stat->prg->elapsed / 100) {
		unsigned char *m = init_str();
		int l = 0;
		add_to_str(&m, &l, get_text("Received "));
		add_xnum_to_str(&m, &l, stat->prg->pos);
		if (stat->prg->size >= 0)
			add_to_str(&m, &l, get_text(" of ")), add_xnum_to_str(&m, &l, stat->prg->size);
		add_to_str(&m, &l, ", ");
		if (stat->prg->elapsed >= CURRENT_SPD_AFTER * SPD_DISP_TIME)
			add_to_str(&m, &l, get_text("avg "));
		add_xnum_to_str(&m, &l, stat->prg->loaded * 10 / (stat->prg->elapsed / 100));
		add_to_str(&m, &l, "/s");
		if (stat->prg->elapsed >= CURRENT_SPD_AFTER * SPD_DISP_TIME) 
			add_to_str(&m, &l, get_text(", cur ")),
			add_xnum_to_str(&m, &l, stat->prg->cur_loaded / (CURRENT_SPD_SEC * SPD_DISP_TIME / 1000)),
			add_to_str(&m, &l, "/s");
		return m;
	}
	return get_err_msg(stat->state);
}

void print_screen_status(struct session *ses)
{
	struct terminal *term = ses->term;
	struct status *stat;
	unsigned char *m;
	fill_area(term, 0, 0, term->x, 1, COLOR_TITLE_BG);
	fill_area(term, 0, term->y - 1, term->x, 1, COLOR_STATUS_BG);
	if (ses->wtd) stat = &ses->loading;
	else if (!list_empty(ses->history)) stat = &cur_loc(ses)->stat;
	else stat = NULL;
	if (stat) {
		if (stat->state == S_OK) if ((m = print_current_link(ses))) goto p;
		if ((m = get_stat_msg(stat))) {
			p:
			print_text(term, 0, term->y - 1, strlen(m), m, COLOR_STATUS);
			mem_free(m);
		}
		if ((m = print_current_title(ses))) {
			int p = term->x - 1 - strlen(m);
			if (p < 0) p = 0;
			print_text(term, p, 0, strlen(m), m, COLOR_TITLE);
			mem_free(m);
		}
	}
	redraw_from_window(ses->win);
}

void print_error_dialog(struct session *ses, struct status *stat, unsigned char *title)
{
	unsigned char *t = get_err_msg(stat->state);
	if (!t) return;
	msg_box(ses->term, getml(t, NULL), title, AL_CENTER, t, ses, 1, get_text("Cancel"), NULL, B_ENTER | B_ESC/*, get_text("Retry"), NULL, 0 !!! FIXME: retry */);
}

void free_wtd(struct session *ses)
{
	if (!ses->wtd) {
		internal("no WTD");
		return;
	}
	if (ses->goto_position) mem_free(ses->goto_position), ses->goto_position = NULL;
	mem_free(ses->loading_url);
	ses->loading_url = NULL;
	ses->wtd = WTD_NO;
}

void ses_forward(struct session *ses)
{
	struct location *l;
	/*struct status *stat;*/
	if (ses->search_word) mem_free(ses->search_word), ses->search_word = NULL;
	if (!(l = mem_alloc(sizeof(struct location) + strlen(ses->loading_url)))) return;
	memset(l, 0, sizeof(struct location));
	init_list(l->frames);
	memcpy(&l->stat, &ses->loading, sizeof(struct status));
	init_vs(&l->vs, ses->loading_url);
	if (ses->goto_position) {
		l->vs.goto_position = ses->goto_position;
		ses->goto_position = NULL;
	}
	add_to_list(ses->history, l);
}

void ses_imgmap(struct session *ses)
{
	struct cache_entry *ce;
	struct fragment *fr;
	struct memory_list *ml;
	struct menu_item *menu;
	if (find_in_cache(ses->loading_url, &ce) || !ce) {
		internal("can't find cache entry");
		return;
	}
	fr = ce->frag.next;
	if ((void *)fr == &ce->frag) return;
	if (get_image_map(ce->head, fr->data, fr->data + fr->length, ses->goto_position, &menu, &ml, ses->imgmap_href_base, ses->imgmap_target_base, get_term_spec(ses->term->term)->charset, ses->assume_cp))
		return;
	add_empty_window(ses->term, (void (*)(void *))freeml, ml);
	do_menu(ses->term, menu, ses);
}

void map_selected(struct terminal *term, struct link_def *ld, struct session *ses)
{
	/* !!! FIXME: frames */
	goto_url(ses, ld->link);
}

void destroy_location(struct location *loc)
{
	struct frame *frame;
	del_from_list(loc);
	foreach(frame, loc->frames) {
		destroy_vs(&frame->vs);
		mem_free(frame->name);
	}
	free_list(loc->frames);
	destroy_vs(&loc->vs);
	mem_free(loc);
}

void ses_back(struct session *ses)
{
	/*struct frame *f;*/
	struct location *loc = ses->history.next;
	if (ses->search_word) mem_free(ses->search_word), ses->search_word = NULL;
	if ((void *)loc == &ses->history) return;
	destroy_location(loc);
	loc = ses->history.next;
	if ((void *)loc == &ses->history) return;
	if (!strcmp(loc->vs.url, ses->loading_url)) return;
	destroy_location(loc);
	ses_forward(ses);
}

void end_load(struct status *, struct session *);
void abort_loading(struct session *);
void abort_preloading(struct session *);

struct session *get_download_ses(struct download *down)
{
	struct session *ses;
	foreach(ses, sessions) if (ses == down->ses) return ses;
	if (!list_empty(sessions)) return sessions.next;
	return NULL;
}

void abort_download(struct download *down)
{
	if (down->win) delete_window(down->win);
	if (down->ask) delete_window(down->ask);
	if (down->stat.state >= 0) change_connection(down->url, &down->stat, NULL, PRI_DOWNLOAD, PRI_CANCEL);
	mem_free(down->url);
	mem_free(down->file);
	if (down->handle != -1) close(down->handle);
	if (down->prog) mem_free(down->prog);
	del_from_list(down);
	mem_free(down);
}

void kill_downloads_to_file(unsigned char *file)
{
	struct download *down;
	foreach(down, downloads) if (!strcmp(down->file, file)) down = down->prev, abort_download(down->next);
}

void undisplay_download(struct download *down)
{
	if (down->win) delete_window(down->win);
}

int dlg_abort_download(struct dialog_data *dlg, struct dialog_item_data *di)
{
	register_bottom_half((void (*)(void *))abort_download, dlg->dlg->udata);
	return 0;
}

int dlg_undisplay_download(struct dialog_data *dlg, struct dialog_item_data *di)
{
	register_bottom_half((void (*)(void *))undisplay_download, dlg->dlg->udata);
	return 0;
}

void download_abort_function(struct dialog_data *dlg)
{
	struct download *down = dlg->dlg->udata;
	down->win = NULL;
}

void download_window_function(struct dialog_data *dlg)
{
	struct download *down = dlg->dlg->udata;
	struct terminal *term = dlg->win->term;
	int max = 0, min = 0;
	int w, x, y;
	int t = 0;
	unsigned char *m, *u;
	struct status *stat = &down->stat;
	redraw_below_window(dlg->win);
	down->win = dlg->win;
	if (stat->state == S_TRANS && stat->prg->elapsed / 100) {
		int l = 0;
		m = init_str();
		t = 1;
		add_to_str(&m, &l, get_text("Received "));
		add_xnum_to_str(&m, &l, stat->prg->pos);
		if (stat->prg->size >= 0)
			add_to_str(&m, &l, get_text(" of ")), add_xnum_to_str(&m, &l, stat->prg->size);
		add_to_str(&m, &l, "\n");
		if (stat->prg->elapsed >= CURRENT_SPD_AFTER * SPD_DISP_TIME)
			add_to_str(&m, &l, get_text("Average speed "));
		else add_to_str(&m, &l, get_text("Speed "));
		add_xnum_to_str(&m, &l, stat->prg->loaded * 10 / (stat->prg->elapsed / 100));
		add_to_str(&m, &l, "/s");
		if (stat->prg->elapsed >= CURRENT_SPD_AFTER * SPD_DISP_TIME) 
			add_to_str(&m, &l, get_text(", current speed ")),
			add_xnum_to_str(&m, &l, stat->prg->cur_loaded / (CURRENT_SPD_SEC * SPD_DISP_TIME / 1000)),
			add_to_str(&m, &l, "/s");
		add_to_str(&m, &l, "\nElapsed time ");
		add_time_to_str(&m, &l, stat->prg->elapsed);
		if (stat->prg->size >= 0 && stat->prg->loaded > 0)
			add_to_str(&m, &l, ", estimated time "),
			/*add_time_to_str(&m, &l, stat->prg->elapsed / 1000 * stat->prg->size / stat->prg->loaded * 1000 - stat->prg->elapsed);*/
			add_time_to_str(&m, &l, (stat->prg->size - stat->prg->pos) / (stat->prg->loaded * 10 / (stat->prg->elapsed / 100)) * 1000);
	} else m = get_err_msg(stat->state);
	if (!m) return;
	u = stracpy(down->url);
	if (strchr(u, POST_CHAR)) *strchr(u, POST_CHAR) = 0;
	max_text_width(u, &max);
	min_text_width(u, &min);
	max_text_width(m, &max);
	min_text_width(m, &min);
	max_buttons_width(dlg->items, dlg->n, &max);
	min_buttons_width(dlg->items, dlg->n, &min);
	w = dlg->win->term->x * 9 / 10 - 2 * DIALOG_LB;
	if (w < min) w = min;
	if (w > dlg->win->term->x - 2 * DIALOG_LB) w = dlg->win->term->x - 2 * DIALOG_LB;
	if (t && stat->prg->size >= 0) {
		if (w < DOWN_DLG_MIN) w = DOWN_DLG_MIN;
	} else {
		if (w > max) w = max;
	}
	if (w < 1) w = 1;
	y = 0;
	dlg_format_text(NULL, u, 0, &y, w, NULL, COLOR_DIALOG_TEXT, AL_LEFT);
	y++;
	if (t && stat->prg->size >= 0) y += 2;
	dlg_format_text(NULL, m, 0, &y, w, NULL, COLOR_DIALOG_TEXT, AL_LEFT);
	y++;
	dlg_format_buttons(NULL, dlg->items, dlg->n, 0, &y, w, NULL, AL_CENTER);
	dlg->xw = w + 2 * DIALOG_LB;
	dlg->yw = y + 2 * DIALOG_TB;
	center_dlg(dlg);
	draw_dlg(dlg);
	y = dlg->y + DIALOG_TB + 1;
	x = dlg->x + DIALOG_LB;
	dlg_format_text(term, u, x, &y, w, NULL, COLOR_DIALOG_TEXT, AL_LEFT);
	if (t && stat->prg->size >= 0) {
		unsigned char q[5];
		int p = w - 6;
		y++;
		set_only_char(term, x, y, '[');
		set_only_char(term, x + w - 5, y, ']');
		fill_area(term, x + 1, y, p * stat->prg->pos / stat->prg->size, 1, COLOR_DIALOG_METER);
		sprintf(q, "%3d%%", 100 * stat->prg->pos / stat->prg->size);
		print_text(term, x + w - 4, y, strlen(q), q, COLOR_DIALOG_TEXT);
		y++;
	}
	y++;
	dlg_format_text(term, m, x, &y, w, NULL, COLOR_DIALOG_TEXT, AL_LEFT);
	y++;
	dlg_format_buttons(term, dlg->items, dlg->n, x, &y, w, NULL, AL_CENTER);
	mem_free(u);
	mem_free(m);
}

void display_download(struct terminal *term, struct download *down, struct session *ses)
{
	struct dialog *dlg;
	struct download *dd;
	foreach(dd, downloads) if (dd == down) goto found;
	return;
	found:
	if (!(dlg = mem_alloc(sizeof(struct dialog) + 3 * sizeof(struct dialog_item)))) return;
	memset(dlg, 0, sizeof(struct dialog) + 3 * sizeof(struct dialog_item));
	undisplay_download(down);
	down->ses = ses;
	dlg->title = get_text("Download");
	dlg->fn = download_window_function;
	dlg->abort = download_abort_function;
	dlg->udata = down;
	dlg->align = AL_CENTER;
	dlg->items[0].type = D_BUTTON;
	dlg->items[0].gid = B_ENTER | B_ESC;
	dlg->items[0].fn = dlg_undisplay_download;
	dlg->items[0].data = get_text("Background");
	dlg->items[1].type = D_BUTTON;
	dlg->items[1].gid = 0;
	dlg->items[1].fn = dlg_abort_download;
	dlg->items[1].data = get_text("Abort");
	dlg->items[2].type = D_END;
	do_dialog(term, dlg, getml(dlg, NULL));
}

void download_data(struct status *stat, struct download *down)
{
	struct cache_entry *ce;
	struct fragment *frag;
	if (stat->state >= S_WAIT && stat->state < S_TRANS) goto end_store;
	if (stat->state >= 0 && stat->state < S_TRANS) goto end_store;
	if (get_cache_entry(down->url, &ce) || !ce) goto end_store;
	if (ce->redirect && down->redirect_cnt++ < MAX_REDIRECTS) {
		unsigned char *u, *p;
		if (stat->state >= 0) change_connection(down->url, &down->stat, NULL, PRI_DOWNLOAD, PRI_CANCEL);
		u = stracpy(ce->redirect);
		if (!bug_302_redirect) if (!ce->redirect_get && (p = strchr(down->url, POST_CHAR))) add_to_strn(&u, p);
		mem_free(down->url);
		down->url = u;
		down->stat.state = S_WAIT_REDIR;
		if (down->win) {
			struct event ev = { EV_REDRAW, 0, 0, 0 };
			ev.x = down->win->term->x;
			ev.y = down->win->term->y;
			down->win->handler(down->win, &ev, 0);
		}
		/*if (!strchr(down->url, POST_CHAR)) {*/
			load_url(down->url, &down->stat, PRI_DOWNLOAD, NC_IF_MOD);
			return;
		/*} else {
			unsigned char *msg = init_str();
			int l = 0;
			add_to_str(&msg, &l, "Do you want to follow redirect and post form data to url ");
			add_bytes_to_str(&msg, &l, down->url, (unsigned char *)strchr(down->url, POST_CHAR) - down->url);
			add_to_str(&msg, &l, "?");
			msg_box(get_download_ses(down)->term, getml(msg, NULL), get_text("Warning"), AL_CENTER, msg, down, 3, get_text("Yes"), down_post_yes, B_ENTER, get_text("No"), down_post_no, 0, get_text("Cancel"), down_post_cancel, B_ESC);
		}*/
	}
	foreach(frag, ce->frag) if (frag->offset <= down->last_pos && frag->offset + frag->length > down->last_pos) {
		int w = write(down->handle, frag->data + down->last_pos - frag->offset, frag->length - (down->last_pos - frag->offset));
		if (w == -1) {
			if (!list_empty(sessions)) {
				unsigned char *msg = init_str();
				int l = 0;
				add_to_str(&msg, &l, get_text("Could not write to file "));
				add_to_str(&msg, &l, down->file);
				add_to_str(&msg, &l, ": ");
				add_to_str(&msg, &l, stracpy(get_text(strerror(errno))));
				msg_box(get_download_ses(down)->term, getml(msg, NULL), get_text("Download error"), AL_CENTER, msg, NULL, 1, "Cancel", NULL, B_ENTER | B_ESC);
			}
			abort_download(down);
			return;
		}
		down->last_pos += w;
	}
	end_store:;
	if (stat->state < 0) {
		if (stat->state != S_OK) {
			unsigned char *t = get_err_msg(stat->state);
			if (t) {
				unsigned char *tt = init_str();
				int tl = 0;
				add_to_str(&tt, &tl, get_text("Error downloading "));
				add_to_str(&tt, &tl, down->url);
				add_to_str(&tt, &tl, ":\n\n");
				add_to_str(&tt, &tl, t);
				mem_free(t);
				msg_box(get_download_ses(down)->term, getml(tt, NULL), get_text("Download error"), AL_CENTER, tt, get_download_ses(down), 1, get_text("Cancel"), NULL, B_ENTER | B_ESC/*, get_text("Retry"), NULL, 0 !!! FIXME: retry */);
			}
		} else {
			if (down->prog) {
				close(down->handle), down->handle = -1;
				exec_on_terminal(get_download_ses(down)->term, down->prog, down->file, down->prog_flags);
			}
		}
		abort_download(down);
		return;
	}
	if (down->win) {
		struct event ev = { EV_REDRAW, 0, 0, 0 };
		ev.x = down->win->term->x;
		ev.y = down->win->term->y;
		down->win->handler(down->win, &ev, 0);
	}
}

int create_download_file(struct terminal *term, unsigned char *file, int safe)
{
	int h;
	int i;
	if ((h = open(file, O_CREAT|O_WRONLY|O_TRUNC|(safe?O_EXCL:0), safe ? 0600 : 0666)) == -1) {
		unsigned char *msg = init_str();
		int l = 0;
		add_to_str(&msg, &l, get_text("Could not create file "));
		add_to_str(&msg, &l, file);
		add_to_str(&msg, &l, ": ");
		add_to_str(&msg, &l, get_text(strerror(errno)));
		msg_box(term, getml(msg, NULL), get_text("Download error"), AL_CENTER, msg, NULL, 1, "Cancel", NULL, B_ENTER | B_ESC);
		return -1;
	}
	set_bin(h);
	if (safe) goto x;
	if (strlen(file) > MAX_STR_LEN) memcpy(download_dir, file, MAX_STR_LEN - 1), download_dir[MAX_STR_LEN - 1] = 0;
	else strcpy(download_dir, file);
	for (i = strlen(download_dir) - 1; i >= 0; i--) if (dir_sep(download_dir[i])) {
		download_dir[i + 1] = 0;
		goto x;
	}
	download_dir[0] = 0;
	x:
	return h;
}

unsigned char *get_temp_name(unsigned char *url)
{
	int l, nl;
	unsigned char *name, *fn, *fnn, *fnnn, *s;
	unsigned char *nm = tempnam(NULL, "links");
	if (!nm) return NULL;
	name = init_str();
	nl = 0;
	add_to_str(&name, &nl, nm);
	free(nm);
	get_filename_from_url(url, &fn, &l);
	fnnn = NULL;
	for (fnn = fn; fnn < fn + l; fnn++) if (*fnn == '.') fnnn = fnn;
	if (fnnn && (s = memacpy(fnnn, l - (fnnn - fn)))) {
		check_shell_security(&s);
		add_to_str(&name, &nl, s);
		mem_free(s);
	}
	return name;
}

unsigned char *subst_file(unsigned char *prog, unsigned char *file)
{
	unsigned char *n = init_str();
	int l = 0;
	while (*prog) {
		int p;
		for (p = 0; prog[p] && prog[p] != '%'; p++) ;
		add_bytes_to_str(&n, &l, prog, p);
		prog += p;
		if (*prog == '%') {
			add_to_str(&n, &l, file);
			prog++;
		}
	}
	return n;
}

void start_download(struct session *ses, unsigned char *file)
{
	struct download *down;
	int h;
	unsigned char *url = ses->dn_url;
	if (!url) return;
	kill_downloads_to_file(file);
	if ((h = create_download_file(ses->term, file, 0)) == -1) return;
	if (!(down = mem_alloc(sizeof(struct download)))) return;
	memset(down, 0, sizeof(struct download));
	down->url = stracpy(url);
	down->stat.end = (void (*)(struct status *, void *))download_data;
	down->stat.data = down;
	down->last_pos = 0;
	down->file = stracpy(file);
	down->handle = h;
	down->ses = ses;
	add_to_list(downloads, down);
	load_url(url, &down->stat, PRI_DOWNLOAD, NC_IF_MOD);
	display_download(ses->term, down, ses);
}

void tp_cancel(struct session *);

void tp_free(struct session *);

void continue_download(struct session *ses, unsigned char *file)
{
	struct download *down;
	int h;
	unsigned char *url = ses->tq_url;
	if (!url) return;
	if (ses->tq_prog) {
		if (!(file = get_temp_name(url))) {
			tp_cancel(ses);
			return;
		}
	}
	kill_downloads_to_file(file);
	if ((h = create_download_file(ses->term, file, !!ses->tq_prog)) == -1) {
		tp_cancel(ses);
		if (ses->tq_prog) mem_free(file);
		return;
	}
	if (!(down = mem_alloc(sizeof(struct download)))) {
		tp_cancel(ses);
		if (ses->tq_prog) mem_free(file);
		return;
	}
	memset(down, 0, sizeof(struct download));
	down->url = stracpy(url);
	down->stat.end = (void (*)(struct status *, void *))download_data;
	down->stat.data = down;
	down->last_pos = 0;
	down->file = stracpy(file);
	down->handle = h;
	down->ses = ses;
	if (ses->tq_prog) {
		down->prog = subst_file(ses->tq_prog, file);
		mem_free(file);
		mem_free(ses->tq_prog);
		ses->tq_prog = NULL;
	}
	down->prog_flags = ses->tq_prog_flags;
	add_to_list(downloads, down);
	change_connection(url, &ses->tq, &down->stat, PRI_MAIN, PRI_DOWNLOAD);
	tp_free(ses);
	display_download(ses->term, down, ses);
}

void tp_free(struct session *ses)
{
	ses->tq_ce->refcount--;
	mem_free(ses->tq_url);
	ses->tq_url = NULL;
	if (ses->tq_goto_position) mem_free(ses->tq_goto_position), ses->tq_goto_position = NULL;
}

void tp_cancel(struct session *ses)
{
	change_connection(ses->tq_url, &ses->tq, NULL, PRI_MAIN, PRI_CANCEL);
	tp_free(ses);
}

void tp_save(struct session *ses)
{
	if (ses->tq_prog) mem_free(ses->tq_prog), ses->tq_prog = NULL;
	query_file(ses, ses->tq_url, continue_download, tp_cancel);
}

void tp_open(struct session *ses)
{
	continue_download(ses, "");
}

void display_timer(struct session *);

void tp_display(struct session *ses)	/* !!! FIXME: frames */
{
	struct location *l;
	if (!(l = mem_alloc(sizeof(struct location) + strlen(ses->tq_url)))) return;
	memset(l, 0, sizeof(struct location));
	init_list(l->frames);
	memcpy(&l->stat, &ses->tq, sizeof(struct status));
	init_vs(&l->vs, ses->tq_url);
	if (ses->tq_goto_position) {
		l->vs.goto_position = ses->tq_goto_position;
		ses->tq_goto_position = NULL;
	}
	add_to_list(ses->history, l);
	cur_loc(ses)->stat.end = (void (*)(struct status *, void *))end_load;
	cur_loc(ses)->stat.data = ses;
	if (ses->tq.state >= 0) change_connection(ses->tq_url, &ses->tq, &cur_loc(ses)->stat, PRI_MAIN, PRI_MAIN);
	else cur_loc(ses)->stat.state = ses->tq.state;
	cur_loc(ses)->vs.plain = 1;
	display_timer(ses);
	tp_free(ses);
}

void type_query(struct session *ses, struct cache_entry *ce, unsigned char *ct, struct assoc *a)
{
	unsigned char *m;
	int l = 0;
	if (ses->tq_prog) mem_free(ses->tq_prog), ses->tq_prog = NULL;
	if (a) ses->tq_prog = stracpy(a->prog), ses->tq_prog_flags = a->cons;
	if (a && !a->ask) {
		tp_open(ses);
		return;
	}
	m = init_str();
	add_to_str(&m, &l, get_text("Content type is "));
	add_to_str(&m, &l, ct);
	add_to_str(&m, &l, ".\n");
	if (!a) add_to_str(&m, &l, get_text("Do you want to save or display this file?"));
	else {
		add_to_str(&m, &l, get_text("Do you want to open file with "));
		if (a->label && *a->label) add_to_str(&m, &l, a->label);
		else add_to_str(&m, &l, a->prog);
		add_to_str(&m, &l, get_text(", save it or display it?"));
	}
	msg_box(ses->term, getml(m, NULL), a ? get_text("What to do?") : get_text("Unknown type"), AL_CENTER, m, ses, 4, a ? get_text("Open") : NULL, tp_open, B_ENTER, get_text("Save"), tp_save, !a ? B_ENTER : 0, get_text("Display"), tp_display, 0, get_text("Cancel"), tp_cancel, B_ESC);
}

int ses_chktype(struct session *ses, struct status **stat, struct cache_entry *ce)
{
	struct assoc *a;
	unsigned char *ct;
	int r = 0;
	if (!(ct = get_content_type(ce->head, ce->url))) goto f;
	if (!strcasecmp(ct, "text/html")) goto ff;
	r = 1;
	if (!strcasecmp(ct, "text/plain")) goto ff;
	if (!(a = get_type_assoc(ses->term, ct)) && strlen(ct) >= 4 && !casecmp(ct, "text", 4)) goto ff;
	if (ses->tq_url) internal("type query to %s already in prgress", ses->tq_url);
	change_connection(ses->tq_url = stracpy(ses->loading_url), &ses->loading, *stat = &ses->tq, PRI_MAIN, PRI_MAIN);
	(ses->tq_ce = ce)->refcount++;
	ses->tq_goto_position = stracpy(ses->goto_position);
	type_query(ses, ce, ct, a);
	mem_free(ct);
	return 1;

	ff:
	mem_free(ct);
	f:
	ses_forward(ses);
	cur_loc(ses)->vs.plain = r;
	return 0;
}

struct wtd_data {
	struct session *ses;
	unsigned char *url;
	int pri;
	int cache;
	int wtd;
	unsigned char *pos;
	void (*fn)(struct status *, struct session *);
};

void post_yes(struct wtd_data *w)
{
	abort_preloading(w->ses);
	w->ses->goto_position = stracpy(w->pos);
	w->ses->loading.end = (void (*)(struct status *, void *))w->fn;
	w->ses->loading.data = w->ses;
	w->ses->loading_url = stracpy(w->url);
	w->ses->wtd = w->wtd;
	load_url(w->ses->loading_url, &w->ses->loading, w->pri, w->cache);
}

void post_no(struct wtd_data *w)
{
	*strchr(w->url, POST_CHAR) = 0;
	post_yes(w);
}

void ses_goto(struct session *ses, unsigned char *url, int pri, int cache, int wtd, unsigned char *pos, void (*fn)(struct status *, struct session *), int redir)
{
	struct wtd_data *w;
	unsigned char *m;
	int l;
	struct cache_entry *e;
	if (!strchr(url, POST_CHAR) || (cache == NC_ALWAYS_CACHE && !find_in_cache(url, &e) && !e->incomplete) || !(w = mem_alloc(sizeof(struct wtd_data)))) {
		ses->goto_position = pos;
		ses->loading.end = (void (*)(struct status *, void *))fn;
		ses->loading.data = ses;
		ses->loading_url = url;
		ses->wtd = wtd;
		load_url(url, &ses->loading, pri, cache);
		return;
	}
	w->ses = ses;
	w->url = url;
	w->pri = pri;
	w->cache = cache;
	w->wtd = wtd;
	w->pos = pos;
	w->fn = fn;
	m = init_str();
	l = 0;
	if (redir) {
		add_to_str(&m, &l, get_text("Do you want to follow redirect and post form data to url "));
		add_bytes_to_str(&m, &l, url, (unsigned char *)strchr(url, POST_CHAR) - url);
		add_to_str(&m, &l, "?");
	}
	else if (wtd == WTD_FORWARD) {
		add_to_str(&m, &l, get_text("Do you want to post form data to url "));
		add_bytes_to_str(&m, &l, url, (unsigned char *)strchr(url, POST_CHAR) - url);
		add_to_str(&m, &l, "?");
	} else {
		add_to_str(&m, &l, get_text("Do you want to repost form data to url "));
		add_bytes_to_str(&m, &l, url, (unsigned char *)strchr(url, POST_CHAR) - url);
		add_to_str(&m, &l, "?");
	}
	msg_box(ses->term, getml(m, w, w->url, w->pos, NULL), get_text("Warning"), AL_CENTER, m, w, 3, get_text("Yes"), post_yes, B_ENTER, get_text("No"), post_no, 0, get_text("Cancel"), NULL, B_ESC);
}

int do_move(struct session *ses, struct status **stat)
{
	struct cache_entry *ce = NULL;
	if (!ses->loading_url) {
		internal("no ses->loading_url");
		return 0;
	}
	/*if ((*stat)->state >= 0 && (*stat)->state < S_TRANS) return 0;*/
	if (find_in_cache(ses->loading_url, &ce) || !ce || (ses->wtd == WTD_IMGMAP && (*stat)->state >= 0)) {
		return 0;
	}
	if (ce->redirect && ses->redirect_cnt++ < MAX_REDIRECTS) {
		unsigned char *u, *p, *gp;
		int w = ses->wtd;
		if (ses->wtd == WTD_BACK && (void *)cur_loc(ses)->next == &ses->history)
			goto b;
		if (!(u = join_urls(ses->loading_url, ce->redirect))) goto b;
		if (!bug_302_redirect) if (!ce->redirect_get && (p = strchr(ses->loading_url, POST_CHAR))) add_to_strn(&u, p);
		/* ^^^^ According to RFC2068 POST must not be redirected to GET, but
			some BUGGY message boards rely on it :-( */
		gp = stracpy(ses->goto_position);
		abort_loading(ses);
		if (!list_empty(ses->history)) *stat = &cur_loc(ses)->stat;
		else *stat = NULL;
		if (w == WTD_FORWARD || w == WTD_IMGMAP) {
			ses_goto(ses, u, PRI_MAIN, NC_CACHE, w, gp, end_load, 1);
			return 2;
		}
		if (gp) mem_free(gp);
		if (w == WTD_BACK) {
			/*struct location *loc = cur_loc(ses)->next;
			free_list(loc->frames);
			del_from_list(loc);
			destroy_vs(&loc->vs);
			mem_free(loc);*/
			ses_goto(ses, u, PRI_MAIN, NC_CACHE, WTD_RELOAD, NULL, end_load, 1);
			return 2;
		}
		if (w == WTD_RELOAD) {
			ses_goto(ses, u, PRI_MAIN, ses->reloadlevel, WTD_RELOAD, NULL, end_load, 1);
			return 2;
		}
	} else ses->redirect_cnt = 0;
	b:
	if (ses->display_timer != -1) kill_timer(ses->display_timer), ses->display_timer = -1;
	if (ses->wtd == WTD_FORWARD) {
		/*ses_forward(ses);*/
		if (ses_chktype(ses, stat, ce)) {
			free_wtd(ses);
			return 2;
		}
	}
	if (ses->wtd == WTD_IMGMAP) ses_imgmap(ses);
	if (ses->wtd == WTD_BACK) ses_back(ses);
	if (ses->wtd == WTD_RELOAD) ses_back(ses), ses_forward(ses);
	/*if (ses->goto_position) mem_free(ses->goto_position);*/
	if ((*stat)->state >= 0) change_connection(ses->loading_url, &ses->loading, *stat = &cur_loc(ses)->stat, PRI_MAIN, PRI_MAIN);
	else cur_loc(ses)->stat.state = ses->loading.state;
	x:
	free_wtd(ses);
	return 1;
}

void display_timer(struct session *ses)
{
	ttime t = get_time();
	html_interpret(ses);
	draw_formatted(ses);
	t = (get_time() - t) * DISPLAY_TIME;
	if (t < DISPLAY_TIME_MIN) t = DISPLAY_TIME_MIN;
	ses->display_timer = install_timer(t, (void (*)(void *))display_timer, ses);
}
	

void end_load(struct status *stat, struct session *ses)
{
	int d = -1;
	if (ses->wtd /*&& (stat->state < S_WAIT || stat->state >= S_TRANS)*/) {
		d = do_move(ses, &stat);
		if (d && d != 2) {
			display_timer(ses);
		}
	}
	if (!stat) return;
	if (stat->state < 0) {
		if (ses->display_timer != -1) kill_timer(ses->display_timer), ses->display_timer = -1;
			/* !!! FIXME: frames */
		if (d != 2 && ses->wtd) {
			free_wtd(ses);
			/*reload(ses, NC_CACHE);*/
		} else if (d == -1) {
			html_interpret(ses);
			draw_formatted(ses);
		}
	}
	if (stat->state < 0 && stat->state != S_OK && d != 2) {
		print_error_dialog(ses, stat, get_text("Error"));
	}
	print_screen_status(ses);
	/*if (d && !ses->wtd) {
		html_interpret(ses);
		draw_formatted(ses);
	}*/
}

struct session *create_session(struct window *win)
{
	struct terminal *term = win->term;
	struct session *ses;
	if ((ses = mem_alloc(sizeof(struct session)))) {
		memset(ses, 0, sizeof(struct session));
		init_list(ses->history);
		ses->term = term;
		ses->win = win;
		ses->id = session_id++;
		ses->screen = NULL;
		ses->wtd = WTD_NO;
		ses->display_timer = -1;
		ses->loading_url = NULL;
		ses->goto_position = NULL;
		ses->assume_cp = assume_cp;
		add_to_list(sessions, ses);
	}
	if (first_use) {
		first_use = 0;
		msg_box(term, NULL, get_text("Welcome"), AL_CENTER, get_text("Welcome to links!"), NULL, 1, get_text("OK"), NULL, B_ENTER | B_ESC);
	}
	return ses;
}

void copy_session(struct session *old, struct session *new)
{
	struct location *l;
	foreachback(l, old->history) {
		struct location *nl;
		if ((nl = mem_alloc(sizeof(struct location) + strlen(l->vs.url) + 1))) {
			struct frame *frm;
			memcpy(nl, l, sizeof(struct location) + strlen(l->vs.url) + 1);
			init_list(nl->frames);
			foreachback(frm, l->frames) {
				struct frame *nfrm;
				if ((nfrm = mem_alloc(sizeof(struct frame) + strlen(frm->vs.url) + 1))) {
					memcpy(nfrm, frm, sizeof(struct frame) + strlen(frm->vs.url) + 1);
					add_to_list(nl->frames, nfrm);
				}
			}
			add_to_list(new->history, nl);
		}
	}
}

void *create_session_info(int cp, unsigned char *url, int *ll)
{
	int l = strlen(url);
	int *i;
	*ll = 2 * sizeof(int) + l;
	if (!(i = mem_alloc(2 * sizeof(int) + l))) return NULL;
	i[0] = cp;
	i[1] = l;
	memcpy(i + 2, url, l);
	return i;
}

int read_session_info(int fd, struct session *ses, void *data, int len)
{
	int cpfrom, sz;
	struct session *s;
	if (len < 2 * sizeof(int)) return -1;
	cpfrom = *(int *)data;
	sz = *((int *)data + 1);
	foreach(s, sessions) if (s->id == cpfrom) {
		copy_session(s, ses);
		break;
	}
	if (sz) {
		char *u;
		if (len < 2 * sizeof(int) + sz) return 0;
		if ((u = mem_alloc(sz + 1))) {
			memcpy(u, (int *)data + 2, sz);
			u[sz] = 0;
			goto_url(ses, u);
			mem_free(u);
		}
	}
	return 0;
}

void abort_preloading(struct session *ses)
{
	if (ses->wtd) {
		change_connection(ses->loading_url, &ses->loading, NULL, PRI_MAIN, PRI_CANCEL);
		free_wtd(ses);
	}
}

void abort_loading(struct session *ses)
{
	struct frame *frm;
	struct location *l = cur_loc(ses);
	if ((void *)l != &ses->history) {
		if (l->stat.state >= 0)
			change_connection(l->vs.url, &l->stat, NULL, PRI_MAIN, PRI_CANCEL);
		foreach(frm, l->frames) if (frm->stat.state >= 0)
			change_connection(frm->vs.url, &frm->stat, NULL, PRI_FRAME, PRI_CANCEL);
	}
	abort_preloading(ses);
}

void destroy_session(struct session *ses)
{
	struct download *d;
	struct location *l;
	if (!ses) return;
	foreach(d, downloads) if (d->ses == ses && d->prog) {
		d = d->prev;
		abort_download(d->next);
	}
	abort_loading(ses);
	while ((void *)(l = ses->history.next) != &ses->history) {
		struct frame *frm;
		while ((void *)(frm = l->frames.next) != &l->frames) {
			destroy_vs(&frm->vs);
			del_from_list(frm);
			mem_free(frm);
		}
		destroy_vs(&l->vs);
		del_from_list(l);
		mem_free(l);
	}
	if (ses->screen) detach_formatted(ses->screen), mem_free(ses->screen);
	if (ses->loading_url) mem_free(ses->loading_url);
	if (ses->display_timer != -1) kill_timer(ses->display_timer);
	if (ses->goto_position) mem_free(ses->goto_position);
	if (ses->imgmap_href_base) mem_free(ses->imgmap_href_base);
	if (ses->imgmap_target_base) mem_free(ses->imgmap_target_base);
	if (ses->tq_url) {
		change_connection(ses->tq_url, &ses->tq, NULL, PRI_MAIN, PRI_CANCEL);
		mem_free(ses->tq_url);
	}
	if (ses->tq_goto_position) mem_free(ses->tq_goto_position);
	if (ses->tq_prog) mem_free(ses->tq_prog);
	if (ses->dn_url) mem_free(ses->dn_url);
	if (ses->search_word) mem_free(ses->search_word);
	if (ses->last_search_word) mem_free(ses->last_search_word);
	del_from_list(ses);
	/*mem_free(ses);*/
}

void destroy_all_sessions()
{
	/*while (!list_empty(sessions)) destroy_session(sessions.next);*/
}

void abort_all_downloads()
{
	while (!list_empty(downloads)) abort_download(downloads.next);
}

void reload(struct session *ses, int no_cache)
{
	struct location *l;
	abort_loading(ses);
	if (no_cache == -1) no_cache = ++ses->reloadlevel;
	else ses->reloadlevel = no_cache;
	if ((void *)(l = ses->history.next) != &ses->history) {
		struct frame *frm;
		l->stat.data = ses;
		l->stat.end = (void *)end_load;
		load_url(l->vs.url, &l->stat, PRI_MAIN, no_cache);
		foreach(frm, l->frames) {
			frm->stat.data = ses;
			frm->stat.end = (void *)end_load;
			load_url(frm->vs.url, &frm->stat, PRI_FRAME, no_cache);
		}
	}
}

/*void ses_load_notify(struct status *stat, struct session *ses)
{
	if (stat->state == S_TRANS || stat->state == S_OK) {
		stat->end = (void (*)(struct status *, void *))end_load;
		ses->wtd = WTD_NO;
		mem_free(ses->loading_url);
		if (ses->wtd == WTD_FORWARD) {
			ses_forward(ses);
		} else internal("bad ses->wtd");
		return;
	}
	if (stat->state >= 0) print_screen_status(ses);
	if (stat->state < 0) print_error_dlg(ses, stat);
}*/

void go_back(struct session *ses)
{
	unsigned char *url;
	ses->reloadlevel = NC_CACHE;
	if (ses->wtd) {
		if (1 || ses->wtd != WTD_BACK) {
			abort_loading(ses);
			print_screen_status(ses);
			reload(ses, NC_CACHE);
		}
		return;
	}
	if (ses->history.next == &ses->history || ses->history.next == ses->history.prev)
		return;
	abort_loading(ses);
	if (!(url = stracpy(((struct location *)ses->history.next)->next->vs.url)))
		return;
	ses_goto(ses, url, PRI_MAIN, NC_ALWAYS_CACHE, WTD_BACK, NULL, end_load, 0);
}

void goto_url_w(struct session *ses, unsigned char *url, int wtd)
{
	unsigned char *u;
	unsigned char *pos;
	ses->reloadlevel = NC_CACHE;
	/*struct location *l = ses->history.next;*/
	if (!(u = translate_url(url))) {
		struct status stat = { NULL, NULL, S_BAD_URL, 0, NULL, NULL };
		print_error_dialog(ses, &stat, get_text("Error"));
		return;
	}
	pos = extract_position(u);
	if (ses->wtd == wtd) {
		if (!strcmp(ses->loading_url, u)) {
			mem_free(u);
			if (ses->goto_position) mem_free(ses->goto_position);
			ses->goto_position = pos;
			return;
		}
	}
	abort_loading(ses);
	ses_goto(ses, u, PRI_MAIN, NC_CACHE, wtd, pos, end_load, 0);
	/*abort_loading(ses);*/
}

void goto_url(struct session *ses, unsigned char *url)
{
	goto_url_w(ses, url, WTD_FORWARD);
}

void goto_imgmap(struct session *ses, unsigned char *url, unsigned char *href, unsigned char *target)
{
	if (ses->imgmap_href_base) mem_free(ses->imgmap_href_base);
	ses->imgmap_href_base = href;
	if (ses->imgmap_target_base) mem_free(ses->imgmap_target_base);
	ses->imgmap_target_base = target;
	goto_url_w(ses, url, WTD_IMGMAP);
}

void win_func(struct window *win, struct event *ev, int fw)
{
	struct session *ses = win->data;
	switch (ev->ev) {
		case EV_INIT:
			if (!(ses = win->data = create_session(win)) ||
			    read_session_info(win->term->fdin, ses, (char *)ev->b + sizeof(int), *(int *)ev->b)) {
				destroy_terminal(win->term);
				return;
			}
			/*make_request(ses, 0);*/
			break;
		case EV_ABORT:
			destroy_session(ses);
			break;
		case EV_RESIZE:
			html_interpret(ses);
		case EV_REDRAW:
			draw_formatted(ses);
			print_screen_status(ses);
			break;
		case EV_KBD:
		case EV_MOUSE:
			send_event(ses, ev);
			break;
		default:
			error("ERROR: unknown event");
	}
}
