/*
** Copyright (C) 26 Aug 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <time.h>

#include "globals.h"
#include "common_gtk.h"
#include "cdrecord_options.h"
#include "linebuffer.h"
#include "isosize.h"
#include "progressbars_gui.h"
#include "locks.h"
#include "preferences.h"
#include "read_image.h"

enum {
	IMAGE_BUF_S = 131072,
	NUM_BUF = 20,
	INFO_BUF = 100,
	WRITE_INTERVALL = 10,
	RETRY_BEFORE_FAIL = 20,
	NO_ERROR = 0,
	INIT_READ_ERROR = 1,
	INIT_WRITE_ERROR = 2,
	INIT_SIZE_ERROR = 3,
	READ_ERROR = 4,
	WRITE_ERROR = 5,
	LSEEK_ERROR = 6
};

typedef struct {
	long written;
	long image_size;
	int speed;
	int read_errors;
	int errorcode;
} progress_info;

pid_t cd_file_child = 0;

static int status_pipe[2];
static GtkWidget *p_win;
int status_pipe_ok;

/* Reads in an iso image from a cd to a file. */
static void cd_to_file(char *from, char *to) {
	size_t read_bytes = 0, n;
	long image_size, image_left;
	int in_fd, out_fd;
	void *buf;
	progress_info status = {0, 0 , 0, 0, 0};
	static struct timeval last_time, now;
	struct timezone tz;

	if (-1 == (out_fd = open(to, O_CREAT | O_WRONLY | O_TRUNC, S_IRWXU))) {
		status.errorcode = INIT_WRITE_ERROR;
		write(status_pipe[1], &status, sizeof(status));
		return;
	}
	if (-1 == (in_fd = open(from, 0))) {
		close(out_fd);
		status.errorcode = INIT_READ_ERROR;
		write(status_pipe[1], &status, sizeof(status));
		return;
	}

	if ((image_size = isosize(in_fd)) <= 0) {
		close(out_fd);
		close(in_fd);
		status.errorcode = INIT_SIZE_ERROR;
		write(status_pipe[1], &status, sizeof(status));
		return;
	}

	status.errorcode = NO_ERROR;
	status.image_size = image_size;
	status.written = 0;
	status.speed = 0;
	status.read_errors = 0;
	write(status_pipe[1], &status, sizeof(status));

	image_left = image_size;

	buf = (void *) malloc(IMAGE_BUF_S);

	gettimeofday(&last_time, &tz);

	while (image_left >= 0) {
		double diff = 0, size_diff = 0;
		static int intervall = WRITE_INTERVALL;
		static int retry = RETRY_BEFORE_FAIL;
		n = read(in_fd, buf, image_left > IMAGE_BUF_S ? IMAGE_BUF_S : image_left);

		if (-1 == n) {
			if (retry-- > 0) {
				status.read_errors++;
				write(status_pipe[1], &status, sizeof(status));
				if (read_bytes != lseek(in_fd, read_bytes, SEEK_SET)) {
					status.errorcode = LSEEK_ERROR;
					write(status_pipe[1], &status, sizeof(status));
					break;
				}
				continue;
			} else {
				status.errorcode = READ_ERROR;
				write(status_pipe[1], &status, sizeof(status));
				break;
			}
		} else if (0 == n) {
			/* */
			status.written = read_bytes;
			write(status_pipe[1], &status, sizeof(status));
			break;
		}

		if (-1 == write(out_fd, buf, n)) {
			status.errorcode = WRITE_ERROR;
			write(status_pipe[1], &status, sizeof(status));
			break;
		}
		read_bytes += n;
		image_left -= n;
		retry = RETRY_BEFORE_FAIL;
		if (intervall-- == 0) {
			gettimeofday(&now, &tz);
			diff = (now.tv_sec - last_time.tv_sec)*1000000 + now.tv_usec - last_time.tv_usec;
			size_diff = read_bytes - status.written;
			if (0 != size_diff)
				status.speed = (int)((1000000*size_diff/diff)/1024);
			else
				status.speed = 0;
			status.written = read_bytes;
			write(status_pipe[1], &status, sizeof(status));
			intervall = WRITE_INTERVALL;
			last_time = now;
		}
	}

	free(buf);
	close(in_fd);
	close(out_fd);
	return;
}

static void update_copy_progress(progress_info status) {
	char buf1[INFO_BUF], buf2[INFO_BUF];
	float p;

	g_snprintf(buf1, INFO_BUF, " %d KB/s \n Read errors corrected: %d ",
			status.speed, status.read_errors);
	g_snprintf(buf2, INFO_BUF, _(" %ld bytes written \n (of %ld bytes)"),
			status.written, status.image_size);
	set_copy_info(buf1, buf2);

	p = (float)status.written/(float)status.image_size;
	if (p >= 0 && p <= 1) {
		estimate_finnish_time_update(p);
		gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), p);
	} else if (status.errorcode == NO_ERROR)
		g_warning("%s %i: strange sizes but no error detected", __FILE__, __LINE__);
}

static int handle_cd_to_file_results(gpointer data) {
	fd_set rset;
	int res;
	ssize_t r;
	struct timeval tv;
	static progress_info status = {0, 0, 0, 0, 0};
	int dirty = FALSE;

	while (status_pipe_ok) {
		tv.tv_sec = 0;
		tv.tv_usec = 0;
		FD_ZERO(&rset);
		FD_SET(status_pipe[0], &rset);
		res = select(status_pipe[0] + 1, &rset, NULL, NULL, &tv);

		if (res < 0)
			g_error("%s %i: select returned negative!", __FILE__, __LINE__);
		if (res == 0)
			break;

		if (status_pipe_ok && FD_ISSET(status_pipe[0], &rset)) {
			if ((r = read(status_pipe[0], &status, sizeof(progress_info))) > 0) {
				dirty = TRUE;
			} else if (0 == r) {
				status_pipe_ok = FALSE;
			} else {
				g_warning("%s %i: read returned returned negative: %s",
						__FILE__, __LINE__, strerror(errno));
				status_pipe_ok = FALSE;
			}
		}
	}

	if ((status.written == status.image_size) && (0 != status.written)) {
		char *m;
		char b[NUM_BUF];
		m = string_append(_(" Copying of cd image to disk compelete! \n "), NULL);
		g_snprintf(b, NUM_BUF, "%ld", status.written);
		m = string_append(m, b);
		m = string_append(m, " bytes copied. ");
		alert_user_of_error(m);
		free(m);
	}

	switch (status.errorcode) {
		case NO_ERROR:
			break;
		case INIT_READ_ERROR:
			alert_user_of_error(_(" Can't open device for reading. "));
			status_pipe_ok = FALSE;
			break;
		case INIT_WRITE_ERROR:
			alert_user_of_error(_(" Can't open file for writing. "));
			status_pipe_ok = FALSE;
			break;
		case INIT_SIZE_ERROR:
			alert_user_of_error(_(" Can't read size of image to read. "));
			status_pipe_ok = FALSE;
			break;
		case READ_ERROR:
			alert_user_of_error(_(" Error reading image from device, retried several times \n"
						" before giving up (probably a bad CDROM drive or a bad CD-R). "));
			status_pipe_ok = FALSE;
			break;
		case WRITE_ERROR:
			alert_user_of_error(_(" Error writing image to disk (out of space?). "));
			status_pipe_ok = FALSE;
			break;
		case LSEEK_ERROR:
			alert_user_of_error(_(" Can't lseek to retry after error reading from device. "));
			status_pipe_ok = FALSE;
			break;
		default:
			g_warning("%s %i: impossible error detected", __FILE__, __LINE__);
			status_pipe_ok = FALSE;
			break;
	}

	if (dirty)
		update_copy_progress(status);

	if (!status_pipe_ok) {
		not_running();
		close(status_pipe[0]);
		gtk_widget_destroy(p_win);
		if (((status.written != status.image_size) || (0 == status.written)) && !bailed_out) {
			alert_user_of_error(_("Unrecoverable error detected, aborting."));
		}
		return FALSE;
	} else
		return TRUE;
}

/* forks off a new process which calls cd_to_file */
void read_image_from_cd(GtkWidget *widget, gpointer data) {
	char *from, *to;

	if (is_running())
		return;

	to = g_strdup(gtk_entry_get_text(GTK_ENTRY(image_path)));
	from = g_strdup(gtk_entry_get_text(GTK_ENTRY(iso_device_path)));

	if (-1 == pipe(status_pipe)) {
		g_warning("%s %i: pipe failed", __FILE__, __LINE__);
		return;
	}

	cd_file_child = fork();

	if (cd_file_child > 0) {
		bailed_out = FALSE;
		status_pipe_ok = TRUE;
		p_win = open_progress_win(_("cd to image copy starting..."));
		if (misc_prefs.own_progresswin)
			gtk_window_set_title(GTK_WINDOW(p_win), _("cd to image copy"));
		close(status_pipe[1]);
		estimate_finnish_time_start();
		gtk_timeout_add(TIMEOUT_GTK, handle_cd_to_file_results, NULL);
            gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), 0.0);
		return;
	} else if (0 == cd_file_child) {
		close(status_pipe[0]);
		cd_to_file(from, to);
		close(status_pipe[1]);
		_exit(0);
	} else {
		g_warning("%s %i: fork failed", __FILE__, __LINE__);
	}
}
