/*================================================================
 * Simple Athena widget control program
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>

#include "config.h"
#include "util.h"
#include "midievent.h"
#include "channel.h"
#include "controls.h"
#include "ext_c.h"

#include "BITMAPS/play.xbm"
#include "BITMAPS/pause.xbm"
#include "BITMAPS/prev.xbm"
#include "BITMAPS/back.xbm"
#include "BITMAPS/stop.xbm"
#include "BITMAPS/fwrd.xbm"
#include "BITMAPS/next.xbm"
#include "BITMAPS/quit.xbm"


/*----------------------------------------------------------------*/

static XtAppContext apps;
static Widget toplevel, topform, title, buttons;
static Widget title_index, title_name, title_time;
static Widget btn_play, btn_pause, btn_prev, btn_back, btn_stop, btn_fwrd, btn_next, btn_quit;
static Widget textwin;

/*----------------------------------------------------------------*/

static int number_of_files;
static char **filename;
static int curfile;
static int total_time, cur_time;
static int curstate;
enum { STAT_CLEAN, STAT_PLAYING, STAT_PAUSE };
static FILE *p_input;

/*----------------------------------------------------------------*/

static void create_window(Widget parent);
static Pixmap MakeBitmap(char *bits, int width, int height);
static void PlayCB(void);
static void NextCB(void);
static void PrevCB(void);
static void FwrdCB(void);
static void BackCB(void);
static void StopCB(void);
static void QuitCB(void);
static void PauseCB(void);
static void InputCB(void);
static void PauseCB(void);
static void TimerCB(void);
static void init_line(void);
static void append_line(void);
static void popup_msg(void);
static void do_play(void);
static void set_index(void);
static void set_title(char *title);
static void set_curtime(int sec);
static void clear_status(void);
static void playing_status(void);
static void paused_status(void);
static void exit_all(void);

/*----------------------------------------------------------------*/

static XtActionsRec actions[] = {
	{"play", (XtActionProc)PlayCB},
	{"stop", (XtActionProc)StopCB},
	{"pause", (XtActionProc)PauseCB},
	{"next", (XtActionProc)NextCB},
	{"prev", (XtActionProc)PrevCB},
	{"forward", (XtActionProc)FwrdCB},
	{"backward", (XtActionProc)BackCB},
	{"quit", (XtActionProc)QuitCB},
};


/*----------------------------------------------------------------*/

#define DELTA_TIME 50

void main(int argc, char **argv)
{
	toplevel = XtAppInitialize(&apps, "Atmidi", NULL, 0, &argc, argv,
				   NULL, NULL, 0);
	if (argc < 2) {
		fprintf(stderr, "no midi file is given\n");
		exit(1);
	}
	number_of_files = argc - 1;
	filename = argv + 1;
	curfile = -1;

	if (! ext_control_init(TRUE, TRUE, TRUE, FALSE, 0, NULL))
		exit(0);
	p_input = fdopen(pipe_in, "r");
	setvbuf(p_input, NULL, _IONBF, 0);

	create_window(toplevel);
	XtAppAddActions(apps, actions, XtNumber(actions));

	XtAppAddInput(apps, pipe_in, (XtPointer)XtInputReadMask,
		      (XtInputCallbackProc)InputCB, NULL);
	XtRealizeWidget(toplevel);
	XtAppAddTimeOut(apps, DELTA_TIME, (XtTimerCallbackProc)TimerCB, NULL);

	/* start automatically */
	PlayCB();

	XtAppMainLoop(apps);

	exit_all();
}

static void exit_all(void)
{
	ext_control_end();
	exit(0);
}

/*----------------------------------------------------------------
 * create widgets
 *----------------------------------------------------------------*/

#define CREATE_BUTTON(name,parent,left,bitmap) CreateButton(name, parent, left, MakeBitmap(bitmap##_bits, bitmap##_width, bitmap##_height))


static Widget CreateButton(char *name, Widget parent, Widget left, Pixmap bitmap)
{
	return XtVaCreateManagedWidget(name, commandWidgetClass, parent,
				       XtNjustify, XtJustifyLeft,
				       XtNresize, False,
				       XtNfromHoriz, left,
				       XtNbitmap, bitmap,
				       NULL);
}

static void create_window(Widget parent)
{
	topform = XtVaCreateManagedWidget
		("topform", formWidgetClass, parent,
		 NULL);

	title = XtVaCreateManagedWidget
		("title", formWidgetClass, topform,
		 XtNleft, XawChainLeft,
		 XtNright, XawChainRight,
		 XtNtop, XawChainTop,
		 NULL);

	title_index = XtVaCreateManagedWidget
		("name", labelWidgetClass, title,
		 XtNlabel, "000/000",
		 XtNtop, XawChainTop,
		 XtNbottom, XawChainBottom,
		 XtNleft, XawChainLeft,
		 NULL);
	title_name = XtVaCreateManagedWidget
		("name", labelWidgetClass, title,
		 XtNtop, XawChainTop,
		 XtNbottom, XawChainBottom,
		 XtNfromHoriz, title_index,
		 NULL);

	title_time = XtVaCreateManagedWidget
		("time", labelWidgetClass, title,
		 XtNwidth, 64,
		 XtNtop, XawChainTop,
		 XtNbottom, XawChainBottom,
		 XtNfromHoriz, title_name,
		 XtNright, XawChainRight,
		 NULL);

	buttons = XtVaCreateManagedWidget
		("buttons", formWidgetClass, topform,
		 XtNleft, XawChainLeft,
		 XtNright, XawChainRight,
		 XtNfromVert, title,
		 NULL);

	btn_play = CREATE_BUTTON("play", buttons, NULL, play);
	btn_pause = CREATE_BUTTON("pause", buttons, btn_play, pause);
	btn_prev = CREATE_BUTTON("prev", buttons, btn_pause, prev);
	btn_back = CREATE_BUTTON("back", buttons, btn_prev, back);
	btn_stop = CREATE_BUTTON("stop", buttons, btn_back, stop);
	btn_fwrd = CREATE_BUTTON("fwrd", buttons, btn_stop, fwrd);
	btn_next = CREATE_BUTTON("next", buttons, btn_fwrd, next);
	btn_quit = CREATE_BUTTON("quit", buttons, btn_next, quit);

	XtAddCallback(btn_play, XtNcallback, (XtCallbackProc)PlayCB, NULL);
	XtAddCallback(btn_pause, XtNcallback, (XtCallbackProc)PauseCB, NULL);
	XtAddCallback(btn_prev, XtNcallback, (XtCallbackProc)PrevCB, NULL);
	XtAddCallback(btn_back, XtNcallback, (XtCallbackProc)BackCB, NULL);
	XtAddCallback(btn_stop, XtNcallback, (XtCallbackProc)StopCB, NULL);
	XtAddCallback(btn_fwrd, XtNcallback, (XtCallbackProc)FwrdCB, NULL);
	XtAddCallback(btn_next, XtNcallback, (XtCallbackProc)NextCB, NULL);
	XtAddCallback(btn_quit, XtNcallback, (XtCallbackProc)QuitCB, NULL);

	textwin = XtVaCreateManagedWidget
		("text", asciiTextWidgetClass, topform,
		 XtNfromVert, buttons,
		 XtNleft, XawChainLeft,
		 XtNeditType, XawtextAppend,
		 XtNwidth, 250,
		 XtNheight, 200,
		 XtNscrollHorizontal, XawtextScrollAlways,
		 XtNscrollVertical, XawtextScrollAlways,
		 XtNtype, XawAsciiString,
		 XtNdisplayCaret, False,
		 NULL);

	init_line();
	clear_status();
}

static Pixmap MakeBitmap(char *bits, int width, int height)
{
	Display *disp;
	Screen *scr;
	Window win;
	Pixel bg, fg;

	scr = XtScreen(toplevel);
	disp = DisplayOfScreen(scr);
	win = RootWindowOfScreen(scr);
	fg = WhitePixelOfScreen(scr);
	bg = BlackPixelOfScreen(scr);
	return XCreatePixmapFromBitmapData(disp, win, bits, width, height, fg, bg, 1);
}

/*----------------------------------------------------------------
 * callback actions
 *----------------------------------------------------------------*/

static void PlayCB()
{
	if (curstate != STAT_PLAYING)
		ext_control_next();
}

static void NextCB()
{
	if (curstate == STAT_PLAYING)
		ext_control_next();
}

static void PrevCB()
{
	if (curstate == STAT_PLAYING)
		ext_control_prev();
}

static void FwrdCB()
{
	if (curstate == STAT_PLAYING)
		ext_control_fwrd();
}

static void BackCB()
{
	if (curstate == STAT_PLAYING)
		ext_control_back();
}

static void StopCB()
{
	if (curstate == STAT_PLAYING) {
		ext_control_stop();
		clear_status();
	}
}

static void QuitCB()
{
	exit_all();
}

static void PauseCB()
{
	if (curstate == STAT_PLAYING) {
		paused_status();
		ext_control_pause();
	} else if (curstate == STAT_PAUSE) {
		playing_status();
		ext_control_pause();
	}
}

/*----------------------------------------------------------------
 * pipe input from player
 *----------------------------------------------------------------*/

static void skip_line(void)
{
	int c;
	while ((c = getc(p_input)) != '\n')
		;
}

static void InputCB(void)
{
	int len;
	char local[100], *cmd, *arg;

	if (fgets(local, sizeof(local), p_input) == NULL)
		return;

	if (*local == 0) return;
	len = strlen(local);
	if (local[len-1] != '\n')
		skip_line();
	else
		local[len-1] = 0;
	local[4] = 0;
	cmd = local;
	arg = local + 5;

	if (strcmp(cmd, "TIME") == 0) {
		total_time = atoi(arg) / 100;
		set_curtime(0);
	} else if (strcmp(cmd, "FILE") == 0) {
		set_title(arg);
	} else if (strcmp(cmd, "TITL") == 0) {
		/*set_title(arg);*/
		;
	} else if (strcmp(cmd, "PREV") == 0) {
		if (curfile > 0)
			curfile--;
		do_play();
	} else if (strcmp(cmd, "NEXT") == 0) {
		curfile++;
		if (curfile >= number_of_files) {
			ext_control_stop();
			clear_status();
		} else
			do_play();
	} else if (strcmp(cmd, "CMSG") == 0) {
		append_line();
	} else if (strcmp(cmd, "CERR") == 0) {
		popup_msg();
		ext_control_next();
	}
}


/*----------------------------------------------------------------
 * display text message from player
 *----------------------------------------------------------------*/

#define MAX_TEXT_LINES		500
#define DISPLAY_LINES		12

static XawTextPosition textpos, linepos[DISPLAY_LINES];
static int curline, lineidx;

static void init_line(void)
{
	int i;
	XtVaSetValues(textwin, XtNstring, "", NULL);
	textpos = 0;
	curline = 0;
	lineidx = 0;
	for (i = 0; i < DISPLAY_LINES; i++)
		linepos[i] = 0;
}

static void append_line(void)
{
	XawTextBlock blk;
	char buf[1000];

	if (curline >= MAX_TEXT_LINES)
		init_line();

	linepos[lineidx++] = textpos;
	if (lineidx >= DISPLAY_LINES) lineidx = 0;

	for (;;) {
		if (fgets(buf, sizeof(buf), p_input) == NULL)
			return;
		blk.firstPos = 0;
		blk.length = strlen(buf);
		blk.ptr = buf;
		blk.format = XawFmt8Bit;
		XawTextReplace(textwin, textpos, textpos, &blk);
		textpos += blk.length;
		if (blk.length > 0 && buf[blk.length-1] == '\n')
			break;
	}
	curline++;
	if (curline >= DISPLAY_LINES) {
		XtVaSetValues(textwin, XtNdisplayPosition, linepos[lineidx], NULL);
	}
}

/*----------------------------------------------------------------
 * popup error message
 *----------------------------------------------------------------*/

static void popup_msg(void)
{
	int c;
	while ((c = getc(p_input)) != '\n')
		putc(c, stderr);
	putc('\n', stderr);
	/* skip to next song */
	ext_control_next();
}

/*----------------------------------------------------------------
 * tell a midi file name to player
 *----------------------------------------------------------------*/

static void do_play(void)
{
	playing_status();
	ext_control_load(filename[curfile]);
	set_index();
}

/*----------------------------------------------------------------
 * status change
 *----------------------------------------------------------------*/

/* display index number */
static void set_index(void)
{
	char local[20];
	sprintf(local, "%03d/%03d", curfile+1, number_of_files);
	XtVaSetValues(title_index, XtNlabel, local, NULL);
}

/* display title */
static void set_title(char *title)
{
	XtVaSetValues(title_name, XtNlabel, title, NULL);
}

/* display total & current time */
static void set_curtime(int sec)
{
	char local[20];
	sprintf(local, "%02d:%02d/%02d:%02d",
		sec / 60, sec % 60, total_time / 60, total_time % 60);
	XtVaSetValues(title_time, XtNlabel, local, NULL);
	cur_time = sec;
}

/* reset all status */
static void clear_status(void)
{
	curfile = -1;
	total_time = 0;
	set_curtime(0);
	set_title("------------------");
	XtVaSetValues(title_index, XtNlabel, "000/000", NULL);
	XtSetSensitive(btn_play, True);
	XtSetSensitive(btn_prev, False);
	XtSetSensitive(btn_next, False);
	XtSetSensitive(btn_pause, False);
	XtSetSensitive(btn_fwrd, False);
	XtSetSensitive(btn_back, False);
	curstate = STAT_CLEAN;
}

/* change to playing mode */
static void playing_status(void)
{
	if (curstate == STAT_PLAYING)
		return;
	XtSetSensitive(btn_play, False);
	XtSetSensitive(btn_prev, True);
	XtSetSensitive(btn_next, True);
	XtSetSensitive(btn_pause, True);
	XtSetSensitive(btn_fwrd, True);
	XtSetSensitive(btn_back, True);
	curstate = STAT_PLAYING;
}

/* change to pause mode */
static void paused_status(void)
{
	if (curstate == STAT_PAUSE)
		return;
	XtSetSensitive(btn_play, False);
	XtSetSensitive(btn_prev, False);
	XtSetSensitive(btn_next, False);
	XtSetSensitive(btn_pause, True);
	XtSetSensitive(btn_fwrd, False);
	XtSetSensitive(btn_back, False);
	curstate = STAT_PAUSE;
}

/*----------------------------------------------------------------
 * timer callback procedure
 *----------------------------------------------------------------*/

static void TimerCB()
{
	int sec;

	/* redisplay the current time */
	sec = extpanel->curcs / 100;
	if (sec != cur_time)
		set_curtime(sec);

	/* reset timer callback again */
	XtAppAddTimeOut(apps, DELTA_TIME, (XtTimerCallbackProc)TimerCB, NULL);
}
