/* FFT and Convolve */

#include "snd.h"

/* fft and convolution of real data in zero-based arrays */

static void shuffle (float* rl, float* im, int n)
{
  /* bit reversal */

  int i,m,j;
  float tempr,tempi;
  j=0;
  for (i=0;i<n;i++)
    {
      if (j>i)
	{
	  tempr = rl[j];
	  tempi = im[j];
	  rl[j] = rl[i];
	  im[j] = im[i];
	  rl[i] = tempr;
	  im[i] = tempi;
	}
      m = n>>1;
      while ((m>=2) && (j>=m))
	{
	  j -= m;
	  m = m>>1;
	}
      j += m;
    }
}

void c_fft (float *rl, float *im, int n, int isign, int ipow)
{
  /* standard fft: real part in rl, imaginary in im,        */
  /* ipow = ceiling (log n / log 2), isign=1 fft, =-1 ifft. */
  /* rl and im are zero-based.                              */
  /*                                                        */
  /* oddly enough, the integer version (using integer ops   */
  /* and "block floating point" scaling) was much slower,   */
  /* and splitting out the no-ops (twiddle factors==0 etc)  */
  /* made no difference at all.                             */

  int mmax,j,pow,prev,lg,i,ii,jj;
  float wrs,wis,tempr,tempi;
  double wr,wi,theta,wtemp,wpr,wpi;

  shuffle(rl,im,n);
  mmax = 2;
  prev = 1;
  pow = (int)(n*0.5);
  theta = (PI*isign);
  for (lg=0;lg<ipow;lg++)
    {
      wpr = cos(theta);
      wpi = sin(theta);
      wr = 1.0;
      wi = 0.0;
      for (ii=0;ii<prev;ii++)
	{
	  wrs = (float) wr;
	  wis = (float) wi;
#ifdef LINUX 
	  if (isnan(wis)) wis=0.0;
#endif
	  i = ii;
	  j = ii + prev;
	  for (jj=0;jj<pow;jj++)
	    {
	      tempr = wrs*rl[j] - wis*im[j];
	      tempi = wrs*im[j] + wis*rl[j];
	      rl[j] = rl[i] - tempr;
	      im[j] = im[i] - tempi;
	      rl[i] += tempr;
	      im[i] += tempi;
	      i += mmax;
	      j += mmax;
	    }
	  wtemp = wr;
	  wr = (wr*wpr) - (wi*wpi);
	  wi = (wi*wpr) + (wtemp*wpi);
	}
      pow = (int)(pow*0.5);
      prev = mmax;
      theta = theta*0.5;
      mmax = mmax*2;
    }
}

void convolve (float* rl1, float* rl2, int n, int ipow)
{
  /* convolves two real arrays.                                           */
  /* rl1 and rl2 are assumed to be set up correctly for the convolution   */
  /* (that is, rl1 (the "signal") is zero-padded by length of             */
  /* (non-zero part of) rl2 and rl2 is stored in wrap-around order)       */
  /* We treat rl2 as the imaginary part of the first fft, then do         */
  /* the split, scaling, and (complex) spectral multiply in one step.     */
  /* result in rl1                                                        */

  int j,n2,nn2;
  float rem,rep,aim,aip,invn;

  c_fft(rl1,rl2,n,1,ipow);
  
  n2=(int)(n*0.5);
  invn = 0.25/n;
  rl1[0] = ((rl1[0]*rl2[0])/n);
  rl2[0] = 0.0;

  for (j=1;j<=n2;j++)
    {
      nn2 = n-j;
      rep = (rl1[j]+rl1[nn2]);
      rem = (rl1[j]-rl1[nn2]);
      aip = (rl2[j]+rl2[nn2]);
      aim = (rl2[j]-rl2[nn2]);

      rl1[j] = invn*(rep*aip + aim*rem);
      rl1[nn2] = rl1[j];
      rl2[j] = invn*(aim*aip - rep*rem);
      rl2[nn2] = -rl2[j];
    }
  
  c_fft(rl1,rl2,n,-1,ipow);
}

/* file-based fft for huge convolutions that can't fit in memory */
/* borrowed, with various changes, from Numerical Recipes */

static void rewind_and_renumber_files(FILE **files, int *na, int *nb, int *nc, int *nd)
{
  int i;
  FILE *ftemp;
  for (i=1;i<=4;i++) rewind(files[i]);
  ftemp = files[2];
  files[2] = files[4];
  files[4] = ftemp;
  ftemp = files[1];
  files[1] = files[3];
  files[3] = ftemp;
  *na=3;
  *nb=4;
  *nc=1;
  *nd=2;
}

static int fft_file(FILE **files, unsigned int n, unsigned int fft_size, int isign, float *afa, float *afb, float *afc)
{
  /* files[1..4] are the data file pointers
         first half of data in file[1], second half in file[2]
     size = desired fft size
     fft_size = fft size we can accomodate in memory
     isign = direction
     return 0 if no error, output split in file[3] and file[4]
  */
  unsigned long j,j12,jk,k,kk,mm,kc=0,kd,ks,kr,nr,ns;
  int cc,na,nb,nc,nd,err = 0;
  float tempr,tempi;
  double wr,wi,wpr,wpi,wtemp,theta;
  static int mate[5] = {0,2,1,4,3};
  jk = n;
  mm = n;
  ns = n/fft_size;
  nr = ns>>1;
  kd = fft_size>>1;
  ks = n;
  rewind_and_renumber_files(files,&na,&nb,&nc,&nd);
  for (;;)
    {
      theta = isign * 3.141592653589793 / (n/mm);
      wtemp = sin(0.5 * theta);
      wpr = -2.0 * wtemp * wtemp;
      wpi = sin(theta);
      wr = 1.0;
      wi = 0.0;
      mm >>= 1;
      for (j12=1;j12<=2;j12++)
	{
	  kr = 0;
	  do {
	    cc = fread(afa,sizeof(float),fft_size,files[na]);
	    if (cc != (int)fft_size) {err = 1; return(err);}
	    cc = fread(afb,sizeof(float),fft_size,files[nb]);
	    if (cc != (int)fft_size) {err = 2; return(err);}
	    for (j=1;j<=fft_size;j+=2)
	      {
		tempr = wr*afb[j-1] - wi*afb[j];
		tempi = wi*afb[j-1] + wr*afb[j];
		afb[j-1] = afa[j-1] - tempr;
		afa[j-1] += tempr;
		afb[j] = afa[j] - tempi;
		afa[j] += tempi;
	      }
	    kc += kd;
	    if (kc == mm)
	      {
		kc = 0;
		wtemp = wr;
		wr = wtemp * wpr - wi * wpi + wr;
		wi = wi * wpr + wtemp * wpi + wi;
	      }
	    cc = fwrite(afa,sizeof(float),fft_size,files[nc]);
	    if (cc != (int)fft_size) {err = 3; return(err);}
	    cc = fwrite(afb,sizeof(float),fft_size,files[nd]);
	    if (cc != (int)fft_size) {err = 4; return(err);}
	  } while (++kr < nr);
	  if ((j12 == 1) && (ks != n) && (ks == fft_size))
	    {
	      na = mate[na];
	      nb = na;
	    }
	  /* if (nr == 0) break; */
	}
      rewind_and_renumber_files(files,&na,&nb,&nc,&nd);
      jk >>= 1;
      if (jk == 1) mm = n;
      ks >>= 1;
      if (ks > fft_size)
	{
	  for (j12=1;j12<=2;j12++)
	    {
	      for (kr=1;kr<=ns;kr+=ks/fft_size)
		{
		  for (k=1;k<=ks;k+=fft_size)
		    {
		      cc = fread(afa,sizeof(float),fft_size,files[na]);
		      if (cc != (int)fft_size) {err = 5; return(err);}
		      cc = fwrite(afa,sizeof(float),fft_size,files[nc]);
		      if (cc != (int)fft_size) {err = 6; return(err);}
		    }
		  nc = mate[nc];
		}
	      na = mate[na];
	    }
	  rewind_and_renumber_files(files,&na,&nb,&nc,&nd);
	}
      else 
	{
	  if (ks == fft_size) 
	    nb = na;
	  else break;
	}
    }
  j = 1;
  for (;;)
    {
      theta = isign * 3.141592653589793 / (n/mm);
      wtemp = sin(0.5 * theta);
      wpr = -2.0 * wtemp * wtemp;
      wpi = sin(theta);
      wr = 1.0;
      wi = 0.0;
      mm >>= 1;
      ks = kd;
      kd >>= 1;
      for (j12=1;j12<=2;j12++)
	{
	  for (kr=1;kr<=ns;kr++)
	    {
	      cc = fread(afc,sizeof(float),fft_size,files[na]);
	      if (cc != (int)fft_size) {err = 7; return(err);}
	      kk = 1;
	      k = ks + 1;
	      for (;;)
		{
		  tempr = wr*afc[kk+ks-1] - wi*afc[kk+ks];
		  tempi = wi*afc[kk+ks-1] + wr*afc[kk+ks];
		  afa[j-1] = afc[kk-1] + tempr;
		  afb[j-1] = afc[kk-1] - tempr;
		  afa[j] = afc[kk] + tempi;
		  afb[j] = afc[kk] - tempi;
		  j+=2;
		  kk+=2;
		  if (kk < k) continue;
		  kc += kd;
		  if (kc == mm)
		    {
		      kc = 0;
		      wtemp = wr;
		      wr = wtemp*wpr - wi*wpi + wr;
		      wi = wi*wpr + wtemp*wpi + wi;
		    }
		  kk += ks;
		  if (kk > fft_size) 
		    break;
		  else k = kk+ks;
		}
	      if (j > fft_size)
		{
		  cc = fwrite(afa,sizeof(float),fft_size,files[nc]);
		  if (cc != (int)fft_size) {err = 8; return(err);}
		  cc = fwrite(afb,sizeof(float),fft_size,files[nd]);
		  if (cc != (int)fft_size) {err = 9; return(err);}
		  j = 1;
		}
	    }
	  na = mate[na];
	}
      rewind_and_renumber_files(files,&na,&nb,&nc,&nd);
      jk >>= 1;
      if (jk > 1) continue;
      break;
    }
  return(err);
}

static char *convolve_strbuf = NULL;

static char *convolve_tempname(char *fname, int current)
{
  /* use user-supplied convolve temp filename to conjure up other such names */
  int i,len,j;
  char *tmp;
  static char mus_itoa[13]={'a','b','c','d','e','f','g','h','i','j','k','l','m'};
  if (convolve_strbuf == NULL) convolve_strbuf = (char *)CALLOC(128,sizeof(char));
  if ((fname) && (*fname))
    tmp = fname;
  else tmp = "mus_tmp";
  len = strlen(tmp);
  if (len >= 128) len = 126;
  for (i=0,j=0;i<len;i++)
    {
      if ((current>0) && (tmp[i] == '.')) {convolve_strbuf[j++] = mus_itoa[current]; current = 0;}
      convolve_strbuf[j++] = tmp[i];
    }
  if (current != 0) /* no extension? */
    convolve_strbuf[j++] = mus_itoa[current];
  convolve_strbuf[j]='\0';
  return(convolve_strbuf);
}

static int file_convolve(int tempfile, float amp, char *fname, int filec, int filterc,
			 int fftsize, int ipow, int chans, int chan, int file_fft_size,
			 float *rl0, float *rl1, float *rl2, int **pbuffer,int *cm,
			 int **fbuffer, int *fcm,int filter_chans,int filter_chan)
{
  /* file-based fft's used for convolution
   * since I'm assuming we want to maintain the sequential reads/writes, I can't combine the ffts 
   * so the process is:
   *   split input1 into 2 halves zero padded to true fft size (files 1,2) (imaginary part 0)
   *   fft writing files 3,4
   *   split input2
   *   fft writing files 5,6
   *   run through 3,4,5,6 doing spectral multiply, writing 1,2
   *   inverse fft 1,2 writing 3,4
   *   get max amp 3,4
   *   combine 3,4 into result sound file with scaling
   *   remove files 1,2,3,4,5,6
   * so we need true fft size * 7 * sizeof(float) bytes of disk space
   * that is, approximately (in1+in2)*20 secs worth 
   */
  FILE *files[6];
  int i,j,k,inctr,infile,fftsize2,filefftsize2,err,curchans,curchan;
  int **curbuf;
  int *curcm;
  float *rl3;
  float temp,maxval;

  cm[chan] = 1;
  fcm[filter_chan] = 1;
  fftsize2 = fftsize>>1;
  filefftsize2 = file_fft_size >> 1;
  rl3 = (float *)CALLOC(file_fft_size,sizeof(float));

  for (inctr=0;inctr<2;inctr++)
    {
      if (inctr==0) 
	{
	  infile = filterc;
	  curcm = fcm;
	  curbuf = fbuffer;
	  curchans = filter_chans;
	  curchan = filter_chan;
	}
      else 
	{
	  infile = filec;
	  curcm = cm;
	  curbuf = pbuffer;
	  curchans = chans;
	  curchan = chan;
	}
      /* first half of first signal in zero-padded file 1 */
      files[0] = fopen(convolve_tempname(fname,1+4*inctr),"w");
      for (i=0;i<fftsize2;i+=file_fft_size)
	{
	  mus_read_any(infile,0,curchans,file_fft_size-1,curbuf,curcm);
	  /* mus_read(infile,0,file_fft_size-1,1,&(pbuffer[chan])); */
	  for (j=0,k=0;j<filefftsize2;j++,k+=2) {rl0[k] = (SNDLIB_SNDFLT * curbuf[curchan][j]); rl0[k+1] = 0.0;}
	  fwrite(rl0,sizeof(float),file_fft_size,files[0]);
	  for (j=filefftsize2,k=0;j<file_fft_size;j++,k+=2) {rl0[k] = (SNDLIB_SNDFLT * curbuf[curchan][j]); rl0[k+1] = 0.0;}
	  fwrite(rl0,sizeof(float),file_fft_size,files[0]);
	}
      for (i=0;i<file_fft_size;i++) rl0[i] = 0.0;
      for (i=0;i<fftsize2;i+=file_fft_size) fwrite(rl0,sizeof(float),file_fft_size,files[0]);
      fclose(files[0]);
      /* second half of first signal to zero-padded file 2 */
      files[0] = fopen(convolve_tempname(fname,2+4*inctr),"w");
      for (i=0;i<fftsize2;i+=file_fft_size)
	{
	  mus_read_any(infile,0,curchans,file_fft_size-1,curbuf,curcm);
	  for (j=0,k=0;j<filefftsize2;j++,k+=2) {rl0[k] = (SNDLIB_SNDFLT * curbuf[curchan][j]); rl0[k+1] = 0.0;}
	  fwrite(rl0,sizeof(float),file_fft_size,files[0]);
	  for (j=filefftsize2,k=0;j<file_fft_size;j++,k+=2) {rl0[k] = (SNDLIB_SNDFLT * curbuf[curchan][j]); rl0[k+1] = 0.0;}
	  fwrite(rl0,sizeof(float),file_fft_size,files[0]);
	}
      for (i=0;i<file_fft_size;i++) rl0[i] = 0.0;
      for (i=0;i<fftsize2;i+=file_fft_size) fwrite(rl0,sizeof(float),file_fft_size,files[0]);
      fclose(files[0]);
      /* create output files */
      files[0] = fopen(convolve_tempname(fname,3+4*inctr),"w");
      for (i=0;i<fftsize;i+=file_fft_size) fwrite(rl0,sizeof(float),file_fft_size,files[0]);
      fclose(files[0]);
      files[0] = fopen(convolve_tempname(fname,4+4*inctr),"w");
      for (i=0;i<fftsize;i+=file_fft_size) fwrite(rl0,sizeof(float),file_fft_size,files[0]);
      fclose(files[0]);

      /* re-open 4 temp files and run the fft */
      files[1] = fopen(convolve_tempname(fname,1+4*inctr),"r+");
      files[2] = fopen(convolve_tempname(fname,2+4*inctr),"r+");
      files[3] = fopen(convolve_tempname(fname,3+4*inctr),"r+");
      files[4] = fopen(convolve_tempname(fname,4+4*inctr),"r+");
      err = fft_file(files,fftsize,file_fft_size,1,rl0,rl1,rl2);
      if (err) {fprintf(stdout,"fft 1: %d\n",err); fflush(stdout);}
      for (i=1;i<=4;i++) fclose(files[i]);

      remove(convolve_tempname(fname,1+4*inctr));
      remove(convolve_tempname(fname,2+4*inctr));
    }

  /* now the spectral multiply (reading 3,4 and 5,6 writing 1,2 */
  files[0] = fopen(convolve_tempname(fname,3),"r");
  files[1] = fopen(convolve_tempname(fname,4),"r");
  files[2] = fopen(convolve_tempname(fname,7),"r");
  files[3] = fopen(convolve_tempname(fname,8),"r");
  files[4] = fopen(convolve_tempname(fname,9),"w");
  files[5] = fopen(convolve_tempname(fname,10),"w");

  for (i=0;i<fftsize;i+=file_fft_size) /* do both halves at once */
    {
      fread(rl0,sizeof(float),file_fft_size,files[0]);
      fread(rl1,sizeof(float),file_fft_size,files[1]);
      fread(rl2,sizeof(float),file_fft_size,files[2]);
      fread(rl3,sizeof(float),file_fft_size,files[3]);
      
      for (j=0;j<file_fft_size;j+=2)
	{
	  temp = rl0[j];
	  rl0[j] = temp*rl2[j] - rl0[j+1]*rl2[j+1];
	  rl0[j+1] = temp*rl2[j+1] + rl0[j+1]*rl2[j];

	  temp = rl1[j];
	  rl1[j] = temp*rl3[j] - rl1[j+1]*rl3[j+1];
	  rl1[j+1] = temp*rl3[j+1] + rl1[j+1]*rl3[j];
	}

      fwrite(rl0,sizeof(float),file_fft_size,files[4]);
      fwrite(rl1,sizeof(float),file_fft_size,files[5]);
    }
  for (i=0;i<=5;i++) fclose(files[i]);

  remove(convolve_tempname(fname,3));
  remove(convolve_tempname(fname,4));
  remove(convolve_tempname(fname,7));
  remove(convolve_tempname(fname,8));

  /* now inverse fft 1,2 -> 3,4 */
  /* re-create output files */
  for (i=0;i<file_fft_size;i++) rl0[i] = 0.0;
  files[0] = fopen(convolve_tempname(fname,11),"w");
  for (i=0;i<fftsize;i+=file_fft_size) fwrite(rl0,sizeof(float),file_fft_size,files[0]);
  fclose(files[0]);
  files[0] = fopen(convolve_tempname(fname,12),"w");
  for (i=0;i<fftsize;i+=file_fft_size) fwrite(rl0,sizeof(float),file_fft_size,files[0]);
  fclose(files[0]);
  files[1] = fopen(convolve_tempname(fname,9),"r+");
  files[2] = fopen(convolve_tempname(fname,10),"r+");
  files[3] = fopen(convolve_tempname(fname,11),"r+");
  files[4] = fopen(convolve_tempname(fname,12),"r+");
  for (i=0;i<file_fft_size;i++) {rl0[i] = 0.0; rl1[i] = 0.0; rl2[i] = 0.0;}
  err = fft_file(files,fftsize,file_fft_size,-1,rl0,rl1,rl2);
  if (err) {fprintf(stdout,"fft 3: %d\n",err); fflush(stdout);}
  for (i=1;i<=4;i++) fclose(files[i]);

  remove(convolve_tempname(fname,9));
  remove(convolve_tempname(fname,10));

  if (amp != 0.0)
    {
      /* get max-amp */
      files[0] = fopen(convolve_tempname(fname,11),"r");
      files[1] = fopen(convolve_tempname(fname,12),"r");
      maxval = 0.0;
      for (i=0;i<fftsize;i+=file_fft_size)
	{
	  fread(rl0,sizeof(float),file_fft_size,files[0]);
	  fread(rl1,sizeof(float),file_fft_size,files[1]);
	  for (j=0;j<file_fft_size;j+=2) /* ignore imaginary part */
	    {
	      if (rl0[j] > maxval) maxval = rl0[j];
	      if (rl1[j] > maxval) maxval = rl1[j];
	      if ((-rl0[j]) > maxval) maxval = -rl0[j];
	      if ((-rl1[j]) > maxval) maxval = -rl1[j];
	    }
	}
      if (maxval != 0.0) maxval = (amp * (SNDLIB_SNDFIX - .1)) / maxval;
      fclose(files[0]);
      fclose(files[1]);
    }
  else maxval = 1.0;
  /* now combine 3,4 into a standard sound file */
  if (amp != 0.0)
    {
      /* prepare temp file (16-bit linear means the convolution results need to be normalized) */
      mus_write_next_header(tempfile,22050,1,28,fftsize*2,SNDLIB_16_LINEAR,NULL,0);
      mus_open_file_descriptors(tempfile,SNDLIB_16_LINEAR,2,28);
    }
  else
    {
      mus_write_next_header(tempfile,22050,1,28,fftsize*4,SNDLIB_32_LINEAR,NULL,0);
      mus_open_file_descriptors(tempfile,SNDLIB_32_LINEAR,4,28);
    }

  /* get the convolution data */
  files[0] = fopen(convolve_tempname(fname,11),"r");
  for (i=0;i<fftsize;i+=file_fft_size)
    {
      fread(rl0,sizeof(float),file_fft_size,files[0]);
      for (j=0,k=0;j<file_fft_size;j+=2,k++) /* just real part */
	pbuffer[chan][k] = (int)(maxval * rl0[j]);
      mus_write(tempfile,0,k - 1,1,&(pbuffer[chan]));
    }
  fclose(files[0]);
  files[0] = fopen(convolve_tempname(fname,12),"r");
  for (i=0;i<fftsize;i+=file_fft_size)
    {
      fread(rl0,sizeof(float),file_fft_size,files[0]);
      for (j=0,k=0;j<file_fft_size;j+=2,k++)
	pbuffer[chan][k] = (int)(maxval * rl0[j]);
      mus_write(tempfile,0,k - 1,1,&(pbuffer[chan]));
    }
  fclose(files[0]);
  mus_close(tempfile);
  FREE(rl3);
  remove(convolve_tempname(fname,11));
  remove(convolve_tempname(fname,12));
  return(0);
}

void c_convolve (char *fname, float amp,
           int filec, int filehdr,
           int filterc, int filterhdr, int filtersize,
           int start, int fftsize, int ipow, int chans, int chan, int max_fft_size, int filter_chans, int filter_chan, int data_size)
{
  float *rl0 = NULL,*rl1 = NULL, *rl2 = NULL;
  int **pbuffer = NULL,**fbuffer = NULL;
  int *cm = NULL,*fcm = NULL;
  int i,allocd,reqd,minsize;
  float scl;
  int tempfile;
  int file_fft_size;
  if (max_fft_size <= 0) max_fft_size = fftsize*2;

  /* need file to hold convolution output */
  tempfile=mus_create(fname);
  if (tempfile != -1)
    {
      /* get to start point in the two sound files */
      mus_seek(filec,filehdr + start*chans*2,SEEK_SET);
      mus_seek(filterc,filterhdr,SEEK_SET);

      if (fftsize < max_fft_size)
	{
	  rl0 = (float *)CALLOC(fftsize,sizeof(float));
	  if (rl0) rl1 = (float *)CALLOC(fftsize,sizeof(float));
	  if (rl1) pbuffer = (int **)CALLOC(chans,sizeof(int *));
	  if (pbuffer) pbuffer[chan] = (int *)CALLOC(data_size,sizeof(int));
	  if (pbuffer[chan]) cm = (int *)CALLOC(chans,sizeof(int));
	  minsize = fftsize;
	  fbuffer = (int **)CALLOC(filter_chans,sizeof(int *));
	  if (fbuffer) fbuffer[filter_chan] = (int *)CALLOC(filtersize,sizeof(int));
	  if (fbuffer[filter_chan]) fcm = (int *)CALLOC(filter_chans,sizeof(int));
	}
      else minsize = max_fft_size;
      if ((rl0 == NULL) || (rl1 == NULL) || 
	  (pbuffer == NULL) || (pbuffer[chan] == NULL) || (cm == NULL) ||
	  (fbuffer == NULL) || (fbuffer[filter_chan] == NULL) || (fcm == NULL))
	{
	  /* try to fallback on file (disk) based fft */
	  /* free current buffers, try 65536, then  4096, then 256, then give up */
	  reqd = (fftsize * 2 * sizeof(float)) + (chans * sizeof(int *)) + (fftsize * sizeof(int)) + (chans * sizeof(int));
	  allocd = 0;
	  if (fftsize < max_fft_size)
	    {
	      if (rl0 != NULL) allocd += (fftsize * sizeof(float));
	      if (rl1 != NULL) allocd += (fftsize * sizeof(float));
	      if (pbuffer != NULL) allocd += (chans * sizeof(int *));
	      if (pbuffer[chan] != NULL) allocd += (data_size * sizeof(int));
	      if (cm != NULL) allocd += (chans * sizeof(int));
	      if (fbuffer != NULL) allocd += (filter_chans * sizeof(int *));
	      if (fbuffer[filter_chan] != NULL) allocd += (filtersize * sizeof(int));
	      if (fcm != NULL) allocd += (filter_chans * sizeof(int));
	    }
	  if (minsize > 65536) 
	    file_fft_size = 65536;
	  else
	    {
	      if (minsize > 4096)
		file_fft_size = 4096;
	      else file_fft_size = 256;
	    }
	  while ((cm == NULL) && (file_fft_size >= 256))
	    {
	      if (rl0) {FREE(rl0); rl0 = NULL;}
	      if (rl1) {FREE(rl1); rl1 = NULL;}
	      if (rl2) {FREE(rl2); rl2 = NULL;}
	      if (pbuffer) 
		{
		  if (pbuffer[chan]) FREE(pbuffer[chan]);
		  FREE(pbuffer); 
		  pbuffer = NULL;
		}
	      if (cm) {FREE(cm); cm = NULL;}
	      if (fbuffer) 
		{
		  if (fbuffer[filter_chan]) FREE(fbuffer[filter_chan]);
		  FREE(fbuffer); 
		  fbuffer = NULL;
		}
	      if (fcm) {FREE(fcm); fcm = NULL;}
	      rl0 = (float *)CALLOC(file_fft_size,sizeof(float));
	      if (rl0) rl1 = (float *)CALLOC(file_fft_size,sizeof(float));
	      if (rl1) rl2 = (float *)CALLOC(file_fft_size,sizeof(float));
	      if (rl2) pbuffer = (int **)CALLOC(chans,sizeof(int *));
	      if (pbuffer) pbuffer[chan] = (int *)CALLOC(file_fft_size,sizeof(int));
	      if (pbuffer[chan]) cm = (int *)CALLOC(chans,sizeof(int));
	      if (cm == NULL) file_fft_size >>= 4;
	      fbuffer = (int **)CALLOC(filter_chans,sizeof(int *));
	      if (fbuffer) fbuffer[filter_chan] = (int *)CALLOC(filtersize,sizeof(int));
	      if (fbuffer[filter_chan]) fcm = (int *)CALLOC(filter_chans,sizeof(int));
	      if (fcm == NULL) file_fft_size >>= 4;
	    }
	  if (rl2 == NULL) rl2 = (float *)CALLOC(file_fft_size,sizeof(float));
	  if (file_fft_size < 256) 
	    mus_error(MUS_MEMORY_ALLOCATION_FAILED,"convolve needs %d bytes, but got %d (fft size: %d)",reqd,allocd,fftsize);
	  else
	    {
	      file_convolve(tempfile,amp,fname,filec,filterc,fftsize,ipow,chans,chan,file_fft_size,rl0,rl1,rl2,
			    pbuffer,cm,fbuffer,fcm,filter_chans,filter_chan);
	    }
	}
      else
	{
	  cm[chan] = 1;
	  fcm[filter_chan] = 1;

	  /* read in the "impulse response" */
	  mus_read_any(filterc,0,filter_chans,filtersize-1,fbuffer,fcm);
	  for (i=0;i<filtersize;i++) 
	    {
	      rl1[i] = (SNDLIB_SNDFLT * fbuffer[filter_chan][i]);
	    }

	  if (amp != 0.0)
	    {
	      /* prepare temp file (16-bit linear means the convolution results need to be normalized) */
	      mus_write_next_header(tempfile,22050,1,28,data_size*2,SNDLIB_16_LINEAR,NULL,0);
	      mus_open_file_descriptors(tempfile,SNDLIB_16_LINEAR,2,28);
	    }
	  else
	    {
	      mus_write_next_header(tempfile,22050,1,28,data_size*4,SNDLIB_32_LINEAR,NULL,0);
	      mus_open_file_descriptors(tempfile,SNDLIB_32_LINEAR,4,28);
	    }
	  /* get the convolution data */
	  mus_read_any(filec,0,chans,data_size-1,pbuffer,cm);
	  for (i=0;i<data_size;i++) rl0[i] = (SNDLIB_SNDFLT * pbuffer[chan][i]);

	  convolve(rl0,rl1,fftsize,ipow);
	  
	  if (amp != 0.0)
	    {
	      /* normalize the results */
	      scl = 0.0;
	      for (i=0;i<data_size;i++) 
		{
		  if (rl0[i] > scl) scl=rl0[i];
		  else if (rl0[i] < -scl) scl=(-rl0[i]);
		}
	      if (scl != 0.0) scl = (amp * (SNDLIB_SNDFIX - .1)) / scl;
	      for (i=0;i<data_size;i++) pbuffer[chan][i] = (int)(scl * rl0[i]);
	    }
	  /* and save as temp file */
	  mus_write(tempfile,0,data_size-1,1,&(pbuffer[chan]));
	  mus_close(tempfile);
	}
      if (rl0) FREE(rl0);
      if (rl1) FREE(rl1);
      if (rl2) FREE(rl2);
      if (pbuffer) 
	{
	  if (pbuffer[chan]) FREE(pbuffer[chan]);
	  FREE(pbuffer);
	}
      if (cm) FREE(cm);
      if (fbuffer) 
	{
	  if (fbuffer[filter_chan]) FREE(fbuffer[filter_chan]);
	  FREE(fbuffer);
	}
      if (fcm) FREE(fcm);
    }
}
