/*
 * dfile.c - AIX file processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.4 99/03/29 07:09:57 abe Exp $";
#endif


#include "lsof.h"


/*
 * Local structures
 */

struct hsfile {
	struct sfile *s;		/* the Sfile table address */
	struct hsfile *next;		/* the next hash bucket entry */
};


/*
 * Local static variables
 */

static struct hsfile *HbyFdi =		/* hash by file buckets */
	(struct hsfile *)NULL;
static int HbyFdiCt = 0;		/* HbyFdi entry count */
static struct hsfile *HbyFsd =		/* hash by file system buckets */
	(struct hsfile *)NULL;
static int HbyFsdCt = 0;		/* HbyFsd entry count */
static struct hsfile *HbyMPC =		/* hash by MPC file buckets */
	(struct hsfile *)NULL;
static int HbyMPCCt = 0;		/* HbyMPC entry count */
static struct hsfile *HbyNm =		/* hash by name buckets */
	(struct hsfile *)NULL;
static int HbyNmCt = 0;			/* HbyNm entry count */


/*
 * Local definitions
 */

#define	SFDIHASH	4094		/* Sfile hash by (device,inode) number
					 * pair bucket count (power of 2!) */
#define	SFFSHASH	128		/* Sfile hash by file system device
					 * number bucket count (power of 2!) */
#define SFHASHDEVINO(maj, min, ino, mod) ((int)(((int)((((int)(maj+1))*((int)((min+1))))+ino)*31415)&(mod-1)))
					/* hash for Sfile by major device,
					 * minor device, and inode, modulo m
					 * (m must be a power of 2) */
#define	SFMPCHASH	1024		/* Sfile hash by MPC device number */
#define	SFNMHASH	4096		/* Sfile hash by name bucket count
					   (power of 2!) */


/*
 * hashSfile() - hash Sfile entries for use in is_file_named() searches
 */

void
hashSfile()
{
	static int hs = 0;
	int i;
	struct sfile *s;
	struct hsfile *sh, *sn;
/*
 * Do nothing if there are no file search arguments cached or if the
 * hashes have already been constructed.
 */
	if (!Sfile || hs)
	    return;
/*
 * Allocate hash buckets by (device,inode), file system device, MPC device,
 * and file name.
 */
	if (!(HbyFdi = (struct hsfile *)calloc((MALLOC_S)SFDIHASH,
					       sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d (dev,ino) hash buckets\n",
		Pn, SFDIHASH);
	    Exit(1);
	}
	if (!(HbyFsd = (struct hsfile *)calloc((MALLOC_S)SFFSHASH,
					       sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d file sys hash buckets\n",
		Pn, SFFSHASH);
	    Exit(1);
	}
	if (!(HbyMPC = (struct hsfile *)calloc((MALLOC_S)SFMPCHASH,
					       sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d MPC file hash buckets\n",
		Pn, SFMPCHASH);
	    Exit(1);
	}
	if (!(HbyNm = (struct hsfile *)calloc((MALLOC_S)SFNMHASH,
					      sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d name hash buckets\n",
		Pn, SFNMHASH);
	    Exit(1);
	}
	hs++;
/*
 * Scan the Sfile chain, building file, file system, MPC file, and file
 * name hash bucket chains.
 */
	for (s = Sfile; s; s = s->next) {
	    for (i = 0; i < 3; i++) {
		if (i == 0) {
		    if (!s->aname)
			continue;
		    sh = &HbyNm[hashbyname(s->aname, SFNMHASH)];
		    HbyNmCt++;
		} else if (i == 1) {
		    if (s->type) {
			sh = &HbyFdi[SFHASHDEVINO(major(s->dev),
						  minor(s->dev),
						  s->i,
						  SFDIHASH)];
			HbyFdiCt++;
		    } else {
			sh = &HbyFsd[SFHASHDEVINO(major(s->dev),
						 minor(s->dev),
						 0,
						 SFFSHASH)];
			HbyFsdCt++;
		    }
		} else {
		    if (!s->type || s->mode != S_IFCHR)
			continue;
		    sh = &HbyMPC[SFHASHDEVINO(major(s->dev),
					      minor(s->dev),
					      0,
					      SFMPCHASH)];
		    HbyMPCCt++;
		}
		if (!sh->s) {
		    sh->s = s;
		    sh->next = (struct hsfile *)NULL;
		    continue;
		} else {
		    if (!(sn = (struct hsfile *)malloc(
				(MALLOC_S)sizeof(struct hsfile))))
		    {
			(void) fprintf(stderr,
			    "%s: can't allocate hsfile bucket for: %s\n",
			    Pn, s->aname);
			Exit(1);
		    }
		    sn->s = s;
		    sn->next = sh->next;
		    sh->next = sn;
		}
	    }
	}
}


/*
 * is_file_named() - is file named?
 */

int
is_file_named(p, ty, ch, ic)
	char *p;			/* path name; NULL = search by device
					 * and inode (from *Lf) */
	enum vtype ty;			/* vnode type */
	chan_t ch;			/* gnode channel */
	int ic;				/* is clone file (4.1.4 and above) */
{
	int f = 0;
	struct sfile *s;
	struct hsfile *sh;

#if	AIXV>=4140
/*
 * Check for a clone match.
 */
	readdev(0);			/* for CloneMaj */
	if (ic
	&&  HbyFdiCt
	&&  CloneMaj >= 0
	&&  Lf->dev_def
	&&  (Lf->inp_ty == 1 || Lf->inp_ty == 3))
	{
	    for (sh = &HbyFdi[SFHASHDEVINO(CloneMaj, major(Lf->dev), Lf->inode,
					   SFDIHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s)
		&&  (major(s->dev) == CloneMaj)
		&&  (minor(s->dev) == major(Lf->dev))
		&&  (s->i == (ino_t)Lf->inode))
		{
		    f = 3;
		    break;
		}
	    }
	}
#endif	/* AIXV>=4140 */

/*
 * Check for a path name match, as requested.
 */
	if (!f && p && HbyNmCt) {
	    for (sh = &HbyNm[hashbyname(p, SFNMHASH)]; sh; sh = sh->next) {
		if ((s = sh->s) && strcmp(p, s->aname) == 0) {
		    f = 2;
		    break;
		}
	    }
	}
/*
 * Check for a regular AIX multiplexed file, matching the channel if
 * it was supplied by the caller.
 */
	if (!f && HbyMPCCt && ty == VMPC && Lf->dev_def) {
	    for (sh = &HbyMPC[SFHASHDEVINO(major(Lf->dev),
					   minor(Lf->dev),
					   0,
					   SFMPCHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s)
		&&  s->dev == Lf->dev
		&&  (s->ch < 0 || ch == s->ch)) {
		    f = 1;
		    break;
		}
	    }
	}
/*
 * Check for a regular file.
 */
	if (!f && HbyFdiCt && Lf->dev_def
	&&  (Lf->inp_ty == 1 || Lf->inp_ty == 3))
	{
	    for (sh = &HbyFdi[SFHASHDEVINO(major(Lf->dev),
					   minor(Lf->dev),
					   Lf->inode,
					   SFDIHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s) && (Lf->dev == s->dev)
		&&  ((ino_t)Lf->inode == s->i)) {
		    f = 1;
		    break;
		}
	    }
	}
/*
 * Check for a file system.
 */
	if (!f && HbyFsdCt && Lf->dev_def) {
	    for (sh = &HbyFsd[SFHASHDEVINO(major(Lf->dev), minor(Lf->dev), 0,
					   SFFSHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s) && Lf->dev == s->dev) {
		    if (!(ty == VCHR && s->mode != S_IFCHR)) {
			f = 1;
			break;
		    }
		}
	    }
	}
/*
 * Convert the name if a match occurred.
 */
	if (f) {
	    if (f == 2) {
		(void) strcpy(Namech, p);

#if	AIXV>=4140
	    } else if (f == 3 && ClonePtc >= 0
		   &&  major(Lf->dev) == ClonePtc)
	    {
		(void) strcpy(Namech, s->name);
		(void) sprintf(endnm(), "/%d", minor(Lf->dev));

#endif	/* AIXV>=4140 */

	    } else if (s->type) {

	    /*
	     * If the search argument isn't a file system, propagate it
	     * to Namech[]; otherwise, let printname() compose the name.
	     */
		(void) strcpy(Namech, s->name);
		if (ty == VMPC && s->ch < 0)
		    (void) sprintf(endnm(), "/%d", ch);
		if (s->devnm)
		    (void) sprintf(endnm(), " (%s)", s->devnm);
	    }
	    s->f = 1;
	    return(1);
	}
	return(0);
}


/*
 * print_dev() - print device
 */

char *
print_dev(lf)
	struct lfile *lf;		/* file whose device to be printed */
{
	static char buf[128];

#if	AIXV>=3200
	if (lf->dev & SDEV_REMOTE) {
	    (void) sprintf(buf, "NFS,%d", minor(lf->dev));
	    return(buf);
	}
#endif	/* AIXV>=3200 */

	(void) sprintf(buf, "%d,%d", major(lf->dev), minor(lf->dev));
	return(buf);
}
