/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@gnu.org>
	Lennert Buytenhek		<buytenh@gnu.org>
	Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <libintl.h>
#define N_(String) String
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include "llseek.h"

#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <linux/hdreg.h>
#include <linux/major.h>
#include <linux/fs.h>		/* FIXME: autoconfuse this */
#include <sys/ioctl.h>
#include <sys/sysmacros.h>
#include <sys/stat.h>
#include <scsi/scsi.h>

#include <parted/parted.h>

#ifndef SCSI_IOCTL_SEND_COMMAND
#define SCSI_IOCTL_SEND_COMMAND 1
#endif

static PedDevice*	devices = NULL;

static PedDevice*
ped_device_new (const char* path);

static void
ped_device_add (PedDevice* dev)
{
	dev->next = devices;
	devices = dev;
}

static void
ped_device_remove (PedDevice* dev)
{
	PedDevice*	walk;
	PedDevice*	last = NULL;

	for (walk = devices; walk != NULL; last = walk, walk = walk->next) {
		if (walk == dev) break;
	}

	if (last)
		last->next = dev->next;
	else
		devices = dev->next;
}

PedDevice*
ped_device_get_next (const PedDevice* dev)
{
	if (dev)
		return dev->next;
	else
		return devices;
}

/* First searches through probed devices, then attempts to open the device
 * regardless.
 */
PedDevice*
ped_device_get (const char* path)
{
	PedDevice*	walk;
	for (walk = devices; walk != NULL; walk = walk->next) {
		if (!strcmp (walk->path, path))
			return walk;
	}

	walk = ped_device_new (path);
	if (!walk)
		return NULL;
	ped_device_add (walk);
	return walk;
}

static int
ped_device_stat (PedDevice* dev, struct stat * dev_stat)
{
	while (1) {
		if (!stat (dev->path, dev_stat)) {
			return 1;
		} else {
			if (ped_exception_throw (
				PED_EXCEPTION_ERROR,
				PED_EXCEPTION_RETRY_CANCEL,
				_("Could not stat device %s - %s."),
				dev->path,
				strerror (errno))
					!= PED_EXCEPTION_RETRY)
				return 0;
		}
	}
}

static char*
strip_name(char* n)
{
	int	len = strlen (n);
	int	end = 0;
	int	i = 0;

	while (i < len) {
		if ((n[i] != ' ') || ((n[i] == ' ') && (n[i + 1] != ' '))) {
			n[end] = n[i];
			end++;
		}
		i++;
	}
	n[end] = 0;
	return (strdup(n));
}

static int
device_probe_type (PedDevice* dev)
{
	struct stat	dev_stat;
	int		dev_major;

	if (!ped_device_stat (dev, &dev_stat))
		return 0;

	if (!S_ISBLK(dev_stat.st_mode)) {
		ped_exception_throw (PED_EXCEPTION_WARNING,
				     PED_EXCEPTION_CANCEL,
				     _("%s is not a block device."),
				     dev->path);
		dev->type = PED_DEVICE_UNKNOWN;
		return 1;
	}

	dev_major = major (dev_stat.st_rdev);

	if (SCSI_DISK_MAJOR (dev_major) && ((dev_stat.st_rdev & 0x0f) == 0)) {
		dev->type = PED_DEVICE_SCSI;
		return 1;
	}

	if (((dev_major == IDE0_MAJOR) || (dev_major == IDE1_MAJOR)
		  || (dev_major == IDE2_MAJOR) || (dev_major == IDE3_MAJOR))
	    && ((dev_stat.st_rdev & 0x3f) == 0)) {
		dev->type = PED_DEVICE_IDE;
		return 1;
	}

	if (ped_exception_throw (PED_EXCEPTION_WARNING,
				 PED_EXCEPTION_IGNORE_CANCEL,
				 _("Device %s is neither a SCSI nor IDE "
				   "drive."),
				  dev->path)
			== PED_EXCEPTION_CANCEL)
		return 0;
	dev->type = PED_DEVICE_UNKNOWN;
	return 1;
}

static int
device_get_sector_size (PedDevice* dev)
{
	int		sector_size;

	PED_ASSERT (dev->open_count, return 0);

#ifdef USE_BLKSSZGET
	if (ioctl (dev->fd, BLKSSZGET, &sector_size))
		return 512;

	if (sector_size != 512) {
		if (ped_exception_throw (
			PED_EXCEPTION_BUG,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("The sector size on %s is %d bytes.  Parted is known "
			"not to work properly with drives with sector sizes "
			"other than 512 bytes"),
			dev->path,
			dev->sector_size)
				== PED_EXCEPTION_IGNORE)
			return sector_size;
		else
			return 0;
	}
#else /* USE_BLKSSZGET */
	return 512;
#endif /* !USE_BLKSSZGET */
}

static PedSector
device_get_length (PedDevice* dev)
{
	unsigned long		size;

	PED_ASSERT (dev->open_count, return 0);

	if (ioctl (dev->fd, BLKGETSIZE, &size)) {
		ped_exception_throw (
			PED_EXCEPTION_BUG,
			PED_EXCEPTION_CANCEL,
			_("Unable to determine the size of %s (%s)"),
			dev->path,
			strerror (errno));
		return 0;
	}

	return size;
}

static int
device_probe_geometry (PedDevice* dev)
{
	struct stat		dev_stat;
	struct hd_geometry	geometry;

	if (!ped_device_stat (dev, &dev_stat))
		goto error;
	PED_ASSERT (S_ISBLK (dev_stat.st_mode), goto error);

	dev->length = device_get_length (dev);
	if (!dev->length)
		goto error;

	if (ioctl (dev->fd, HDIO_GETGEO, &geometry)) {
		ped_exception_throw (PED_EXCEPTION_FATAL,
				     PED_EXCEPTION_CANCEL,
				     _("Could not read geometry of %s - %s."),
				     dev->path, strerror (errno));
		goto error;
	}

	dev->sector_size = device_get_sector_size (dev);
	if (!dev->sector_size)
		goto error;

	dev->sectors = geometry.sectors;
	dev->heads = geometry.heads;
	if (!dev->sectors || !dev->heads)
		goto error_dodgey_geometry;

	dev->cylinders = dev->length / (dev->heads * dev->sectors
						   * (dev->sector_size / 512));

	return 1;

error_dodgey_geometry:
	ped_exception_throw (
		PED_EXCEPTION_ERROR,
		PED_EXCEPTION_CANCEL,
		_("Device %s has dodgey geometry."),
		dev->path);
error:
	return 0;
}

static int
init_ide (PedDevice* dev)
{
	struct stat		dev_stat;
	int			dev_major;
	struct hd_driveid	hdi;

	if (!ped_device_stat (dev, &dev_stat))
		goto error;
	dev_major = major (dev_stat.st_rdev);

	if (!ped_device_open (dev))
		goto error;

        if (ioctl (dev->fd, HDIO_GET_IDENTITY, &hdi)) {
		if (ped_exception_throw (PED_EXCEPTION_WARNING,
			     PED_EXCEPTION_IGNORE_CANCEL,
			     _("Could not get identity of device %s - %s"),
			     dev->path, strerror (errno))
		    == PED_EXCEPTION_CANCEL)
			goto error_close_dev;
		dev->model = strdup(_("unknown"));
	} else {
		dev->model = strip_name (hdi.model);
	}

	if (!device_probe_geometry (dev))
		goto error_close_dev;

	dev->host = (dev_major == IDE1_MAJOR) + (dev_major == IDE2_MAJOR) * 2
		    + (dev_major == IDE3_MAJOR) * 3;
	dev->did = (dev_stat.st_rdev & 0x40) >> 6;

	ped_device_close (dev);
	return 1;

error_close_dev:
	ped_device_close (dev);
error:
	return 0;
}

static int
init_scsi (PedDevice* dev)
{
	unsigned char		idlun [8];
	unsigned char		buffer [128];
	unsigned char*		cmd;
	struct hd_geometry	geometry;

	if (!ped_device_open (dev))
		goto error;

	memset (buffer, 0, 96);

	*((int *) buffer) = 0;  /* length of input data */
	*(((int *) buffer) + 1) = 96;   /* length of output buffer */

	cmd = (char *) (((int *) buffer) + 2);

	cmd[0] = 0x12;          /* INQUIRY */
	cmd[1] = 0x00;          /* lun=0, evpd=0 */
	cmd[2] = 0x00;          /* page code = 0 */
	cmd[3] = 0x00;          /* (reserved) */
	cmd[4] = 96;            /* allocation length */
	cmd[5] = 0x00;          /* control */

	if (ioctl(dev->fd, SCSI_IOCTL_SEND_COMMAND, buffer)) {
		buffer[40] = 0;
		dev->model = strip_name (buffer + 16);
	} else {
		dev->model = _("Unknown SCSI");
	}

	if (ioctl(dev->fd, SCSI_IOCTL_GET_IDLUN, idlun)) {
			ped_exception_throw (PED_EXCEPTION_FATAL,
				     PED_EXCEPTION_CANCEL,
				     _("Error initialising SCSI device "
				       "%s - %s"),
				     dev->path, strerror (errno));
		goto error_close_dev;
	}

	if (!device_probe_geometry (dev))
		goto error_close_dev;

	dev->host = *((unsigned long *) (idlun + 4));
	dev->did = idlun [0];

	ped_device_close (dev);
	return 1;

error_close_dev:
	ped_device_close (dev);
error:
	return 0;
}

static int
init_unknown (PedDevice* dev)
{
	struct stat		dev_stat;
	struct hd_geometry	geometry;

	if (!ped_device_stat (dev, &dev_stat))
		goto error;

	if (!ped_device_open (dev))
		goto error;

	ped_exception_fetch_all ();
	if (device_probe_geometry (dev)) {
		ped_exception_leave_all ();
	} else {
		/* hack to allow use of files, for testing */
		ped_exception_catch ();
		ped_exception_leave_all ();

		dev->length = dev_stat.st_size / 512;
		if (ped_exception_throw (PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("Unable to determine geometry of "
				"file/device.  You should not use Parted "
				"unless you REALLY know what you're doing!"))
				!= PED_EXCEPTION_IGNORE)
			goto error;

		/* what should we stick in here? */
		dev->cylinders = dev->length / 4 / 32;
		dev->heads = 4;
		dev->sectors = 32;
	}

	ped_device_close (dev);
	return 1;

error_close_dev:
	ped_device_close (dev);
error:
	return 0;
}

static PedDevice*
ped_device_new (const char* path)
{
	PedDevice*	dev;

	PED_ASSERT (path != NULL, return NULL);

	dev = (PedDevice*) ped_malloc (sizeof (PedDevice));
	if (!dev)
		goto error;

	dev->path = strdup (path);
	dev->open_count = 0;
	dev->dirty = 0;
	dev->geom_known = 1;

	if (!device_probe_type (dev))
		goto error_free_dev;

	switch (dev->type) {
	case PED_DEVICE_IDE:
		if (!init_ide (dev))
			goto error_free_dev;
		break;

	case PED_DEVICE_SCSI:
		if (!init_scsi (dev))
			goto error_free_dev;
		break;

	case PED_DEVICE_UNKNOWN:
		if (!init_unknown (dev))
			goto error_free_dev;
		break;

	default:
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				PED_EXCEPTION_CANCEL,
				_("ped_device_new()  Unsupported device type"));
		goto error_free_dev;
	}
	return dev;

error_free_dev:
	ped_free (dev);
error:
	return NULL;
}

static void
ped_device_destroy (PedDevice* dev)
{
	PED_ASSERT (dev != NULL, return);

	while (dev->open_count)
		ped_device_close (dev);

	ped_free (dev->path);
	ped_free (dev);
}

int
ped_device_open (PedDevice* dev)
{
	PED_ASSERT (dev != NULL, return 0);

	if (dev->open_count++)
		return 1;

	dev->fd = open (dev->path, O_RDWR);
	if (dev->fd == -1) {
		ped_exception_throw (PED_EXCEPTION_FATAL, PED_EXCEPTION_CANCEL,
				     "%s: %s", dev->path, strerror (errno));
		return 0;
	}

	return 1;
}

int
ped_device_close (PedDevice* dev)
{
	int	retry_count = 5;	/* a bad guess */
	PED_ASSERT (dev != NULL, return 0);

	if (--dev->open_count) return 1;

#ifdef linux
	if (dev->dirty && dev->type != PED_DEVICE_UNKNOWN) {
		sync();
		while (ioctl (dev->fd, BLKRRPART)) {
			retry_count--;
			sync();
	    		if (!retry_count) {
				ped_exception_throw (
					PED_EXCEPTION_WARNING,
					PED_EXCEPTION_OK,
		_("The kernel was unable to re-read the partition "
		  "table on %s (%s).  This means Linux knows nothing "
		  "about any modifications you made.  You should "
		  "reboot your computer before doing anything with %s."),
					dev->path, strerror (errno), dev->path);

	    			break;
	    		}
		}
	}
#endif /* linux */

	close (dev->fd);
	return 1;
}

static int
ped_device_seek (const PedDevice* dev, PedSector sector)
{
	PED_ASSERT (dev != NULL, return 0);

#ifdef linux
	if (sizeof (off_t) < 8) {
		loff_t		pos = sector * 512;
		return ped_llseek (dev->fd, pos, SEEK_SET) == pos;
	} else
#endif
	{
		off_t		pos = sector * 512;
		return lseek (dev->fd, pos, SEEK_SET) == pos;
	}
}

int
ped_device_read (const PedDevice* dev, void* buffer, PedSector start,
		 PedSector count)
{
	int		status;
	int		exception_status;

	PED_ASSERT (dev != NULL, return 0);
	PED_ASSERT (buffer != NULL, return 0);

	while (1) {
		if (ped_device_seek (dev, start))
			break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during seek for read on %s"),
			strerror (errno), dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY) return 0;
	}

	while (1) {
		status = read (dev->fd, buffer, count * PED_SECTOR_SIZE);
		if (status == count * PED_SECTOR_SIZE) break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during read on %s"),
			strerror (errno),
			dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY) return 0;
	}

	return 1;
}

int
ped_device_sync (PedDevice* dev)
{
	int		status;
	int		exception_status;

	PED_ASSERT (dev != NULL, return 0);

	while (1) {
		status = fsync (dev->fd);
		if (status >= 0) break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during write on %s"),
			strerror (errno), dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY) return 0;
	} 

	return 1;
}

int
ped_device_write (PedDevice* dev, const void* buffer, PedSector start,
		  PedSector count)
{
	int		status;
	int		exception_status;

	PED_ASSERT (dev != NULL, return 0);
	PED_ASSERT (buffer != NULL, return 0);

	while (1) {
		if (ped_device_seek (dev, start))
			break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during seek for write on %s"),
			strerror (errno), dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY) return 0;
	}

#ifdef READONLY
	printf ("ped_device_write (\"%s\", %p, %d, %d)\n",
		dev->path, buffer, (int) start, (int) count);
#else
	dev->dirty = 1;
	while (1) {
		status = write (dev->fd, buffer, count * PED_SECTOR_SIZE);
		if (status == count * PED_SECTOR_SIZE) break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during write on %s"),
			strerror (errno), dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY) return 0;
	}
#endif
	return 1;
}

/* returns the number of sectors that are ok.
 */
PedSector
ped_device_check (PedDevice* dev, void* buffer, PedSector start,
		  PedSector count)
{
	int		status;
	int		exception_status;

	PED_ASSERT (dev != NULL, return 0);
	PED_ASSERT (buffer != NULL, return 0);

	while (1) {
		if (ped_device_seek (dev, start))
			break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during seek for check on %s"),
			strerror (errno), dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY) return 0;
	}

	while (1) {
		status = read (dev->fd, buffer, count * PED_SECTOR_SIZE);
		if (status == count * PED_SECTOR_SIZE) break;

		exception_status = ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_RETRY_IGNORE_CANCEL,
			_("%s during check on %s"),
			strerror (errno),
			dev->path);

		if (exception_status == PED_EXCEPTION_IGNORE) break;
		if (exception_status != PED_EXCEPTION_RETRY)
			return status / PED_SECTOR_SIZE;
	}

	return count;
}

static void
probe (char* path)
{
	PedDevice*	dev;

	PED_ASSERT (path != NULL, return);

	for (dev = devices; dev; dev = dev->next) {
		if (!strcmp (dev->path, path))
			return;
	}

	ped_exception_fetch_all ();

	dev = ped_device_new (path);
	if (!dev)
		goto error;

	if (!ped_device_open (dev)) {
		ped_device_destroy (dev);
		goto error;
	}
	ped_exception_leave_all ();

	ped_device_close (dev);
	ped_device_add (dev);

	return;

error:
	ped_exception_catch ();
	ped_exception_leave_all ();
}

/* FIXME */
void
ped_device_probe_all ()
{
	probe ("/dev/hdh");
	probe ("/dev/hdg");
	probe ("/dev/hdf");
	probe ("/dev/hde");
	probe ("/dev/hdd");
	probe ("/dev/hdc");
	probe ("/dev/hdb");
	probe ("/dev/hda");

	probe ("/dev/sdf");
	probe ("/dev/sde");
	probe ("/dev/sdd");
	probe ("/dev/sdc");
	probe ("/dev/sdb");
	probe ("/dev/sda");
}

void
ped_device_free_all ()
{
	PedDevice*	dev;
	while (devices)
	{
		dev = devices;
		ped_device_remove (dev);
		ped_device_destroy (dev);
	}
}
