/*  Screem:  support.c,
 *  some useful functions
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <errno.h>
#include <gnome.h>
#include <stdio.h>
#include <string.h>

#include "support.h"

GList *icon_list = NULL;

static void file_sel_delete( GtkWidget *widget, GtkWidget **file_sel );
static void file_selected( GtkWidget *widget, gboolean *selected );
static GtkWidget *results_dialog( void );

/**
 * change_state:
 *
 * toggles widget sensitivity
 *
 * return values: none
 */
void change_state( GtkWidget *widget, gpointer data )
{
        gtk_widget_set_sensitive( widget, !GTK_WIDGET_IS_SENSITIVE( widget ) );
}

/**
 * file_select:
 *
 * handles selection of files/directories modally
 * the return value is NULL if the select was cancelled, otherwise it
 * returns the filename
 *
 * return values: a string
 */
gchar* file_select( gchar *title )
{
	GtkWidget *file_sel;
	gchar *filename;
	gboolean selected = FALSE;

	file_sel = gtk_file_selection_new( title );
	gtk_window_set_modal( GTK_WINDOW( file_sel ), TRUE );

	gtk_signal_connect( GTK_OBJECT( file_sel ), "destroy", 
                            GTK_SIGNAL_FUNC( file_sel_delete ), &file_sel );
	
        gtk_signal_connect( GTK_OBJECT( GTK_FILE_SELECTION( file_sel )->cancel_button ), "clicked", GTK_SIGNAL_FUNC( file_sel_delete ), &file_sel );
        
        gtk_signal_connect( GTK_OBJECT( GTK_FILE_SELECTION( file_sel )->ok_button ), "clicked", GTK_SIGNAL_FUNC( file_selected ), &selected );

        gtk_widget_show( file_sel );

	while( ( ! selected ) && ( file_sel ) )
		gtk_main_iteration();

	/* canceled or window was closed */
	if( ! selected )
		return NULL;

	/* ok */
	filename = g_strdup( gtk_file_selection_get_filename( GTK_FILE_SELECTION( file_sel ) ) );
	gtk_widget_destroy( file_sel );
	return filename;
}

static void file_sel_delete( GtkWidget *widget, GtkWidget **file_sel )
{
	gtk_widget_destroy( *file_sel );
	*file_sel = NULL;
}

static void file_selected( GtkWidget *widget, gboolean *selected )
{
	*selected = TRUE;
}

gboolean execute_command( gchar *command )
{
        FILE *results;
        GtkWidget *text;
        gchar buffer[ 1024 ];
        gint retval;
        gint size;
        GtkWidget *dialog;

        results = popen( command, "r" );

	dialog = results_dialog();
        text = gtk_object_get_data( GTK_OBJECT( dialog ), "output_window" );
        gtk_widget_show_all( dialog );

        gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL,
                         "Running: ", -1 );
        gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL, command,
                         -1  );
        gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL,
                         "\n\n", -1 );
        g_free( command );

	if( results ) {
		while( ( size = fread( buffer, 1, 1024, results ) ) ) {
			gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL,
					 buffer, size );
		}
		
		retval = pclose( results );
	} else {
		retval = -1;
	}

        gnome_dialog_run_and_close( GNOME_DIALOG( dialog ) );

        return ( retval != -1 );
}

static GtkWidget *results_dialog()
{
  GtkWidget *dialog1;
  GtkWidget *dialog_vbox1;
  GtkWidget *frame1;
  GtkWidget *output_window;
  GtkWidget *dialog_action_area1;
  GtkWidget *button3;

  dialog1 = gnome_dialog_new (NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (dialog1), "dialog1", dialog1);
  gtk_window_set_policy (GTK_WINDOW (dialog1), FALSE, FALSE, FALSE);

  dialog_vbox1 = GNOME_DIALOG (dialog1)->vbox;
  gtk_object_set_data (GTK_OBJECT (dialog1), "dialog_vbox1", dialog_vbox1);
  gtk_widget_show (dialog_vbox1);

  frame1 = gtk_frame_new (_("Output"));
  gtk_widget_ref (frame1);
  gtk_object_set_data_full (GTK_OBJECT (dialog1), "frame1", frame1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (frame1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), frame1, TRUE, TRUE, 0);

  output_window = gtk_text_new (NULL, NULL);
  gtk_widget_ref (output_window);
  gtk_object_set_data_full (GTK_OBJECT (dialog1), "output_window", output_window,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (output_window);
  gtk_container_add (GTK_CONTAINER (frame1), output_window);
  gtk_widget_set_usize (output_window, 320, 240);

  dialog_action_area1 = GNOME_DIALOG (dialog1)->action_area;
  gtk_object_set_data (GTK_OBJECT (dialog1), "dialog_action_area1", dialog_action_area1);
  gtk_widget_show (dialog_action_area1);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area1), 8);

  gnome_dialog_append_button (GNOME_DIALOG (dialog1), GNOME_STOCK_BUTTON_CLOSE);
  button3 = g_list_last (GNOME_DIALOG (dialog1)->buttons)->data;
  gtk_widget_ref (button3);
  gtk_object_set_data_full (GTK_OBJECT (dialog1), "button3", button3,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button3);
  GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);

  return dialog1;
}

gchar* escape_path( gchar *pathname )
{
	gchar *temp;
	gchar *path;
	gint i = 0;
	
	GList *list;

	/* we need to escape any _'s in the path */
	pathname = g_strdup( pathname );
	while( i < strlen( pathname ) ) {
		if( pathname[ i ] == '_' ) {
			temp = g_strndup( pathname, ++i );
			path = g_strconcat( temp, "_", &pathname[ i ],
					    NULL );
			g_free( temp );
			g_free( pathname );
			pathname = path;
		}
		i ++;
	}
	
	return pathname;
}

Icons *icon_from_mime_type( const gchar *mime_type )
{
	GList *list;
	Icons *icon = NULL;
	const gchar *file;

	g_return_val_if_fail( mime_type != NULL, NULL );

	for( list = icon_list; ( ! icon ) && list; list = list->next ) {
			icon = list->data;
			if( strcmp( icon->mime_type, mime_type ) )
				icon = NULL;
	}

	if( ! icon ) {
		/* add an icon type */
		file = gnome_mime_get_value( mime_type, 
					     "icon-filename" );
		icon = g_new( Icons, 1 );
		icon->mime_type = mime_type;
		icon->image = gdk_imlib_load_image( (gchar*)file );
		gdk_imlib_render( icon->image, 20, 20 );
		icon->open = gdk_imlib_copy_image( icon->image );
		icon->open_mask = gdk_imlib_copy_mask( icon->image );
		
		icon_list = g_list_append( icon_list, icon );
	}

	return icon;
}
