/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

/* Do the simple clock and calendar here.  Also, hostname stuff.
*/

#include "gkrellm.h"


static Panel	uptime;
static Panel	clock_cal;


typedef struct
	{
	GdkFont	*font;
	gint	lbearing,
			rbearing,
			width,
			ascent,
			descent;
	}
	Extents;


static gint			clock_visible;
static gint			clock_needs_refresh;

static Extents		wday_extents,
					mday_extents,
					month_extents,
					time_extents,
					mode_extents;	/* am/pm or seconds for 12 hour clock */

static gint			x_month, y_month,
					x_wday, y_wday,
					x_mday, y_mday,
					x_time, y_time,
					x_mode, y_mode;

static gint			time_effect, label_effect;

static struct tm	*pTm;
struct tm			current_tm;

static time_t		time_now;


void
string_extents(GdkFont *font, gchar *string, Extents *ext)
	{
	ext->font = font;
	gdk_string_extents(font, string, &ext->lbearing, &ext->rbearing,
			&ext->width, &ext->ascent, &ext->descent);
	}

void
draw_string(GdkDrawable *drawable, GdkFont *font, GdkColor *color,
				gint effect, gint x, gint y, gchar *s)
	{
	if (effect)
		{
		gdk_gc_set_foreground(GK.text_GC, &GK.background_color);
		gdk_draw_string(drawable, font, GK.text_GC, x + 1, y + 1, s);
		}
	gdk_gc_set_foreground(GK.text_GC, color);
	gdk_draw_string(drawable, font, GK.text_GC, x, y, s);
	}

void
draw_clock(struct tm *t)
	{
	Panel		*p;
	gchar		buf[32];
	gint		x_adjust;

	if (t == NULL || clock_visible == FALSE)
		return;
	p = &clock_cal;

	if (t->tm_min != current_tm.tm_min || clock_needs_refresh)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->background,
				0, 0,  0, 0,  p->w, p->h);

		strftime(buf, sizeof(buf), "%a", t);  /* Abbreviated weekday name */
		draw_string(p->pixmap, wday_extents.font, &GK.label_color,
					label_effect, x_wday, y_wday, buf);

		strftime(buf, sizeof(buf), "%e", t);  /* Day in month 1-31 */
		x_adjust = (t->tm_mday < 10) ? 2 : 0;
		draw_string(p->pixmap, mday_extents.font, &GK.time_color,
				time_effect, x_mday - x_adjust, y_mday, buf);

		strftime(buf, sizeof(buf), "%b", t); /* Abbreviated month name */
		draw_string(p->pixmap, month_extents.font, &GK.label_color,
				label_effect, x_month, y_month, buf);

		if (UC.clock_24hr_format)
			strftime(buf, sizeof(buf), "%k:%M", t);   /* 24 hour:minute */
		else
			{
			if (! UC.clock_12hr_seconds)
				{
				strftime(buf, sizeof(buf), "%P", t);   /* am/pm */
				draw_string(p->pixmap, mode_extents.font, &GK.label_color,
							label_effect, x_mode, y_mode, buf);
				}
			strftime(buf, sizeof(buf), "%l:%M", t);   /* 12 hour:minute */
			}
		draw_string(p->pixmap, time_extents.font, &GK.time_color,
				time_effect, x_time, y_time, buf);
		gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					0, 0,  0, 0,  p->w, p->h);
		}
	if (UC.clock_12hr_seconds || UC.clock_24hr_format)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->background,
				x_mode - mode_extents.lbearing, y_mode - mode_extents.ascent,
				x_mode - mode_extents.lbearing, y_mode - mode_extents.ascent,
				mode_extents.lbearing + mode_extents.rbearing + 2,
				mode_extents.ascent + mode_extents.descent);
		strftime(buf, sizeof(buf), "%S", t);   /* seconds 00-59 */
		draw_string(p->pixmap, mode_extents.font, &GK.time_color,
							label_effect, x_mode, y_mode, buf);
		gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
				x_mode - mode_extents.lbearing, y_mode - mode_extents.ascent,
				x_mode - mode_extents.lbearing, y_mode - mode_extents.ascent,
				mode_extents.lbearing + mode_extents.rbearing,
				mode_extents.ascent + mode_extents.descent);
		}
	clock_needs_refresh = FALSE;
	}

void
update_clock(void)
	{
	GK.ten_second_tick = FALSE;
	GK.minute_tick = FALSE;
	GK.hour_tick = FALSE;
	GK.day_tick = FALSE;

	if (GK.second_tick)
		{
		time(&time_now);
		pTm = localtime(&time_now);
		if (pTm->tm_sec != current_tm.tm_sec)
			draw_clock(pTm);
		GK.ten_second_tick = ((pTm->tm_sec % 10) == 0) ? TRUE : FALSE;
		GK.minute_tick = (pTm->tm_min  != current_tm.tm_min)  ? TRUE : FALSE;
		GK.hour_tick   = (pTm->tm_hour != current_tm.tm_hour) ? TRUE : FALSE;
		GK.day_tick   = (pTm->tm_mday != current_tm.tm_mday) ? TRUE : FALSE;
		current_tm = *pTm;
		}
	}

static gint
clock_expose_event(GtkWidget *widget, GdkEventExpose *event)
	{
	if (widget == clock_cal.drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				clock_cal.pixmap,
				event->area.x, event->area.y,
				event->area.x, event->area.y,
				event->area.width, event->area.height);
		}
	else if (widget == uptime.drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				uptime.pixmap,
				event->area.x, event->area.y,
				event->area.x, event->area.y,
				event->area.width, event->area.height);
		}
	return FALSE;
	}

void
create_clock(GtkWidget *vbox)
	{
	GdkImlibBorder	*border;
	gint			y_split, h;

	border = &GK.bordered_bg_border;
	time_effect = effect_string_value(GK.time_effect_string);
	label_effect = effect_string_value(GK.label_effect_string);

	string_extents(GK.alt_font, "Wed", &wday_extents);
	string_extents(GK.large_font, "88", &mday_extents);
	string_extents(GK.alt_font, "Aug", &month_extents);
	string_extents(GK.large_font, "00:00", &time_extents);
	string_extents(GK.alt_font, "p7", &mode_extents);	/* am/pm or seconds */

	x_mday = (UC.chart_width - mday_extents.width) / 2;
	y_mday = mday_extents.ascent + border->top;
	x_wday = x_mday - wday_extents.width - 2;
	y_wday = y_mday - (mday_extents.ascent - wday_extents.ascent) / 2;

	x_month = x_mday + mday_extents.width + 2;
	y_month = y_wday;

	y_split = y_mday + border->bottom + time_effect;

	x_time = (UC.chart_width - time_extents.width - mode_extents.width) / 2;
	y_time = y_split + border->top + time_extents.ascent;
	x_mode = x_time + time_extents.width + 2;
	y_mode = y_time;

	h = y_time + border->bottom + time_effect;

	clock_cal.label.h_panel = h;
	clock_cal.label.string = NULL;

	create_panel_area(vbox, &clock_cal, GK.bg_spacer_image);
	if (UC.enable_clock)
		{
		GK.monitor_height += clock_cal.h;
		clock_visible = TRUE;
		}
	else
		gtk_widget_hide(clock_cal.hbox);


	gdk_imlib_paste_image(GK.bg_bordered_image, clock_cal.background,
			0, 0, UC.chart_width, y_split);
	gdk_imlib_paste_image(GK.bg_bordered_image, clock_cal.background,
			0, y_split, UC.chart_width, h - y_split);

	gtk_signal_connect(GTK_OBJECT (clock_cal.drawing_area), "expose_event",
		(GtkSignalFunc) clock_expose_event, NULL);

	time(&time_now);
	pTm = localtime(&time_now);
	clock_needs_refresh = TRUE;
	draw_clock(pTm);
	}

void
apply_clock_config()
	{
	enable_visibility(UC.enable_clock, &clock_visible, clock_cal.hbox,
						clock_cal.h);
	clock_needs_refresh = TRUE;
	draw_clock(pTm);
	}


/* ----------------------------------------------------------------------*/
/* Hostname label on the top frame event_box. */

static Panel			host_panel;			/* Just use the label part */
static Style			host_style;

static GtkWidget		*hostname_vbox;
static GtkWidget		*host_pixmapwid;
static GdkPixmap		*host_pixmap;

static gint				hostname_visible,
						hostname_short,
						hostname_height;

static gchar			hostname[128];

void
get_hostname()
	{
	gchar	*s;

	if (gethostname(hostname, sizeof(hostname)))
		strcpy(hostname, "unknown");
	else if (UC.hostname_short)
		{
		s = strchr(hostname, (int) '.');
		if (s)
			*s = '\0';
		s = strchr(hostname, (int) '@');	/* ??? */
		if (s)
			*s = '\0';
		}
	/* Configuring a panel, but I have no panel.  Just getting the
	|  height I need for the hostname and its border.
	*/
	default_textstyle(&host_panel.label.textstyle, TEXTSTYLE_ALT1);
	configure_panel(&host_panel, hostname, &host_style);
	if (host_panel.label.width > UC.chart_width)
		{
		host_panel.label.textstyle.font = GK.alt_font;
		configure_panel(&host_panel, hostname, &host_style);
		}
	hostname_short = UC.hostname_short;
	}

void
draw_label(Label *label, GdkPixmap *pixmap)
	{
	gint	x;

	x = (UC.chart_width - host_panel.label.width) / 2;
	if (x < 0)
		x = 0;
	if (label->textstyle.effect)
		{
		gdk_gc_set_foreground(GK.text_GC, &GK.background_color);
		gdk_draw_string(pixmap, label->textstyle.font, GK.text_GC, x + 1,
				label->y_baseline + 1, label->string);
		}
	gdk_gc_set_foreground(GK.text_GC, &label->textstyle.color);
	gdk_draw_string(pixmap, label->textstyle.font, GK.text_GC, x,
				label->y_baseline, label->string);
	}

void
draw_hostname(GtkWidget *vbox, gint visible)
	{
	GdkBitmap	*mask		= NULL;
	gint		w, h;

	if (host_pixmap)
		gdk_pixmap_unref(host_pixmap);
	if (host_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(vbox), host_pixmapwid);
	GK.monitor_height -= hostname_height;

	w = UC.chart_width;
	h = visible ? host_panel.label.h_panel : 4;
	hostname_height = h;
	host_pixmap = gdk_pixmap_new(vbox->window, w, h, -1);
	gdk_imlib_paste_image(GK.bg_spacer_image, host_pixmap, 0, 0, w, h);

	if (visible)
		draw_label(&host_panel.label, host_pixmap);

	host_pixmapwid = gtk_pixmap_new(host_pixmap, mask);
	gtk_box_pack_start (GTK_BOX(vbox), host_pixmapwid, FALSE, FALSE, 0);
	gtk_widget_show(host_pixmapwid);
	GK.monitor_height += hostname_height;
	pack_side_frames();
	}

void
create_hostname(GtkWidget *vbox)
	{
	hostname_vbox = vbox;
	host_style.label_position = LABEL_CENTER;	/* XXX not used */
	host_style.border_panel.top = 1;
	host_style.border_panel.bottom = 1;

	if (GK.trace)
		printf("create_hostname()\n");
	get_hostname();

	hostname_visible = UC.enable_hostname ? TRUE : FALSE;
	draw_hostname(vbox, hostname_visible);
	if (GK.trace)
		printf("  <-\n");
	}

void
apply_hostname_config()
	{
	int		short_changed	= FALSE;

	if (UC.hostname_short != hostname_short)
		{
		get_hostname();
		short_changed = TRUE;
		}
	if (UC.enable_hostname && (! hostname_visible || short_changed))
		{
		hostname_visible = TRUE;
		draw_hostname(hostname_vbox, hostname_visible);
		}
	else if (!UC.enable_hostname && hostname_visible)
		{
		hostname_visible = FALSE;
		draw_hostname(hostname_vbox, hostname_visible);
		}
	}


/* ----------------------------------------------------------------------*/
/* Uptime Monitor		*/

static void
draw_upminutes(gint minutes)
	{
	GdkImlibBorder	*border;
	GdkFont			*font;
	gint			x, w1, w2, w, w_field;
	gint			days, hours;
	Panel			*g;
	Label			*lbl;
	gchar			buf1[16], buf2[16], *s;

	g = &uptime;
	lbl = &g->label;
	border = &GK.bordered_bg_border;

	hours = minutes / 60;
	minutes %= 60;

	days = hours / 24;
	hours %= 24;

	w_field = UC.chart_width - border->left - border->right;
	s = buf1;
	font = lbl->textstyle.font;
	sprintf(buf1, "%dd %2d:%02d", days, hours, minutes); 
	sprintf(buf2, "%dd%2d:%02d", days, hours, minutes); 
	w = w1 = gdk_string_width(font, buf1);
	if (w1 > w_field)
		{
		if ((w2 = gdk_string_width(font, buf2)) > w_field)
			{
			font = GK.alt_font;
			w = gdk_string_width(font, buf1);
			}
		else
			{
			s = buf2;
			w = w2;
			}
		}
	/* Last chance to fit it in.
	*/
	if (w > w_field)
		{
		sprintf(buf1, "%dd%2d:", days, hours);
		s = buf1; 
		}
	x = (UC.chart_width - w) / 2;
	if (x < border->left)
		x = border->left;	/* Will truncate to right */

	gdk_draw_pixmap(g->pixmap, GK.draw1_GC, g->background,
			0, 0,  0, 0,  g->w, g->w);
	gdk_draw_pixmap(g->drawing_area->window, GK.draw1_GC, g->background,
			0, 0,  0, 0,  g->w, g->w);
	if (lbl->textstyle.effect)
		{
		gdk_gc_set_foreground(GK.text_GC, &GK.background_color);
		gdk_draw_string(g->pixmap, font, GK.text_GC,
				x + 1, lbl->y_baseline + 1, s);
		gdk_draw_string(g->drawing_area->window, font, GK.text_GC,
				x + 1, lbl->y_baseline + 1, s);
		}
	gdk_gc_set_foreground(GK.text_GC, &lbl->textstyle.color);
	gdk_draw_string(g->pixmap, font, GK.text_GC,
			x, lbl->y_baseline, s);
	gdk_draw_string(g->drawing_area->window, font, GK.text_GC,
			x, lbl->y_baseline, s);
	}


void
update_uptime(void)
	{
	glong	up_minutes;

	/* Once every 10 seconds is default update period.
	*/
	if (GK.ten_second_tick)
		{
		up_minutes = (time(0) - GK.start_time + GK.base_uptime) / 60;
		if (GK.up_minutes != up_minutes)
				draw_upminutes(up_minutes);
		GK.up_minutes = up_minutes;
		}
	}

void
create_uptime(GtkWidget *vbox)
	{
	Style		uptime_style;
	FILE		*f;


	/* Give a dummy string to configure_panel() to get panel height right.
	|  then, null out the label string.
	*/
	if (GK.trace)
		printf("  uptime...\n");
	uptime_style = *GK.meter_style[0];	/* To get the border I want */
	uptime_style.label_position = LABEL_CENTER;
	default_textstyle(&uptime.label.textstyle, TEXTSTYLE_TIME);
	configure_panel(&uptime, "0d :99", &uptime_style);
	uptime.label.string = NULL;
	create_panel_area(vbox, &uptime, GK.bg_meter_image[0]);
	GK.monitor_height += uptime.h;

	gtk_signal_connect(GTK_OBJECT (uptime.drawing_area), "expose_event",
			(GtkSignalFunc) clock_expose_event, NULL);

	if ((f = fopen("/proc/uptime", "r")) != NULL)
		{
		fscanf(f, "%d", &GK.base_uptime);
		fclose(f);
		}
	GK.up_minutes = -1;
	}

