/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"

  /* Smooth the krell response with an exponential moving average.
  |  Eponential MA is quicker responding than pure moving average.
  |  a   =  2 / (period + 1)
  |  ema = ema + a * (reading - ema)
  |  Don't need floating point precision here, so do the int math in an
  |  order to minimize roundoff error and scale by 256 for some precision.
  */
gint
exp_MA(Krell *k)
	{
	gint	ema, p, reading, round_up;

	/* First, help the krell settle to zero and full scale. This gives
	|  the illusion of an overall fast response while the ema is actually
	|  smoothing.
	*/
	if (k->reading == 0 && k->last_reading == 0)
		return 0;
	if (k->reading >= k->full_scale && k->last_reading >= k->full_scale)
		return k->full_scale;
	if (k->last_reading == 0)	/* Fast liftoff as well */
		return k->reading;
	ema = k->ema << 8;
	p   = k->period + 1;		/* Don't scale this! */
	reading = (k->reading) << 8;

	ema = ema + 2 * reading / p - 2 * ema / p;
	round_up = ((ema & 0xff) > 127) ? 1 : 0;

	return (ema >> 8) + round_up;
	}

  /* If the Krell has moved, redraw its layer on its stencil.
  */
void
update_krell(Panel *p, Krell *krell, unsigned long value)
	{
	gint	xnew, x_src, y_src, h_src, x_dst, w1, w2, w_overlap, d, frame;

	if (krell == NULL || krell->full_scale == 0)
		return;
	if (value < krell->previous)		/* unsigned long overflow? */
		{
		krell->previous = value;
		return;
		}

	krell->reading = (gint) (value - krell->previous) * krell->full_scale_expand;
	if (krell->reading > krell->full_scale)
		krell->reading = krell->full_scale;

	krell->ema = (krell->period > 1) ? exp_MA(krell) : krell->reading;
	krell->previous = value;
	krell->last_reading = krell->reading;

	xnew = krell->x0 + krell->ema * krell->w_scale / krell->full_scale;
	if (xnew == krell->x_position)
		return;

	krell->x_position = xnew;

	/* If the krell has depth, pick the frame to display as function of
	|  ema.  Depth == 2 means display frame 0 at xnew == 0, and frame 1
	|  everywhere else.  Depth > 2 means same thing for frame 0,
	|  diplay frame depth - 1 at xnew == full_scale, and display other
	|  frames in between at xnew proportional to ema/full_scale.
	*/
	d = krell->depth;
	h_src = krell->h / d;

	if (krell->ema == 0 || xnew == krell->x0) /* Krell can settle before ema*/
		frame = 0;
	else if (krell->ema == krell->full_scale)
		frame = d - 1;
	else
		{
		if (d == 1)
			frame = 0;
		else if (d == 2)
			frame = 1;
		else
			frame = 1 + ((d - 2) * krell->ema / krell->full_scale);
		}
	y_src = h_src * frame;

	/* Clear the krell stencil bitmap.
	*/
	gdk_gc_set_foreground(p->stencil_gc, &GK.background_color);
	gdk_draw_rectangle(krell->stencil, p->stencil_gc, TRUE,
				0,0,  krell->w, krell->y0 + h_src);

	x_src = krell->x_hot - xnew;
	if (x_src < 0)
		x_src = 0;

	x_dst = xnew - krell->x_hot;
	if (x_dst < 0)
		x_dst = 0;

	w1 = krell->w - x_src;
	w2 = p->w - x_dst;
	w_overlap = (w2 > w1) ? w1 : w2;

#if 0
if (strcmp(krell->name, "ppp0") == 0)
{
printf("%s: ema=%d xnew=%d x0=%d full_scale=%d reading=%d\n", krell->name,
krell->ema, xnew, krell->x0, krell->full_scale, krell->reading);
printf("    frame=%d xsrc=%d ysrc=%d xdst=%d ydst=%d w=%d h=%d hot=%d\n",
frame, x_src, y_src, x_dst, krell->y0, w_overlap, h_src, krell->x_hot);
}
#endif

	gdk_draw_pixmap(krell->stencil, p->stencil_gc, krell->mask,
			x_src, y_src, x_dst, krell->y0, w_overlap, h_src);

	krell->old_draw = krell->draw;

	krell->draw.x_src = x_src;
	krell->draw.y_src = y_src;
	krell->draw.x_dst = x_dst;
	krell->draw.y_dst = krell->y0;
	krell->draw.w = w_overlap;
	krell->draw.h = h_src;

	krell->modified = TRUE;
	}


  /* Push decal pixmaps through their stencils onto a Panel expose pixmap
  */
static void
push_decal_pixmaps(Panel *p)
	{
	Decal	*l;

	gdk_gc_set_clip_mask(GK.text_GC, p->stencil);
	for (l = p->decal; l; l = l->next)
		{
		if (l->type == DECAL_TRANSPARENCY)
			gdk_draw_pixmap(p->pixmap, GK.text_GC, l->pixmap,
						0, l->y_src, l->x, l->y, l->w, l->h);
		else if (l->type == DECAL_OPAQUE)
			gdk_draw_pixmap(p->pixmap, GK.draw1_GC, l->pixmap,
						0, l->y_src, l->x, l->y, l->w, l->h);
		}
	gdk_gc_set_clip_mask(GK.text_GC, NULL);
	}

  /* Push krell pixmaps through their stencils onto a Panel expose pixmap
  */
static void
push_krell_pixmaps(Panel *p)
	{
	Krell		*k;
	Draw_rec	*d;

	for (k = p->krell; k; k = k->next)
		{
		gdk_gc_set_clip_mask(GK.text_GC, k->stencil);
		d = &k->draw;
		gdk_draw_pixmap(p->pixmap, GK.text_GC, k->pixmap,
					d->x_src, d->y_src, d->x_dst, d->y_dst, d->w, d->h);
		}
	gdk_gc_set_clip_mask(GK.text_GC, NULL);
	}


void
draw_decal(Panel *p, Decal *decal, gint index)
	{
	gint	y_src;

	if (decal->value != index)
		{
		/* Write the new decal mask onto the panel stencil XXX
		*/
		y_src = index * decal->h;
		decal->y_src = y_src;
		if (decal->mask)		/* Can be NULL if no transparency	*/
			gdk_draw_pixmap(p->stencil, p->stencil_gc, decal->mask,
					0, y_src, decal->x, decal->y, decal->w, decal->h);
		else	/* Fill decal extent with white	*/
			{
			gdk_gc_set_foreground(p->stencil_gc, &GK.white_color);
			gdk_draw_rectangle(p->stencil, p->stencil_gc,
					TRUE, decal->x, decal->y, decal->w, decal->h);
			}
		decal->modified = TRUE;
		}
	decal->value = index;
	}

void
draw_layers(Panel *p)
	{
	Krell		*k;
	Decal		*l;			/* LED's or other graphic	*/
	Draw_rec	*d;

	/* Restore background of panel where objects have changed.
	|  I should test for size of krell, and if large enough, just do this:
	|		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->background,
	|					0, 0,   0, 0,   p->w, p->h);
	|
	*/
	/* Restore background under old decal positions */
	for (l = p->decal; l; l = l->next)
		{
		if (l->modified)
			{
			if (l->type == DECAL_TRANSPARENCY)
				gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->background,
						l->x, l->y,  l->x, l->y,   l->w, l->h);
			p->modified = TRUE;
			}
		}
	/* Restore background under old krell positions */
	for (k = p->krell; k; k = k->next)
		if (k->modified)
			{
			d = &k->old_draw;
			gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->background,
					d->x_dst, d->y_dst,   d->x_dst, d->y_dst,   d->w, d->h);
			p->modified = TRUE;
			}

	/* For each layer, push new layer image onto the expose pixmap.
	*/
	if (p->modified)
		{
		push_decal_pixmaps(p);
		push_krell_pixmaps(p);

		/* Draw regions from the expose pixmap onto the screen.  Draw the
		|  Old krell region and the new krell region.
		|  I should test here for region sizes to minimize screen draws.
		*/
#if 0
		if (regions_to_draw_are_large_enough)	/* Blast expose pixmap */
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					0, 0,   0, 0,   p->w, p->h);
		else
#endif
		for (l = p->decal; l; l = l->next)
			{
			if (l->type != DECAL_ZOMBIE)
				gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC,
						p->pixmap, l->x, l->y,  l->x, l->y,   l->w, l->h);
			l->modified = FALSE;
			}
		for (k = p->krell; k; k = k->next)
			{
			d = &k->old_draw;
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					d->x_dst, d->y_dst,   d->x_dst, d->y_dst,   d->w, d->h);
			d = &k->draw;
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					d->x_dst, d->y_dst,   d->x_dst, d->y_dst,   d->w, d->h);
			k->modified = FALSE;
			}
		}
	p->modified = FALSE;
	}

void
destroy_krell(Krell *k)
	{
	if (k)
		{
		gdk_pixmap_unref(k->stencil);
		gdk_imlib_free_pixmap(k->pixmap);
		gdk_imlib_free_pixmap(k->mask);

		g_free(k);
		}
	}

void
create_krell(gchar *name, GdkImlibImage *im, Krell **krell, Style *style)
	{
	Krell	*k, *knew;
	gint	w, h, map_x, w_render;

	if (GK.trace)
		printf("create_krell()\n");

	knew = (Krell *) g_new0(Krell, 1);
	if (*krell == NULL)
		*krell = knew;
	else
		for (k = *krell; k; k = k->next)
			if (k->next == NULL)
				{
				k->next = knew;
				break;
				}
	k = knew;
	k->name = name;

	if (im == NULL || style == NULL)
		{
		printf("create_krell: NULL image or style\n");
		exit(0);
		}
	k->x0 = 0;
	k->y0 = style->krell_yoff;
	if (k->y0 < 0)
		k->y0 = style->border_panel.top;

	k->w_scale  = UC.chart_width - k->x0 - 1;

	w = im->rgb_width;
	h = im->rgb_height;

	if (style->krell_x_hot < 0)
		style->krell_x_hot = w / 2;
	w_render = w;
	k->x_hot = style->krell_x_hot;

	if (GK.debug)
		printf("Krell %s: x0=%d y0=%d w_scale=%d hot=%d w=%d h=%d exp=%d",
			k->name, k->x0, k->y0, k->w_scale,
			k->x_hot, w, h, style->krell_expand);

	if (style->krell_expand)
		{
		switch (style->krell_expand)
			{
			case LEFT:
				if (style->krell_x_hot > 0)
					{
					map_x = UC.chart_width /* - style->border_panel.right */;
					w_render = map_x * w / style->krell_x_hot;
					k->x_hot = map_x;
					}
				break;
			case RIGHT:
				if (w > style->krell_x_hot)
					{
					map_x = UC.chart_width /* - style->border_panel.left */;
					w_render = map_x * w / (w - style->krell_x_hot);
					k->x_hot = style->krell_x_hot * w_render / w;
					}
				break;
			}
		if (GK.debug)
			printf(" => w=%d hot=%d", w_render, k->x_hot);
		}
	else
		k->x_hot = style->krell_x_hot;

	gdk_imlib_render(im, w_render, h);

	k->pixmap = gdk_imlib_move_image(im);
	k->mask	  = gdk_imlib_move_mask(im);
	k->h = h;
	k->w = w_render;

	k->depth = style->krell_depth;
	if (k->depth < 1)
		k->depth = 1;

	k->h_stencil = k->y0 + h / k->depth;
	if (style->krell_yoff < 0)			/* Fit krell inside of borders? */
		k->h_stencil += style->border_panel.bottom;

	k->stencil = gdk_pixmap_new(top_window->window,
					UC.chart_width, k->h_stencil, 1);
	k->period = style->krell_ema_period;
	if (k->period >= 4 * UC.update_HZ)
		k->period = 4 * UC.update_HZ;
	if (GK.debug)
		printf("\n  h_stencil=%d depth=%d, period=%d\n",
			k->h_stencil, k->depth, k->period);
	k->x_position = -1;      /* Force initial draw   */
	k->full_scale_expand = 1;
	if (GK.trace)
		printf("  <-\n");
	}
