/* sfm -- Simple File Manager
   Copyright (C) 1998 Pixel (Pascal Rigaux)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */
#include "sfm_doc.h"
#include "filetype.h"
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/un.h>
#include <fcntl.h>


static void sfm_doc_set_directory(SfmDoc *doc, char *name);
static void sfm_doc_update_directory(SfmDoc *doc);
static boolean sfm_doc_a_begins_with_b(char *a, char *b);



extern SfmDoc *sfm_doc_new(int number, int *argc, char ***argv)
{
  char pwd[tmpSize];
  int arg = 1;
  SfmDoc *doc = malloc(sizeof(SfmDoc));
  doc->initial_cwd = (getcwd(pwd, tmpSize)) ? strdup(pwd) : NULL;
  doc->prog_name = (*argv)[0];
  doc->number = number;
  doc->dir_prefix = NULL;
  doc->dir_suffix = NULL;
  doc->dir_list.elements = NULL;

  doc->nb_dir_entries = 0;
  memset(doc->dir_entries, 0, NB_DIRENTRIES_MAX * sizeof(char *));

  parse_ls_color();

  doc->cdm = (arg < *argc) && streq((*argv)[arg], optionCdm);
  if (doc->cdm) arg++;

  sfm_doc_new_directory(doc, arg < *argc ? (*argv)[arg] : ".");
  
  return doc;
}

extern void sfm_doc_free(SfmDoc *doc)
{
  FREE_NULL(doc->initial_cwd);
  FREE_NULL(doc->dir_prefix);
  FREE_NULL(doc->dir_suffix);
  dir_list_free(doc->dir_list);
  pchar_free(doc->dir_entries, NB_DIRENTRIES_MAX);
  free(doc);
}

extern void sfm_doc_quit(SfmDoc *doc)
{
  if (doc->cdm) {
    if (doc->cdm != -1) printf("%s\n", doc->dir_prefix);
    else if (doc->initial_cwd) printf("%s\n", doc->initial_cwd);
  }
}

extern void sfm_doc_new_directory(SfmDoc *doc, char *name)
{
  sfm_doc_set_directory(doc, name);
}

extern boolean sfm_doc_up_directory(SfmDoc *doc)
{
  char *p;
  
  if (streq(doc->dir_suffix, "")) {
    if (streq(doc->dir_prefix, "/")) return false;
    chop(doc->dir_prefix);
    p = strrchr(doc->dir_prefix, '/');
    if (p == NULL)
      char_replace(&doc->dir_prefix, strdup("/"));
    else
      p[1] = '\0'; 
  }
  char_replace(&doc->dir_suffix, strdup(""));
  chdir(doc->dir_prefix);
  return true;
}

static void sfm_doc_set_directory(SfmDoc *doc, char *name)
{
  char *prefix, *suffix, *p;
  
  prefix = expandEnvVars(name);
  if (prefix[0] != '/' && doc->initial_cwd) {
    p = prefix;
    if (sfm_doc_a_begins_with_b(p, "./")) p += 2;
    else if (streq(p, ".")) p = "";
    p = strconcat3(doc->initial_cwd, "/", p);
    free(prefix);
    prefix = p;
  }
  supprDoubleSlashes(prefix);
  if ((p = strrchr(prefix, '/')) == NULL) {
    suffix = prefix;
    prefix = strdup("/");
  } else {
    suffix = strdup(p + 1);
    p[1] = '\0';
  }
  char_replace(&doc->dir_prefix, prefix);
  char_replace(&doc->dir_suffix, suffix);
  sfm_doc_directory_has_changed(doc);

  if (doc->dir_list.size == 1 && !streq(doc->dir_suffix, "")) {
    p = strconcat(prefix, doc->dir_list.elements[0].name);
    
    if (is_directory(p)) {
      char_replace(&doc->dir_prefix, p);
      char_replace(&doc->dir_suffix, strdup(""));
       
      sfm_doc_directory_has_changed(doc);
    } else free(p);
  } 
  chdir(doc->dir_prefix);
}

extern void sfm_doc_directory_has_changed(SfmDoc *doc)
{
  sfm_doc_update_directory(doc);
}

static void sfm_doc_update_directory(SfmDoc *doc)
{
  dir_list_free(doc->dir_list);
  doc->dir_list = dir_list_with_suffix(doc->dir_prefix, doc->dir_suffix);
}

static boolean sfm_doc_a_begins_with_b(char *a, char *b)
{
  return strncmp(a, b, strlen(b)) == 0;
}

extern int sfm_doc_search_starts_with(SfmDoc *doc, char *word, int startIndice)
{ 
  int i, j;
  
  for (i = 0; i < doc->dir_list.size; i++) { 
    j = (startIndice + i) % doc->dir_list.size;
    if (sfm_doc_a_begins_with_b(doc->dir_list.elements[j].name, word)) return j;
  }
  return -1;
}

extern void sfm_doc_delete(SfmDoc *doc, int *tab, int nb)
{
  char **args, **q;
  int i;
    
  args = pchar_malloc(nb + 3);
  args[0] = "rm";
  args[1] = "-rf";
  q = args + 2;
  for (i = 0; i < nb; i++) q[i] = strconcat(doc->dir_prefix, doc->dir_list.elements[tab[i]].name);
  q[nb] = NULL;

  exec_it_and_mem(args, "deleting");

  pchar_free(q, nb);
  free(args);
}

extern int sfm_doc_get_color_of_file(SfmDoc *doc, int indice)
{  
  return color_of_extension(doc->dir_list.elements[indice].name, doc->dir_list.elements[indice].mode);
}

extern boolean sfm_doc_rename(SfmDoc *doc, char *before, char *after)
{
  boolean b;
  char *compl_before = strconcat(doc->dir_prefix, before);
  char *compl_after = strconcat(doc->dir_prefix, after);
  
  if (lastchar(compl_before) == '/') chop(compl_before);
  if (lastchar(compl_after) == '/') chop(compl_after);
  b = rename(compl_before, compl_after) == 0;

  free(compl_before);
  free(compl_after);
  return b;
}

extern void sfm_doc_copy_or_cut(SfmDoc *doc, 
				     TypePaste type_paste, 
				     TypeAddOrOverride add_or_override, 
				     int *tab, int nb)
{
  int i, fd;

  if (!connect_server(&fd)) die("connect_server");
  send_msg(fd, "set\n");
  send_int(fd, type_paste);
  send_int(fd, add_or_override);
  for (i = 0; i < nb; i++) {
    send_msg(fd, doc->dir_prefix);
    send_msg(fd, doc->dir_list.elements[tab[i]].name);
    send_msg(fd, "\n");
  }
  send_msg(fd, "\n");
  close(fd);
}

extern void sfm_doc_paste(SfmDoc *doc)
{
  char *args[4];

  args[0] = doc->prog_name;
  args[1] = optionPaste;
  args[2] = doc->dir_prefix;
  args[3] = NULL;

  chdir(doc->initial_cwd);
  exec_it_and_mem(args, "pasting");
  chdir(doc->dir_prefix);
}

static void find_file(SfmDoc *doc, char *name, char *base_name)
{
  char *p;
  int i;
  
  strput3(name, doc->dir_prefix, doc->dir_suffix, base_name, tmpSize - 2);
  p = strend(name);
  for (i = 0; sprintf(p, "%d", i), is_file(name); i++);
}

extern boolean sfm_doc_create_directory(SfmDoc *doc)
{
  char name[tmpSize];

  find_file(doc, name, defaultCreateDirectoryName);
  return mkdir(name, 0777) == 0;
}

extern boolean sfm_doc_create_file(SfmDoc *doc)
{
  char name[tmpSize];
  int fd;  

  find_file(doc, name, defaultCreateFileName);
  if ((fd = creat(name, 0777)) == -1) return false;
  close(fd);
  return true;
}


extern boolean sfm_doc_new_view(SfmDoc *doc, int num)
{
  int fd;

  if (!connect_server(&fd)) die("connect_server");

  send_msg(fd, "new\n");
  send_int(fd, num);
  close(fd);
  return true;
}

extern boolean sfm_doc_action(SfmDoc *doc, char *file)
{
  char *prog, *absolute = strconcat(doc->dir_prefix, file);
  int fd;

  if (!connect_server(&fd)) die("connect_server");
  send_msg(fd, "get_prog\n");
  send_msg(fd, absolute); send_msg(fd, "\n");
  init_line_by_line_read();
  prog = strdup(line_by_line_read(fd));
  end_line_by_line_read();  
  free(absolute);

  if (strlen(prog) > 0) launch_action(&file, 1, prog);
  return strlen(prog) > 0;
}

extern void sfm_doc_many_actions(SfmDoc *doc, int *tab, int nb)
{
  char tmp[tmpSize];
  int i;

  for (i = 0; i < nb; i++) {
    strncpy(tmp, doc->dir_list.elements[tab[i]].name, tmpSize);
    sfm_doc_action(doc, tmp);
  }
}

extern void sfm_doc_that_action(SfmDoc *doc, char *file, char *prog)
{
  char *absolute = strconcat(doc->dir_prefix, file);
  int fd;

  launch_action(&file, 1, prog);

  if (!connect_server(&fd)) die("connect_server");
  send_msg(fd, "add_prog\n");
  send_msg(fd, absolute); send_msg(fd, "\n");
  send_msg(fd, prog); send_msg(fd, "\n");
  free(absolute);
}

extern char **sfm_doc_choose_action(SfmDoc *doc, char *file, int *nb)
{
  char **r, *absolute = strconcat(doc->dir_prefix, file);
  int i, fd;

  if (!connect_server(&fd)) die("connect_server");
  send_msg(fd, "get_progs\n");
  send_msg(fd, absolute); send_msg(fd, "\n");

  init_line_by_line_read();
  *nb = atoi(line_by_line_read(fd));
  if (*nb == 0) return NULL;
  r = pchar_malloc(*nb);
  for (i = 0; i < *nb; i++) r[i] = strdup(line_by_line_read(fd));
  end_line_by_line_read();  

  free(absolute);
  return r;
}

extern void sfm_doc_action_many_files(SfmDoc *doc, int *tab, int nb, char *prog)
{
  char **files;
  int i;

  if (nb == 0) return;
  files = pchar_malloc(nb);

  printf("\n"); /* for actions like ``ls -l'', acts like a separator */
  for (i = 0; i < nb; i++) files[i] = doc->dir_list.elements[tab[i]].name;
  launch_action(files, nb, prog);

  free(files);
}

extern int sfm_doc_find_name(SfmDoc *doc, char *name)
{
  int i;

  for (i = 0; i < doc->dir_list.size; i++) if (streq(name, doc->dir_list.elements[i].name)) return i;
  return -1;
}
 
extern char *sfm_doc_set_dir_entry(SfmDoc *doc)
{
  char *new_dir_entry = strconcat(doc->dir_prefix, doc->dir_suffix);
  char *old_dir_entry = doc->dir_entries[doc->nb_dir_entries];
  if (old_dir_entry == NULL || !streq(new_dir_entry, old_dir_entry)) {
    doc->current_dir_entry = doc->nb_dir_entries = (doc->nb_dir_entries + 1) % NB_DIRENTRIES_MAX;
    char_swap(&doc->dir_entries[doc->nb_dir_entries], &new_dir_entry);
  }
  doc->current_dir_entry = doc->nb_dir_entries;
  FREE_NULL(new_dir_entry);
  return doc->dir_entries[doc->nb_dir_entries];
}

extern char *sfm_doc_up_stack_dir_entries(SfmDoc *doc)
{
  int i = (doc->current_dir_entry - 1 + NB_DIRENTRIES_MAX) % NB_DIRENTRIES_MAX;
  if (i == doc->nb_dir_entries) return NULL;
  if (doc->dir_entries[i]) doc->current_dir_entry = i;
  return doc->dir_entries[i];
}
extern char *sfm_doc_down_stack_dir_entries(SfmDoc *doc)
{
  int i = (doc->current_dir_entry + 1) % NB_DIRENTRIES_MAX;
  if (doc->current_dir_entry == doc->nb_dir_entries) return NULL;
  if (doc->dir_entries[i]) doc->current_dir_entry = i;
  return doc->dir_entries[i];
}
