# ps.tcl - a tcl dialog to the ps command
# version 1.0, Feb 2, 1999
# Author: Samy Zafrany  (samy@netanya.ac.il // samy@math.technion.ac.il)
#         www.netanya.ac.il/~samy
#
namespace eval ps {
    variable ps

    # Define the ps array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set ps {
       dialog ""
       format:long ""
       format:user ""
       format:jobs ""
       format:signal ""
       format:vm ""
       format:memory_info ""
       format:forest ""
       format:other_users ""
       format:no_ctrl_term ""
       format:cpu_time_page_faults ""
       format:task_struct_cmd_name ""
       format:show_env ""
       format:wide ""
       format:no_header ""
       format:running_procs_only ""
       format:num_output_user_wchan ""
       sort:cmd          ""
       sort:cmdline      ""
       sort:flags        ""
       sort:pgrp         ""
       sort:tpgid        ""
       sort:cutime       ""
       sort:cstime       ""
       sort:utime        ""
       sort:stime        ""
       sort:min_flt      ""
       sort:maj_flt      ""
       sort:cmin_flt     ""
       sort:cmaj_flt     ""
       sort:session      ""
       sort:pid          ""
       sort:ppid         ""
       sort:rss          ""
       sort:resident     ""
       sort:size         ""
       sort:share        ""
       sort:tty          ""
       sort:start_time   ""
       sort:uid          ""
       sort:user         ""
       sort:vsize        ""
       sort:priority     ""
       sort:cmd:order          "+"
       sort:cmdline:order      "+"
       sort:flags:order        "+"
       sort:pgrp:order         "+"
       sort:tpgid:order        "+"
       sort:cutime:order       "+"
       sort:cstime:order       "+"
       sort:utime:order        "+"
       sort:stime:order        "+"
       sort:min_flt:order      "+"
       sort:maj_flt:order      "+"
       sort:cmin_flt:order     "+"
       sort:cmaj_flt:order     "+"
       sort:session:order      "+"
       sort:pid:order          "+"
       sort:ppid:order         "+"
       sort:rss:order          "+"
       sort:resident:order     "+"
       sort:size:order         "+"
       sort:share:order        "+"
       sort:tty:order          "+"
       sort:start_time:order   "+"
       sort:uid:order          "+"
       sort:user:order         "+"
       sort:vsize:order        "+"
       sort:priority:order     "+"
       controlling_device      ""
       tty:controlling:entry   ""
       tty:controlling:textvar ""
       cu:controlling:entry    ""
       cu:controlling:textvar  ""
    }
}

# ps::create --
#
#   Method to create the dialog box for the ps command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc ps::create { } {
    global tkWorld
    variable ps

    # Put the focus on the ps dialog if it is already open.
    if [winfo exists $ps(dialog)] {
	switch -- [wm state $ps(dialog)] {
	    normal {
		raise $ps(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $ps(dialog)
	    }
	}
	focus $ps(dialog)
	return
    } else {
	set ps(dialog) [dialog::create .ps Ps]
    }

    # The first tab deals with the standard ps command line options.
    set tab1 [tabnotebook::page [dialog::interior $ps(dialog)] "Format"]

    # Now build the individual checkbutton options available to
    # the user.
    set f1 [frame $tab1.f1 -class TabnotebookFrame]
    set format_checks {
       l  format:long  "Long Format"
       u  format:user  "User Format"
       a  format:other_users  "Show Procs of Other Users Too"
       w  format:wide  "Wide Format"
       j  format:jobs  "Jobs Format"
       v  format:vm  "Vm Format"
       m  format:memory_info  "Memory Info"
       f  format:forest  "Forest Format"
       x  format:no_ctrl_term  "Procs With no Controlling Term too"
       S  format:cpu_time_page_faults  "Add Child Cpu Time and Page Faults"
       c  format:task_struct_cmd_name  "Command Name from Task_struct"
       e  format:show_env  "Show Environment After Command Line"
       h  format:no_header  "No Header"
       s  format:signal  "Signal Format"
       r  format:running_procs_only  "Running Procs Only"
       n  format:num_output_user_wchan  "Numeric Output for USER and WCHAN"
    }

    set n 0
    foreach {onval var text} $format_checks {
        set cb $f1.$var
        checkbutton $cb \
	    -text "$text" \
	    -variable ps::ps($var) \
	    -onvalue "$onval" \
	    -offvalue ""
        set i [expr $n/2]
        set j [expr $n%2]
        incr n
        grid config $cb -row $i -column $j -padx 2 -pady 2 -sticky "w"

    }

    # Build the first tab.
    pack $f1 \
        -side top \
        -fill x   \
        -padx 5   \
        -pady 5   \
        -ipadx 5  \
        -ipady 5


    # The second tab has to do with ps sort options
    set tab2 [tabnotebook::page [dialog::interior .ps] "Sort By"]

    set f2 [frame $tab2.f2 \
	    -class TabnotebookFrame]


    set sort_checks {
       u   sort:user         "User Name"
       U   sort:uid          "User ID Number"
       k   sort:utime        "User Time"
       c   sort:cmd          "Simple Name of Executable"
       C   sort:cmdline      "Full Command Line"
       p   sort:pid          "Process ID"
       P   sort:ppid         "Parent Process ID"
       g   sort:pgrp         "Process Group ID"
       y   sort:priority     "Kernel Scheduling Priority"
       G   sort:tpgid        "Controlling TTY Process Group ID"
       j   sort:cutime       "Cumulative User Time"
       J   sort:cstime       "Cumulative System Time"
       K   sort:stime        "System Time"
       m   sort:min_flt      "Number of Minor Page Faults"
       M   sort:maj_flt      "Number of Major Page Faults"
       n   sort:cmin_flt     "Cumulative Minor Page Faults"
       N   sort:cmaj_flt     "Cumulative Major Page Faults"
       s   sort:size         "Memory Size in Kilobytes"
       o   sort:session      "Session ID"                       
       R   sort:resident     "Resident Pages"
       r   sort:rss          "Resident Set Size"
       v   sort:vsize        "Total VM Size in Bytes"
       f   sort:flags        "Flags As in Long Format F Field"
       S   sort:share        "Amount of Shared Pages"
       t   sort:tty          "The Minor Device Number of TTY"
       T   sort:start_time   "Time Process Was Started"
    }

    set n 0
    foreach {onval var text} $sort_checks {
        set b $f2.$var:order
        set cb $f2.$var
        button $b -textvariable "ps::ps($var:order)" \
                  -command [list ps::Toggle_textvar $b] \
                  -font fixed
        checkbutton $cb \
               -text "$text" \
               -variable ps::ps($var) \
               -onvalue "$onval" \
               -offvalue ""
        set i [expr $n/2]
        set j [expr $n%2]
        incr n
        grid config $b -row $i -column [expr 2*$j] \
               -padx 2 -pady 2 -sticky "w"
        grid config $cb -row $i -column [expr 2*$j+1] \
               -padx 2 -pady 2 -sticky "w"
    }

    pack $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

# The controlling tty/cu frame.
    set tab3 \
        [tabnotebook::page [dialog::interior $ps(dialog)] "Controlling tty/cu"]

    set f3 [frame $tab3.f3 -class TabnotebookFrame]
    radiobutton $f3.tty:controlling \
	    -text "Show Only  Procs  with  Controlling tty: " \
	    -variable ps::ps(controlling_device) \
	    -value "tty"
    set ps(tty:controlling:entry) [entry $f3.tty:controlling:entry \
	    -width 12 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable ps::ps(tty:controlling:textvar)]
    radiobutton $f3.cu:controlling \
	    -text "Show Only  Procs  with  Controlling cu: " \
	    -variable ps::ps(controlling_device) \
	    -value "cu"
    set ps(cu:controlling:entry) [entry $f3.cu:controlling:entry \
	    -width 12 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable ps::ps(cu:controlling:textvar)]
    radiobutton $f3.off \
	    -text "Off" \
	    -variable ps::ps(controlling_device) \
	    -value ""
    grid $f3.tty:controlling $f3.tty:controlling:entry \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3.cu:controlling $f3.cu:controlling:entry \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3.off - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    bind $f3.tty:controlling <ButtonPress-1> {ps::toggle_tty_entry tty}
    bind $f3.cu:controlling <ButtonPress-1> {ps::toggle_tty_entry cu}
    bind $f3.off <ButtonPress-1> {ps::toggle_tty_entry off}

    pack $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    set mf [frame $tab3.mf -class TabnotebookFrame]
    set m $mf.message
    message $m -aspect 200 -font fixed -text "
Only  procs  with  controlling tty xx; for xx you may use either the name
of a device file under \"/dev\" or that name with either tty or cu sliced off.
This is the reverse heuristic that ps uses to print out the abbreviated
tty name in the TTY field."

    pack $mf \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5
    pack $m -side top -fill x


# The field descriptions frame.
    set tab4 \
        [tabnotebook::page [dialog::interior $ps(dialog)] "Field Descriptions"]

    set f4 [frame $tab4.f4 -class TabnotebookFrame]
    set m $f4.message
    message $m -aspect 200 -font fixed -text "
                     FIELD DESCRIPTIONS
RSS      Resident set size; kilobytes of program in memory.
PID      Process id.
PPID     Parent process id.
UID      User id.
TT       Controlling tty.
TRS      Text resident size.
PRI      time in HZ of the process's possible timeslice.
NI       Standard unix nice value; a positive value means less cpu time.
SIZE     Virtual image size; size of text+data+stack.
SWAP     Kilobytes on swap device.
SHARE    Shared memory.
WCHAN    Name of the kernel function where the process is sleeping.
STAT     Information about the status  of the process. The first field
         is R for runnable, S for sleeping, D for uninterruptible sleep,
         T for stopped or traced, or Z for a zombie process.
PAGEIN   Number of major page faults (page faults that cause pages to be
         read from disk, including pages read from the buffer cache)."

    pack $f4 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5
    pack $m -side top -fill x
}


################# Procedures

# ps::Toggle_textvar --
#
#   Method to invert the +/- signs of the sort:xxx:order text variables
#
proc ps::Toggle_textvar {b} {
  variable ps
  set textvar [$b cget -textvariable]
  if {[subst $$textvar]=="+"} {
    set $textvar "-"
  } else {
    set $textvar "+"
  }
}

# ps::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc ps::ok { } {
    global tkWorld
    variable ps

    # Build the command line argument based on format, sort, and tty options
    set cmd_arg ""
    set format ""
    set sort ""
    set tty ""
    foreach x [array names ps] {
      if ![regexp {([^:]+):} $x dummy vartype] {continue}
      if [regexp {:order$} $x] {continue}
      switch $vartype {
         format  {
             if [string length $ps($x)] {
                  append format $ps($x)
             }
         }
         sort    {
             if [string length $ps($x)] {
                  append sort   "$ps($x:order)$ps($x)"
             }
         }
      }
    }
    if {$ps(controlling_device)=="tty"} {
      set tty $ps(tty:controlling:textvar)
    }
    if {$ps(controlling_device)=="cu"} {
      set tty $ps(cu:controlling:textvar)
    }
    if [string length $format] {
      append cmd_arg $format
    }
    if [string length $sort] {
      append cmd_arg " O$sort"
    }
    if [string length $tty] {
      append cmd_arg " t$tty"
    }
    if [string length $cmd_arg] {
    }

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert \
		[subst {ps $cmd_arg}]
    } else {
	$tkWorld(cmd_center) insert insert "ps"
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Change to the default background/foreground of the file entry.
    # $ps(entry.file) configure -bg white -fg black
}

# ps::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc ps::reset { } {
    variable ps

    set save {
      dialog
      tty:controlling:entry
      cu:controlling:entry
    }

    foreach x [array names ps] {
      if {[lsearch $save $x]>=0} {continue}
      if [regexp {:order$} $x] {
         set ps($x) "+"
         continue
      }
      set ps($x) ""
    }
}

# ps::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc ps::clear { } {
    variable ps

    # Reset the data structure elements and bg/fg.
    set ps(controlling_device)      ""
    set ps(tty:controlling:textvar) ""
    set ps(cu:controlling:textvar)  ""
    $ps(tty:controlling:entry) delete 0 end
    $ps(cu:controlling:entry) delete 0 end
    $ps(cu:controlling:entry) configure -bg #d9d9d9 -state disabled
    $ps(tty:controlling:entry) configure -bg #d9d9d9 -state disabled
}

# ps::help --
#
#   Method to invoke the Ps Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc ps::help { } {
    global tkWorld

    help::create "ps/index.html" "Ps Command Help"
}

# ps::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc ps::close { } {
    variable ps
    
    balloonhelp::cancel
    destroy $ps(dialog)
}


proc ps::toggle_tty_entry { type } {
    variable ps

    switch $type {
	tty {
	    $ps(tty:controlling:entry) configure \
		    -bg #ffffff \
		    -state normal
	    $ps(cu:controlling:entry) delete 0 end
	    $ps(cu:controlling:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	cu {
	    $ps(cu:controlling:entry) configure \
		    -bg #ffffff \
		    -state normal
	    $ps(tty:controlling:entry) delete 0 end
	    $ps(tty:controlling:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	off {
	    $ps(tty:controlling:entry) delete 0 end
	    $ps(tty:controlling:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $ps(cu:controlling:entry) delete 0 end
	    $ps(cu:controlling:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
    }
}
