/*
 *  simple.c:		Simple dialog widgets
 *
 *  Written by:		Ullrich Hafner
 *  
 *  Copyright (C) 1999 Ullrich Hafner <hafner@bigfoot.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1999/12/05 17:27:42 $
 *  $Author: hafner $
 *  $Revision: 1.13 $
 *  $State: Exp $
 */

#include "config.h"

#include <stdio.h>

#include <gtk/gtk.h>
#include <proplist.h>

#include "simple.h"
#include "dialog.h"
#include "menu.h"
#include "texture.h"
#include "misc.h"
#include "window.h"
#include "themes.h"

#include "error.h"

/*******************************************************************************

			     global variables
  
*******************************************************************************/

extern bool_t	  changed;		/* window.c */
extern proplist_t *windowmaker;		/* window.c */
 
/*******************************************************************************

			     local variables
  
*******************************************************************************/

#ifdef HAVE_LIBWMFUN
static const char **color_types = NULL;

/*******************************************************************************

			       prototypes
  
*******************************************************************************/

static void
color_toggle (GtkWidget *widget, gpointer ptr);
static void
show_color_buttons (text_color_t *text_color);
static void
update_text_color (proplist_t *key, gpointer ptr, proplist_t *value,
		   const char *path);
#endif /* HAVE_LIBWMFUN */
static void
set_bool (GtkWidget *widget, gpointer ptr);
static void
set_ibool (GtkWidget *widget, gpointer ptr);
static void
append_dialog (GtkWidget *container, GtkWidget *left, GtkWidget *right,
	       GtkAttachOptions attach);
static void
set_string (GtkWidget *widget, gpointer ptr);
static void
set_int (GtkWidget *widget, gpointer ptr);
static void
set_entry_text (GtkWidget *widget, gpointer ptr);
static void
set_fontname (GtkWidget *widget, gpointer ptr);
static void
fontsel_dialog (GtkWidget *widget, gpointer ptr);
static void
update_entry (proplist_t *key, gpointer ptr, proplist_t *value,
	      const char *path);
static void
update_color (proplist_t *key, gpointer ptr, proplist_t *value,
	      const char *path);
static void
update_boolean (proplist_t *key, gpointer ptr, proplist_t *value,
		const char *path);
static void
update_iboolean (proplist_t *key, gpointer ptr, proplist_t *value,
		 const char *path);
static void
update_int (proplist_t *key, gpointer ptr, proplist_t *value,
	    const char *path);

/*******************************************************************************

			       public code
  
*******************************************************************************/

void
boolean_dialog (GtkWidget *table, proplist_t *key, proplist_t *value,
		proplist_t *info)
{
   GtkWidget *label  = gtk_label_new (D_(PLGetString (info)));
   GtkWidget *button = gtk_check_button_new ();

   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				 convert_bool (value));
   gtk_signal_connect (GTK_OBJECT (button), "toggled",
		       GTK_SIGNAL_FUNC (set_bool), (gpointer) key);
   append_dialog (table, label, button, GTK_SHRINK);

   connect_update_function (key, button, update_boolean);
}

void
iboolean_dialog (GtkWidget *table, proplist_t *key, proplist_t *value,
		 proplist_t *info)
{
   GtkWidget *label  = gtk_label_new (D_(PLGetString (info)));
   GtkWidget *button = gtk_check_button_new ();

   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				 convert_bool (value) ? FALSE : TRUE);
   gtk_signal_connect (GTK_OBJECT (button), "toggled",
		       GTK_SIGNAL_FUNC (set_ibool), (gpointer) key);
   append_dialog (table, label, button, GTK_SHRINK);

   connect_update_function (key, button, update_iboolean);
}

void
string_dialog (GtkWidget *table, proplist_t *key, proplist_t *value,
	       proplist_t *range, proplist_t *rinfo, proplist_t *info)
{
   GtkWidget	*option_menu;
   GtkWidget	*label = gtk_label_new (D_(PLGetString (info)));
   const char	**values;
   unsigned	n, current = 0;
   
   values = Calloc (PLGetNumberOfElements (range) + 1, sizeof (char *));
   for (n = 0; n < PLGetNumberOfElements (range); n++)
   {
      proplist_t *element = PLGetArrayElement (range, n);
	       
      if (strcaseeq (PLGetString (element), PLGetString (value)))
	 current = n;
      values [n] = PLGetString (element);
   }
   values [n] = NULL;
		  
   option_menu = generate_option_menu (NULL, NULL, NULL, values,
				       values [current], rinfo, set_string, key);
   append_dialog (table, option_menu, label, GTK_FILL | GTK_SHRINK);

   {
      omenudata_t *data = Calloc (1, sizeof (omenudata_t));
      
      data->omenu  = GTK_OPTION_MENU (option_menu);
      data->values = values;
      connect_update_function (key, data, update_optionmenu);
   }
}

void
int_dialog (GtkWidget *table, proplist_t *key, proplist_t *value,
	    proplist_t *range, proplist_t *rinfo, proplist_t *info)
{
   GtkWidget    *label 	= gtk_label_new (D_(PLGetString (info)));
   GtkWidget    *hbox  	= gtk_hbox_new (FALSE, 5);
   GtkEntry	**array = Calloc (PLGetNumberOfElements (range) / 2 + 1,
				  sizeof (GtkEntry *));
   unsigned	n;
   
   for (n = 0; n < PLGetNumberOfElements (range); n += 2)
   {
      GtkWidget *spinner;
      GtkObject *adj;
      char	*m = PLGetString (PLGetArrayElement (range, n));
      char	*M = PLGetString (PLGetArrayElement (range, n + 1));
      char 	*v;
	       
      if (PLIsArray (value))
	 v = PLGetString (PLGetArrayElement (value, n / 2));
      else
	 v = PLGetString (value);
	       
      adj = gtk_adjustment_new (strtod (v, NULL),
				strtod (m, NULL), strtod (M, NULL),
				1.0, 5.0, 0.0);
      spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
      gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinner), TRUE);
      gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
      gtk_widget_show_all (spinner);
      gtk_object_set_user_data (GTK_OBJECT (adj), (gpointer) spinner);
      gtk_box_pack_start (GTK_BOX (hbox), spinner, TRUE, TRUE, 0);
      gtk_object_set_user_data (GTK_OBJECT (spinner),
				(gpointer) (spinner + n / 2));

      gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			  GTK_SIGNAL_FUNC (set_int), (gpointer) key);
      gtk_signal_connect (GTK_OBJECT (&GTK_SPIN_BUTTON (spinner)->entry),
			  "changed", GTK_SIGNAL_FUNC (set_entry_text),
			  (gpointer) key);

      array [n / 2] = &GTK_SPIN_BUTTON (spinner)->entry;
   }
   array [n / 2] = NULL;

   append_dialog (table, hbox, label, GTK_FILL | GTK_SHRINK);

   connect_update_function (key, array, update_int);
}

void
text_dialog (GtkWidget *table, proplist_t *key, proplist_t *value,
	     proplist_t *info)
{
   GtkWidget *entry = gtk_entry_new ();
   GtkWidget *label = gtk_label_new (D_(PLGetString (info)));
		  
   gtk_widget_set_usize (entry, 100, 0);
   gtk_entry_set_text (GTK_ENTRY (entry), PLGetString (value));
   gtk_signal_connect (GTK_OBJECT (entry), "changed",
		       GTK_SIGNAL_FUNC (set_entry_text), key);
   append_dialog (table, entry, label, GTK_FILL | GTK_SHRINK);
   connect_update_function (key, entry, update_entry);
}

GtkWidget *
font_dialog (GtkWidget *page, proplist_t *key, proplist_t *value,
	     proplist_t *info)
{
   GtkWidget *entry  = gtk_entry_new ();
   GtkWidget *frame  = gtk_frame_new (D_(PLGetString (info)));
   GtkWidget *hbox   = gtk_hbox_new (FALSE, 0);
   GtkWidget *bbox   = gtk_hbutton_box_new ();
   GtkWidget *button = gtk_button_new_with_label (_("Browse..."));

   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
   gtk_container_add (GTK_CONTAINER (frame), hbox);

   gtk_entry_set_text (GTK_ENTRY (entry), PLGetString (value));
   gtk_signal_connect (GTK_OBJECT (entry), "changed",
		       GTK_SIGNAL_FUNC (set_entry_text), key);
   gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 5);
   gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 5);
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);

   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (fontsel_dialog), (gpointer) entry);

   gtk_box_pack_end (GTK_BOX (page), frame, TRUE, TRUE, 5);
   gtk_widget_show_all (frame);

   connect_update_function (key, entry, update_entry);
   
   return frame;
}

GtkWidget *
color_dialog (GtkWidget *page, proplist_t *key, proplist_t *value,
	      GtkTooltips *tooltips, proplist_t *info, proplist_t *title)
{
   GtkWidget *frame;
   GtkWidget *button = color_button (NULL, key, value, tooltips,
				     D_(PLGetString (info)));
   GtkWidget *hbox = gtk_hbox_new (TRUE, 0);

   if (title)
      frame = gtk_frame_new (D_(PLGetString (title)));
   else
      frame = gtk_frame_new (PLGetString (key));

   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_container_add (GTK_CONTAINER (frame), hbox);
   gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
   
   gtk_box_pack_start (GTK_BOX (page), frame, FALSE, TRUE, 5);
   gtk_widget_show_all (frame);

   connect_update_function (key,
			    gtk_object_get_user_data (GTK_OBJECT (button)),
			    update_color);
   
   return frame;
}

#ifdef HAVE_LIBWMFUN
GtkWidget *
extended_color_dialog (GtkWidget *page, proplist_t *key, proplist_t *value,
		       GtkTooltips *tooltips, proplist_t *info, proplist_t *title)
{
   GtkWidget   	*frame;
   GtkWidget   	*option_menu;
   text_color_t *text_color = Calloc (1, sizeof (text_color_t));
   GtkWidget   	*hbox 	    = gtk_hbox_new (FALSE, 5);

   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   
   frame = gtk_frame_new (_("Color of title text"));
   text_color->key = key;

   if (!color_types)
   {
      color_types = Calloc (TCOLOR_LAST, sizeof (char *));
      
      color_types [TCOLOR_SIMPLE]   = _("Solid color");
      color_types [TCOLOR_COLORSET] = _("Drop shadow");
      color_types [TCOLOR_LAST]     = NULL;
   }

   option_menu = generate_option_menu (NULL, tooltips,
				       _("A solid color or a drop shadow."),
				       color_types, color_types [0], NULL,
				       color_toggle, text_color);
   text_color->option_menu = option_menu;
   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_container_add (GTK_CONTAINER (frame), hbox);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 0);

   {
      GtkWidget *button;

      text_color->solid_color = color_button (NULL, key, NULL, NULL, NULL);
      gtk_box_pack_start (GTK_BOX (hbox), text_color->solid_color,
			  FALSE, TRUE, 0);
      text_color->color_set [0] = color_button (NULL, key, NULL,
						tooltips, _("Foreground"));
      gtk_box_pack_start (GTK_BOX (hbox), text_color->color_set [0],
			  FALSE, TRUE, 0);
      text_color->color_set [1] = color_button (NULL, key, NULL,
						tooltips, _("Shadow 1"));
      gtk_box_pack_start (GTK_BOX (hbox), text_color->color_set [1],
			  FALSE, TRUE, 0);
      text_color->color_set [2] = color_button (NULL, key, NULL,
						tooltips, _("Shadow 2"));
      gtk_box_pack_start (GTK_BOX (hbox), text_color->color_set [2],
			  FALSE, TRUE, 0);
   }
   
   gtk_box_pack_start (GTK_BOX (page), frame, FALSE, FALSE, 0);
   gtk_widget_show_all (frame);

   show_color_buttons (text_color);
   
   connect_update_function (key, text_color, update_text_color);
   
   return frame;
}
#endif /* HAVE_LIBWMFUN */

void
update_optionmenu (proplist_t *key, gpointer ptr, proplist_t *value,
		   const char *path)
{
   omenudata_t *data = (omenudata_t *) ptr;

   if (PLIsString (value))
      set_option_menu_default (data->omenu, data->values, PLGetString (value));
}

bool_t
convert_bool (proplist_t *p)
/*
 *  Parse boolean values of WindowMaker file.
 *
 *  Return value:
 *	converted value
 */
{
   char *val;

   assert (p && PLIsSimple (p));

   val = PLGetString (p);

   if (strcaseeq (val, "y") || strcaseeq (val, "yes")
       || strcaseeq (val, "t") || strcaseeq (val, "true")
       || strcaseeq (val, "1"))
      return TRUE;
   else if (strcaseeq (val, "n") || strcaseeq (val, "no")
	    || strcaseeq (val, "f") || strcaseeq (val, "false")
	    || strcaseeq (val, "0"))
      return FALSE;
   else
   {
      warning (_("Undefined boolean value '%s' converted to true."), val);
      return TRUE;
   }
}

/*******************************************************************************

			       private code
  
*******************************************************************************/

#ifdef HAVE_LIBWMFUN
static void
color_toggle (GtkWidget *widget, gpointer ptr)
{
   char 	*text 	    = gtk_object_get_user_data (GTK_OBJECT (widget));
   text_color_t *text_color = (text_color_t *) ptr;
   proplist_t 	*value 	    = PLGetDictionaryEntry (windowmaker, text_color->key);
   
   if (text == color_types [TCOLOR_SIMPLE])
   {
      if (!PLIsString (value))
      {
	 PLInsertDictionaryEntry (windowmaker, text_color->key,
				  PLMakeString ("tan"));
	 toggle_save (changed = YES, text_color->key);
	 show_color_buttons (text_color);
      }
   }
   else
   {
      if (!PLIsArray (value))
      {
	 proplist_t *pl;
	 pl = PLMakeArrayFromElements (PLMakeString ("function"),
				       PLMakeString ("libwmfun.so"),
				       PLMakeString ("drawPlainString"),
				       PLMakeString ("white"),
				       PLMakeString ("black"),
				       PLMakeString ("red"), NULL);
      
	 PLInsertDictionaryEntry (windowmaker, text_color->key, pl);
	 toggle_save (changed = YES, text_color->key);
	 show_color_buttons (text_color);
      }
   }
}

static void
show_color_buttons (text_color_t *text_color)
{
   proplist_t *value = PLGetDictionaryEntry (windowmaker, text_color->key);
   
   if (PLIsArray (value))		/* check syntax of value */
   {
      if (!(PLGetNumberOfElements (value) == 6
	    && strcaseeq (PLGetString (PLGetArrayElement (value, 0)),
			  "function")
	    && strncaseeq (PLGetString (PLGetArrayElement (value, 1)),
			   "libwmfun", strlen ("libwmfun"))
	    && strcaseeq (PLGetString (PLGetArrayElement (value, 2)),
			  "drawPlainString")))
      {
	 proplist_t *pl;
	 pl = PLMakeArrayFromElements (PLMakeString ("function"),
				       PLMakeString ("libwmfun.so"),
				       PLMakeString ("drawPlainString"),
				       PLMakeString ("white"),
				       PLMakeString ("black"),
				       PLMakeString ("red"), NULL);
	 
	 PLInsertDictionaryEntry (windowmaker, text_color->key, pl);
	 value = pl;
      }
      set_option_menu_default (GTK_OPTION_MENU (text_color->option_menu),
			       color_types, color_types [TCOLOR_COLORSET]);

      /*
       *  Set colors
       */
      {
	 GtkWidget *preview;

	 preview = gtk_object_get_user_data (GTK_OBJECT (text_color->color_set [0]));
	 fill_preview (GTK_PREVIEW (preview), PLGetString (PLGetArrayElement (value, 3)));
	 gtk_object_set_user_data (GTK_OBJECT (preview), PLGetArrayElement (value, 3));
	 preview = gtk_object_get_user_data (GTK_OBJECT (text_color->color_set [1]));
	 fill_preview (GTK_PREVIEW (preview), PLGetString (PLGetArrayElement (value, 4)));
	 gtk_object_set_user_data (GTK_OBJECT (preview), PLGetArrayElement (value, 4));
	 preview = gtk_object_get_user_data (GTK_OBJECT (text_color->color_set [2]));
	 fill_preview (GTK_PREVIEW (preview), PLGetString (PLGetArrayElement (value, 5)));
	 gtk_object_set_user_data (GTK_OBJECT (preview), PLGetArrayElement (value, 5));
      }
      gtk_widget_show (text_color->color_set [0]);
      gtk_widget_show (text_color->color_set [1]);
      gtk_widget_show (text_color->color_set [2]);
      gtk_widget_hide (text_color->solid_color);
   }
   else
   {
      gtk_widget_hide (text_color->color_set [0]);
      gtk_widget_hide (text_color->color_set [1]);
      gtk_widget_hide (text_color->color_set [2]);
      gtk_widget_show (text_color->solid_color);
      set_option_menu_default (GTK_OPTION_MENU (text_color->option_menu),
			       color_types, color_types [TCOLOR_SIMPLE]);
      /*
       *  Set default color
       */
      {
	 GtkWidget *preview;

	 preview = gtk_object_get_user_data (GTK_OBJECT (text_color->solid_color));
	 fill_preview (GTK_PREVIEW (preview), PLGetString (value));
	 gtk_object_set_user_data (GTK_OBJECT (preview), value);
      }
   }
}
#endif /* HAVE_LIBWMFUN */

static void
append_dialog (GtkWidget *container, GtkWidget *left, GtkWidget *right,
	       GtkAttachOptions attach)
{
   if (GTK_IS_TABLE (container))
   {
      unsigned *number = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (container));
      unsigned row, col;
   
      row  = ++*number;
      col  = (row % 2) * 2;
      row /= 2;

      gtk_table_attach (GTK_TABLE (container), left, col, col + 1, row, row + 1,
			GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
      gtk_table_attach (GTK_TABLE (container), right, col + 1, col + 2, row, row + 1,
			attach, attach, 0, 0);

      gtk_widget_show_all (left);
      gtk_widget_show_all (right);
   }
   else if (GTK_IS_VBOX (container))
   {
      GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
      gtk_container_set_border_width (GTK_CONTAINER (container), 5);
      gtk_box_pack_start (GTK_BOX (hbox), right, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox), left, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (container), hbox, FALSE, FALSE, 0);
      gtk_widget_show_all (hbox);
   }
}

static void
set_bool (GtkWidget *widget, gpointer ptr)
/*
 *  Update boolean value.
 *
 *  No return value.
 */
{
   proplist_t *key  = (proplist_t *) ptr;
   char       *text = GTK_TOGGLE_BUTTON (widget)->active ? "Yes" : "No";
   proplist_t *pl   = PLMakeString (text);
   
   PLInsertDictionaryEntry (windowmaker, key, pl);
   PLRelease (pl);
   toggle_save (changed = YES, key);

   if (strcaseeq (PLGetString (key), "NewStyle"))
      check_widgetstyle ();
}

static void
set_ibool (GtkWidget *widget, gpointer ptr)
/*
 *  Update boolean value.
 *
 *  No return value.
 */
{
   proplist_t *key  = (proplist_t *) ptr;
   char       *text = GTK_TOGGLE_BUTTON (widget)->active ? "No" : "Yes";
   proplist_t *pl   = PLMakeString (text);
   
   PLInsertDictionaryEntry (windowmaker, key, pl);
   PLRelease (pl);
   toggle_save (changed = YES, key);

   if (strcaseeq (PLGetString (key), "NewStyle"))
      check_widgetstyle ();
}

static void
set_string (GtkWidget *widget, gpointer ptr)
/*
 *  Update string value.
 *
 *  No return value.
 */
{
   proplist_t *key  = (proplist_t *) ptr;
   char       *text = gtk_object_get_user_data (GTK_OBJECT (widget));
   proplist_t *pl   = PLMakeString (text);
   
   PLInsertDictionaryEntry (windowmaker, key, pl);
   PLRelease (pl);
   toggle_save (changed = YES, key);
}

static void
set_int (GtkWidget *widget, gpointer ptr)
/*
 *  Update unsigned value.
 *
 *  No return value.
 */
{
   proplist_t	 *key = (proplist_t *) ptr;
   GtkSpinButton *spinner;
   int		 val;
   char		 tmp [MAXSTRLEN];
   
   spinner = GTK_SPIN_BUTTON (gtk_object_get_user_data (GTK_OBJECT (widget)));
   val     = gtk_spin_button_get_value_as_int (spinner);
   
   sprintf (tmp, "%d", val);

   if (PLIsArray (PLGetDictionaryEntry (windowmaker, key)))
   {
      proplist_t *array = PLGetDictionaryEntry (windowmaker, key);
      unsigned	 n;
      
      n = (GtkWidget *) gtk_object_get_user_data (GTK_OBJECT (spinner))
	  - GTK_WIDGET (spinner);
      PLRemoveArrayElement (array, n);
      PLInsertArrayElement (array, PLMakeString (tmp), n);
   }
   else
   {
      proplist_t *pl = PLMakeString (tmp);
      PLInsertDictionaryEntry (windowmaker, key, pl);
      PLRelease (pl);
   }
   
   toggle_save (changed = YES, key);
}

static void
set_entry_text (GtkWidget *widget, gpointer ptr)
/*
 *  Update string value.
 *
 *  No return value.
 */
{
   proplist_t *element;
   proplist_t *key   = (proplist_t *) ptr;
   proplist_t *array = PLGetDictionaryEntry (windowmaker, key);

   element = PLMakeString ((char *) gtk_entry_get_text (GTK_ENTRY (widget)));

   if (PLIsArray (array))
   {
      PLRemoveArrayElement (array, 1);	/* filename */
      PLInsertArrayElement (array, element, 1);
   }
   else
   {
      PLInsertDictionaryEntry (windowmaker, key, element);
      PLRelease (element);
   }
   
   toggle_save (changed = YES, key);
}

static void
fontsel_dialog (GtkWidget *widget, gpointer ptr)
{
  static GtkWidget *fontsel = NULL;

  if (!fontsel)
  {
     GtkWidget			*entry;
     char			*text;
     GtkFontSelectionDialog	*fs;
     
     entry   = GTK_WIDGET (ptr);
     text    = gtk_entry_get_text (GTK_ENTRY (entry));
     fontsel = gtk_font_selection_dialog_new (_("Font selection"));
     fs	     = GTK_FONT_SELECTION_DIALOG (fontsel);
     
     gtk_window_set_position (GTK_WINDOW (fontsel), GTK_WIN_POS_MOUSE);
     gtk_object_set_user_data (GTK_OBJECT (fontsel), entry);
     gtk_font_selection_dialog_set_font_name (fs, text);

     gtk_object_set_data (GTK_OBJECT (entry), "fontsel", fontsel);
     gtk_signal_connect (GTK_OBJECT (fontsel), "destroy",
			 GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			 &fontsel);
     gtk_signal_connect  (GTK_OBJECT (fs->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (set_fontname), fontsel);
     gtk_signal_connect_object (GTK_OBJECT (fs->ok_button), "clicked",
				GTK_SIGNAL_FUNC (gtk_widget_destroy),
				GTK_OBJECT (fontsel));
     gtk_signal_connect_object (GTK_OBJECT (fs->cancel_button), "clicked",
				GTK_SIGNAL_FUNC (gtk_widget_destroy),
				GTK_OBJECT (fontsel));
  }

  if (!GTK_WIDGET_VISIBLE (fontsel))
     gtk_widget_show (fontsel);
  else
     gtk_widget_destroy (fontsel);
}

static void
set_fontname (GtkWidget *widget, gpointer ptr)
{
   GtkWidget *fontsel = GTK_WIDGET (ptr);
   GtkWidget *entry   = gtk_object_get_user_data (GTK_OBJECT (fontsel));
   gchar     *fontname;

   fontname = gtk_font_selection_dialog_get_font_name (GTK_FONT_SELECTION_DIALOG (fontsel));
   gtk_entry_set_text (GTK_ENTRY (entry), fontname);
}

static void
update_entry (proplist_t *key, gpointer ptr, proplist_t *value,
	      const char *path)
{
   if (PLIsString (value))
      gtk_entry_set_text (GTK_ENTRY (ptr), PLGetString (value));
}

static void
update_color (proplist_t *key, gpointer ptr, proplist_t *value,
	      const char *path)
{
   if (PLIsString (value))
   {
      GtkWidget  *preview  = GTK_WIDGET (ptr);
      proplist_t *oldvalue = gtk_object_get_user_data (GTK_OBJECT (preview));

      PLRelease (oldvalue);
      PLRetain (value);
      gtk_object_set_user_data (GTK_OBJECT (preview), value);
      fill_preview (GTK_PREVIEW (preview), PLGetString (value));
      PLInsertDictionaryEntry (windowmaker, key, value);
      toggle_save (changed = YES, key);
   }
}

#ifdef HAVE_LIBWMFUN
static void
update_text_color (proplist_t *key, gpointer ptr, proplist_t *value,
		   const char *path)
{
   if (PLIsString (value) || PLIsArray (value))
   {
      text_color_t *text_color = (text_color_t *) ptr;

      PLInsertDictionaryEntry (windowmaker, key, value);
      toggle_save (changed = YES, key);
      show_color_buttons (text_color);
   }
}
#endif /* HAVE_LIBWMFUN */

static void
update_boolean (proplist_t *key, gpointer ptr, proplist_t *value,
		const char *path)
{
   if (PLIsString (value))
   {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ptr),
				    convert_bool (value));
      set_bool (GTK_WIDGET (ptr), key);
   }
}

static void
update_iboolean (proplist_t *key, gpointer ptr, proplist_t *value,
		 const char *path)
{
   if (PLIsString (value))
   {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ptr),
				    convert_bool (value) ? FALSE : TRUE);
      set_bool (GTK_WIDGET (ptr), key);
   }
}

static void
update_int (proplist_t *key, gpointer ptr, proplist_t *value,
	    const char *path)
{
   GtkEntry **entry = (GtkEntry **) ptr;
   
   if (PLIsString (value))
      gtk_entry_set_text (entry [0], PLGetString (value));
   else if (PLIsArray (value))
   {
      unsigned n;

      for (n = 0; n < PLGetNumberOfElements (value); n++)
	 if (entry [n])
	    gtk_entry_set_text (entry [n],
				PLGetString (PLGetArrayElement (value, n)));
	 else
	    break;
   }
}
