/***************************************************************************

	Midway MCR-68k system
	
	Currently implemented:
		* Zwackery (Chip Squeak Deluxe)
		* Xenopohobe (Sounds Good)
		* Spy Hunter 2 (Turbo Chip Squeak)
		* Blasted (Unknown Sound)
		* Arch Rivals
		* Tri-Sports
		* Pigskin 621AD

	Emulation by Bryan McPhail, mish@tendril.force9.net and
	Aaron Giles agiles@pobox.com

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1
	========================================================================
	000000-03FFFF   R     xxxxxxxx xxxxxxxx    Program ROM
	060000-063FFF   R/W   xxxxxxxx xxxxxxxx    Program RAM
	070000-070FFF   R/W   xxxxxxxx xxxxxxxx    Background video RAM
	071000-071FFF   R/W   xxxxxxxx xxxxxxxx    Additional RAM
	080000-080FFF   R/W   xxxxxxxx xxxxxxxx    Sprite RAM
	090000-0900FF     W   xxxxxxxx xxxxxxxx    Palette RAM
	0A0000-0A000F   R/W   xxxxxxxx --------    M6840 I/O
	0B0000-0B0001     W   -------- --------    Watchdog
	0C0000-0C0001     W   -------- xxxxxxxx    Sound I/O
	0D0000-0D0001   R     xxxxxxxx xxxxxxxx    External inputs
	0E0000-0E0001   R     xxxxxxxx xxxxxxxx    External inputs
	0F0000-0F0001   R     xxxxxxxx xxxxxxxx    External inputs
	========================================================================
	Interrupts:
		NMI ???
		INT generated by CTC
	========================================================================
	NOTE: Pigskin and Tri-Sports have similar memory maps; the addresses
			are simply shuffled around
	========================================================================

***************************************************************************/

#include "driver.h"
#include "machine/mcr.h"
#include "sndhrdw/mcr.h"
#include "vidhrdw/generic.h"


extern UINT16 mcr68_char_code_mask;
extern UINT16 mcr68_sprite_code_mask;
extern UINT8 mcr68_sprite_clip;
extern INT8 mcr68_sprite_xoffset;

static UINT8 *control_word;


void mcr68_videoram_w(int offset, int data);
void mcr68_paletteram_w(int offset, int data);
void mcr68_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);

void zwackery_videoram_w(int offset, int data);
void zwackery_paletteram_w(int offset, int data);
void zwackery_spriteram_w(int offset, int data);
void zwackery_convert_color_prom(unsigned char *palette, unsigned short *colortable, const unsigned char *color_prom);
void zwackery_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);



/*************************************
 *
 *	Zwackery-specific handlers
 *
 *************************************/

int zwackery_port_2_r(int offset)
{
	int result = input_port_2_r(offset);
	int wheel = input_port_5_r(offset);
	
	return result | ((wheel >> 2) & 0x3e);
}


static int zwackery_6840_r(int offset)
{
	/* Zwackery does a timer test:                          */
	/* It loads $1388 into one of the timers clocked by E   */
	/* Then it sits in a tight loop counting down from $4E4 */
	/*       BTST #$1,($2,A0)                               */
	/*       DBNE D1,*-6                                    */
	/* It expects D1 to end up between 0 and 5; in order to */
	/* make this happen, we must assume that reads from the */
	/* 6840 take 14 additional cycles                       */
	*cpuintf[Machine->drv->cpu[0].cpu_type & ~CPU_FLAGS_MASK].icount -= 14;
	return mcr68_6840_upper_r(offset);
}



/*************************************
 *
 *	Xenophobe-specific handlers
 *
 *************************************/

static void xenophobe_control_w(int offset, int data)
{
	int oldword = READ_WORD(&control_word[offset]);
	int newword = COMBINE_WORD(oldword, data);
	WRITE_WORD(&control_word[offset], newword);
	
/*	soundsgood_reset_w(~newword & 0x0020);*/
	soundsgood_data_w(offset, ((newword & 0x000f) << 1) | ((newword & 0x0010) >> 4));
}



/*************************************
 *
 *	Spy Hunter 2-specific handlers
 *
 *************************************/

static int spyhunt2_port_0_r(int offset)
{
	int result = input_port_0_r(offset);
	int which = (READ_WORD(control_word) >> 3) & 3;
	int analog = readinputport(3 + which);
	return result | (analog << 8);
}


static void spyhunt2_control_w(int offset, int data)
{
	int oldword = READ_WORD(&control_word[offset]);
	int newword = COMBINE_WORD(oldword, data);
	WRITE_WORD(&control_word[offset], newword);

	turbocs_reset_w(~newword & 0x0080);
	turbocs_data_w(offset, (newword >> 8) & 0x001f);

	csdeluxe_reset_w(~newword & 0x2000);
}



/*************************************
 *
 *	Blasted-specific handlers
 *
 *************************************/

static void blasted_control_w(int offset, int data)
{
	int oldword = READ_WORD(&control_word[offset]);
	int newword = COMBINE_WORD(oldword, data);
	WRITE_WORD(&control_word[offset], newword);
	
	if (errorlog) fprintf(errorlog, "extcontrol = %04X\n", data);
}



/*************************************
 *
 *	Arch Rivals-specific handlers
 *
 *************************************/

static int archrivl_port_1_r(int offset)
{
	int joystick = input_port_3_r(offset);
	int result = 0;

	/* each axis of the 49-way joystick is mapped like this:*/
	/*      0 8      = neutral                              */
	/*      1        = slightly left/up                     */
	/*      2 3      = middle left/up                       */
	/*      4 5 6 7  = full left/up                         */
	/*      C        = slightly right/down                  */
	/*      A E      = middle right/down                    */
	/*      9 B D F  = full right/down                      */

	if (joystick & 0x0001) result |= 0x0040;
	else if (joystick & 0x0002) result |= 0x0090;

	if (joystick & 0x0004) result |= 0x0004;
	else if (joystick & 0x0008) result |= 0x0009;

	if (joystick & 0x0010) result |= 0x4000;
	else if (joystick & 0x0020) result |= 0x9000;

	if (joystick & 0x0040) result |= 0x0400;
	else if (joystick & 0x0080) result |= 0x0900;

	return result;
}


static void archrivl_control_w(int offset, int data)
{
	int oldword = READ_WORD(&control_word[offset]);
	int newword = COMBINE_WORD(oldword, data);
	WRITE_WORD(&control_word[offset], newword);

	advaudio_reset_w(~newword & 0x0400);
	advaudio_data_w(offset, newword & 0x3ff);
}



/*************************************
 *
 *	Pigskin-specific handlers
 *
 *************************************/

static UINT8 protection_data[5];
static void pigskin_protection_w(int offset, int data)
{
	/* ignore upper-byte only */
	if (data & 0x00ff0000) return;
	
	/* track the last 5 bytes */
	protection_data[0] = protection_data[1];
	protection_data[1] = protection_data[2];
	protection_data[2] = protection_data[3];
	protection_data[3] = protection_data[4];
	protection_data[4] = data;
	
	if (errorlog) fprintf(errorlog, "%06X:protection_w=%02X\n", cpu_getpreviouspc(), data & 0xff);
}


static int pigskin_protection_r(int offset)
{
	/* based on the last 5 bytes return a value */
	if (protection_data[4] == 0xe3 && protection_data[3] == 0x94)
		return 0x00;	/* must be <= 1 */
	if (protection_data[4] == 0xc7 && protection_data[3] == 0x7b && protection_data[2] == 0x36)
		return 0x00;	/* must be <= 1 */
	if (protection_data[4] == 0xc7 && protection_data[3] == 0x7b)
		return 0x07;	/* must be > 5 */
	if (protection_data[4] == 0xc7 && protection_data[3] == 0x1f && protection_data[2] == 0x03 &&
		protection_data[1] == 0x25 && protection_data[0] == 0x36)
		return 0x00;	/* must be < 3 */
	
	if (errorlog) fprintf(errorlog, "Protection read after unrecognized sequence: %02X %02X %02X %02X %02X\n",
			protection_data[0], protection_data[1], protection_data[2], protection_data[3], protection_data[4]);

	return 0x00;
}


static int pigskin_port_1_r(int offset)
{
	int joystick = input_port_3_r(offset);
	int result = input_port_1_r(offset);

	/* see archrivl_port_1_r for 49-way joystick description */

	if (joystick & 0x0001) result |= 0x4000;
	else if (joystick & 0x0002) result |= 0x9000;

	if (joystick & 0x0004) result |= 0x0400;
	else if (joystick & 0x0008) result |= 0x0900;

	return result;
}


static int pigskin_port_2_r(int offset)
{
	int joystick = input_port_3_r(offset);
	int result = input_port_2_r(offset);

	/* see archrivl_port_1_r for 49-way joystick description */

	if (joystick & 0x0010) result |= 0x4000;
	else if (joystick & 0x0020) result |= 0x9000;

	if (joystick & 0x0040) result |= 0x0400;
	else if (joystick & 0x0080) result |= 0x0900;

	return result;
}



/*************************************
 *
 *	Tri-Sports-specific handlers
 *
 *************************************/

static int trisport_port_1_r(int offset)
{
	int xaxis = (INT8)input_port_3_r(offset);
	int yaxis = (INT8)input_port_4_r(offset);
	int result = input_port_1_r(offset);
	
	result |= (xaxis & 0x3c) << 6;
	result |= (yaxis & 0x3c) << 10;

	return result;
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static struct MemoryReadAddress mcr68_readmem[] =
{
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x060000, 0x063fff, MRA_BANK2 },
	{ 0x070000, 0x070fff, MRA_BANK3 },
	{ 0x071000, 0x071fff, MRA_BANK4 },
	{ 0x080000, 0x080fff, MRA_BANK5 },
	{ 0x0a0000, 0x0a000f, mcr68_6840_upper_r },
	{ 0x0d0000, 0x0dffff, input_port_0_r },
	{ 0x0e0000, 0x0effff, input_port_1_r },
	{ 0x0f0000, 0x0fffff, input_port_2_r },
	{ -1 }  /* end of table */
};


static struct MemoryWriteAddress mcr68_writemem[] =
{
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x060000, 0x063fff, MWA_BANK2 },
	{ 0x070000, 0x070fff, mcr68_videoram_w, &videoram, &videoram_size },
	{ 0x071000, 0x071fff, MWA_BANK4 },
	{ 0x080000, 0x080fff, MWA_BANK5, &spriteram, &spriteram_size },
	{ 0x090000, 0x09007f, mcr68_paletteram_w, &paletteram },
	{ 0x0a0000, 0x0a000f, mcr68_6840_upper_w },
	{ 0x0b0000, 0x0bffff, watchdog_reset_w },
	{ 0x0c0000, 0x0cffff, MWA_BANK6, &control_word },
	{ -1 }  /* end of table */
};



/*************************************
 *
 *	Zwackery main CPU memory handlers
 *
 *************************************/

static struct MemoryReadAddress zwackery_readmem[] =
{
	{ 0x000000, 0x037fff, MRA_ROM },
	{ 0x080000, 0x080fff, MRA_BANK2 },
	{ 0x084000, 0x084fff, MRA_BANK3 },
	{ 0x100000, 0x10000f, zwackery_6840_r },
	{ 0x104000, 0x104007, pia_2_r },
	{ 0x108000, 0x108007, pia_3_r },
	{ 0x10c000, 0x10c007, pia_4_r },
	{ 0x800000, 0x800fff, MRA_BANK4 },
	{ 0x802000, 0x803fff, paletteram_word_r },
	{ 0xc00000, 0xc00fff, MRA_BANK5 },
	{ -1 }  /* end of table */
};


static struct MemoryWriteAddress zwackery_writemem[] =
{
	{ 0x000000, 0x037fff, MWA_ROM },
	{ 0x080000, 0x080fff, MWA_BANK2 },
	{ 0x084000, 0x084fff, MWA_BANK3 },
	{ 0x100000, 0x10000f, mcr68_6840_upper_w },
	{ 0x104000, 0x104007, pia_2_w },
	{ 0x108000, 0x108007, pia_3_w },
	{ 0x10c000, 0x10c007, pia_4_w },
	{ 0x800000, 0x800fff, zwackery_videoram_w, &videoram, &videoram_size },
	{ 0x802000, 0x803fff, zwackery_paletteram_w, &paletteram },
	{ 0xc00000, 0xc00fff, zwackery_spriteram_w, &spriteram, &spriteram_size },
	{ -1 }  /* end of table */
};



/*************************************
 *
 *	Pigskin CPU memory handlers
 *
 *************************************/

static struct MemoryReadAddress pigskin_readmem[] =
{
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x080000, 0x08ffff, pigskin_port_1_r },
	{ 0x0a0000, 0x0affff, pigskin_port_2_r },
	{ 0x100000, 0x100fff, MRA_BANK2 },
	{ 0x120000, 0x120001, pigskin_protection_r },
	{ 0x140000, 0x143fff, MRA_BANK3 },
	{ 0x160000, 0x1607ff, MRA_BANK4 },
	{ 0x180000, 0x18000f, mcr68_6840_upper_r },
	{ 0x1e0000, 0x1effff, input_port_0_r },
	{ -1 }  /* end of table */
};


static struct MemoryWriteAddress pigskin_writemem[] =
{
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x0c0000, 0x0c007f, mcr68_paletteram_w, &paletteram },
	{ 0x0e0000, 0x0effff, watchdog_reset_w },
	{ 0x100000, 0x100fff, mcr68_videoram_w, &videoram, &videoram_size },
	{ 0x120000, 0x120001, pigskin_protection_w },
	{ 0x140000, 0x143fff, MWA_BANK3 },
	{ 0x160000, 0x1607ff, MWA_BANK4, &spriteram, &spriteram_size },
	{ 0x180000, 0x18000f, mcr68_6840_upper_w },
	{ 0x1a0000, 0x1affff, MWA_BANK6, &control_word },
	{ -1 }  /* end of table */
};



/*************************************
 *
 *	Tri-Sports CPU memory handlers
 *
 *************************************/

static struct MemoryReadAddress trisport_readmem[] =
{
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x080000, 0x08ffff, trisport_port_1_r },
	{ 0x0a0000, 0x0affff, input_port_2_r },
	{ 0x100000, 0x103fff, MRA_BANK2 },
	{ 0x140000, 0x1407ff, MRA_BANK3 },
	{ 0x160000, 0x160fff, MRA_BANK4 },
	{ 0x180000, 0x18000f, mcr68_6840_upper_r },
	{ 0x1e0000, 0x1effff, input_port_0_r },
	{ -1 }  /* end of table */
};


static struct MemoryWriteAddress trisport_writemem[] =
{
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x100000, 0x103fff, MWA_BANK2 },
	{ 0x120000, 0x12007f, mcr68_paletteram_w, &paletteram },
	{ 0x140000, 0x1407ff, MWA_BANK3, &spriteram, &spriteram_size },
	{ 0x160000, 0x160fff, mcr68_videoram_w, &videoram, &videoram_size },
	{ 0x180000, 0x18000f, mcr68_6840_upper_w },
	{ 0x1a0000, 0x1affff, MWA_BANK6, &control_word },
	{ 0x1c0000, 0x1cffff, watchdog_reset_w },
	{ -1 }  /* end of table */
};



/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( zwackery_input_ports )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE( 0x0010, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* sword */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )	/* sound communications */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* spell up */
	PORT_BIT( 0x3e, IP_ACTIVE_HIGH, IPT_UNUSED )	/* encoder wheel */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* shield */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 )	/* spell down */

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
	
	PORT_START
	PORT_DIPNAME( 0x07, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 6C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x38, 0x00, "Buy-in" )
	PORT_DIPSETTING(    0x00, "1 coin" )
	PORT_DIPSETTING(    0x08, "2 coins" )
	PORT_DIPSETTING(    0x10, "3 coins" )
	PORT_DIPSETTING(    0x18, "4 coins" )
	PORT_DIPSETTING(    0x20, "5 coins" )
	PORT_DIPSETTING(    0x28, "6 coins" )
	PORT_DIPSETTING(    0x30, "7 coins" )
	PORT_DIPSETTING(    0x38, "None" )
	PORT_DIPNAME( 0xc0, 0x00, "Difficulty" )
	PORT_DIPSETTING(    0xc0, "Easier" )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x40, "Harder" )
	PORT_DIPSETTING(    0x80, "Hardest" )
	
	PORT_START
	PORT_ANALOGX( 0xff, 0x00, IPT_DIAL | IPF_REVERSE, 50, 10, 0, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )
INPUT_PORTS_END


INPUT_PORTS_START( xenophob_input_ports )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0018, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0003, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Coins per Life Unit" )
	PORT_DIPSETTING(      0x0008, "1" )
	PORT_DIPSETTING(      0x0000, "2" )
	PORT_DIPNAME( 0x0010, 0x0010, "Life Unit" )
	PORT_DIPSETTING(      0x0010, "1000" )
	PORT_DIPSETTING(      0x0000, "2000" )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( On ) )
	PORT_DIPNAME( 0x00c0, 0x0000, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0040, "Easy" )
	PORT_DIPSETTING(      0x0000, "Medium" )
	PORT_DIPSETTING(      0x0080, "Hard" )
/*	PORT_DIPSETTING(      0x00c0, "Medium" )*/
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( spyhunt2_input_ports )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 ) /* 1st gear */
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 ) /* 2nd gear */
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON7 | IPF_PLAYER2 ) /* 3rd gear */
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 ) /* 1st gear */
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 ) /* 2nd gear */
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON7 | IPF_PLAYER1 ) /* 3rd gear */
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 ) /* Left Trigger */
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 ) /* Left Button */
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 ) /* Right Trigger */
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 ) /* Right Button */
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 ) /* Left Trigger */
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 ) /* Left Button */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 ) /* Right Trigger */
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 ) /* Right Button */

	PORT_START	/* IN3 -- dipswitches */
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_2C ) )
	/*PORT_DIPSETTING(      0x0000, DEF_STR( 1C_2C ) )*/
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0018, 0x0008, "Point Threshholds" )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x0018, "Medium" )
	PORT_DIPSETTING(      0x0010, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0060, 0x0060, "Free Timer After" )
	PORT_DIPSETTING(      0x0000, "30 sec" )
	PORT_DIPSETTING(      0x0040, "45 sec" )
	PORT_DIPSETTING(      0x0060, "60 sec" )
	PORT_DIPSETTING(      0x0020, "90 sec" )
	PORT_BITX( 0x0080,    0x0080, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Advance", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
	
	PORT_START	/* analog ports for steering and pedals */
	PORT_ANALOG( 0xff, 0x30, IPT_PEDAL | IPF_PLAYER2 | IPF_REVERSE, 100, 10, 0, 0x30, 0xff )

	PORT_START
	PORT_ANALOG( 0xff, 0x30, IPT_PEDAL | IPF_PLAYER1 | IPF_REVERSE, 100, 10, 0, 0x30, 0xff )
	
	PORT_START
	PORT_ANALOG( 0xff, 0x80, IPT_PADDLE | IPF_PLAYER2 | IPF_REVERSE, 80, 10, 0, 0x10, 0xf0 )

	PORT_START
	PORT_ANALOG( 0xff, 0x80, IPT_PADDLE | IPF_PLAYER1 | IPF_REVERSE, 80, 10, 0, 0x10, 0xf0 )
INPUT_PORTS_END


INPUT_PORTS_START( blasted_input_ports )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNUSED/* credit w/bill */ )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED/* credit 1 w/bill */ )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNUSED/* credit 2 w/bill */ )
	PORT_BIT( 0xf000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_2C ) )
	/*PORT_DIPSETTING(      0x0000, DEF_STR( 1C_2C ) )*/
	PORT_DIPNAME( 0x000c, 0x0000, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x0000, "Medium" )
	PORT_DIPSETTING(      0x0004, "Hard" )
/*	PORT_DIPSETTING(      0x000c, "Medium" )*/
	PORT_DIPNAME( 0x0020, 0x0020, "Dollar Receptor" )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_BITX( 0x0080,    0x0080, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Advance", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( archrivl_input_ports )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0c00, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )

	PORT_START
	PORT_BIT( 0xffff, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 1/2 joysticks go here */

	PORT_START
	PORT_DIPNAME( 0x0003, 0x0003, "Game Time" )
	PORT_DIPSETTING(      0x0003, "Preset Time" )
	PORT_DIPSETTING(      0x0002, "Preset + 10sec" )
	PORT_DIPSETTING(      0x0001, "Preset + 20sec" )
	PORT_DIPSETTING(      0x0000, "Preset + 30sec" )
	PORT_DIPNAME( 0x001c, 0x001c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0014, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x001c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Team Names" )
	PORT_DIPSETTING(      0x0020, "Default" )
	PORT_DIPSETTING(      0x0000, "Hometown Heroes" )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 49-way joystick simulator */
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
INPUT_PORTS_END


INPUT_PORTS_START( pigskin_input_ports )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x000f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 1 joystick goes here */

	PORT_START
	PORT_DIPNAME( 0x0003, 0x0003, "Game Time" )
	PORT_DIPSETTING(      0x0000, "Shortest" )
	PORT_DIPSETTING(      0x0002, "Short" )
	PORT_DIPSETTING(      0x0003, "Medium" )
	PORT_DIPSETTING(      0x0001, "Long" )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0004, "Set Your Own" )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Test Switch" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Coin Chutes" )
	PORT_DIPSETTING(      0x0000, "Individual" )
	PORT_DIPSETTING(      0x0040, "Common" )
	PORT_DIPNAME( 0x0080, 0x0080, "Joystick" )
	PORT_DIPSETTING(      0x0080, "Standard" )
	PORT_DIPSETTING(      0x0000, "Rotated" )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 2 joystick goes here */

	PORT_START	/* 49-way joystick simulator */
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
INPUT_PORTS_END


INPUT_PORTS_START( trisport_input_ports )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0f00, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0xc000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* analog controls go here */

	PORT_START
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0000, "Battery Options" )
	PORT_DIPNAME( 0x0018, 0x0018, "Pool Turns" )
	PORT_DIPSETTING(      0x0010, "5" )
	PORT_DIPSETTING(      0x0008, "6" )
	PORT_DIPSETTING(      0x0018, "7" )
	PORT_DIPSETTING(      0x0000, "8" )
	PORT_DIPNAME( 0x0020, 0x0020, "Bowling Difficulty" )
	PORT_DIPSETTING(      0x0020, "Standard" )
	PORT_DIPSETTING(      0x0000, "Advanced" )
	PORT_DIPNAME( 0x0040, 0x0040, "Shot Timer" )
	PORT_DIPSETTING(      0x0000, "Slower" )
	PORT_DIPSETTING(      0x0040, "Standard" )
	PORT_DIPNAME( 0x0080, 0x0080, "Golf Holes" )
	PORT_DIPSETTING(      0x0080, "3" )
	PORT_DIPSETTING(      0x0000, "4" )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_X | IPF_PLAYER1, 100, 20, 0x7f, 0, 0 )

	PORT_START
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_Y | IPF_REVERSE | IPF_PLAYER1, 100, 20, 0x7f, 0, 0 )
INPUT_PORTS_END



/*************************************
 *
 *	Graphics definitions
 *
 *************************************/

MCR_CHAR_LAYOUT(charlayout, 4096);
MCR_SPRITE_LAYOUT(spritelayout, 1024);

static struct GfxLayout zwackery_layout =
{
	16,16,
	1024,
	8,
	{ 0, 0, 0, 0, 0, 0, 0, 0 },
	{  3,  2,  1,  0, 11, 10,  9,  8,
	  19, 18, 17, 16, 27, 26, 25, 24 },
	{ 0x0000*8+4, 0x4000*8+4, 0x0000*8+0, 0x4000*8+0, 0x0000*8+36, 0x4000*8+36, 0x0000*8+32, 0x4000*8+32,
	  0x0000*8+68, 0x4000*8+68, 0x0000*8+64, 0x4000*8+64, 0x0000*8+100, 0x4000*8+100, 0x0000*8+96, 0x4000*8+96 },
	128
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ 1, 0x00000, &charlayout,   0, 4 },
	{ 1, 0x20000, &spritelayout, 0, 4 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo zwackery_gfxdecodeinfo[] =
{
	{ 1, 0x00000, &zwackery_layout,      0, 16 },
	{ 1, 0x20000, &spritelayout,     0x800, 32 },
	{ 1, 0x00000, &zwackery_layout,      0, 16 },	/* yes, an extra copy */
	{ -1 } /* end of array */
};



/*************************************
 *
 *	Machine drivers
 *
 *************************************/

/*=================================================================

	Timing for these games is crucial. They all use the 6840 timer
	to preemptively multitask during each frame. The clock on the
	6840 timer is taken from the 68000's E clock, which runs at
	1/10th the speed of the 68000 itself.
	
	All the games run in a sequence of six steps per frame, using
	counter 1 on the 6840 to time each step. The sum total of the
	6 programmed steps for each game determines how many E clocks
	should be generated per frame, which in turn determines the
	clock we should expect the CPU to have.
 
	Ideal CPU timings, based on counter usage:

		Zwackery:     7652400
		Xenophobe:    7723800
		Spy Hunter 2: 7723800
		Blasted:      7798800
		Arch Rivals:  7799100
		Pigskin:      9211200
		Tri-Sports:   9211200

	Currently, we are using the Xenophobe CPU for the first four 
	until we spot problems.

=================================================================*/

static struct MachineDriver zwackery_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7652400,	/* 8 Mhz */
			0,
			zwackery_readmem,zwackery_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_CHIP_SQUEAK_DELUXE(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	zwackery_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	zwackery_gfxdecodeinfo,
	4096, 4096,
	zwackery_convert_color_prom,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	zwackery_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_CHIP_SQUEAK_DELUXE
	}
};


static struct MachineDriver xenophob_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7723800,	/* 8 Mhz */
			0,
			mcr68_readmem,mcr68_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_SOUNDS_GOOD(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr68_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	8*16, 8*16,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr68_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_SOUNDS_GOOD
	}
};


static struct MachineDriver spyhunt2_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7723800,	/* 8 Mhz */
			0,
			mcr68_readmem,mcr68_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_TURBO_CHIP_SQUEAK(2),
		SOUND_CPU_CHIP_SQUEAK_DELUXE(3)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr68_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	8*16, 8*16,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr68_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_TURBO_CHIP_SQUEAK_PLUS_CSDELUXE
	}
};


static struct MachineDriver blasted_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7723800,	/* 8 Mhz */
			0,
			mcr68_readmem,mcr68_writemem,0,0,
			mcr68_interrupt,1
		},
/*		SOUND_CPU_SOUNDS_GOOD(2)	-- no sound ROMs yet! */
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr68_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	8*16, 8*16,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr68_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_SOUNDS_GOOD
	}
};


static struct MachineDriver archrivl_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7723800,	/* 8 Mhz */
			0,
			mcr68_readmem,mcr68_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_ADVANCED_AUDIO(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr68_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	8*16, 8*16,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr68_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_ADVANCED_AUDIO
	}
};


static struct MachineDriver pigskin_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7723800,	/* 8 Mhz */
			0,
			pigskin_readmem,pigskin_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_ADVANCED_AUDIO(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr68_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	8*16, 8*16,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr68_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_ADVANCED_AUDIO
	}
};


static struct MachineDriver trisport_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7723800,	/* 8 Mhz */
			0,
			trisport_readmem,trisport_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_ADVANCED_AUDIO(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr68_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	8*16, 8*16,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_DIRTY | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr68_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_ADVANCED_AUDIO
	}
};



/*************************************
 *
 *	Driver initialization
 *
 *************************************/

static void zwackery_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(MCR_CHIP_SQUEAK_DELUXE);

	/* Zwackery doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);
}


static void xenophob_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(MCR_SOUNDS_GOOD);

	mcr68_char_code_mask = 0x7ff;
	mcr68_sprite_code_mask = 0x1ff;
	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = 0;
	
	/* Xenophobe doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* install control port handler */
	install_mem_write_handler(0, 0x0c0000, 0x0cffff, xenophobe_control_w);
}


static void spyhunt2_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(MCR_TURBO_CHIP_SQUEAK | MCR_CHIP_SQUEAK_DELUXE);

	mcr68_char_code_mask = 0x7ff;
	mcr68_sprite_code_mask = 0x3ff;
	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = -6;
	
	/* Spy Hunter 2 doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* analog port handling is a bit tricky */
	install_mem_write_handler(0, 0x0c0000, 0x0cffff, spyhunt2_control_w);
	install_mem_read_handler(0, 0x0d0000, 0x0dffff, spyhunt2_port_0_r);
}


static void blasted_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(0);

	mcr68_char_code_mask = 0x7ff;
	mcr68_sprite_code_mask = 0x3ff;
	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = 0;

	/* Blasted checks the timing of VBLANK relative to the 493 interrupt */
	/* VBLANK is required to come within 220-256 E clocks (i.e., 2200-2560 CPU clocks) */
	/* after the 493; we also allow 16 E clocks for latency  */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* handle control writes */
	install_mem_write_handler(0, 0x0c0000, 0x0cffff, blasted_control_w);

	/* 6840 is mapped to the lower 8 bits */
	install_mem_write_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_w);
	install_mem_read_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_r);
}


static void archrivl_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(MCR_ADVANCED_AUDIO);

	mcr68_char_code_mask = 0xfff;
	mcr68_sprite_code_mask = 0x3ff;
	mcr68_sprite_clip = 16;
	mcr68_sprite_xoffset = 0;
	
	/* Arch Rivals doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* handle control writes */
	install_mem_write_handler(0, 0x0c0000, 0x0cffff, archrivl_control_w);

	/* 49-way joystick handling is a bit tricky */
	install_mem_read_handler(0, 0x0e0000, 0x0effff, archrivl_port_1_r);
	
	/* 6840 is mapped to the lower 8 bits */
	install_mem_write_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_w);
	install_mem_read_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_r);
	
	/* expand the sound ROMs */
	memcpy(&Machine->memory_region[2][0x18000], &Machine->memory_region[2][0x10000], 0x08000);
	memcpy(&Machine->memory_region[2][0x28000], &Machine->memory_region[2][0x20000], 0x08000);
	memcpy(&Machine->memory_region[2][0x38000], &Machine->memory_region[2][0x30000], 0x08000);
	memcpy(&Machine->memory_region[2][0x40000], &Machine->memory_region[2][0x30000], 0x10000);
	
	/* CVSD data:
		Interrupt rate = 55931Hz / 3 = 18643Hz
		1 bit/interrupt --> final sample rate = 18643Hz
		
		count & data stored at $023E
		current address in Y
		stop address in $0242
		bank in $022B
	*/
}


static void pigskin_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(MCR_ADVANCED_AUDIO);

	/* handle control writes */
	install_mem_write_handler(0, 0x1a0000, 0x1affff, archrivl_control_w);

	/* Pigskin doesn't care too much about this value; currently taken from Tri-Sports */
	mcr68_timing_factor = 115.0 / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	mcr68_char_code_mask = 0xfff;
	mcr68_sprite_code_mask = 0x3ff;
	mcr68_sprite_clip = 16;
	mcr68_sprite_xoffset = 0;

	/* expand the sound ROMs */
	memcpy(&Machine->memory_region[2][0x40000], &Machine->memory_region[2][0x30000], 0x10000);
	
	/* CVSD data:
		Interrupt rate = 55931Hz / 9 = 6215Hz
		4 bits/interrupt --> final sample rate = 24858Hz
		
		count & data stored at $023F
		current address in X
		stop address in $0243
		bank in $022B
		
		also DAC data (2 bytes/interrupt = 12429Hz)
		current address in Y
		stop address in $0234
		bank in $022C
	*/
}


static void trisport_init(void)
{
	MCR_CONFIGURE_NO_HISCORE;
	MCR_CONFIGURE_SOUND(MCR_ADVANCED_AUDIO);

	/* Tri-Sports checks the timing of VBLANK relative to the 493 interrupt */
	/* VBLANK is required to come within 87-119 E clocks (i.e., 870-1190 CPU clocks) */
	/* after the 493 */
	mcr68_timing_factor = 115.0 / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* handle control writes */
	install_mem_write_handler(0, 0x1a0000, 0x1affff, archrivl_control_w);

	mcr68_char_code_mask = 0xfff;
	mcr68_sprite_code_mask = 0x3ff;
	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = 0;
	
	/* expand the sound ROMs */
	memcpy(&Machine->memory_region[2][0x38000], &Machine->memory_region[2][0x30000], 0x08000);
	memcpy(&Machine->memory_region[2][0x40000], &Machine->memory_region[2][0x30000], 0x10000);
	
	/* same CVSD as pigskin */
}



/*************************************
 *
 *	ROM decoding
 *
 *************************************/

static void rom_decode(void)
{
	int i;

	/* tile graphics are inverted */
	for (i = 0; i < 0x20000; i++)
		Machine->memory_region[1][i] ^= 0xff;
}



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( zwackery_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "pro0.bin",   0x00000, 0x4000, 0x6fb9731c )
	ROM_LOAD_ODD ( "pro1.bin",   0x00000, 0x4000, 0x84b92555 )
	ROM_LOAD_EVEN( "pro2.bin",   0x08000, 0x4000, 0xe6977a2a )
	ROM_LOAD_ODD ( "pro3.bin",   0x08000, 0x4000, 0xf5d0a60e )
	ROM_LOAD_EVEN( "pro4.bin",   0x10000, 0x4000, 0xec5841d9 )
	ROM_LOAD_ODD ( "pro5.bin",   0x10000, 0x4000, 0xd7d99ce0 )
	ROM_LOAD_EVEN( "pro6.bin",   0x18000, 0x4000, 0xb9fe7bf5 )
	ROM_LOAD_ODD ( "pro7.bin",   0x18000, 0x4000, 0x5e261b3b )
	ROM_LOAD_EVEN( "pro8.bin",   0x20000, 0x4000, 0x55e380a5 )
	ROM_LOAD_ODD ( "pro9.bin",   0x20000, 0x4000, 0x12249dca )
	ROM_LOAD_EVEN( "pro10.bin",  0x28000, 0x4000, 0x6a39a8ca )
	ROM_LOAD_ODD ( "pro11.bin",  0x28000, 0x4000, 0xad6b45bc )
	ROM_LOAD_EVEN( "pro12.bin",  0x30000, 0x4000, 0xe2d25e1f )
	ROM_LOAD_ODD ( "pro13.bin",  0x30000, 0x4000, 0xe131f9b8 )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "tileh.bin",    0x00000, 0x4000, 0xa7237eb1 )
	ROM_LOAD( "tileg.bin",    0x04000, 0x4000, 0x626cc69b )
	ROM_LOAD( "spr6h.bin",    0x20000, 0x4000, 0xa51158dc )
	ROM_LOAD( "spr7h.bin",    0x24000, 0x4000, 0x941feecf )
	ROM_LOAD( "spr6j.bin",    0x40000, 0x4000, 0xf3eef316 )
	ROM_LOAD( "spr7j.bin",    0x44000, 0x4000, 0xa8a34033 )
	ROM_LOAD( "spr10h.bin",   0x60000, 0x4000, 0xa99daea6 )
	ROM_LOAD( "spr11h.bin",   0x64000, 0x4000, 0xc1a767fb )
	ROM_LOAD( "spr10j.bin",   0x80000, 0x4000, 0x4dd04376 )
	ROM_LOAD( "spr11j.bin",   0x84000, 0x4000, 0xe8c6a880 )

	ROM_REGION(0x20000)
	ROM_LOAD_EVEN( "csd7.bin",  0x00000, 0x2000, 0x5501f54b )
	ROM_LOAD_ODD ( "csd17.bin", 0x00000, 0x2000, 0x2e482580 )
	ROM_LOAD_EVEN( "csd8.bin",  0x04000, 0x2000, 0x13366575 )
	ROM_LOAD_ODD ( "csd18.bin", 0x04000, 0x2000, 0xbcfe5820 )

	ROM_REGION(0x8000)	/* bg color maps */
	ROM_LOAD_GFX_EVEN( "tilef.bin",  0x0000, 0x4000, 0xa0dfcd7e )
	ROM_LOAD_GFX_ODD ( "tilee.bin",  0x0000, 0x4000, 0xab504dc8 )
ROM_END


ROM_START( xenophob_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "xeno_pro.3c",  0x00000, 0x10000, 0xf44c2e60 )
	ROM_LOAD_ODD ( "xeno_pro.3b",  0x00000, 0x10000, 0x01609a3b )
	ROM_LOAD_EVEN( "xeno_pro.2c",  0x20000, 0x10000, 0xe45bf669 )
	ROM_LOAD_ODD ( "xeno_pro.2b",  0x20000, 0x10000, 0xda5d39d5 )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "xeno_bg.11d",  0x00000, 0x08000, 0x3d2cf284 )
	ROM_LOAD( "xeno_bg.12d",  0x10000, 0x08000, 0xc32288b1 )
	ROM_LOAD( "xeno_fg.7j",   0x20000, 0x10000, 0xb12eddb2 )
	ROM_LOAD( "xeno_fg.8j",   0x40000, 0x10000, 0x20e682f5 )
	ROM_LOAD( "xeno_fg.9j",   0x60000, 0x10000, 0x82fb3e09 )
	ROM_LOAD( "xeno_fg.10j",  0x80000, 0x10000, 0x6a7a3516 )

	ROM_REGION(0x40000)  /* Sounds Good board */
	ROM_LOAD_EVEN( "xeno_snd.u7",  0x00000, 0x10000, 0x77561d15 )
	ROM_LOAD_ODD ( "xeno_snd.u17", 0x00000, 0x10000, 0x837a1a71 )
	ROM_LOAD_EVEN( "xeno_snd.u8",  0x20000, 0x10000, 0x6e2915c7 )
	ROM_LOAD_ODD ( "xeno_snd.u18", 0x20000, 0x10000, 0x12492145 )
ROM_END


ROM_START( spyhunt2_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "3c",  0x00000, 0x10000, 0x5b92aadf )
	ROM_LOAD_ODD ( "3b",  0x00000, 0x10000, 0x6ed0a25f )
	ROM_LOAD_EVEN( "2c",  0x20000, 0x10000, 0xbc834f3f )
	ROM_LOAD_ODD ( "2b",  0x20000, 0x10000, 0x8a9f7ef3 )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "bg0.11d",  0x00000, 0x08000, 0x81efef7a )
	ROM_LOAD( "bg1.12d",  0x10000, 0x08000, 0x6a902e4d )
	ROM_LOAD( "fg0.7j",   0x20000, 0x20000, 0x55ce12ea )
	ROM_LOAD( "fg1.8j",   0x40000, 0x20000, 0x692afb67 )
	ROM_LOAD( "fg2.9j",   0x60000, 0x20000, 0xf1aba383 )
	ROM_LOAD( "fg3.10j",  0x80000, 0x20000, 0xd3475ff8 )

	ROM_REGION(0x10000)  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbo-cs.u5", 0x08000, 0x4000, 0x4b1d8a66 )
	ROM_LOAD( "turbo-cs.u4", 0x0c000, 0x4000, 0x3722ce48 )

	ROM_REGION(0x8000)  /* 32k for the Chip Squeak Deluxe */
	ROM_LOAD_EVEN( "csd_u7a.u7",   0x00000, 0x2000, 0x6e689fe7 )
	ROM_LOAD_ODD ( "csd_u17b.u17", 0x00000, 0x2000, 0x0d9ddce6 )
	ROM_LOAD_EVEN( "csd_u8c.u8",   0x04000, 0x2000, 0x35563cd0 )
	ROM_LOAD_ODD ( "csd_u18d.u18", 0x04000, 0x2000, 0x63d3f5b1 )
ROM_END


ROM_START( blasted_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "3c",  0x00000, 0x10000, 0xb243b7df )
	ROM_LOAD_ODD ( "3b",  0x00000, 0x10000, 0x627e30d3 )
	ROM_LOAD_EVEN( "2c",  0x20000, 0x10000, 0x026f30bf )
	ROM_LOAD_ODD ( "2b",  0x20000, 0x10000, 0x8e0e91a9 )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "11d",  0x00000, 0x08000, 0xd8ed5cbc )
	ROM_LOAD( "12d",  0x10000, 0x08000, 0x60d00c69 )
	ROM_LOAD( "fg0",  0x20000, 0x20000, 0x5034ae8a )
	ROM_LOAD( "fg1",  0x40000, 0x20000, 0x4fbdba58 )
	ROM_LOAD( "fg2",  0x60000, 0x20000, 0x8891f6f8 )
	ROM_LOAD( "fg3",  0x80000, 0x20000, 0x18e4a130 )

	ROM_REGION(0x20000)  /* Unknown */
	ROM_LOAD( "blasted.snd",  0x00000, 0x08000, 0x00000000 )	/* placeholder */
ROM_END


ROM_START( archrivl_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "3c-rev2",  0x00000, 0x10000, 0x60d4b760 )
	ROM_LOAD_ODD ( "3b-rev2",  0x00000, 0x10000, 0xe0c07a8d )
	ROM_LOAD_EVEN( "2c-rev2",  0x20000, 0x10000, 0xcc2893f7 )
	ROM_LOAD_ODD ( "2b-rev2",  0x20000, 0x10000, 0xfa977050 )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "11d-rev1",  0x00000, 0x10000, 0x7eb3d7c6 )
	ROM_LOAD( "12d-rev1",  0x10000, 0x10000, 0x31e68050 )
	ROM_LOAD( "7j-rev1",   0x20000, 0x20000, 0x148ce28c )
	ROM_LOAD( "8j-rev1",   0x40000, 0x20000, 0x58187ac2 )
	ROM_LOAD( "9j-rev1",   0x60000, 0x20000, 0x0dd1204e )
	ROM_LOAD( "10j-rev1",  0x80000, 0x20000, 0xeb3d0344 )

	ROM_REGION(0x50000)  /* Audio System board */
	ROM_LOAD( "u4.snd",  0x10000, 0x08000, 0x96b3c652 )
	ROM_LOAD( "u19.snd", 0x20000, 0x08000, 0xc4b3dc23 )
	ROM_LOAD( "u20.snd", 0x30000, 0x08000, 0xf7907a02 )
ROM_END


ROM_START( pigskin_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "pigskin.a5",  0x00000, 0x10000, 0xab61c29b )
	ROM_LOAD_ODD ( "pigskin.b5",  0x00000, 0x10000, 0x55a802aa )
	ROM_LOAD_EVEN( "pigskin.a6",  0x20000, 0x10000, 0x4d8b7e50 )
	ROM_LOAD_ODD ( "pigskin.b6",  0x20000, 0x10000, 0x1194f187 )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "pigskin.e2",  0x00000, 0x10000, 0x12d5737b )
	ROM_LOAD( "pigskin.e1",  0x10000, 0x10000, 0x460202a9 )
	ROM_LOAD( "pigskin.h15", 0x20000, 0x20000, 0x2655d03f )
	ROM_LOAD( "pigskin.h17", 0x40000, 0x20000, 0x31c52ea7 )
	ROM_LOAD( "pigskin.h18", 0x60000, 0x20000, 0xb36c4109 )
	ROM_LOAD( "pigskin.h14", 0x80000, 0x20000, 0x09c87104 )

	ROM_REGION(0x50000)  /* Audio System board */
	ROM_LOAD( "pigskin.u4",  0x10000, 0x10000, 0x6daf2d37 )
	ROM_LOAD( "pigskin.u19", 0x20000, 0x10000, 0x56fd16a3 )
	ROM_LOAD( "pigskin.u20", 0x30000, 0x10000, 0x5d032fb8 )
ROM_END


ROM_START( trisport_rom )
	ROM_REGION(0x40000)
	ROM_LOAD_EVEN( "la3.a5",  0x00000, 0x10000, 0xfe1e9e37 )
	ROM_LOAD_ODD ( "la3.b5",  0x00000, 0x10000, 0xf352ec81 )
	ROM_LOAD_EVEN( "la3.a6",  0x20000, 0x10000, 0x9c6a1398 )
	ROM_LOAD_ODD ( "la3.b6",  0x20000, 0x10000, 0x597b564c )

	ROM_REGION_DISPOSE(0xa0000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "la2.e2",  0x00000, 0x10000, 0xf61149a0 )
	ROM_LOAD( "la2.e1",  0x10000, 0x10000, 0xcf753497 )
	ROM_LOAD( "la2.h15", 0x20000, 0x20000, 0x18a44d43 )
	ROM_LOAD( "la2.h17", 0x40000, 0x20000, 0x874cd237 )
	ROM_LOAD( "la2.h18", 0x60000, 0x20000, 0xf7637a18 )
	ROM_LOAD( "la2.h14", 0x80000, 0x20000, 0x403f9401 )

	ROM_REGION(0x50000)  /* Audio System board */
	ROM_LOAD( "sl1-snd.u4",  0x10000, 0x10000, 0x0ed8c904 )
	ROM_LOAD( "sl1-snd.u19", 0x20000, 0x10000, 0xb57d7d7e )
	ROM_LOAD( "sl1-snd.u20", 0x30000, 0x08000, 0x3ae15c08 )
ROM_END



/*************************************
 *
 *	Game drivers
 *
 *************************************/

#define MCR68_DRIVER(name,year,rotate,fullname) \
	struct GameDriver name##_driver =				\
	{												\
		__FILE__,									\
		0,											\
		#name,										\
		fullname,									\
		#year,										\
		"Bally Midway",								\
		"Brian McPhail\nAaron Giles",		  		\
		0,											\
		&name##_machine_driver,						\
		name##_init,								\
													\
		name##_rom,									\
		rom_decode, 0,								\
		0,											\
		0,	/* sound_prom */						\
													\
		name##_input_ports,							\
													\
		0, 0,0,										\
		rotate,										\
													\
		mcr_hiload,mcr_hisave						\
	};

MCR68_DRIVER(zwackery, 1984, ORIENTATION_DEFAULT,    "Zwackery")
MCR68_DRIVER(xenophob, 1987, ORIENTATION_DEFAULT,    "Xenophobe")
MCR68_DRIVER(spyhunt2, 1987, ORIENTATION_DEFAULT,    "Spy Hunter 2")
MCR68_DRIVER(blasted,  1988, ORIENTATION_DEFAULT,    "Blasted")
MCR68_DRIVER(archrivl, 1989, ORIENTATION_DEFAULT,    "Arch Rivals")
MCR68_DRIVER(trisport, 1989, ORIENTATION_ROTATE_270, "Tri-Sports")
MCR68_DRIVER(pigskin,  1990, ORIENTATION_DEFAULT,    "Pigskin 621AD")
